# carmanconfig.py - Reader for global and user files
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements L{CarmanConfig} and L{CarmanOptionParser}.
"""

import os
from optparse import OptionParser, SUPPRESS_USAGE
from ConfigParser import SafeConfigParser, DEFAULTSECT
from common.singleton import Singleton
from common.carlog import INFO, WARNING, ERROR, LOGLEVELS
try:
    import models.isservice
    IMPORT_IS_SERVICE = True
except:
    IMPORT_IS_SERVICE = False

__NAME__ = 'carman-evas'
__VERSION__ = '0.7-beta2-7'
__WEBSITE__ = 'http://openbossa.indt.org/carman'
__DEV_WEBSITE__ = 'http://garage.maemo.org/projects/carman'
__CONFIG_FILE__ = '/usr/share/carman/data/carman.conf'
__HOME_CARMAN_DIR__ = '.carman'
__HOME_CONFIG_DIR__ = '%s' % __HOME_CARMAN_DIR__
__HOME_CACHE_DIR__ = '%s/cache' % __HOME_CARMAN_DIR__
__HOME_PURPLE_DIR__ = '%s/purple' % __HOME_CARMAN_DIR__
__HOME_CONFIG_FILE__ = 'carman.conf'
__USER_PROFILES_FILE__ = "userprofiles"
__DEFAULT_PROFILE_DB_FILENAME__ = '/usr/share/carman/profiles/profiles.db'
__DEFAULT_VIEWER_PLUGINS__ = '/usr/share/carman/plugins/viewer'
__DEFAULT_REPOSITORY_DIR__ = '/usr/share/carman/repositories'
__DEFAULT_REPOSITORY_PLUGINS__ = '/usr/share/carman/plugins/default_map_repo'
__DEFAULT_USER_REPOSITORY_PLUGINS__ = '/home/user/.carman/user_map_repo'
__DEFAULT_CURRENT_THEME__ = '/usr/share/carman/themes/tuning'
__DEFAULT_ROOT_THEME_PATH__ = "/usr/share/carman/themes"
__DEFAULT_REPOSITORY_NAME__ = "default_map_repo/osmrepo"
__DEFAULT_MAX_SPEED__ = 60
__DEFAULT_SPEED_ALERT__ = 'OFF'
__CARMAN_EVAS_SECTION__ = 'carman-evas'
METRIC = 'Metric'
IMPERIAL = 'Imperial'
__UNIT__ = [METRIC, IMPERIAL]


class CarmanConfig(Singleton):
    """
    Abstraction of the global configuration file handling. The
    configuration file can be found at /etc/carman/carman.conf.
    """

    def __init__(self):
        super(CarmanConfig, self).__init__(self)
        self._config = SafeConfigParser()           # COnfiguration file
        self._userconfig = SafeConfigParser()           # COnfiguration file
        self._cb_funcs = []
        self._app_cmd_opts = None
        self._unit_cache = None
        self._bdaddr_cache = None
        self._gpsaddr_cache = None
        self._trip_folder = None
        self._max_speed = None
        self._speed_alert = None
        self.is_service = IMPORT_IS_SERVICE

        self._parse_app_args()

        home_config_file = os.path.join(self.get_user_directory(), __HOME_CONFIG_FILE__)

        base_dir = os.path.dirname(home_config_file)
        if not os.path.exists(base_dir):
            os.makedirs(base_dir)

        files = self._config.read([__CONFIG_FILE__, home_config_file])

        if __CONFIG_FILE__ not in files:
            WARNING('Unable to read %s file.' % __CONFIG_FILE__)
        if home_config_file not in files:
            INFO('Unable to read %s file.' % home_config_file)
#        self._read_userprofiles_file()

    def _parse_app_args(self):
        """
        Parses the command line options.
        """
        optparser = CarmanOptionParser(usage=SUPPRESS_USAGE)
        optparser.add_option('-l', '--enable-log', action='store_true',
                             dest='is_log_enabled')
        optparser.add_option('--log-format', action='store',
                             dest='log_format')
        optparser.add_option('--log-level', action='store',
                             dest='log_level')
        optparser.add_option('--log-path', action='store',
                             dest='log_path')
        optparser.add_option('-w', '--window-mode', action='store_true',
                             dest='is_window')
        self._app_cmd_opts = optparser.parse_args()[0]
        if  self._app_cmd_opts is not None and \
                 self._app_cmd_opts.log_format is not None or\
                 self._app_cmd_opts.log_level is not None or\
                 self._app_cmd_opts.log_path is not None:
            self._app_cmd_opts.is_log_enabled = True

    def _write_home_config_file(self):
        """
        Writes the user home folder configuration file to the disk.
        """
        carman_conf  = os.path.join(self.get_user_directory(),
                                __HOME_CONFIG_FILE__)
        """try:
            os.stat(path)
        except OSError:
            os.mkdir(path, 0755)"""
        path = os.path.split(carman_conf)[0]
        if not os.path.exists(path):
            os.mkdir(path, 0755)

        try:
            filep = open(carman_conf, 'w')
            self._config.write(filep)
            filep.close()
        except IOError, err:
            ERROR('Unable to write %s file: %s' % (carman_conf, err))

    def _write_user_profile_file(self):
        """
        Writes the user profile configuration to the disk.
        """
        filename = os.path.join(self.get_user_directory(),
                                __USER_PROFILES_FILE__)
        path = os.path.split(filename)[0]
        try:
            os.stat(path)
        except OSError:
            os.mkdir(path, 0755)

        try:
            filep = open(filename, 'w')
            self._userconfig.write(filep)
            filep.close()
        except IOError, err:
            ERROR('Unable to write %s file: %s' % (filename, err))

    @staticmethod
    def get_app_name():
        """
        Returns the application name.

        @rtype: string
        @return: Application name.
        """
        return __NAME__

    @staticmethod
    def get_app_version():
        """
        Returns the application version.

        @rtype: string
        @return: Application version.
        """
        return __VERSION__

    @staticmethod
    def get_app_website():
        """
        Returns the application website.

        @rtype: string
        @return: Application website.
        """
        return __WEBSITE__

    @staticmethod
    def get_app_dev_website():
        """
        Returns the application development website.

        @rtype: string
        @return: Development website.
        """
        return __DEV_WEBSITE__

    @staticmethod
    def get_user_directory():
        """
        Returns the carman directory inside user home folder.

        @rtype: string
        @return: Carman diretory.
        """
        return os.path.join(os.path.expanduser('~'), __HOME_CONFIG_DIR__)

    def is_log_enabled(self):
        """
        Verifies if log is enabled.

        @rtype: boolean
        @return: C{True} if log is enabled, C{False} otherwise.
        """
        if self._app_cmd_opts and \
            self._app_cmd_opts.is_log_enabled is not None:
            return True
        elif self._config.has_option(DEFAULTSECT, 'enable-log'):
            logopt = self._config.get(DEFAULTSECT, 'enable-log')
            if 'yes' == logopt.lower() or 'true' == logopt.lower():
                return True
        else:
            return False

    def get_log_level(self):
        """
        Returns the desired log level from the command line or config.

        @rtype: string
        @return: Log level.
        """
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_level is not None and \
                self._app_cmd_opts.log_level.upper() in LOGLEVELS:
            return self._app_cmd_opts.log_level.upper()
        elif self._config.has_option(DEFAULTSECT, 'log-level') and \
                self._config.get(DEFAULTSECT, 'log-level').upper() in LOGLEVELS:
            return self._config.get(DEFAULTSECT, 'log-level').upper()
        else:
            return None

    def get_log_path(self):
        """
        Returns the desired log filename from the command line or config.

        @rtype: string
        @return: Log filename.
        """
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_path is not None:
            return self._app_cmd_opts.log_path
        elif self._config.has_option(DEFAULTSECT, 'log-path'):
            return self._config.get(DEFAULTSECT, 'log-path')
        else:
            return None

    def get_log_format(self):
        """
        Returns the log format from the command line or config.

        @rtype: string
        @return: Log format.
        """
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_format is not None:
            return self._app_cmd_opts.log_format.upper()
        elif self._config.has_option(DEFAULTSECT, 'log-format'):
            return self._config.get(DEFAULTSECT, 'log-format').upper()
        else:
            return None

    def get_profiles_db_file(self):
        """
        Returns the path to the profile database file.

        @rtype: string
        @return: Profile database path.
        """
        if self._config.has_option(DEFAULTSECT, 'profiledb'):
            return self._config.get(DEFAULTSECT, 'profiledb')
        else:
            WARNING('Unable to get profiledb from %s. Using %s'\
                    % (__CONFIG_FILE__, __DEFAULT_PROFILE_DB_FILENAME__))
            return __DEFAULT_PROFILE_DB_FILENAME__
    profiles_db_file = property(get_profiles_db_file)

    def get_profiles_path(self):
        """
        Returns the path to the profiles directory. Every profile must be
        installed under this directory.

        @rtype: string
        @return: Profile directory path.
        """
        prof_db = self.get_profiles_db_file()
        path = os.path.split(prof_db)[0]
        return path
    profiles_path = property(get_profiles_path)

    def get_viewer_plugins_path(self):
        """
        Returns the viewer plugins path.

        @rtype: string
        @return: Viewer plugins path.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'viewer-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'viewer-plugins')
        else:
            WARNING('Unable to get plugins path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_VIEWER_PLUGINS__))
            return __DEFAULT_VIEWER_PLUGINS__
    viewer_plugins_path = property(get_viewer_plugins_path)

    def get_repository_plugins_path(self):
        """
        Returns the repository plugins path.

        @rtype: string
        @return: Repository plugins path.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'repository-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'repository-plugins')
        else:
            WARNING('Unable to get repository path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_REPOSITORY_PL))
            return __DEFAULT_REPOSITORY_PLUGINS__
    repository_plugins_path = property(get_repository_plugins_path)

    def get_user_repository_plugins_path(self):
        """
        Returns the user repository plugins path.

        @rtype: string
        @return: User repository plugins path.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'user-repository-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'user-repository-plugins')
        else:
            WARNING('Unable to get user repository path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_USER_REPOSITORY_PLUGINS__))
            return __DEFAULT_USER_REPOSITORY_PLUGINS__
    repository_plugins_path = property(get_repository_plugins_path)

    def get_repository_default_name(self):
        """
        Returns the repository default name.

        @rtype: string
        @return: Repository default name.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'default-repository'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'default-repository')
        else:
            WARNING('Unable to get repository name from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_REPOSITORY_NAME__))
            return __DEFAULT_REPOSITORY_NAME__

    def set_repository_default_name(self, repository):
        """
        Sets the repository default name.

        @type   repository: string
        @param  repository: Repository default name.
        """
        self._config.set(__CARMAN_EVAS_SECTION__, 'default-repository', repository)
        self._write_home_config_file()
    repository_default_name = property(get_repository_default_name)

    def get_last_fix(self):
        """
        Returns the GPS last fix position.

        @rtype: tuple
        @return: Tuple of C{(float, float, int)}.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'last-fix'):
            last_fix = self._config.get(__CARMAN_EVAS_SECTION__,
                'last-fix').split()
            return (float(last_fix[0]), float(last_fix[1]), int(last_fix[2]))
        else:
            WARNING('Unable to get last fix from %s. Using %s' \
                % (__CONFIG_FILE__, (0, 0, 15)))
            return (0, 0, 15)

    def set_last_fix(self, lat, lon, zoom):
        """
        Updates the GPS last fix position.

        @type   lat: number
        @param  lat: Latitude value.
        @type   lon: number
        @param  lon: Longitude value.
        @type   zoom: number
        @param  zoom: Zoom value.
        """
        self._config.set(__CARMAN_EVAS_SECTION__, 'last-fix', "%f %f %d" % \
            (lat, lon, zoom))
        self._write_home_config_file()
    last_fix = property(get_last_fix, set_last_fix)

    def get_last_trip_fix(self):
        """
        Returns the last trip fix.

        @rtype: tuple
        @return: Tuple of C{(float, float, int)}.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'last-trip-fix'):
            last_fix = self._config.get(__CARMAN_EVAS_SECTION__,
                'last-trip-fix').split()
            return (float(last_fix[0]), float(last_fix[1]), int(last_fix[2]))
        else:
            WARNING('Unable to get last trip fix from %s. Using %s' \
                % (__CONFIG_FILE__, (0, 0, 15)))
            return (0, 0, 15)

    def set_last_trip_fix(self, lat, lon, zoom):
        """
        Sets the last trip fix.

        @type   lat: number
        @param  lat: Latitude value.
        @type   lon: number
        @param  lon: Longitude value.
        @type   zoom: number
        @param  zoom: Zoom value.
        """
        self._config.set(__CARMAN_EVAS_SECTION__, 'last-trip-fix', "%f %f %d" % \
            (lat, lon, zoom))
        self._write_home_config_file()
    last_trip_fix = property(get_last_trip_fix, set_last_trip_fix)

    def get_axisY(self):
        """
        Returns the axis-Y default value.

        @rtype: string
        @return: Axis-Y default value.
        """
        if self._config.has_option(__CARMAN_TRIPS_SECTION__, 'axis_y'):
            return self._config.get(__CARMAN_TRIPS_SECTION__, 'axis_y')
        else:
            WARNING('Unable to get axis-y default value from %s. Using %s' \
                % (__CONFIG_FILE__, "0D"))
            return "0D"

    def set_axisY(self, value):
        """
        Sets the axis-Y value.

        @type   value: number
        @param  value: Axis-Y value.
        """
        self._config.set(__CARMAN_TRIPS_SECTION__, 'axis_y', value)
        self._write_home_config_file()

    def get_theme_root_path(self):
        """
        Returns the theme root path.

        @rtype: string
        @return: Theme root path.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'theme-root'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'theme-root')
        else:
            WARNING('Unable to get theme root path from %s. Using %s' % (__CONFIG_FILE__, __DEFAULT_ROOT_THEME_PATH__))
            return __DEFAULT_ROOT_THEME_PATH__

    def set_theme_path(self, path):
        """
        Sets the current theme path.

        @type   path: string
        @param  path: Theme path.
        """
        if os.path.isdir(path):
            self._config.set(__CARMAN_EVAS_SECTION__, 'current-theme', path)
            self._write_home_config_file()
            return True
        else:
            return False

    def get_current_theme_path(self):
        """
        Returns the current theme path.

        @rtype: string
        @return: Current theme path.
        """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'current-theme'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'current-theme')
        else:
            WARNING('Unable to get current theme from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_CURRENT_THEME__))
            return __DEFAULT_CURRENT_THEME__
    current_theme_path = property(get_current_theme_path)

    def get_current_theme(self):
        """
        Returns current theme path.

        @rtype: string
        @return: Current theme path.
        """
        return os.path.join(self.get_current_theme_path(), "theme.edj")

    def get_carman_home_dir(self):
        """
        Returns the Carman home diretory.

        @rtype: string
        @return: Carman home diretory.
        """
        return os.path.expanduser("~/%s" % __HOME_CARMAN_DIR__)

    def get_carman_purple_dir(self):
        """
        Returns Purple Carman diretory.

        @rtype: string
        @return: Purple Carman diretory.
        """
        return os.path.expanduser("~/%s" % __HOME_PURPLE_DIR__)

    def set_unit(self, unit):
        """
        Sets the unit property. This should be 'Metric' or 'Imperial'.

        @type   unit: string
        @param  unit: 'Metric' or 'Imperial'.
        """
        if unit not in self.get_all_units():
            raise ValueError ('set_unit: unit should be one of %s'\
                               % self.get_all_units())
        self._config.set(DEFAULTSECT, 'unit', unit)
        #self._home_config.set(DEFAULTSECT, 'unit', unit)
        self._write_home_config_file()
        self._unit_cache = unit
        self._call_cb('unit-updated', unit)

    def get_unit(self):
        """
        Returns the unit property value.

        @rtype: string
        @return: Unit value.
        """
        if self._unit_cache is None:
            if self._config.has_option(DEFAULTSECT, 'unit'):
                self._unit_cache = self._config.get(DEFAULTSECT, 'unit')
            else:
                self._unit_cache = METRIC
            return self._unit_cache
        else:
            return self._unit_cache
    unit = property(get_unit, set_unit)

    def set_max_speed(self, speed):
        """
        Sets the max speed value.

        @type   speed: number
        @param  speed: Max speed value.
        """
        self._config.set(DEFAULTSECT, 'speed', str(speed))
        self._write_home_config_file()
        self._max_speed = speed
        #self._call_cb('speed-updated', speed)

    def get_max_speed(self):
        """
        Returns the max speed value.

        @rtype: number
        @return: Max speed value.
        """
        if self._max_speed is None:
            if self._config.has_option(DEFAULTSECT, 'speed'):
                self._max_speed = self._config.get(DEFAULTSECT, 'speed')
            else:
                self._max_speed = __DEFAULT_MAX_SPEED__

        return int(self._max_speed)

    max_speed = property(get_max_speed, set_max_speed)

    def set_speed_alert(self, activate):
        """
        Sets the speed alert state.

        @type   activate: string
        @param  activate: Speed alert state, C{ON} or C{OFF}.
        """
        self._config.set(DEFAULTSECT, 'speed-alert', activate)
        self._write_home_config_file()
        self._speed_alert = activate

    def get_speed_alert(self):
        """
        Returns the speed alert state.

        @rtype: string
        @return: C{ON} if speed alert is enable, C{OFF} otherwise.
        """
        if self._speed_alert is None:
            if self._config.has_option(DEFAULTSECT, 'speed-alert'):
                self._speed_alert = self._config.get(DEFAULTSECT, 'speed-alert')
            else:
                self._speed_alert = __DEFAULT_SPEED_ALERT__

        return self._speed_alert

    speed_alert = property(get_speed_alert, set_speed_alert)

    @staticmethod
    def get_all_units():
        """
        Returns all the possible unit values.

        @rtype: list
        @return: Unit values list.
        """
        return __UNIT__

    def _call_cb(self, message, value):
        """
        Called all the registered callable function. Used when any configuration
        has changed. The purpose of this funcion is to emulate the gobject
        signal without loading the object to the carman-sdl.

        @type   message: string
        @param  message: A string indicating what has changed.
        @type   value: string
        @param  value: Value to be changed.
        """
        for cb_func in self._cb_funcs:
            try:
                cb_func(message, value)
            except Exception, err:
                WARNING('Got an error in the callback function: %s' % err)


class CarmanOptionParser(OptionParser):
    """
    Carman Parser for command line options.
    """

    def error(self, msg):
        pass

    def _process_args(self, largs, rargs, values):
        """
        Processes arguments from the command.

        @type   largs: list
        @param  largs: The list of leftover arguments.
        @type   rargs: list
        @param  rargs: The argument list currently being parsed.
        @type   values: dictionary
        @param  values: Dictionary of C{{option1:value1, option2:value2, ...}}
        """
        while rargs:
            arg = rargs[0]
            # We handle bare "--" explicitly, and bare "-" is handled by the
            # standard arg handler since the short arg case ensures that the
            # len of the opt string is greater than 1.
            if arg == "--":
                del rargs[0]
                return
            elif arg[0:2] == "--":
                # process a single long option (possibly with value(s))
                try:
                    self._process_long_opt(rargs, values)
                except ValueError, err:
                    print "Command line error: '%s'" % err
            elif arg[:1] == "-" and len(arg) > 1:
                # process a cluster of short options (possibly with
                # value(s) for the last one only)
                try:
                    self._process_short_opts(rargs, values)
                except ValueError, err:
                    print "Command line error: '%s'" % err
            elif self.allow_interspersed_args:
                largs.append(arg)
                del rargs[0]
            else:
                return
