#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import dbus
from common.carlog import DEBUG
from models.btmodel import BtModel
from common.singleton import Singleton
from main.pairingdlgctrl import PairingDlgController
from models.dbusmodel import CarmandDbusModel, DBUS_BUS_NAME, \
    DBUS_OBJECT_PATH, DBUS_IFACE_GPS

class GPSModel(Singleton, dbus.Interface):

    DISCONNECTED = "Disconnected"
    CONNECTING   = "Connecting"
    FIXING       = "Fixing"
    FIXED        = "Fixed"

    def __init__(self):
        """ GPS Model constructor """
        Singleton.__init__(self)
        carmandbusmodel = CarmandDbusModel()
        dbus.Interface.__init__(self, carmandbusmodel.get_bus_object(),
                        DBUS_IFACE_GPS)

        self._last_data = None
        self._status_changed_cbs = []
        self._data_available_cbs = []
        self._connection_lost_cbs = []
        self._disconnecting = False
        self._add_carmand_listeners(carmandbusmodel.get_bus())
        self.pairingctrl = None
    # __init__

    def _on_status_changed(self, status):
        """ Called when the carmand change the GPS status connection """
        DEBUG("GPS status changed to %s" % status)
        for cb in self._status_changed_cbs:
            cb(status)
        if status == self.DISCONNECTED:
            if not self._disconnecting:
                for cb in self._connection_lost_cbs:
                    cb()
            self._disconnecting = False
            if self.pairingctrl:
                self.pairingctrl.hide()
            self.pairingctrl = None
        elif status == self.FIXING or status == self.FIXED:
            self.pairingctrl = None
    # _on_fixed

    def _on_data_available(self, mode, *data):
        """
        Called when the carmand have information about a location to report.
        """
        self._last_data = data
        for cb in self._data_available_cbs:
            cb(*data)
    # _on_data_available

    def _add_carmand_listeners(self, bus):
        """ Add interesting listeners """
        bus.add_signal_receiver(self._on_status_changed,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_GPS, signal_name = 'StatusChanged')

        bus.add_signal_receiver(self._on_data_available,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_GPS, signal_name = 'DataAvailable')

    def add_status_changed_cb(self, cb):
        if callable(cb) and cb not in self._status_changed_cbs:
            self._status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        if cb in self._status_changed_cbs:
            self._status_changed_cbs.remove(cb)

    def add_connection_lost_cb(self, cb):
        if callable(cb) and cb not in self._connection_lost_cbs:
            self._connection_lost_cbs.append(cb)

    def del_connection_lost_cb(self, cb):
        if cb in self._connection_lost_cbs:
            self._connection_lost_cbs.remove(cb)

    def add_data_available_cb(self, cb):
        if callable(cb) and cb not in self._data_available_cbs:
            self._data_available_cbs.append(cb)

    def del_data_available_cb(self, cb):
        if cb in self._data_available_cbs:
            self._data_available_cbs.remove(cb)

    def get_last_data(self):
        return self._last_data

    def connect(self, address):
        bt_model = BtModel()
        if address != "none" and not bt_model.has_bonding(address):
            self.pairingctrl = PairingDlgController()
            bt_model.passkey_register(address, self.pairingctrl.show)
        self.Connect()

    def disconnect(self):
        self._disconnecting = True
        self.Disconnect()
