# connmodel - Starts the connection
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import dbus
import common.ossohelper as osso
from common.singleton import Singleton
from common.carlog import DEBUG, WARNING
try:
    import conic
except ImportError:
    conic = None
    WARNING("conic module not found - are you running on a desktop?")

DBUS_ADDRESS     = "unix:path=/var/run/dbus/system_bus_socket"
DBUS_BUS_NAME    = "com.nokia.icd"
DBUS_OBJECT_PATH = "/com/nokia/icd"
DBUS_IFACE       = "com.nokia.icd"

class ConnectionModel(Singleton):

    CONNECTING    = "CONNECTING"
    CONNECTED     = "CONNECTED"
    DISCONNECTING = "DISCONNECTING"
    IDLE          = "IDLE"

    def __init__(self):
        Singleton.__init__(self)
        self._conn = None
        self._status_changed_cbs = []
        self._connection_lost_cbs = []

        if osso.get_device_state() != osso.DEVICE_STATE_NORMAL:
            self._last_status = self.DISCONNECTING
        else:
            self._last_status = self.CONNECTED

        if conic:
            self._start()
    # __init__

    def _event_cb(self, *args):
        """ """
        DEBUG("Event conic callback %s" % args)
    # _event_cb

    def _changed_status(self, name, infra, status, aux):
        """ """
        DEBUG("Changed status %s" % status)
        self._last_status = status
        for cb in self._status_changed_cbs:
            cb(status)
        if status == self.DISCONNECTING:
            for cb in self._connection_lost_cbs:
                cb()
    # changed_status

    def _start(self):
        self._bus = dbus.bus.BusConnection(DBUS_ADDRESS)
        self._conn = conic.Connection()
        self._bus.add_signal_receiver(self._changed_status, "status_changed",
            DBUS_IFACE, DBUS_BUS_NAME, DBUS_OBJECT_PATH)
        self._conn.connect("connection-event", self._event_cb)
        self._conn.set_property("automatic-connection-events", True)
        self._conn.request_connection(conic.CONNECT_FLAG_NONE)
    # start

    def add_status_changed_cb(self, cb):
        if callable(cb) and cb not in self._status_changed_cbs:
            self._status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        if cb in self._status_changed_cbs:
            self._status_changed_cbs.remove(cb)

    def add_connection_lost_cb(self, cb):
        if callable(cb) and cb not in self._connection_lost_cbs:
            self._connection_lost_cbs.append(cb)

    def del_connection_lost_cb(self, cb):
        if cb in self._connection_lost_cbs:
            self._connection_lost_cbs.remove(cb)

    def try_connect(self):
        if self._conn:
            self._conn.request_connection(conic.CONNECT_FLAG_NONE)

    def Status(self):
        return self._last_status
# ConnectionStatus
