#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import ecore
from models.obdmodel import OBDModel
from models.mapmodel import MapModel
from common.carmanconfig import CarmanConfig, METRIC
from maps.infosharingview import BuddyView, BuddyInformationView
from models.dbusmodel import CarmandDbusModel
from models.infosharingmodel import ISManagerModel, ISBuddyModel, \
    PLUGIN_BUS_NAME
from main.infosharingagent import InfoSharingAgent, AGENT_OBJECT_PATH

class InfoSharingController(object):

    def __init__(self, controller, canvas):
        self.zoom = None
        self.timer = None
        self.canvas = canvas
        self.__move_cb = None
        self.last_data = None
        self.buddy_model = {}
        self.last_friend_xy = None
        self.sensors_handle = None
        self.show_msg_disconnecting = True
        self.__buddy_status_changed_cb = None

        self.obd_model = OBDModel()
        self.map_model = MapModel()

        self.agent = InfoSharingAgent()
        self.manager_model = ISManagerModel()
        self.manager_model.add_buddy_created_cb(self.__buddy_created_cb)
        self.manager_model.add_buddy_removed_cb(self.__buddy_removed_cb)
        if self.manager_model.connect():
            self.manager_model.register_agent(AGENT_OBJECT_PATH)

        CarmandDbusModel().add_name_owner_changed_cb(
            self.__name_owner_changed)

        metric = CarmanConfig().get_unit() == METRIC
        self.buddy_view = BuddyView(controller, canvas)
        self.buddy_information = BuddyInformationView(canvas, metric)

    def __name_owner_changed(self, *data):
        if PLUGIN_BUS_NAME == data[0] and data[2]:
            if self.manager_model.connect():
                self.manager_model.register_agent(AGENT_OBJECT_PATH)

    def __buddy_data_available(self, lat, lon, alt, speed, track, rpm):
        self.buddy_information.set_speed_value(speed)
        self.buddy_information.set_rpm_value(rpm)

        self.last_data = (lat, lon)
        x, y = self.map_model.latlon_to_xy(lat, lon)
        if self.zoom is not None:
            self.last_friend_xy = x >> self.zoom, y >> self.zoom
            self.buddy_view.activate()
            self.buddy_view.set_buddy_position(*self.last_friend_xy)
            if self.__move_cb:
                self.__move_cb(*self.last_friend_xy)

    def get_last_data(self):
        return self.last_data

    def __hide_buddy_information(self):
        self.buddy_information.hide()

    def __show_msg_error(self, error):
        self.buddy_information.show_message(error)
        if self.timer:
            self.timer.delete()
        self.timer = ecore.timer_add(5, self.__hide_buddy_information)

    def __sensor_data(self, model, pid, *data):
        pass

    def __buddy_created_cb(self, path, buddy_model):
        buddy_model.add_data_available_cb(self.__buddy_data_available)

        if self.__buddy_status_changed_cb:
            self.__buddy_status_changed_cb(True)

        self.buddy_information.set_speed_value(-1)
        self.buddy_information.set_rpm_value(-1)
        if not self.buddy_information.is_visible():
            properties = buddy_model.GetInfo()
            self.buddy_information.set_buddy_name(properties['Name'])
            self.buddy_information.show_data_values()
            self.buddy_information.show()

        if not self.sensors_handle:
            self.sensors_handle = self.obd_model.add_data_available_cb(
                (("0C", 1, 0), ), self.__sensor_data)

    def __buddy_removed_cb(self, path, buddy_model):
        if buddy_model:
            buddy_model.del_data_available_cb(self.__buddy_data_available)
        if not self.manager_model.has_buddies():
            if self.__buddy_status_changed_cb:
                self.__buddy_status_changed_cb(False)
            if self.sensors_handle:
                self.obd_model.del_data_available_cb(self.sensors_handle)
                self.sensors_handle = None

        if self.show_msg_disconnecting:
            self.buddy_view.deactivate()
            self.__show_msg_error("Disconnecting")
        self.show_msg_disconnecting = True

    def set_move_cb(self, cb):
        if callable(cb):
            self.__move_cb = cb
            if self.last_friend_xy:
                cb(self.last_friend_xy[0], self.last_friend_xy[1], False)
        else:
            self.__move_cb = None

    def get_lock(self):
         return callable(self.__move_cb) 
	
    def set_buddy_status_changed_cb(self, cb):
        if callable(cb):
            self.__buddy_status_changed_cb = cb

    def __authorization_ok(self):
        self.buddy_information.show_data_values()

    def __authorization_failed(self, error):
        self.__show_msg_error(error.message)

    def request_authorization(self, name, email):
        self.buddy_information.set_buddy_name(name)
        self.buddy_information.show_message("Waiting authorization")
        self.buddy_information.show()

        self.manager_model.request_authorization(email,
            self.__authorization_ok, self.__authorization_failed)

    def disconnect_buddy(self, path, show_msg):
        self.show_msg_disconnecting = show_msg
        self.manager_model.remove_buddy(path)

    def set_unit_system(self, value):
        self.buddy_information.set_unit_system(value)

    def set_view(self, theme, view):
        self.buddy_view.set_theme(theme)
        self.buddy_view.set_buddy_area(view.part_geometry_get("area_view"))
        view.part_swallow("buddy_view", self.buddy_view)
        self.buddy_information.set_view(theme, view)
        if self.manager_model.has_buddies():
            if self.__buddy_status_changed_cb:
                self.__buddy_status_changed_cb(True)

    def set_fullscreen(self, fullscreen):
        self.buddy_view.set_fullscreen(fullscreen)

    def get_buddy_position(self):
        return self.buddy_view.get_buddy_position()

    def set_position(self, x, y, zoom):
        self.buddy_view.set_position(x, y)
        if self.zoom != zoom:
            self.zoom = zoom
            if self.last_data:
                x, y = self.map_model.latlon_to_xy(self.last_data[0],
                    self.last_data[1])
                self.last_friend_xy = x >> zoom, y >> zoom
                self.buddy_view.set_buddy_position(*self.last_friend_xy)

    def finalize(self):
        self.manager_model.remove_all_buddies()
        self.manager_model.unregister_agent(AGENT_OBJECT_PATH)
