# tile_downloader.py - Downloads tiles
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#


#### EXAMPLE

# import tile_downloader
# import select
# import sys
# ### Some dummy test code

# def hello(x, y, z):
#     """ """
#     print "Completed:", x, y, z
#     return True
# # hello

# def bogus(x, y, z, errmsg):
#     """ """
#     print "Error:", errmsg, x, y, z
#     return True
# # bogus

# bla = tile_downloader.TileDownloader()

# bla.set_plugins_dir('/home/work/carman/branches/prototypes_0.7/gps/maps/plugins/')

# for i in bla.get_repos():
#     print i.NAME

# bla.choose_repo(0)

# bla.set_cache_dir('~/.gcache/')

# fd = bla.initialize(hello, bogus)

# fd_poll = select.poll()

# fd_poll.register(fd, select.POLLIN | select.POLLPRI)

# for i in range(10, 12):
#     for j in range(20, 40):
#         for z in range(15, 17):
#             bla.add_download(i, j, z, True)
# try:
#     while 1:
#         fd_poll.poll()
#         bla.process_results()
# except KeyboardInterrupt:
#     bla.close()

# sys.exit(0)

import os
from common.singleton import Singleton
from models.reposet import RepositoriesSet
from models.download_mng import DownloadManager

class TileDownloader(Singleton):
    """
    High level class to download tiles.
    """
    def __init__(self):
        """ """
        self._mgr = None
        self._list_repos = []
        self._plug_dir = None
        self._repo = None
        self._cache_dir = None
        self._complete_cb = None
        self._errcb = None
    # __init__

    def list_repos(self):
        """Return a list of the avaliable repositories."""
        return self._list_repos
    # list_repos

    def choose_repo(self, name):
        """
        Chooses and initializes an repository specified by an index in
        the repository list.
        """
        reposet = RepositoriesSet()
        self._repo = reposet.load_repository(name)
        return self._repo
    # choose_repo

    def get_plugins_dir(self):
        """ """
        return self._plug_dir
    # get_plugins_ditr

    def initialize(self):
        """
        Sets up the callbacks of the tile_downloader

        @return file descriptor to notify of events
        """
        self._mgr = DownloadManager()

        return self._mgr.initialize(self._wrapper_cb, self._wrapper_cb)
    # initialize

    def _wrapper_cb(self, filename, data, errmsg=None):
        """ """
        x, y, z, callback_list = data

        if errmsg:
            for cb in callback_list:
                if not callable(cb):
                    continue
                cb(None, x, y, z)
            return

        for cb in callback_list:
            if not callable(cb):
                continue
            cb(filename, x, y, z)
    # _wrapper_complete_cb

    def get_tile_filename(self, tile_x, tile_y, zoom):
        """ """
        return os.path.join(self.get_cache_dir(), str(zoom), str(tile_y),
            str(tile_x) + self._repo.get_extension())
    # get_tile_filename

    def request_tile(self, tile_x, tile_y, zoom, foreground=True, callback=None):
        """
        Request some tile to the download manager.
        """
        if self._cache_dir is None:
            raise ValueError("The cache dir must exist.")

        filename = self.get_tile_filename(tile_x, tile_y, zoom)

        if os.path.exists(filename):
            return filename

        data = (tile_x, tile_y, zoom, [callback])

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.add_download(url, filename, data, foreground)
    # request_tile

    def process_results(self):
        """
        All the callbacks are called here.
        """
        self._mgr.process_results()
    # process_results

    def change_state(self, tile_x, tile_y, zoom, foreground):
        """ """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.change_state(url, foreground)
    # change_priority

    def has_fg_downloads(self):
        return self._mgr.has_fg_downloads()

    def has_bg_downloads(self):
        return self._mgr.has_bg_downloads()

    def has_downloads(self):
        return self._mgr.has_downloads()

    def get_cache_dir(self):
        """ """
        return self._cache_dir
    # get_cache_dir

    def set_cache_dir(self, cache_dir):
        """ """
        path = os.path.join(cache_dir, self._repo.get_cache_name())

        if not os.path.exists(path):
            os.makedirs(path)

        self._cache_dir = path
    # set_cache_dir

    def stop_all_downloads(self):
        """ """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        self._mgr.stop_all_downloads()
    # stop_all_downloads

    def clear_fg_queue(self):
        self._mgr.clear_fg_queue()

    def clear_bg_queue(self):
        self._mgr.clear_bg_queue()

    def clear_queue(self):
        """ """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        self._mgr.clear_queues()
    # clear_queue

    def cancel_download(self, tile_x, tile_y, zoom):
        """ """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.cancel_download(url)
    # cancel_download

    def close(self):
        """ """
        self._mgr.close()
        self._mgr.join()
    # close
# TileDownloader
