#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import evas

class GaugeController(object):

    def __init__(self, screen, sensor, metric):
        """
        Gauge Controller constructor
        @param screen: Edje screen object
        @param sensor: sensor object
        @param metric: True for metric or False for imperial
        """
        self.screen = screen
        self.sensor = sensor
        self.metric = metric
        self.__last_value = None
        self.unit_part_name = ""
        self.scale = [1, 1]
        self.offset = 0
        self.frequency = 1
        self.pid = sensor.get_pid()
        self.min = [sensor.get_min(False), sensor.get_min(True)]
        self.max = [sensor.get_max(False), sensor.get_max(True)]

    def destroy(self):
        if isinstance(self.view, evas.SmartObject):
            self.screen.part_unswallow(self.view)
            self.view.delete()

    def set_title(self, part_name):
        """
        Set object title
        @param part_name: object part name
        """
        if part_name:
            title = self.sensor.get_short_description()
            if title:
                self.screen.part_text_set(part_name, title.encode("utf-8"))

    def __set_unit_text(self):
        """ Set unit text """
        if self.unit_part_name:
            unit = self.sensor.get_unit(self.metric)
            if unit:
                if self.scale[self.metric] > 1:
                    unit += " x%d" % self.scale[self.metric]
                self.screen.part_text_set(self.unit_part_name,
                    unit.encode("utf-8"))

    def set_unit(self, part_name):
        """
        Set object unit
        @param part_name: object part name
        """
        self.unit_part_name = part_name
        self.__set_unit_text()

    @staticmethod
    def __conv_to_int(value, default):
        """
        Convert string to integer
        @param value: value to be converted
        @param default: default value if error converting value
        """
        try:
            return int(value)
        except:
            WARNING("Error converting value \"%s\"" % value)
            return default

    def set_min(self, metric, imperial):
        """
        Set minimum values for metric and imperial units
        @param metric: metric value
        @param imperial: imperial value
        """
        if metric:
            self.min[True] = self.__conv_to_int(metric, self.min[True])
            if self.min[True] == self.max[True]:
                self.min[True] -= 1
        if imperial:
            self.min[False] = self.__conv_to_int(imperial, self.min[False])
            if self.min[False] == self.max[False]:
                self.min[False] -= 1

    def set_max(self, metric, imperial):
        """
        Set maximum values for metric and imperial units
        @param metric: metric value
        @param imperial: imperial value
        """
        if metric:
            self.max[True] = self.__conv_to_int(metric, self.max[True])
            if self.max[True] == self.min[True]:
                self.max[True] += 1
        if imperial:
            self.max[False] = self.__conv_to_int(imperial, self.max[False])
            if self.max[False] == self.min[False]:
                self.max[False] += 1

    def set_scale(self, metric, imperial):
        """
        Set scale values for metric and imperial units
        @param metric: metric value
        @param imperial: imperial value
        """
        if metric:
            self.scale[True] = self.__conv_to_int(metric, self.scale[True])
            if self.scale[True] == 0:
                self.scale[True] = 1
        if imperial:
            self.scale[False] = self.__conv_to_int(imperial,
                self.scale[False])
            if self.scale[False] == 0:
                self.scale[False] = 1

    def set_frequency(self, value):
        """
        Set OBD command frequency
        @param value: command frequency
        """
        if value:
            self.frequency = self.__conv_to_int(value, self.frequency)

    def set_offset(self, value):
        """
        Set OBD command offset
        @param value: command offset
        """
        if value:
            self.offset = self.__conv_to_int(value, self.offset)

    def get_obd_command(self):
        """ Return OBD command """
        return (self.pid[:2], self.frequency, self.offset)

    def set_unit_system(self, metric):
        """
        Set unit system value
        @param metric: True for metric and False for imperial
        """
        self.metric = metric
        self.__set_unit_text()
        self.view.set_unit_system(self.min[metric], self.max[metric],
            self.scale[metric], self.sensor.get_unit(metric))

    def set_view(self, part_name, view):
        """
        Set object widget
        @param part_name: object part name
        @param obj: object widget
        """
        self.view = view
        self.set_unit_system(self.metric)
        if isinstance(view, evas.SmartObject):
            self.screen.part_swallow(part_name, view)

    def set_value(self, data):
        """
        Set current value
        @param value: value absolute
        """
        if data:
            index = not self.metric and 2 or 0
            if ":" in self.pid:
                index += int(self.pid.split(":")[-1]) - 1
            value = data[index]

            if value != self.__last_value:
                value = max(value, self.min[self.metric])
                value = min(value, self.max[self.metric])
                value_rel = float(value - self.min[self.metric]) / \
                    (self.max[self.metric] - self.min[self.metric])
                self.view.set_value(value, value_rel)
                self.__last_value = value
