#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import math, time, ecore
from main import evasutil
from common.carlog import DEBUG
from gauges.gaugeview import GaugeView

__ANIMATOR_TIMER__ = 0.25

class AnalogueView(GaugeView):

    def __init__(self, canvas, image=None, radius=105, initial_angle=225,
      total_angle=270, clock_wise="yes", pieces=9, font=None):
        """
        Gauge Analogue viewer constructor
        @param canvas: Evas canvas object
        @param image: pointer image key
        @param radius: radius of the marks
        @param initial_angle: initial angle
        @param total_angle: total angle
        @param clock_wise: move orientation
        @param pieces: number of the marks
        @param font: font preset name
        """
        GaugeView.__init__(self, canvas)
        self.marks = []
        self.value = None
        self.last_value = None
        self.current_value = 0
        self.animator = None
        self.dx = self.dy = 0
        self.radius = radius
        self.pieces = pieces >= 2 and pieces or 2
        self.initial_angle = initial_angle
        self.total_angle = total_angle
        self.clock_wise = clock_wise.lower() == "yes"
        if font is None:
            raise ValueError("Font not specified")
        for i in range(pieces):
            text = self.Text(**font)
            text.clip_set(self.clipper)
            text.show()
            self.marks.append(text)
        self.new_pointer = self.Image()
        self.new_pointer.alpha_set(True)
        self.new_pointer.clip_set(self.clipper)
        self.new_pointer.show()
        if image is None:
            raise ValueError("Image not specified")
        self.pointer = self.Image(file=image)
        self.__rotate_pointer(0)

    def __del__(self):
        DEBUG("deleting analogue gauge %s" % self)

    def __move_marks(self):
        """ Set marks position """
        for i in range(self.pieces):
            degree = i * self.total_angle / (self.pieces - 1)
            if self.clock_wise:
                degree = self.initial_angle - degree
            else:
                degree = self.initial_angle + degree
            degree = degree * math.pi / 180
            w, h = self.marks[i].size_get()
            self.marks[i].move(self.dx + int(round(self.radius *
                math.cos(degree))) - w / 2, self.dy - int(round(self.radius *
                math.sin(degree))) - h / 2)

    def move(self, x, y):
        """ Set pointer and marks position """
        if self.dx != x or self.dy != y:
            self.dx = x
            self.dy = y
            w, h = self.new_pointer.image_size_get()
            self.new_pointer.move(self.dx - w / 2, self.dy - h / 2)
            self.__move_marks()

    def __rotate_pointer(self, value):
        """
        Rotate and centralize pointer image
        @param value: relative value from 0.0 to 1.0
        """
        angle = value * self.total_angle
        if self.clock_wise:
            angle = self.initial_angle - angle
        else:
            angle = self.initial_angle + angle

        evasutil.rotate(self.new_pointer, self.pointer, angle)
        w, h = self.new_pointer.image_size_get()
        self.new_pointer.move(self.dx - w / 2, self.dy - h / 2)

    def __refresh_pointer_cb(self):
        """ Animate pointer rotation """
        time_elapsed = (time.time() - self.start_time) / __ANIMATOR_TIMER__
        self.current_value = self.last_value + (self.value - \
            self.last_value) * (time_elapsed > 1 and 1 or time_elapsed)
        self.__rotate_pointer(self.current_value)

        return time_elapsed < 1

    def set_value(self, value_abs, value_rel):
        """
        Change current value
        @param value_abs: absolute value (not used)
        @param value_rel: relative value from 0.0 to 1.0
        """
        if value_rel < 0:
            value_rel = 0
        elif value_rel > 1:
            value_rel = 1
        if value_rel != self.value:
            self.value = value_rel
            self.last_value = self.current_value
            if self.animator:
                self.animator.delete()
            self.start_time = time.time()
            self.animator = ecore.animator_add(self.__refresh_pointer_cb)

    def set_unit_system(self, min, max, scale, unit):
        """
        Change unit system values
        @param min: integer mininum value
        @param max: integer maximum value
        @param scale: integer scale value
        @param unit: string unit value
        """
        for i in range(self.pieces):
            mark = (min + i * (max - min) / (self.pieces - 1)) / scale
            self.marks[i].text_set(str(mark))
        self.__move_marks()

