# carmanconfig.py - Reader for global and user files
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os, sys
from optparse import OptionParser, SUPPRESS_USAGE
from ConfigParser import SafeConfigParser, DEFAULTSECT
from common.singleton import Singleton
from common.carlog import INFO, WARNING, ERROR, LOGLEVELS

__CONFIG_FILE__ = '/usr/share/carman/data/carman.conf'
__HOME_CARMAN_DIR__ = '.carman'
__HOME_CONFIG_DIR__ = '%s' % __HOME_CARMAN_DIR__
__HOME_CACHE_DIR__ = '%s/cache' % __HOME_CARMAN_DIR__
__HOME_CONFIG_FILE__ = 'carman.conf'
__USER_PROFILES_FILE__ = "userprofiles"
__DEFAULT_PROFILE_DB_FILENAME__ = '/usr/share/carman/profiles/profiles.db'
__DEFAULT_VIEWER_PLUGINS__ = '/usr/share/carman/plugins/viewer'
__DEFAULT_REPOSITORY_DIR__ = '/usr/share/carman/repositories'
__DEFAULT_REPOSITORY_PLUGINS__ = '/usr/share/carman/plugins/default_map_repo'
__DEFAULT_USER_REPOSITORY_PLUGINS__ = '/home/user/MyDocs/Carman/.plugins/user_map_repo'
__DEFAULT_CURRENT_THEME__ = '/usr/share/carman/themes/tuning'
__DEFAULT_ROOT_THEME_PATH__ = "/usr/share/carman/themes"
__DEFAULT_REPOSITORY_NAME__ = "default_map_repo/osmrepo"
__DEFAULT_MAX_SPEED__ = 60
__DEFAULT_SPEED_ALERT__ = 'OFF'
__CARMAN_EVAS_SECTION__ = 'carman-evas'
METRIC = 'Metric'
IMPERIAL = 'Imperial'
__UNIT__ = [METRIC, IMPERIAL]


class CarmanConfig(Singleton):
    '''
    This class abstract the global configuration file reading. The
    configuration file can be found at /etc/carman/carman.conf.
    '''

    def __init__(self):
        '''
        GlobalConfig constructor.
        '''
        super(CarmanConfig, self).__init__(self)
        self._config = SafeConfigParser()           # COnfiguration file
        self._userconfig = SafeConfigParser()           # COnfiguration file
        self._cb_funcs = []
        self._app_cmd_opts = None
        self._unit_cache = None
        self._bdaddr_cache = None
        self._gpsaddr_cache = None
        self._trip_folder = None
        self._max_speed = None
        self._speed_alert = None

        self._parse_app_args()

        home_config_file = os.path.join(self.get_user_directory(), __HOME_CONFIG_FILE__)

        base_dir = os.path.dirname(home_config_file)
        if not os.path.exists(base_dir):
            os.makedirs(base_dir)

        files = self._config.read([__CONFIG_FILE__, home_config_file])

        if __CONFIG_FILE__ not in files:
            WARNING('Unable to read %s file.' % __CONFIG_FILE__)
        if home_config_file not in files:
            INFO('Unable to read %s file.' % home_config_file)

#        self._read_userprofiles_file()

    # __init__

    def _parse_app_args(self):
        """
        Parse command line options
        """
        optparser = CarmanOptionParser(usage=SUPPRESS_USAGE)
        optparser.add_option('-l', '--enable-log', action='store_true',
                             dest='is_log_enabled')
        optparser.add_option('--log-format', action='store',
                             dest='log_format')
        optparser.add_option('--log-level', action='store',
                             dest='log_level')
        optparser.add_option('--log-path', action='store',
                             dest='log_path')
        optparser.add_option('-w', '--window-mode', action='store_true',
                             dest='is_window')
        self._app_cmd_opts = optparser.parse_args()[0]
        if  self._app_cmd_opts is not None and \
                 self._app_cmd_opts.log_format is not None or\
                 self._app_cmd_opts.log_level is not None or\
                 self._app_cmd_opts.log_path is not None:
            self._app_cmd_opts.is_log_enabled = True
    # _parse_command_args

    def _write_home_config_file(self):
        '''
        Write the current user configuration to the disk.
        '''
        carman_conf  = os.path.join(self.get_user_directory(),
                                __HOME_CONFIG_FILE__)
        """try:
            os.stat(path)
        except OSError:
            os.mkdir(path, 0755)"""
        path = os.path.split(carman_conf)[0]
        if not os.path.exists(path):
            os.mkdir(path, 0755)

        try:
            filep = open(carman_conf, 'w')
            self._config.write(filep)
            filep.close()
        except IOError, err:
            ERROR('Unable to write %s file: %s' % (carman_conf, err))
    # _write_config

    def _write_user_profile_file(self):
        '''
        Write the current user configuration to the disk.
        '''
        filename = os.path.join(self.get_user_directory(),
                                __USER_PROFILES_FILE__)
        path = os.path.split(filename)[0]
        try:
            os.stat(path)
        except OSError:
            os.mkdir(path, 0755)

        try:
            filep = open(filename, 'w')
            self._userconfig.write(filep)
            filep.close()
        except IOError, err:
            ERROR('Unable to write %s file: %s' % (filename, err))
    # _write_config

    @staticmethod
    def get_app_name():
        """
        Return the application name
        """
        return os.path.basename(sys.argv[0])
    # get_application_name

    @staticmethod
    def get_user_directory():
        """
        Return the carman directory under the user home dir
        """
        return os.path.join(os.path.expanduser('~'), __HOME_CONFIG_DIR__)
    # get_user_directory

    def is_log_enabled(self):
        """
        Return if we should enable the log module.
        """
        if self._app_cmd_opts and \
            self._app_cmd_opts.is_log_enabled is not None:
            return True
        elif self._config.has_option(DEFAULTSECT, 'enable-log'):
            logopt = self._config.get(DEFAULTSECT, 'enable-log')
            if 'yes' == logopt.lower() or 'true' == logopt.lower():
                return True
        else:
            return False
    # is_log_enabled

    def get_log_level(self):
        """
        Return the desired log level from the command line or config.
        """
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_level is not None and \
                self._app_cmd_opts.log_level.upper() in LOGLEVELS:
            return self._app_cmd_opts.log_level.upper()
        elif self._config.has_option(DEFAULTSECT, 'log-level') and \
                self._config.get(DEFAULTSECT, 'log-level').upper() in LOGLEVELS:
            return self._config.get(DEFAULTSECT, 'log-level').upper()
        else:
            return None
    # get_log_level

    def get_log_path(self):
        """__logformat__
        Return the desired log filename from the command line or config.
        """
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_path is not None:
            return self._app_cmd_opts.log_path
        elif self._config.has_option(DEFAULTSECT, 'log-path'):
            return self._config.get(DEFAULTSECT, 'log-path')
        else:
            return None
    # get_log_path

    def get_log_format(self):
        if self._app_cmd_opts and \
                self._app_cmd_opts.log_format is not None:
            return self._app_cmd_opts.log_format.upper()
        elif self._config.has_option(DEFAULTSECT, 'log-format'):
            return self._config.get(DEFAULTSECT, 'log-format').upper()
        else:
            return None
    # get_log_format

    def get_profiles_db_file(self):
        '''
        Return the path to the profile database file.
        '''
        if self._config.has_option(DEFAULTSECT, 'profiledb'):
            return self._config.get(DEFAULTSECT, 'profiledb')
        else:
            WARNING('Unable to get profiledb from %s. Using %s'\
                    % (__CONFIG_FILE__, __DEFAULT_PROFILE_DB_FILENAME__))
            return __DEFAULT_PROFILE_DB_FILENAME__
    # get_profiles_db_file
    profiles_db_file = property(get_profiles_db_file)

    def get_profiles_path(self):
        '''
        Return the path to the profiles directory. Every profile must be
        installed under this directory.
        '''
        prof_db = self.get_profiles_db_file()
        path = os.path.split(prof_db)[0]
        return path
    # get_profiles_path
    profiles_path = property(get_profiles_path)

    def get_viewer_plugins_path(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'viewer-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'viewer-plugins')
        else:
            WARNING('Unable to get plugins path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_VIEWER_PLUGINS__))
            return __DEFAULT_VIEWER_PLUGINS__
    # get_viewer_plugins_path
    viewer_plugins_path = property(get_viewer_plugins_path)

    def get_repository_plugins_path(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'repository-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'repository-plugins')
        else:
            WARNING('Unable to get repository path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_REPOSITORY_PLUGINS__))
            return __DEFAULT_REPOSITORY_PLUGINS__
    # get_repository_plugins_path
    repository_plugins_path = property(get_repository_plugins_path)

    def get_user_repository_plugins_path(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'user-repository-plugins'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'user-repository-plugins')
        else:
            WARNING('Unable to get user repository path from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_USER_REPOSITORY_PLUGINS__))
            return __DEFAULT_USER_REPOSITORY_PLUGINS__
    # get_user_repository_plugins_path
    repository_plugins_path = property(get_repository_plugins_path)

    def get_repository_default_name(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'default-repository'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'default-repository')
        else:
            WARNING('Unable to get repository name from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_REPOSITORY_NAME__))
            return __DEFAULT_REPOSITORY_NAME__
    # get_repository_default_name

    def set_repository_default_name(self, repository):
        self._config.set(__CARMAN_EVAS_SECTION__, 'default-repository', repository)
        self._write_home_config_file()
    # set_repository_default_name

    repository_default_name = property(get_repository_default_name)

    def get_last_fix(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'last-fix'):
            last_fix = self._config.get(__CARMAN_EVAS_SECTION__,
                'last-fix').split()
            return (float(last_fix[0]), float(last_fix[1]), int(last_fix[2]))
        else:
            WARNING('Unable to get last fix from %s. Using %s' \
                % (__CONFIG_FILE__, (0, 0, 15)))
            return (0, 0, 15)
    # get_last_fix

    def set_last_fix(self, lat, lon, zoom):
        self._config.set(__CARMAN_EVAS_SECTION__, 'last-fix', "%f %f %d" % \
            (lat, lon, zoom))
        self._write_home_config_file()
    # set_last_fix
    last_fix = property(get_last_fix, set_last_fix)

    def get_last_trip_fix(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'last-trip-fix'):
            last_fix = self._config.get(__CARMAN_EVAS_SECTION__,
                'last-trip-fix').split()
            return (float(last_fix[0]), float(last_fix[1]), int(last_fix[2]))
        else:
            WARNING('Unable to get last trip fix from %s. Using %s' \
                % (__CONFIG_FILE__, (0, 0, 15)))
            return (0, 0, 15)
    # get_last_fix

    def set_last_trip_fix(self, lat, lon, zoom):
        self._config.set(__CARMAN_EVAS_SECTION__, 'last-trip-fix', "%f %f %d" % \
            (lat, lon, zoom))
        self._write_home_config_file()
    # set_last_fix
    last_trip_fix = property(get_last_trip_fix, set_last_trip_fix)

    def get_axisY(self):
        """ """
        if self._config.has_option(__CARMAN_TRIPS_SECTION__, 'axis_y'):
            return self._config.get(__CARMAN_TRIPS_SECTION__, 'axis_y')
        else:
            WARNING('Unable to get axis-y default value from %s. Using %s' \
                % (__CONFIG_FILE__, "0D"))
            return "0D"
    # get_axisY

    def set_axisY(self, value):
        self._config.set(__CARMAN_TRIPS_SECTION__, 'axis_y', value)
        self._write_home_config_file()
    # set_axisY

    def get_theme_root_path(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'theme-root'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'theme-root')
        else:
            WARNING('Unable to get theme root path from %s. Using %s' % (__CONFIG_FILE__, __DEFAULT_ROOT_THEME_PATH__))
            return __DEFAULT_ROOT_THEME_PATH__
    # get_theme_root_path

    def set_theme_path(self, path):
        """ """
        if os.path.isdir(path):
            self._config.set(__CARMAN_EVAS_SECTION__, 'current-theme', path)
            self._write_home_config_file()
            return True
        else:
            return False
    # set_theme_root_path

    def get_current_theme_path(self):
        """ """
        if self._config.has_option(__CARMAN_EVAS_SECTION__, 'current-theme'):
            return self._config.get(__CARMAN_EVAS_SECTION__, 'current-theme')
        else:
            WARNING('Unable to get current theme from %s. Using %s' \
                 % (__CONFIG_FILE__, __DEFAULT_CURRENT_THEME__))
            return __DEFAULT_CURRENT_THEME__
    # get_current_theme_path
    current_theme_path = property(get_current_theme_path)

    def get_current_theme(self):
        return os.path.join(self.get_current_theme_path(), "theme.edj")

    def get_carman_home_dir (self):
        return os.path.expanduser("~/%s" % __HOME_CARMAN_DIR__)

    def set_unit(self, unit):
        '''
        Set the unit property. This should be 'Metric' or 'Imperial'.
        @param unit: 'Metric' or 'Imperial'.
        '''
        if unit not in self.get_all_units():
            raise ValueError ('set_unit: unit should be one of %s'\
                               % self.get_all_units())
        self._config.set(DEFAULTSECT, 'unit', unit)
        #self._home_config.set(DEFAULTSECT, 'unit', unit)
        self._write_home_config_file()
        self._unit_cache = unit
        self._call_cb('unit-updated', unit)
    # set_unit

    def get_unit(self):
        '''
        Get the unit property.
        '''
        if self._unit_cache is None:
            if self._config.has_option(DEFAULTSECT, 'unit'):
                self._unit_cache = self._config.get(DEFAULTSECT, 'unit')
            else:
                self._unit_cache = METRIC
            return self._unit_cache
        else:
            return self._unit_cache
    # get_unit
    unit = property(get_unit, set_unit)

    def set_max_speed(self, speed):
        self._config.set(DEFAULTSECT, 'speed', str(speed))
        self._write_home_config_file()
        self._max_speed = speed
        #self._call_cb('speed-updated', speed)

    def get_max_speed(self):
        if self._max_speed is None:
            if self._config.has_option(DEFAULTSECT, 'speed'):
                self._max_speed = self._config.get(DEFAULTSECT, 'speed')
            else:
                self._max_speed = __DEFAULT_MAX_SPEED__

        return int(self._max_speed)

    max_speed = property(get_max_speed, set_max_speed)

    def set_speed_alert(self, activate):
        self._config.set(DEFAULTSECT, 'speed-alert', activate)
        self._write_home_config_file()
        self._speed_alert = activate

    def get_speed_alert(self):
        if self._speed_alert is None:
            if self._config.has_option(DEFAULTSECT, 'speed-alert'):
                self._speed_alert = self._config.get(DEFAULTSECT, 'speed-alert')
            else:
                self._speed_alert = __DEFAULT_SPEED_ALERT__

        return self._speed_alert

    speed_alert = property(get_speed_alert, set_speed_alert)

    @staticmethod
    def get_all_units():
        '''
        List all the possible unit values.
        '''
        return __UNIT__
    # get_all_units

    def _call_cb(self, message, value):
        '''
        Call all the registered callable function. Used when any configuration
        has changed. The purpose of this funcion is to emulate the gobject
        signal without loading the object to the carman-sdl.
        @param message: a string indicating what has changed.
        '''
        for cb_func in self._cb_funcs:
            try:
                cb_func(message, value)
            except Exception, err:
                WARNING('Got an error in the callback function: %s' % err)
    # _call_cb
# CarmanConfig


class CarmanOptionParser(OptionParser):

    def error(self, msg):
        pass
    # error

    def _process_args(self, largs, rargs, values):
        while rargs:
            arg = rargs[0]
            # We handle bare "--" explicitly, and bare "-" is handled by the
            # standard arg handler since the short arg case ensures that the
            # len of the opt string is greater than 1.
            if arg == "--":
                del rargs[0]
                return
            elif arg[0:2] == "--":
                # process a single long option (possibly with value(s))
                try:
                    self._process_long_opt(rargs, values)
                except ValueError, err:
                    print "Command line error: '%s'" % err
            elif arg[:1] == "-" and len(arg) > 1:
                # process a cluster of short options (possibly with
                # value(s) for the last one only)
                try:
                    self._process_short_opts(rargs, values)
                except ValueError, err:
                    print "Command line error: '%s'" % err
            elif self.allow_interspersed_args:
                largs.append(arg)
                del rargs[0]
            else:
                return
        # _process_args

# CarmanOptionParser
