#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import evas, ecore
from main import evasutil

__MIN_SIZE__ = 15
__SHOW_DELAY__ = 1
__THICKNESS__ = 3
__BORDER__ = 50

class Track(evas.SmartObject):

    def __init__(self, canvas, clipper, view, color=(255, 0, 0)):
        evas.SmartObject.__init__(self, canvas)
        self.polys = []
        self.points = []
        self.prev = None
        self.view = view
        self.horizontal = None
        self.clipper = clipper
        self.track_color = color

    def clear(self):
        for poly in self.polys:
            self.member_del(poly)
            poly.delete()

        self.polys = []
        self.points = []
        self.prev = None
        self.horizontal = None
        self.clipper.hide()

    def move(self, dx, dy):
        i = 0
        for poly in self.polys:
            poly.points_clear()
            for point in self.points[i]:
                poly.point_add(point[0] + dx, point[1] + dy)
            i += 1

    def draw_poly(self, point, px, py):
        if not self.prev:
            self.prev = point
            self.horizontal = None
            return

        dx = abs(point[0] - self.prev[0])
        dy = abs(point[1] - self.prev[1])
        if dx >= __MIN_SIZE__ or dy >= __MIN_SIZE__:
            if self.horizontal is None:
                self.clipper.show()
            if dx > dy:
                if self.horizontal != True:
                    self.horizontal = True
                    self.last_points = ((self.prev[0], self.prev[1] +
                        __THICKNESS__), (self.prev[0], self.prev[1] -
                        __THICKNESS__), (point[0], point[1] - __THICKNESS__),
                        (point[0], point[1] + __THICKNESS__))
                else:
                    self.last_points = (self.last_points[3],
                        self.last_points[2], (point[0], point[1] -
                        __THICKNESS__), (point[0], point[1] + __THICKNESS__))
            else:
                if self.horizontal != False:
                    self.horizontal = False
                    self.last_points = ((self.prev[0] - __THICKNESS__,
                        self.prev[1]), (self.prev[0] + __THICKNESS__,
                        self.prev[1]), (point[0] + __THICKNESS__, point[1]),
                        (point[0] - __THICKNESS__, point[1]))
                else:
                    self.last_points = (self.last_points[3],
                        self.last_points[2], (point[0] + __THICKNESS__,
                        point[1]), (point[0] - __THICKNESS__, point[1]))

            poly = self.Polygon(color=self.track_color,
                points=[(x + px, y + py) for x, y in self.last_points])
            poly.clip_set(self.clipper)
            poly.show()
            self.polys.append(poly)
            self.points.append(self.last_points)
            self.prev = point

    def update(self, track, px, py):
        self.clear()
        for points in track:
            self.prev = points[0]
            self.horizontal = None
            for point in points[1:]:
                self.draw_poly(point, px, py)

        if self.prev and self.view.is_point_out(self.prev):
            self.prev = None

    def is_empty(self):
        return len(self.polys) == 0

class TrackView(evas.SmartObject):

    def __init__(self, canvas, controller, gps=True, trip=False):
        evas.SmartObject.__init__(self, canvas)
        self.angle = None
        self.posx = self.posy = 0
        self.gpsx = self.gpsy = 0
        self.tripx = self.tripy = 0
        self.centerx = self.centery = 0
        self.pointer_w = self.pointer_h = 0
        self.w = self.h = 0
        self.zoom = None
        self.timer = None
        self.controller = controller
        self.clipper = self.Rectangle()
        self.clipper_gps = self.Rectangle()
        self.track = Track(canvas, self.clipper, self)
        self.track.pass_events = True
        self.member_add(self.track)

        if trip:
            self.trip_pointer = self.Image()
            self.trip_pointer.alpha_set(True)
            self.trip_pointer.show()
            self.trip_pointer.clip_set(self.clipper_gps)
            self.clipper_gps.color = (255, 255, 255, 0)
        else:
            self.trip_pointer = None

        if gps:
            self.pointer = self.Image()
            self.new_pointer = self.Image()
            self.new_pointer.alpha_set(True)
            self.new_pointer.clip_set(self.clipper_gps)
            self.new_pointer.show()
        else:
            self.new_pointer = None

        if not trip and not gps:
            self.clipper_gps.color = (255, 255, 255, 0)

    def set_theme(self, theme, color):
        self.track.track_color = color

        if self.new_pointer:
            self.pointer.file_set(theme, "images/gps-position")
        if self.trip_pointer:
            self.trip_pointer.file_set(theme, "images/trip-pin")
            w, h = self.trip_pointer.image_size_get()
            self.trip_pointer.size_set(w, h)
            self.trip_pointer.fill_set(0, 0, w, h)
            self.trip_pointer_w, self.trip_pointer_h = w / 2, h / 2
            self.trip_pointer.show()

    def __update_track(self):
        points = self.controller.get_track(self.centerx - self.w / 2 -
            __BORDER__, self.centery - self.h / 2 - __BORDER__,
            self.centerx + self.w / 2 + __BORDER__, self.centery +
            self.h / 2 + __BORDER__)
        self.track.update(points, self.posx + self.w / 2 - self.centerx,
            self.posy + self.h / 2 - self.centery)

    def __move_pointer(self):
        if self.new_pointer:
            x = self.w / 2 - self.centerx + self.gpsx
            y = self.h / 2 - self.centery + self.gpsy
            self.new_pointer.move(self.posx + x - self.pointer_w,
                self.posy + y - self.pointer_h)
        if self.trip_pointer:
            dx = self.w / 2 - self.centerx + self.tripx
            dy = self.h / 2 - self.centery + self.tripy
            self.trip_pointer.move(self.posx + dx - self.trip_pointer_w,
                self.posy + dy - self.trip_pointer_h)

    def new_track(self):
        self.track.prev = None

    def get_gps_position(self):
        return self.gpsx, self.gpsy

    def set_gps_position(self, x, y, angle):
        if not self.new_pointer:
            return

        self.gpsx = x
        self.gpsy = y
        if not self.is_point_out((x, y)):
            self.track.draw_poly((x, y), self.posx + self.w / 2 - \
                self.centerx, self.posy + self.h / 2 - self.centery)

        if self.angle != angle:
            self.angle = angle
            evasutil.rotate(self.new_pointer, self.pointer, -angle)
            w, h = self.new_pointer.image_size_get()
            self.pointer_w, self.pointer_h = w / 2, h / 2
        self.__move_pointer()

    def set_trip_position(self, x, y):
        if self.trip_pointer:
            if self.clipper_gps.color[3] == 0:
                self.clipper_gps.color = (255, 255, 255, 255)
            self.tripx = x
            self.tripy = y
            self.__move_pointer()

    def set_position(self, x, y, zoom):
        self.centerx = x
        self.centery = y
        if self.zoom != zoom:
            self.zoom = zoom
            self.track.clear()
        else:
            self.__move_pointer()
            if not self.track.is_empty():
                self.track.move(self.posx + self.w / 2 - x, self.posy + \
                    self.h / 2 - y)
        if self.timer:
            self.timer.delete()
        self.timer = ecore.timer_add(__SHOW_DELAY__, self.__update_track)

    def is_point_out(self, point):
        px = point[0] - self.centerx + self.w / 2
        py = point[1] - self.centery + self.h / 2
        return px < 0 or px > self.w or py < 0 or py > self.h

    def clear(self):
        self.track.clear()

    def clip_set(self, obj):
        self.clipper.clip_set(obj)
        self.clipper_gps.clip_set(obj)

    def clip_unset(self):
        self.clipper.clip_unset()
        self.clipper_gps.clip_unset()

    def show(self):
        if not self.track.is_empty():
            self.clipper.show()
        self.clipper_gps.show()

    def hide(self):
        self.clipper.hide()
        self.clipper_gps.hide()

    def move(self, x, y):
        self.posx = x
        self.posy = y
        evas.SmartObject.move(self, x, y)

    def resize(self, w, h):
        self.w = w
        self.h = h
        self.clipper.resize(w, h)
        self.clipper_gps.resize(w, h)
