/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carman-pythnon.
 *
 *  carman-python is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carman-python is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <Python.h>
#include <Evas.h>
#include <evas/evas.c_evas.h>
#include <carmand-trip.h>

#define TOTAL_TIME (3600 * 10)
#define COMPATIBLE_TRIP_VERSION 1

enum {
	FIELD_ALTITUDE,
	FIELD_RPM,
	FIELD_SPEED,
	FIELD_COUNT
};

static Evas_Object *_img = NULL;
static Evas_Colorspace _colorspace;
static struct trip_data *_trip = NULL;
static time_t _start_time = 0, _finish_time = 0, _total_time = 0;
static unsigned int _color32 = 0, _color16 = 0;
static double _sum_alt, _min_alt, _max_alt, _cnt_alt = 0;
static int _sum_rpm, _min_rpm, _max_rpm, _cnt_rpm = 0;
static int _sum_speed, _min_speed, _max_speed, _cnt_speed = 0;
static int _min_scale = 0, _max_scale = 100;
static int _current_field = FIELD_SPEED, _field_mask = TRIP_SPEED;
static int _width_real, _width_view, _height_real, _height_view, _pitch;
static int _trip_size = 0, _trip_buffer_size = 0, _precision = 1;
static int _last_x = -1, _last_y = -1, _line_width = 1, _offset = 0;
static void (*__draw_line)(void *, int, int, int, int, int) = NULL;

static PyObject *_init(PyObject *self, PyObject *args);
static PyObject *_load_file(PyObject *self, PyObject *args);
static PyObject *_set_theme(PyObject *self, PyObject *args);
static PyObject *_set_graph(PyObject *self, PyObject *args);
static PyObject *_update_graph(PyObject *self);
static PyObject *_add_data(PyObject *self, PyObject *args);
static PyObject *_get_altitude_stats(PyObject *self);
static PyObject *_get_rpm_stats(PyObject *self);
static PyObject *_get_speed_stats(PyObject *self);
static PyObject *_get_latlon(PyObject *self, PyObject *args);
static PyObject *_get_track(PyObject *self);
static PyObject *_get_period(PyObject *self);
static PyObject *_get_all_speed_samples(PyObject *self);
static PyObject *_get_all_rpm_samples(PyObject *self);
static PyObject *_get_trip_distance(PyObject *self);

static PyMethodDef trip_methods[] = {
	{ "init", _init, METH_VARARGS,
		"init(image) - Initialize graph module" },
	{ "load_file", _load_file, METH_VARARGS,
		"load_file(filename) - Load a trip file" },
	{ "set_theme", _set_theme, METH_VARARGS,
		"set_theme(color, line_width, height) - "
		"Define theme attributes" },
	{ "set_graph", _set_graph, METH_VARARGS,
		"set_graph(field, width, (min_scale, max_scale), zoom) - "
		"Define current graph attributes" },
	{ "update_graph", (PyCFunction)_update_graph, METH_NOARGS,
		"Update all graph with current attributes" },
	{ "add_data", _add_data, METH_VARARGS,
		"add_data(lat, lon, alt, rpm, speed) - Add new data to trip" },
	{ "get_altitude_stats", (PyCFunction)_get_altitude_stats, METH_NOARGS,
		"get_altitude_stats() - Return altitude statistics (min, max, avg)" },
	{ "get_rpm_stats", (PyCFunction)_get_rpm_stats, METH_NOARGS,
		"get_rpm_stats() - Return RPM statistics (min, max, avg)" },
	{ "get_speed_stats", (PyCFunction)_get_speed_stats, METH_NOARGS,
		"get_speed_stats() - Return speed statistics (min, max, avg)" },
	{ "get_latlon", _get_latlon, METH_VARARGS,
		"get_latlon(time) - Return latitude and longitude" },
	{ "get_track", (PyCFunction)_get_track, METH_NOARGS,
		"get_track() - Return the complete track from trip" },
	{ "get_period", (PyCFunction)_get_period, METH_NOARGS,
		"get_period() - Return start, finish and total times" },
	{ "get_all_speed_samples", (PyCFunction)_get_all_speed_samples, METH_NOARGS,
		"get_all_speed_samples() - Return all speed samples" },
	{ "get_all_rpm_samples", (PyCFunction)_get_all_rpm_samples, METH_NOARGS,
		"get_all_rpm_samples() - Return all rpm samples" },
	{ "get_trip_distance", (PyCFunction)_get_trip_distance, METH_NOARGS,
		"get_trip_distance() - Return trip distance" },
	{ NULL, NULL, 0, NULL }
};

DL_EXPORT(void)
inittrip(void)
{
	Py_InitModule ("trip", trip_methods);
	if (PyErr_Occurred ())
		Py_FatalError ("can't initialise module trip");
}

static void _draw_line_32(void *data, int pitch, int signx,
							int deltax, int deltay, int pixelxy)
{
	int i, tmp, x = 0, y = 0;
	unsigned int *pixel, *t_pixel;

	pixel = (unsigned int *)data + pixelxy;

	for (; x < deltax; x++, pixel += signx) {
		for (i = 0; i < _line_width; i++) {
			tmp = i - _line_width / 2;
			t_pixel = pixel + pitch * tmp;
			*t_pixel = _color32;
		}

		y += deltay;
		if (y >= deltax) {
			y -= deltax;
			pixel += pitch;
		}
	}
}

static void _draw_line_16(void *data, int pitch, int signx,
							int deltax, int deltay, int pixelxy)
{
	int i, tmp, x = 0, y = 0;
	unsigned char *pixel_a, *pixel_atmp;
	unsigned short *pixel, *pixel_tmp;

	pixel = (unsigned short *)data + pixelxy;
	pixel_a = data + _pitch * _height_real * 2 + pixelxy;

	for (; x < deltax; x++, pixel += signx, pixel_a += signx) {
		for (i = 0; i < _line_width; i++) {
			tmp = i - _line_width / 2;
			pixel_tmp = pixel + pitch * tmp;
			pixel_atmp = pixel_a + pitch * tmp;
			*pixel_tmp = _color16;
			*pixel_atmp = 0x1F;
		}

		y += deltay;
		if (y >= deltax) {
			y -= deltax;
			pixel += pitch;
			pixel_a += pitch;
		}
	}
}

static void _draw_line(void *data, int x1, int y1, int x2, int y2)
{
	int deltax, deltay, signx, signy, pitch, pixelxy, swaptmp;

	if (y1 < _offset)
		y1 = _offset;
	else if (y1 > _height_view + _offset - 1)
		y1 = _height_view + _offset - 1;
	if (y2 < _offset)
		y2 = _offset;
	else if (y2 > _height_view + _offset - 1)
		y2 = _height_view + _offset - 1;

	deltax = x2 - x1;
	deltay = y2 - y1;
	signx = (deltax < 0) ? -1 : 1;
	signy = (deltay < 0) ? -1 : 1;
	deltax = signx * deltax + 1;
	deltay = signy * deltay + 1;

	pixelxy = x1 + _pitch * y1;
	pitch = _pitch * signy;
	if (deltax < deltay) {
		swaptmp = deltax; deltax = deltay; deltay = swaptmp;
		swaptmp = signx; signx = pitch; pitch = swaptmp;
	}

	__draw_line(data, pitch, signx, deltax, deltay, pixelxy);
}

static PyObject *_init(PyObject *self, PyObject *args)
{
	PyObject *obj;

	if (!PyArg_ParseTuple(args, "O", &obj))
		return NULL;

	_img = ((struct PyEvasObject *)obj)->obj;

	_colorspace = evas_object_image_colorspace_get(_img);
	if (_colorspace == EVAS_COLORSPACE_RGB565_A5P)
		__draw_line = _draw_line_16;
	else if (_colorspace == EVAS_COLORSPACE_ARGB8888)
		__draw_line = _draw_line_32;

	Py_RETURN_NONE;
}

static PyObject *_load_file(PyObject *self, PyObject *args)
{
	int fd, i;
	ssize_t len;
	char *filename;
	struct stat stat;
	struct trip_header header;

	if (!PyArg_ParseTuple(args, "s", &filename))
		return NULL;

	fd = open(filename, O_RDONLY);
	if (fd == -1) {
		PyErr_SetString(PyExc_IOError, "could not open file");
		return NULL;
	}

	len = read(fd, &header, sizeof(struct trip_header));

	if (len < sizeof(struct trip_header) ||
		strncmp(header.file_header, "CMAN", 4)) {

		PyErr_SetString(PyExc_IOError, "trip file is not valid");
		close(fd);
		return NULL;
	}

	if (header.file_version > COMPATIBLE_TRIP_VERSION) {
		PyErr_SetString(PyExc_IOError, "trip file is generated by "
				"a newer version of Carmand. Format is not "
				"recognized.");
		close(fd);
		return NULL;
	}

	fstat(fd, &stat);
	_trip_size = (stat.st_size - len) / sizeof(struct trip_data);
	_trip_buffer_size = (_trip_size / 100 + 1) * 100;
	_precision = header.precision;
	if (_precision <= 0)
		_precision = 10;
	_start_time = header.start_time;
	_finish_time = header.end_time;
	_total_time = _trip_size * _precision;

	if (_trip)
		free(_trip);
	_trip = (struct trip_data *) calloc(sizeof(struct trip_data),
		_trip_buffer_size);

	// FIXME: Check if read with success
	read(fd, _trip, sizeof(struct trip_data) * _trip_size);
	close(fd);

	_cnt_alt = _cnt_rpm = _cnt_speed = 0;
	_sum_alt = _sum_rpm = _sum_speed = 0;
	_min_alt = _min_rpm = _min_speed = 1000000000;
	_max_alt = _max_rpm = _max_speed = -1000000000;

	for (i = 0; i < _trip_size; i++) {
		if (_trip[i].flag & TRIP_ALTITUDE) {
			_cnt_alt++;
			_sum_alt += _trip[i].alt;
			if (_trip[i].alt < _min_alt)
				_min_alt = _trip[i].alt;
			if (_trip[i].alt > _max_alt)
				_max_alt = _trip[i].alt;
		}

		if (_trip[i].flag & TRIP_RPM) {
			_cnt_rpm++;
			_sum_rpm += _trip[i].rpm;
			if (_trip[i].rpm < _min_rpm)
				_min_rpm = _trip[i].rpm;
			if (_trip[i].rpm > _max_rpm)
				_max_rpm = _trip[i].rpm;
		}

		if (_trip[i].flag & TRIP_SPEED) {
			_cnt_speed++;
			_sum_speed += _trip[i].speed;
			if (_trip[i].speed < _min_speed)
				_min_speed = _trip[i].speed;
			if (_trip[i].speed > _max_speed)
				_max_speed = _trip[i].speed;
		}
	}

	Py_RETURN_NONE;
}

static PyObject *_set_theme(PyObject *self, PyObject *args)
{
	int r, g, b;

	if (!PyArg_ParseTuple(args, "(iii)ii", &r, &g, &b, &_line_width,
							&_height_view))
		return NULL;

	_offset = _line_width / 2;
	_width_real = _width_view + 2 * _offset;
	_height_real = _height_view + 2 * _offset;
	_color16 = ((r & 0xF8) << 8) | ((g & 0xFC) << 3) |
				((b & 0xF8) >> 3);
	_color32 = 0xFF000000 | ((r & 0xFF) << 16) | ((g & 0xFF) << 8) |
				(b & 0xFF);

	Py_RETURN_NONE;
}

static PyObject *_set_graph(PyObject *self, PyObject *args)
{
	int width, field, min, max, zoom;

	if (!PyArg_ParseTuple(args, "ii(ii)i", &field, &width, &min, &max,
							&zoom))
		return NULL;

	if (field < FIELD_ALTITUDE || field > FIELD_COUNT) {
		PyErr_SetString(PyExc_ValueError, "field invalid");
		return NULL;
	}
	if (min >= max) {
		PyErr_SetString(PyExc_ValueError, "max scale is not bigger than min "
			"scale");
		return NULL;
	}
	if (zoom < 0 || zoom > 5) {
		PyErr_SetString(PyExc_ValueError, "zoom invalid");
		return NULL;
	}

	_min_scale = min;
	_max_scale = max;
	_current_field = field;
	if (field == FIELD_ALTITUDE)
		_field_mask = TRIP_ALTITUDE;
	else if (field == FIELD_RPM)
		_field_mask = TRIP_RPM;
	else
		_field_mask = TRIP_SPEED;
	if (zoom == 5)
		_width_view = width;
	else
		_width_view = width * 20.0 / (1 << zoom);
	_width_real = _width_view + 2 * _offset;

	Py_RETURN_NONE;
}

static inline void _draw_next_line(void *data, int pos)
{
	int x, y;

	if (_trip[pos].flag & _field_mask) {
		x = pos * _precision * _width_view / TOTAL_TIME + _offset;
		if (_current_field == FIELD_ALTITUDE)
			y = _height_view - (_trip[pos].alt - _min_scale) * _height_view /
				(_max_scale - _min_scale) + _offset - 1;
		else if (_current_field == FIELD_RPM)
			y = _height_view - (_trip[pos].rpm - _min_scale) * _height_view /
				(_max_scale - _min_scale) + _offset - 1;
		else
			y = _height_view - (_trip[pos].speed - _min_scale) *
				_height_view / (_max_scale - _min_scale) + _offset - 1;
		if (_last_x != -1)
			_draw_line(data, _last_x, _last_y, x, y);
		_last_x = x;
		_last_y = y;
	} else
		_last_x = -1;
}

static PyObject *_update_graph(PyObject *self)
{
	void *data;
	int i, last_width, last_height;

	if (!_img) {
		PyErr_SetString(PyExc_ValueError, "trip not initialized");
		return NULL;
	}

	evas_object_image_size_get(_img, &last_width, &last_height);
	if (last_width != _width_real || last_height != _height_real) {
		evas_object_image_size_set(_img, _width_real, _height_real);
		evas_object_resize(_img, _width_view, _height_view);
		evas_object_image_fill_set(_img, -_offset, -_offset, _width_real,
			_height_real);
		_pitch = evas_object_image_stride_get(_img);
	}

	data = evas_object_image_data_get(_img, 1);
	if (_colorspace == EVAS_COLORSPACE_RGB565_A5P)
		memset(data, 0, _width_real * _height_real * 3);
	else if (_colorspace == EVAS_COLORSPACE_ARGB8888)
		memset(data, 0, _width_real * _height_real * 4);

	_last_x = _last_y = -1;
	for (i = 0; i < _trip_size; i++)
		_draw_next_line(data, i);

	evas_object_image_data_set(_img, data);
	evas_object_image_data_update_add(_img, 0, 0, _width_real, _height_real);

	Py_RETURN_NONE;
}

static PyObject *_add_data(PyObject *self, PyObject *args)
{
	void *data;
	struct trip_data *trip;
	int rpm = 0, speed = 0, flag = 0;
	double lat = 0.0, lon = 0.0, alt = 0.0, dist = 0.0;
	PyObject *py_lat, *py_lon, *py_alt, *py_rpm, *py_speed, *py_dist;

	if (!PyArg_ParseTuple(args, "OOOOOO", &py_lat, &py_lon, &py_alt, &py_rpm,
							&py_speed, &py_dist))
		return NULL;

	if (_total_time + _precision > TOTAL_TIME) {
		PyErr_SetString(PyExc_ValueError, "trip is full");
		return NULL;
	}
	_total_time += _precision;

	if (PyFloat_Check(py_lat) && PyFloat_Check(py_lon)) {
		flag |= TRIP_LAT_LON;
		lat = PyFloat_AS_DOUBLE(py_lat);
		lon = PyFloat_AS_DOUBLE(py_lon);
	} else if (py_lat != Py_None || py_lon != Py_None) {
		PyErr_SetString(PyExc_TypeError,
			"a float or None are required to latitude and longitude");
		return NULL;
	}

	if (PyFloat_Check(py_alt)) {
		alt = PyFloat_AS_DOUBLE(py_alt);
		if (alt == alt) {
			flag |= TRIP_ALTITUDE;
			_cnt_alt++;
			_sum_alt += alt;
			if (alt < _min_alt)
				_min_alt = alt;
			if (alt > _max_alt)
				_max_alt = alt;
		}
	} else if (py_alt != Py_None) {
		PyErr_SetString(PyExc_TypeError,
			"a float or None are required to altitude");
		return NULL;
	}

	if (PyInt_Check(py_rpm) || PyLong_Check(py_rpm)) {
		flag |= TRIP_RPM;
		rpm = PyInt_AS_LONG(py_rpm);
		_cnt_rpm++;
		_sum_rpm += rpm;
		if (rpm < _min_rpm)
			_min_rpm = rpm;
		if (rpm > _max_rpm)
			_max_rpm = rpm;
	} else if (py_rpm != Py_None) {
		PyErr_SetString(PyExc_TypeError,
			"an integer or None are required to RPM");
		return NULL;
	}

	if (PyInt_Check(py_speed) || PyLong_Check(py_speed)) {
		flag |= TRIP_SPEED;
		speed = PyInt_AS_LONG(py_speed);
		_cnt_speed++;
		_sum_speed += speed;
		if (speed < _min_speed)
			_min_speed = speed;
		if (speed > _max_speed)
			_max_speed = speed;
	} else if (py_speed != Py_None) {
		PyErr_SetString(PyExc_TypeError,
			"an integer or None are required to speed");
		return NULL;
	}

	if (PyFloat_Check(py_dist)) {
		flag |= TRIP_DISTANCE;
		dist = PyFloat_AS_DOUBLE(py_dist);
	} else if (py_dist != Py_None) {
		PyErr_SetString(PyExc_TypeError,
			"a float or None are required to distance");
		return NULL;
	}


	_trip_size++;
	if (_trip_size > _trip_buffer_size) {
		_trip_buffer_size = (_trip_size / 100 + 1) * 100;
		_trip = (struct trip_data *) realloc(_trip, _trip_buffer_size *
				sizeof(struct trip_data));
	}

	trip = &_trip[_trip_size - 1];
	trip->flag = flag;
	trip->lat = lat;
	trip->lon = lon;
	trip->alt = alt;
	trip->distance = dist;
	trip->rpm = rpm;
	trip->speed = speed;

	data = evas_object_image_data_get(_img, 1);
	_draw_next_line(data, _trip_size - 1);
	evas_object_image_data_set(_img, data);
	// FIXME: Don't update all image
	evas_object_image_data_update_add(_img, 0, 0, _width_real, _height_real);

	Py_RETURN_NONE;
}

static PyObject *_get_altitude_stats(PyObject *self)
{
	if (!_cnt_alt)
		Py_RETURN_NONE;

	return Py_BuildValue("ddd", _min_alt, _max_alt, _sum_alt / _cnt_alt);
}

static PyObject *_get_rpm_stats(PyObject *self)
{
	if (!_cnt_rpm)
		Py_RETURN_NONE;

	return Py_BuildValue("iii", _min_rpm, _max_rpm, _sum_rpm / _cnt_rpm);
}

static PyObject *_get_speed_stats(PyObject *self)
{
	if (!_cnt_speed)
		Py_RETURN_NONE;

	return Py_BuildValue("iii", _min_speed, _max_speed, _sum_speed /
		_cnt_speed);
}

static PyObject *_get_latlon(PyObject *self, PyObject *args)
{
	double lat, lon;
	int time, index, mod;

	if (!PyArg_ParseTuple(args, "i", &time))
		return NULL;

	if (time < 0 || time > _total_time) {
		PyErr_SetString(PyExc_ValueError,
			"the time is out of the trip period");
		return NULL;
	}

	index = time / _precision;
	mod = time % _precision;
	if (!(_trip[index].flag & TRIP_LAT_LON))
		Py_RETURN_NONE;

	lat = _trip[index].lat;
	lon = _trip[index].lon;
	if (mod && _trip[index + 1].flag & TRIP_LAT_LON) {
		lat = (lat * (_precision - mod) + _trip[index + 1].lat * mod) /
			_precision;
		lon = (lon * (_precision - mod) + _trip[index + 1].lon * mod) /
			_precision;
	}

	return Py_BuildValue("dd", lat, lon);
}

static PyObject *_get_track(PyObject *self)
{
	int i;
	PyObject *tuple, *data;

	tuple = PyTuple_New(_trip_size);
	for (i = 0; i < _trip_size; i++) {
		if (_trip[i].flag & TRIP_LAT_LON) {
			data = PyTuple_New(2);
			PyTuple_SetItem(data, 0, PyFloat_FromDouble(_trip[i].lat));
			PyTuple_SetItem(data, 1, PyFloat_FromDouble(_trip[i].lon));
			PyTuple_SetItem(tuple, i, data);
		} else {
			Py_INCREF(Py_None);
			PyTuple_SetItem(tuple, i, Py_None);
		}
	}

	return tuple;
}

static PyObject *_get_all_speed_samples(PyObject *self)
{
	int i;
	PyObject *tuple;

	tuple = PyTuple_New(_trip_size);
	for (i = 0; i < _trip_size; i++) {
		if (_trip[i].flag & TRIP_SPEED) {
			PyTuple_SetItem(tuple, i,
					PyFloat_FromDouble(_trip[i].speed));
		} else {
			Py_INCREF(Py_None);
			PyTuple_SetItem(tuple, i, Py_None);
		}
	}

	return tuple;
}

static PyObject *_get_all_rpm_samples(PyObject *self)
{
	int i;
	PyObject *tuple;

	tuple = PyTuple_New(_trip_size);
	for (i = 0; i < _trip_size; i++) {
		if (_trip[i].flag & TRIP_RPM) {
			PyTuple_SetItem(tuple, i,
					PyFloat_FromDouble(_trip[i].rpm));
		} else {
			Py_INCREF(Py_None);
			PyTuple_SetItem(tuple, i, Py_None);
		}
	}

	return tuple;
}

static PyObject *_get_trip_distance(PyObject *self)
{
	if (_trip_size <= 0)
		Py_RETURN_NONE;

	if (_trip[_trip_size - 1].distance <= 0)
		return PyFloat_FromDouble(0);

	return PyFloat_FromDouble(_trip[_trip_size - 1].distance);
}

static PyObject *_get_period(PyObject *self)
{
	return Py_BuildValue("iii", _start_time, _finish_time, _total_time);
}
