#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

from gauges.gaugeview import GaugeView
from common.carlog import DEBUG

class BarView(GaugeView):

    def __init__(self, canvas, image=None, pieces=10, mode="full"):
        """
        Guage Bar viewer constructor
        @param canvas: Evas canvas object
        @param image: bar image key
        @param pieces: number of the bars
        @param mode: full, up or down
        """
        GaugeView.__init__(self, canvas)
        self.dx = self.dy = self.py = 0
        self.pieces = pieces >= 1 and pieces or 1
        self.mode = mode.lower()
        if image is None:
            raise ValueError("Image not specified")
        self.image = self.Image(file=image)
        sx, self.sy = self.image.image_size_get()
        self.clipper.resize(sx, self.sy)
        self.bar_height = self.sy / pieces
        self.image.size_set(sx, self.sy)
        self.image.fill_set(0, 0, sx, self.sy)
        self.image.clip_set(self.clipper)
        self.image.show()
        if self.mode == "down":
            self.set_value(0, 1)
        else:
            self.set_value(0, 0)

    def __del__(self):
        DEBUG("deleting bar gauge %s" % self)

    def calculate(self):
        self.clipper.move(self.dx, self.dy + self.py)

    def move(self, x, y):
        """ Move objects """
        if self.dx != x or self.dy != y:
            self.dx = x
            self.dy = y
            self.image.move(x, y)
            self.changed()

    def set_value(self, value_abs, value_rel):
        """
        Change current value
        @param value_abs: absolute value (not used)
        @param value_rel: relative value from 0.0 to 1.0 for full mode,
        0.5 to 1.0 for up mode and 0.0 to 0.5 for down mode
        """
        if self.mode == "up":
            if value_rel <= 0.5:
                self.py = self.sy
            elif value_rel >= 1:
                self.py = 0
            else:
                value = (value_rel - 0.5) * 2
                self.py = int(self.sy - round(value * self.pieces) * \
                    self.bar_height)

        elif self.mode == "down":
            if value_rel <= 0:
                self.py = 0
            elif value_rel >= 0.5:
                self.py = -self.sy
            else:
                value = 1 - value_rel * 2
                self.py = -int(self.sy - round(value * self.pieces) * \
                    self.bar_height)

        else:
            if value_rel <= 0:
                self.py = self.sy
            elif value_rel >= 1:
                self.py = 0
            else:
                self.py = int(self.sy - round(value_rel * self.pieces) * \
                    self.bar_height)

        self.changed()
