#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import dbus
from common.carlog import DEBUG, ERROR
from common.singleton import Singleton
from models.dbusmodel import CarmandDbusModel

PIDGIN_BUS_NAME     = "im.pidgin.purple.PurpleService"
PIDGIN_OBJECT_PATH  = "/im/pidgin/purple/PurpleObject"
PIDGIN_IFACE        = "im.pidgin.purple.PurpleInterface"

PLUGIN_BUS_NAME     = "org.indt.carmanplugin"
PLUGIN_OBJECT_PATH  = "/"
PLUGIN_IFACE        = "org.indt.carmanplugin.Manager"
PLUGIN_IFACE_BUDDY  = "org.indt.carmanplugin.Buddy"

class ISPurpleModel(Singleton):

    def __init__(self):
        Singleton.__init__(self)
        self._iface = None
        self._bus = dbus.SessionBus()
        self._bus.add_signal_receiver(self.__name_owner_changed,
            dbus_interface="org.freedesktop.DBus",
            signal_name="NameOwnerChanged")

    def __name_owner_changed(self, *data):
        if data[0] == PIDGIN_BUS_NAME and data[1]:
            self._iface = None
            self._bus_object = None

    def connect(self):
        if self._iface:
            return True

        if PIDGIN_BUS_NAME not in self._bus.list_names():
            return False

        self._bus_object = self._bus.get_object(PIDGIN_BUS_NAME,
            PIDGIN_OBJECT_PATH)
        self._iface = dbus.Interface(self._bus_object, PIDGIN_IFACE)
        return True

    def get_accounts_active(self):
        accounts = []
        if self._iface:
            for account in self._iface.PurpleAccountsGetAllActive():
                if self._iface.PurpleAccountIsConnected(account):
                    accounts.append(account)

        return accounts

    def get_buddies_online(self, accounts):
        def cmp(a, b):
            a = a[0].strip().upper()
            b = b[0].strip().upper()
            if a < b:
                return -1
            elif a > b:
                return 1
            else:
                return 0

        buddies = []
        if self._iface:
            for account in accounts:
                for buddy in self._iface.PurpleFindBuddies(account, ""):
                    if self._iface.PurpleBuddyIsOnline(buddy):
                        name = self._iface.PurpleBuddyGetName(buddy)
                        alias = self._iface.PurpleBuddyGetAlias(buddy)
                        buddies.append((alias, name))
            buddies.sort(cmp=cmp)

        return buddies


class ISManagerModel(Singleton):

    def __init__(self):
        Singleton.__init__(self)
        self._iface = None
        self._buddies_model = {}
        self._buddy_created_cbs = []
        self._buddy_removed_cbs = []
        self._bus = CarmandDbusModel().get_bus()
        CarmandDbusModel().add_name_owner_changed_cb(
            self.__name_owner_changed)

        self._bus.add_signal_receiver(self.__on_buddy_created,
            bus_name=PLUGIN_BUS_NAME, path=PLUGIN_OBJECT_PATH,
            dbus_interface=PLUGIN_IFACE, signal_name="BuddyCreated")
        self._bus.add_signal_receiver(self.__on_buddy_removed,
            bus_name=PLUGIN_BUS_NAME, path=PLUGIN_OBJECT_PATH,
            dbus_interface=PLUGIN_IFACE, signal_name="BuddyRemoved")

    def __on_buddy_created(self, path):
        DEBUG("buddy created %s" % path)
        buddy_model = ISBuddyModel(path)
        self._buddies_model[path] = buddy_model
        for cb in self._buddy_created_cbs:
            cb(path, buddy_model)

    def __on_buddy_removed(self, path):
        DEBUG("buddy removed %s" % path)
        buddy_model = None
        if path in self._buddies_model:
            buddy_model = self._buddies_model.pop(path)
            buddy_model.finalize()
        for cb in self._buddy_removed_cbs:
            cb(path, buddy_model)

    def __name_owner_changed(self, *data):
        if PLUGIN_BUS_NAME == data[0] and data[1]:
            self._iface = None
            self._bus_object = None

    def connect(self):
        if self._iface:
            return True

        if PLUGIN_BUS_NAME not in self._bus.list_names():
            ERROR("failure to connect: pidgin plugin is disabled")
            return False

        self._bus_object = self._bus.get_object(PLUGIN_BUS_NAME,
            PLUGIN_OBJECT_PATH)
        self._iface = dbus.Interface(self._bus_object, PLUGIN_IFACE)

        return True

    def add_buddy_created_cb(self, cb):
        if callable(cb) and cb not in self._buddy_created_cbs:
            self._buddy_created_cbs.append(cb)

    def del_buddy_created_cb(self, cb):
        if cb in self._buddy_created_cbs:
            self._buddy_created_cbs.remove(cb)

    def add_buddy_removed_cb(self, cb):
        if callable(cb) and cb not in self._buddy_removed_cbs:
            self._buddy_removed_cbs.append(cb)

    def del_buddy_removed_cb(self, cb):
        if cb in self._buddy_removed_cbs:
            self._buddy_removed_cbs.remove(cb)

    def register_agent(self, path):
        if self._iface:
            self._iface.RegisterAgent(path)

    def remove_buddy(self, path):
        if self._iface:
            self._iface.RemoveBuddy(path)

    def request_authorization(self, buddy, ok_cb, error_cb):
        if self._iface:
            self._iface.RequestAuthorization(buddy, reply_handler=ok_cb,
                error_handler=error_cb)

    def unregister_agent(self, path):
        if self._iface:
            try:
                self._iface.UnregisterAgent(path)
            except Exception, e:
                DEBUG("unregister agent failed: %s" % e)

    def get_buddies_info(self):
        buddies = []
        for path, model in self._buddies_model.items():
            info = model.GetInfo()
            buddies.append((path, info["Name"], info["Email"].split("/")[0]))
        return buddies

    def has_buddies(self):
        return len(self._buddies_model) > 0

    def remove_all_buddies(self):
        if self._iface:
            for path in self._buddies_model:
                self._iface.RemoveBuddy(path)


class ISBuddyModel(dbus.Interface):

    def __init__(self, path):
        self._path = path
        self._data_available_cbs = []
        self._bus = CarmandDbusModel().get_bus()
        self._bus_object = self._bus.get_object(PLUGIN_BUS_NAME, path)
        dbus.Interface.__init__(self, self._bus_object, PLUGIN_IFACE_BUDDY)

        self._bus.add_signal_receiver(self.__data_available,
            bus_name=PLUGIN_BUS_NAME, path=path,
            dbus_interface=PLUGIN_IFACE_BUDDY, signal_name="DataAvailable")

    def __del__(self):
        DEBUG("deleting buddy model %s" % self)

    def __data_available(self, *data):
        for cb in self._data_available_cbs:
            cb(*data)

    def add_data_available_cb(self, cb):
        if callable(cb) and cb not in self._data_available_cbs:
            self._data_available_cbs.append(cb)

    def del_data_available_cb(self, cb):
        if cb in self._data_available_cbs:
            self._data_available_cbs.remove(cb)

    def finalize(self):
        self._bus.remove_signal_receiver(self.__data_available,
            bus_name=PLUGIN_BUS_NAME, path=self._path,
            dbus_interface=PLUGIN_IFACE_BUDDY, signal_name="DataAvailable")
