# carlog.py - logging facilities for carman
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sys, os, fcntl, time, errno

(NOT_INITIALIZED, INITIALIZING, INITIALIZED) = range(3)
__state__ = NOT_INITIALIZED
__logstream__ = None
__loglevel__ = 2
__msg_buffer__ = []

LOGLEVELS = ['DEBUG', 'INFO', 'WARNING', 'ERROR']

__CARMAN_DEBUG__ = 'CARMAN_DEBUG'

def setup():
    """
    Configure where the logging messages will be written.
    """
    global __state__
    global __msg_buffer__
    global DEBUG, INFO, WARNING, ERROR

    if __state__ == NOT_INITIALIZED:
        __state__ = INITIALIZING
        from common.carmanconfig import CarmanConfig

        config = CarmanConfig()

        if not config.is_log_enabled():
            __state__ = INITIALIZED
            __msg_buffer__ = []
            DEBUG = INFO = WARNING = ERROR = _get_log_dumb_method
            return

        _setup_log_level(config.get_log_level())
        _setup_log_path(config.get_log_path(), config)
        _setup_log_format(config.get_log_format())

        __state__ = INITIALIZED

        for level, msg in __msg_buffer__:
            if __logstream__ and level >= __loglevel__:
                _write_log(msg)
        __msg_buffer__ = []
# setup()

def _setup_log_level(loglevel):
    """
    Init the loglevel
    @param loglevel: one of LOGLEVELS
    """
    global __loglevel__
    if loglevel in LOGLEVELS:
        __loglevel__ = LOGLEVELS.index(loglevel)
    else:
        __loglevel__ = LOGLEVELS.index('WARNING')
# _setup_log_level

def _setup_log_path(logpath, carman_config):
    """
    Init logpath. The log path is the path where the log file should be saved.
    If empty, we use the [~user/.carman]. If not informed or an error happens 
    while trying to create a log file, we use the stderror.
    @param logpath: the path to create a file
    @param carman_config: the carman config.
    """
    global __logstream__
    if logpath is None:
        __logstream__ = sys.stderr
    else:
        if not os.path.dirname(logpath):
            filename = os.path.join(carman_config.get_user_directory(),
                    os.path.splitext(carman_config.get_app_name())[0] \
                        +'.log')
        else:
            filename = os.path.join(logpath,
                    os.path.splitext(carman_config.get_app_name())[0] \
                    +'.log')
        try:
            __logstream__ = open(filename, 'w')
        except IOError, err:
            ERROR('Unable to open %s log file: %s. Using stderr instead.' \
                  % (filename, err))
            __logstream__ = sys.stderr

        filedesc = __logstream__.fileno()
        arg = fcntl.fcntl(filedesc, fcntl.F_GETFL)
        fcntl.fcntl(filedesc, fcntl.F_SETFL, arg | os.O_NONBLOCK)
# _setup_log_path

def _setup_log_format(logformat):
    """
    Init the logformat
    @param logformat: one of LOGFORMATS
    """
    global __logformat__
    if logformat in LOGFORMATS.keys():
        __logformat__ = LOGFORMATS[logformat]
    else:
        __logformat__ = LOGFORMATS['COMPACT']
# _setup_log_format

def finalize():
    """
    Close any used resource.
    """
    global __logstream__
    global __initialized__

    if __logstream__ is not None and __logstream__ != sys.stderr:
        __logstream__.close()
    __logstream__ = None
    __state__ = NOT_INITIALIZED
# finalize()

def _log_format_compact(level, msg):
    """
    Write the log message to the stream.
    @param level: the message level
    @param msg: the message itself
    """
    format = '%(level)s: %(message)s\n'

    kargs = {
        'level' : level,
        'message' : msg,
    }

    return format % kargs

# _log_format_compact

def _log_format_normal(level, msg):
    """
    Write the log message to the stream.
    @param level: the message level
    @param msg: the message itself
    """
    frame = sys._getframe(2)
    if frame.f_code.co_name != '?':
        frameinfo = '%s:%s' % (os.path.splitext(
                            os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_code.co_name)
    else:
        frameinfo = '%s' % os.path.splitext(
                            os.path.basename(frame.f_code.co_filename))[0]

    format = '%(level)s: %(frameinfo)s - %(message)s\n'

    kargs = {
        'frameinfo' : frameinfo,
        'level' : level,
        'message' : msg,
    }

    return format % kargs
# _log_format_normal

def _log_format_verbose(level, msg):
    """
    Write the log message to the stream.
    @param level: the message level
    @param msg: the message itself
    """
    currenttime = time.time()
    frame = sys._getframe(2)
    
    if frame.f_code.co_name != '?':
        frameinfo = '%s:%s:%s' % (os.path.splitext(
                              os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_code.co_name,
                               frame.f_lineno)
    else:
        frameinfo = '%s:%s' % (os.path.splitext(
                              os.path.basename(frame.f_code.co_filename))[0],
                               frame.f_lineno)

    msecs = (currenttime - long(currenttime)) * 1000
    asctime = time.strftime('%H:%M:%S', time.localtime(currenttime))
    format = '%(asctime)s:%(msecs)03d: %(level)s: %(frameinfo)s -' \
             ' %(message)s\n'

    kargs = {
        'asctime' : asctime,
        'msecs' : msecs,
        'frameinfo' : frameinfo,
        'level' : level,
        'message' : msg,
    }

    return format % kargs

# _log_format_verbose

def _write_log(msg):
    """
    Write the message into the logstream
    @param msg: tre message to be written
    """
    try:
        __logstream__.write(msg)
        __logstream__.flush()
    except IOError:
        pass
# _write_log

def log_method_call(method) :
    """Decorator to debug method calls."""
    def wrap(*args, **kargs) :
        """
        Decorator wrap function.
        """
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'CALL: %s.%s(%r, %r)' % \
                               (args[0].__class__.__name__,
                                method.__name__, args[1:], kargs)))
        reply = method(*args, **kargs)
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'RET: %s.%s(%r, %r) = %r' \
                               % (args[0].__class__.__name__,
                                method.__name__, args[1:], kargs, reply)))
        return reply
    # wrap()
    if os.environ.has_key(__CARMAN_DEBUG__) and \
        os.environ[__CARMAN_DEBUG__] == '1':
        return wrap
    else:
        return method
# log_method_call()

def log_function_call(function) :
    """Decorator to debug function calls."""
    def wrap(*args, **kargs) :
        """
        Decorator wrap function
        """
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'CALL: %s.%s(%r, %r)' % \
                               (function.__module__, function.__name__,
                                args, kargs)))
        reply = function(*args, **kargs)
        if __logstream__ is not None and __loglevel__ == 0:
            _write_log(_log_format_verbose('DEBUG', 'RET: %s.%s(%r, %r) = %r' \
                               % (function.__module__, function.__name__,
                                args, kargs, reply)))
        return reply
    # wrap()
    if os.environ.has_key(__CARMAN_DEBUG__) and \
        os.environ[__CARMAN_DEBUG__] == '1':
        return wrap
    else:
        return function
# log_function_call()

def _get_log_dumb_method(msg):
    """
    This method will do nothing, that is, ignore the message
    @param msg:
    """
    pass
# _get_log_dumb_method

def _get_log_method(level) :
    """
    Wrap function to the debug, info, warning and error object.
    @param level: the log level
    """
    def do_log(msg) :
        """
        Write to the log stream.
        @param msg: 
        """
        global __msg_buffer__
        if __state__ == INITIALIZED:
            if __logstream__ is not None and num_level >= __loglevel__:
                _write_log(__logformat__(level, msg))
        elif __state__ == INITIALIZING:
            __msg_buffer__.append((num_level,__logformat__(level, msg)))

    # do_log()
    num_level = LOGLEVELS.index(level)
    return do_log
# _get_log_method()

def get_log_stream():
    """
    Return the stream that the messages should be sent to.
    """
    return __logstream__
# get_log_stream

def get_log_level():
    """
    Return the loglevel string.
    """
    return LOGLEVELS[__loglevel__]
# get_log_level

def get_log_format():
    """
    Return the current log format string.
    """
    for format in LOGFORMATS.keys():
        if __logformat__ == LOGFORMATS[format]:
            return format
    return 'COMPACT' # should never happen
# get_log_format

LOGFORMATS = {'COMPACT': _log_format_compact, 
              'NORMAL': _log_format_normal,
              'VERBOSE': _log_format_verbose,}

__logformat__ = LOGFORMATS['COMPACT']

DEBUG = _get_log_method('DEBUG')
INFO = _get_log_method('INFO')
WARNING = _get_log_method('WARNING')
ERROR = _get_log_method('ERROR')
