#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ManageBuddiesController}.
"""

from main.listctrl import ListView
from main.messagedlgctrl import MessageDlgController
from maps.connectbuddyctrl import ConnectBuddyController
from maps.deletebuddyctrl import DeleteBuddyController
from maps.disconnectbuddyctrl import DisconnectBuddyController
from maps.editbuddyctrl import EditBuddyController
from maps.newbuddyctrl import NewBuddyController
from models.buddymodel import BuddyModel
from models.infosharingmodel import InfoSharingModel

class ManageBuddiesController(object):
    """
    Controls buddies options (connect/disconnect/add/remove/edit).

    @type parent: L{GlobalOptionsView}
    @param parent: Parent window.
    """
    def __init__(self, parent):
        self._parent = parent
        self.buddy_model = BuddyModel()
        self.view = ListView("list-normal", "list-item-normal", \
            "MANAGE BUDDIES")
        self.is_model = InfoSharingModel()

        if len(self.buddy_model.get_connected_buddies()) > 0:
            self.view.add_item("Disconnect all buddies", \
                    self.__disconnect_all_buddies_cb)
            self.view.add_item("Disconnect buddy", self.__disconnect_buddy_cb)

        if self.buddy_model.size < BuddyModel.MAX_BUDDIES:
            self.view.add_item("Connect buddy", self.__connect_buddy_cb)

        self.view.add_item("New buddy", self.__new_buddy_cb)
        self.view.add_item("Edit buddy settings", self.__edit_buddy_cb)
        self.view.add_item("Delete buddy", self.__delete_buddy_cb)

    def __disconnect_all_buddies_cb(self, obj, item, param):
        """
        Called when "Disconnect all buddies" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        def disconnect_buddy():
            self.buddy_model.disconnect_all_buddies()
            obj.hide(True)
            self._parent.hide(True)

        msg = MessageDlgController(confirm_cb=disconnect_buddy)
        msg.show_message("Do you want to disconnect<br>from all buddies?", \
                buttons=2, title="DISCONNECT ALL BUDDIES")

    def __disconnect_buddy_cb(self, obj, item, param):
        """
        Called when "Disconnect buddy" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        disconnect = DisconnectBuddyController(obj)
        if disconnect.load_list():
            disconnect.show()
            self._parent.hide(True)

    def __connect_buddy_cb(self, obj, item, param):
        """
        Called when "Connect buddy" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        connect = ConnectBuddyController(obj)
        if connect.load_list():
            connect.show()
            self._parent.hide(True)

    def __new_buddy_cb(self, obj, item, param):
        """
        Called when "New buddy" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        new_buddy = NewBuddyController(obj)
        new_buddy.show()

    def __edit_buddy_cb(self, obj, item, param):
        """
        Called when "Edit buddy" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        edit = EditBuddyController(obj)
        if edit.load_list():
            edit.show()

    def __delete_buddy_cb(self, obj, item, param):
        """
        Called when "Delete buddy" option is selected.

        @type obj: L{ListView}
        @param obj: Instance of L{ListView} which originated the signal.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        delete = DeleteBuddyController(obj)
        if delete.load_list():
            delete.show()

    def show(self):
        """
        Displays the L{ListView} object.
        """
        self.view.show()
