#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{BuddyView} and L{BuddyInformationView}.
"""

import evas, edje
from models.gpsmodel import GPSModel


class BuddyView(evas.SmartObject):

    """
    Implements view for buddy specific data: buddy icon localization and buddy
    information balloon.

    @type controller: class
    @param controller: Instance of L{MapsController}.
    @type canvas: class
    @param canvas: Instance of Main canvas.
    """

    def __init__(self, controller, canvas):
        evas.SmartObject.__init__(self, canvas)
        self.active = False
        self.controller = controller
        self.clipper = self.Rectangle()

        self.__theme = None
        self.buddy = []
        for i in range(9):
            buddy = self.Image()
            buddy.clip_set(self.clipper)
            buddy.on_mouse_down_add(self.__buddy_mouse_down)
            self.buddy.append([buddy, -1, -1])
        self.last_buddy = None

        self.x = self.y = 0
        self.buddy_x = self.buddy_y = 0
        self.center_x = self.center_y = 0

    def __load_image(self, buddy, key):
        """
        Loads buddy icon image on maps.

        @type buddy: object
        @param buddy: Buddy instance.
        @type key: string
        @param key: icon image file path.
        """
        buddy[0].file_set(*key)
        w, h = buddy[0].image_size_get()
        buddy[0].size_set(w, h)
        buddy[0].fill_set(0, 0, w, h)
        buddy[1], buddy[2] = w / 2, h / 2

    def set_image(self, buddy_gps):
        """
        Sets an icon image for connected buddy.

        @type buddy_gps: boolean
        @param buddy_gps: C{True} if connected buddy has active GPS, C{False}
        otherwise.
        """
        if buddy_gps:
            IMG = "images/buddy%s"
        else:
            IMG = "images/buddytrans%s"

        for i in range(9):
            self.__load_image(self.buddy[i], (self.__theme, IMG % i))

    def set_theme(self, theme, set_image=None):
        """
        Sets L{BuddyView} view with given theme.

        @type theme: string
        @param theme: Theme filename with full path.
        @type set_image: callback
        @param set_image: Callback to be called when setting the icon image for
        buddy icon.
        """
        self.__theme = theme
        if set_image != None:
            if set_image:
                self.set_image(True)
            else:
                self.set_image(False)

    def set_buddy_area(self, area):
        """
        Sets L{BuddyView} area.

        @type area: tuple
        @param area: Tuple containing four coordinates bounds for buddy view.
        """
        self.buddy_area = area
        if self.active:
            self.changed()

    def calculate(self):
        """
        Calculates the buddy view position on map.
        """
        x = self.x + self.w / 2 - self.center_x + self.buddy_x
        y = self.y + self.h / 2 - self.center_y + self.buddy_y

        if y < self.buddy_area[1]:
            if x < self.buddy_area[0]:
                buddy = self.buddy[8]
            elif x > self.buddy_area[2]:
                buddy = self.buddy[2]
            else:
                buddy = self.buddy[1]

        elif y > self.buddy_area[3]:
            if x < self.buddy_area[0]:
                buddy = self.buddy[6]
            elif x > self.buddy_area[2]:
                buddy = self.buddy[4]
            else:
                buddy = self.buddy[5]

        elif x < self.buddy_area[0]:
            buddy = self.buddy[7]
        elif x > self.buddy_area[2]:
            buddy = self.buddy[3]
        else:
            buddy = self.buddy[0]

        if self.last_buddy != buddy[0]:
            if self.last_buddy:
                self.last_buddy.hide()
            buddy[0].show()
        self.last_buddy = buddy[0]

        x = min(max(x, self.buddy_area[0]), self.buddy_area[2])
        y = min(max(y, self.buddy_area[1]), self.buddy_area[3])
        buddy[0].move(x - buddy[1], y - buddy[2])

    def set_position(self, x, y):
        """
        Sets buddy icon position on map.

        @type x: number
        @param x: X coordinate.
        @type y: number
        @param y: Y coordinate.
        """
        self.center_x = x
        self.center_y = y
        if self.active:
            self.changed()

    def set_buddy_position(self, x, y):
        """
        Sets a remote connected buddy icon position on map.

        @type x: number
        @param x: X coordinate.
        @type y: number
        @param y: Y coordinate.
        """
        self.buddy_x = x
        self.buddy_y = y
        if self.active:
            self.changed()

    def __buddy_mouse_down(self, *param):
        """
        Wrapper for buddy button click operation.

        @type param: object
        @param param: Not used.
        """
        self.controller.set_center_buddy()

    def activate(self):
        """
        Activates the buddy.
        """
        self.active = True

    def deactivate(self):
        """
        Deactivates the buddy.
        """
        self.active = False
        self.last_buddy = None
        for buddy in self.buddy:
            buddy[0].hide()

    def clip_set(self, obj):
        """
        Clips object into area.

        @type   obj: C{evas.Rectangle}
        @param  obj: Object to clip.
        """
        self.clipper.clip_set(obj)

    def clip_unset(self):
        """
        Unclips all objects inside area.
        """
        self.clipper.clip_unset()

    def show(self):
        """
        Shows the C{evas.Rectangle} object.
        """
        self.clipper.show()

    def move(self, x, y):
        """
        Moves the C{evas.Rectangle} object.

        @type   x: number
        @param  x: M{X} position.
        @type   y: number
        @param  y: M{Y} position.
        """
        self.x = x
        self.y = y
        self.clipper.move(x, y)

    def resize(self, w, h):
        """
        Resizes the C{evas.Rectangle} object.
        """
        self.w = w
        self.h = h
        self.clipper.resize(w, h)


class BuddyInformationView(object):
    """
    Implements buddy balloon information view when have Infoshare buddy(ies)
    connected.

    @type canvas: class
    @param canvas: Instance of L{evas.Canvas}
    @type metric: boolean
    @param metric: Indicates if metric system is International or Imperial.
    C{True} if it is International, C{False} otherwise.
    @ivar view: Instance of L{BuddyView}.
    @ivar visible: C{True} if balloon is visible, C{False} otherwise.
    @ivar buddy_name: Buddy name to be shown in balloon.
    @ivar speed: Speed data.
    @ivar car_stopped: C{True} if infoshare buddy car is stopped, C{False}
    otherwise.
    @ivar gps_value: GPS data.
    """

    def __init__(self, canvas, metric):
        self.view = None
        self.canvas = canvas
        self.metric = metric
        self.visible = False
        self.buddy_name = ""
        self.speed = self.rpm = -1
        self.car_stopped = False
        self.gps_value = GPSModel.DISABLED

    def set_view(self, theme, view):
        """
        Instantiates a new view and Evas smart object.

        @type theme: string
        @param theme: Full path to theme file.
        @type view: class
        @param view: Instance of L{BuddyView}.
        """
        self.parent_view = view
        if self.view:
            self.view.delete()
        self.view = edje.Edje(self.canvas, file=theme,
            group="buddy_information")
        view.part_swallow("buddy_information", self.view)

        if self.visible:
            self.show()
            self.set_buddy_name(self.buddy_name)
            self.set_speed_value(self.speed)
            self.set_rpm_value(self.rpm)

    def set_unit_system(self, value):
        """
        Sets metric system used on balloon information speed and RPM.

        @type value: boolean
        @param value: C{True} if metric system is International, C{False} if it
        is Imperial.
        """
        self.metric = value
        self.set_speed_value(self.speed)

    def set_buddy_name(self, name):
        """
        Sets on balloon information the buddy name.

        @type name: string
        @param name: Buddy name.
        """
        self.buddy_name = name
        self.view.part_text_set("buddy_name", name.encode("utf-8"))

    def set_speed_value(self, value):
        """
        Updates on balloon information the speed value sent by infoshare buddy.

        @type value: string
        @param value: Speed value to be shown in balloon information.
        """
        self.speed = value
        self.car_stopped = False
        if value <= 0:
            if self.gps_value == GPSModel.FIXED:
                self.car_stopped = True
            else:
                if self.metric:
                    self.view.part_text_set("speed", "KM/H")
                else:
                    self.view.part_text_set("speed", "MPH")
                self.view.part_text_set("speed-value", "--")
        else:
            if self.metric:
                self.view.part_text_set("speed", "KM/H")
            else:
                self.view.part_text_set("speed", "MPH")
                value = round(value * 62 / 100)
            self.view.part_text_set("speed-value", "%d" % value)

    def set_rpm_value(self, value):
        """
        Updates on balloon information the RPM value sent by infoshare buddy.

        @type value: string
        @param value: RPM value to be shown in balloon information.
        """
        self.rpm = value
        if value < 0:
            self.view.part_text_set("RPM-value", "--")
        else:
            self.view.part_text_set("RPM-value", "%d" % value)

    def set_gps_status(self, value):
        """
        Updates GPS status by infoshare buddy.

        @type value: string
        @param value: GPS status from infoshare buddy.
        """
        self.gps_value = value
        if value == GPSModel.FIXED:
            self.view.signal_emit("gps_on", "")
        elif value == GPSModel.DISABLED:
            self.view.signal_emit("gps_disabled", "")
        else:
            self.view.signal_emit("gps_off", "")

    def show_data_values(self):
        """
        Shows GPS, RPM and Speed information on balloon.
        """
        if self.car_stopped:
            if self.gps_value == GPSModel.FIXED:
                self.view.signal_emit("car_stopped", "")
            else:
                self.view.part_text_set("speed-value", "--")
                self.view.part_text_set("RPM-value", "--")
                self.view.signal_emit("show-data", "")
        else:
            self.view.signal_emit("show-data", "")

    def show_buddy_connecting(self):
        """
        Shows C{Connecting} message on balloon information for a connecting
        buddy.
        """
        self.view.signal_emit("connecting", "")

    def show(self):
        """
        Displays the information balloon.
        """
        self.visible = True
        self.parent_view.signal_emit("show-buddy-info", "")

    def hide(self):
        """
        Hides the balloon.
        """
        self.visible = False
        self.parent_view.signal_emit("hide-buddy-info", "")

    def is_visible(self):
        """
        Returns if a balloon is visible or not.

        @return: C{True} if balloon is visible, C{False} otherwise.
        """
        return self.visible
