#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{DisconnectBuddyController}.
"""

from main.listctrl import ListView
from main.messagedlgctrl import MessageDlgController
from models.buddymodel import BuddyModel
from models.infosharingmodel import InfoSharingModel


class DisconnectBuddyController(object):
    """
    Implements buddy disconnection features (controls
    L{DisconnectBuddyController} in order to display a window to user
    containing disconnectable buddies).

    @type   parent: class
    @param  parent: Instance of L{MapsController}.
    """

    def __init__(self, parent):
        self._parent = parent
        self.buddy_model = BuddyModel()
        self.is_model = InfoSharingModel()
        self.view = ListView("list-normal", "list-item-normal", \
            "DISCONNECT BUDDY")

    def load_list(self):
        """
        Verifies if all exceptions are validated before displaying list of
        connectable buddies.

        @rtype: boolean
        @return: C{True} if all exceptions are validated, C{False} otherwise.
        """
        def buddy_selected_cb(obj, item, param):
            def disconnect_buddy():
                self.buddy_model.disconnect_buddy(param)
                hide_view()

            def hide_view():
                self.view.hide(True)

            self._parent.hide(True)
            msg = MessageDlgController(confirm_cb=disconnect_buddy, cancel_cb=hide_view)
            msg.show_message("Do you want to disconnect from<br>%s?" % param, \
                buttons=2, title="DISCONNECT_BUDDY")

        for buddy in self.buddy_model.get_connected_buddies():
            alias = self.is_model.get_buddy_alias(buddy)
            has_alias = lambda alias: alias if alias else buddy
            self.view.add_item(has_alias(alias), buddy_selected_cb, buddy)

        if self.view.has_items():
            return True
        else:
            msg = MessageDlgController()
            msg.show_message("No buddies left to disconnect", \
                    title="DISCONNECT BUDDY")
            self.view.hide(True)
            return False

    def show(self):
        """
        Shows L{ListView}.
        """
        self.view.show()
