#!/usr/bin/env python
# -*- coding: UTF8 -*-
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements L{SingletonMeta} and L{Singleton}.
"""


class SingletonMeta(type):
    """
    Meta class which assures that only one instance of the parent class is
    instantiated.
    """

    def __init__(mcs, name, base, dic):
        super(SingletonMeta, mcs).__init__(mcs, name, base, dic)
        mcs.__instance = None
        mcs.__copy__ = lambda self: self
        mcs.__deepcopy__ = lambda self, memo=None: self

    def __call__(cls, *args, **kwargs):
        """
        Called when the parent class is called as a function. Executes an
        instance of the parent class, if not instantiated yet, or returns
        the existing instance.
        """
        if not cls.__instance:
            cls.__instance = super(SingletonMeta, cls).__call__(*args, **kwargs)
        return cls.__instance


class Singleton(object):
    """
    Abstraction of a singleton class. Uses L{SingletonMeta} as an inherited
    metaclass which assures that only one instance of an object will be
    instantiated.

    @ivar   __metaclass__: L{SingletonMeta} class object.
    """
    __metaclass__ = SingletonMeta
    pass
