#!/usr/bin/env python
# -*- coding: UTF8 -*-
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Implements L{BluetoothAdapter}.

@var __DBUS_BLUEZ_SERVICE__: Bluez service.
@var __DBUS_BLUEZ_PATH__: Bluez path.
@var __DBUS_BLUEZ_ADAPTER__: Bluez adapter.
@var __DBUS_BLUEZ_ADAPTER_SERVICE__: Bluez adapter service.
@var __DBUS_SERVICE__: DBus service.
@var __DBUS_PATH__: DBus path.
"""

import dbus
from common.singleton import Singleton
from common.carlog import WARNING

__DBUS_BLUEZ_SERVICE__ = 'org.bluez'
__DBUS_BLUEZ_PATH__ = '/org/bluez'
__DBUS_BLUEZ_ADAPTER__ = '/org/bluez/hci0'
__DBUS_BLUEZ_ADAPTER_SERVICE__ = __DBUS_BLUEZ_SERVICE__ + '.Adapter'
__DBUS_SERVICE__ = 'org.freedesktop.DBus'
__DBUS_PATH__ = '/org/freedesktop/DBus'

class BluetoothAdapter(Singleton):
    """
    Implements methods that handle the bluetooth mode.
    """
    def __init__(self):
        Singleton.__init__(self)
        self._bus = dbus.SystemBus()
        bluez_object = self._bus.get_object(__DBUS_BLUEZ_SERVICE__,
                                           __DBUS_BLUEZ_ADAPTER__,
                                           introspect=False)
        self._bluez_adapter = dbus.Interface(bluez_object,
                                            __DBUS_BLUEZ_ADAPTER_SERVICE__)

    def get_mode(self):
        """
        Returns the bluetooth mode.

        @rtype: string
        @return: Bluetooth mode.

        @raise Exception: Unable to get the current bluetooth device mode: I{error_type}
        """
        mode = None

        try: # old BlueZ D-Bus API
            mode = self._bluez_adapter.GetMode()
        except dbus.DBusException, err:
            pass

        if mode == None:
            try: # new BlueZ D-Bus API
                mode = self._bluez_adapter.GetProperties()['Mode']
            except dbus.DBusException, err:
                WARNING('Unable to get the current bluetooth device mode: %s' \
                    % err)

        return mode

    def turn_on(self):
        """
        Turns the bluetooth on.

        @rtype: string
        @return: Bluetooth mode.

        @raise Exception: Error setting bluetooth mode.
        """
        mode = self.get_mode()
        if mode != None and mode == 'off':
            try:
                self._bluez_adapter.SetMode('connectable')
            except:
                self._bluez_adapter.SetProperty('Mode', \
                    dbus.String('connectable', variant_level=1))

    def turn_off(self):
        """
        Turns the bluetooth off.

        @raise Exception: Error setting bluetooth mode.
        """
        mode = self.get_mode()
        if mode != None and mode != 'off':
            try:
                self._bluez_adapter.SetMode('off')
            except:
                self._bluez_adapter.SetProperty('Mode', \
                    dbus.String('off', variant_level=1))

