/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <time.h>
#include <sys/time.h>
#include <gdbus.h>

#include "purple.h"
#include "dbus.h"
#include "commands.h"
#include "log.h"
#include "dbus-common.h"
#include "server.h"

/* D-Bus definitions */
#define INFOSHARING		"org.indt.carmanplugin" /*FIXME: Change this name*/
#define SERVICE_IFACE		"org.indt.carmanplugin.Service"
#define ERROR_IFACE		"org.indt.carmanplugin.Error"
#define CARMAND_DBUS_PATH	"/org/indt/carmand"

static DBusConnection *connection = NULL;

static DBusMessage *send_msg(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *message, *buddy;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &buddy,
				DBUS_TYPE_STRING, &message,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = send_im_message(buddy, message);

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");
	else if (-EINVAL == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.BuddyNotExists",
				"Buddy not found");
	else
		return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *authorize_buddy(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *buddy;
	gboolean authorize;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &buddy,
				DBUS_TYPE_BOOLEAN, &authorize,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = request_authorize_buddy_cb(buddy, authorize);

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.NoPendingRequests",
				"There are no pending requests");
	else if (-EINVAL == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.PendingRequestNotFound",
				"Pending request not found");
	else
		return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *remove_account(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (TRUE != account_remove())
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");
	else
		return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *set_acc_username(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret =  set_account_username(username);

        if (-ENOENT == ret)
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *set_acc_alias(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *alias;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &alias,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret = set_account_alias(alias);

        if (-ENOENT == ret)
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *set_acc_password(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *pwd;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &pwd,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret = set_account_password(pwd);

	if (-ENOENT == ret)
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *set_acc_rem_pwd(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	gboolean value;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_BOOLEAN, &value,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret = set_account_rem_pwd(value);

	if (-ENOENT == ret)
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *set_acc_protocol_id(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *protocol_id;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &protocol_id,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (0 != set_account_protocol_id(protocol_id))
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *get_acc_username(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	char *username;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	username = get_account_username();

        if (NULL == username) {

	    g_free(username);

            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_STRING, &username,
                        DBUS_TYPE_INVALID);

}

static DBusMessage *get_acc_protocol_id(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	char *id;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	id = get_account_protocol_id();

        if (NULL == id) {

	    g_free(id);

            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_STRING, &id,
                        DBUS_TYPE_INVALID);
}

static DBusMessage *get_acc_alias(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	char *alias;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	alias = get_account_alias();

	if (NULL == alias) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
			"Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_STRING, &alias,
			DBUS_TYPE_INVALID);
}

static DBusMessage *get_acc_pwd(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	char *pwd;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	pwd = g_strdup(get_account_password());

        if (NULL == pwd) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
			"Infosharing account not configured");
            //return g_dbus_create_error(msg,
            //            "org.indt.carmanplugin.Error.PasswordNotPresent",
            //           "Account password is not set");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_STRING, &pwd,
                        DBUS_TYPE_INVALID);
}

static DBusMessage *get_acc_rem_pwd(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	gboolean rem_pwd;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	rem_pwd = get_account_rem_pwd();

	//if (FALSE == rem_pwd) {
        //    return g_dbus_create_error(msg,
        //                "org.indt.carmanplugin.Error.AccountNotExists",
	//		"Infosharing account not configured");
        //}
        //else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_BOOLEAN, &rem_pwd,
			DBUS_TYPE_INVALID);
}

static DBusMessage *new_account(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username, *protocol_id;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_STRING, &protocol_id,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret = account_new(username, protocol_id);

        if (-EEXIST == ret) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountAlreadyExists",
                        "Given account already exists");
        }
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *connect_account(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
        int ret;

        infosharingd_debug("%s:%d\n", __func__, __LINE__);

        ret = account_connect();

        if (-ENOENT == ret) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *disconnect_account(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

        ret = account_disconnect();

        if (-ENOENT == ret) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *account_exist(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	gboolean ret = account_exists();

	return g_dbus_create_reply(msg, DBUS_TYPE_BOOLEAN, &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *account_enabled(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	gboolean ret = account_is_enabled();

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
			"org.indt.carmanplugin.Error.AccountNotExists",
			"Infosharing account not configured");
        else
		return g_dbus_create_reply(msg,
			DBUS_TYPE_BOOLEAN, &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *account_connected(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	gboolean ret = account_is_connected();

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (-ENOENT == ret) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_BOOLEAN, &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *account_connecting(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	gboolean ret = account_is_connecting();

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (-ENOENT == ret) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_BOOLEAN, &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *account_disconnected(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	gboolean ret = account_is_disconnected();

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (-ENOENT == ret) {
            return g_dbus_create_error(msg,
                    "org.indt.carmanplugin.Error.AccountNotExists",
                    "Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg,
                        DBUS_TYPE_BOOLEAN, &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *protocol_options_get(DBusConnection *conn,
			DBusMessage *msg, void *data)
{
	DBusMessage *reply;
	DBusMessageIter iter;
	DBusMessageIter dict;
	GHashTable *prpl_opt;
	DBusError derr;
	gchar *connect_server, *server, *protocol_id;
	gint port = 0;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &protocol_id,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	prpl_opt = get_protocol_options(protocol_id);

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.NoMemory",
				"No memory available");

	dbus_message_iter_init_append(reply, &iter);

	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
			DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
			DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
			DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &dict);

	if (prpl_opt) {
		if (g_hash_table_lookup_extended(prpl_opt,
				"connect_server",
				NULL, (gpointer)&connect_server)) {
			if (connect_server == NULL)
				connect_server = "";
			dbus_message_iter_append_dict_entry(&dict,
					"connect_server",
					DBUS_TYPE_STRING, &connect_server);
		}
		if (g_hash_table_lookup_extended(prpl_opt,
				"server", NULL, (gpointer)&server)) {
			dbus_message_iter_append_dict_entry(&dict, "server",
					DBUS_TYPE_STRING, &server);
		}

		if (g_hash_table_lookup_extended(prpl_opt, "port", NULL,
					(gpointer)&port)) {
			dbus_message_iter_append_dict_entry(&dict,
					"port",
					DBUS_TYPE_INT32, &port);
		}
	} else
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.ProtocolNotExist",
				"Given protocol id does not exist");

	dbus_message_iter_close_container(&iter, &dict);

	return reply;
}

static DBusMessage *account_get_prot_opts(DBusConnection *conn,
			DBusMessage *msg, void *data)
{
	DBusMessage *reply;
	DBusMessageIter iter;
	DBusMessageIter dict;
	GHashTable *prpl_opt;
	char *connect_server, *server;
	int port;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	prpl_opt = get_account_protocol_options();

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.NoMemory",
				"No memory available");

	dbus_message_iter_init_append(reply, &iter);

	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
			DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
			DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
			DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &dict);

	if (prpl_opt) {
		if (g_hash_table_lookup_extended(prpl_opt,
				"connect_server",
				NULL, (gpointer)&connect_server)) {
			dbus_message_iter_append_dict_entry(&dict,
					"connect_server",
					DBUS_TYPE_STRING, &connect_server);
		}

		if (g_hash_table_lookup_extended(prpl_opt,
				"server", NULL, (gpointer)&server)) {
			dbus_message_iter_append_dict_entry(&dict, "server",
					DBUS_TYPE_STRING, &server);
		}

		if (g_hash_table_lookup_extended(prpl_opt, "port", NULL,
					(gpointer)&port)) {
			dbus_message_iter_append_dict_entry(&dict,
					"port",
					DBUS_TYPE_INT32, &port);
		}
	} else
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.ProtocolNotExist",
				"Given protocol id does not exist");

	dbus_message_iter_close_container(&iter, &dict);

	return reply;
}

static DBusMessage *account_set_prot_opts(DBusConnection *conn,
				DBusMessage *msg, void *data)
{
	DBusMessageIter iter;
	GHashTable *prpl_opt;
	DBusError derr;

	dbus_error_init(&derr);
	prpl_opt = NULL;

	if (!dbus_message_iter_init(msg, &iter))
		goto error;

	prpl_opt = dbus_iter_hash_table(iter, &derr);
	if (NULL == prpl_opt)
		goto error;

	if (set_account_protocol_options(prpl_opt) != 0)
		goto error;

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);

error:
	if (prpl_opt)
		g_hash_table_destroy(prpl_opt);

	return g_dbus_create_error(msg,
			"org.indt.carmanplugin.Error.InvalidArguments",
			"Error setting protocol options");
}

static DBusMessage *set_account_status(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *type, *message;
        int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &type,
				DBUS_TYPE_STRING, &message,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

        ret = account_set_active_status(type, message);

        if (-ENOENT == ret) {
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");
        }
        else
            return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *buddy_add(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username, *alias, *group;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_STRING, &alias,
				DBUS_TYPE_STRING, &group,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = add_buddy(username, group, alias);

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.error.AccountNotExists",
				"Infosharing account not configured");
	else
            if (-EINVAL == ret)
                return g_dbus_create_error(msg,
                            "org.indt.carmanplugin.error.BuddyNameNull",
                            "Buddy Name is NULL");
            else
                if (-EEXIST == ret)
                    return g_dbus_create_error(msg,
                                "org.indt.carmanplugin.error.ConnectionNotFound",
                                "Connection not found");
                else
                    return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *check_buddy_exists(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = buddy_exists(username);

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.error.AccountNotExists",
				"Infosharing account not configured");

	return g_dbus_create_reply(msg, DBUS_TYPE_BOOLEAN, (gboolean) &ret,
			DBUS_TYPE_INVALID);
}

static DBusMessage *buddy_remove(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = remove_buddy(username);

	if (-ENOENT == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");
	else if (-EINVAL == ret)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.BuddyNotExists",
				"Buddy not found");
	else
		return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *list_buddies(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	DBusMessageIter iter, array_iter;
	char *buddy;
	GSList *l;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.Unknown",
				"Error retrieving buddies list");

	if (NULL == get_buddies()) {
		/* FIXME: Free reply message */
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.BuddyListEmpty",
				"No Buddies added to the Buddy List");
	}

	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
				DBUS_TYPE_STRING_AS_STRING, &array_iter);

	for (l = get_buddies(); l; l = l->next) {
		buddy = (char *) l->data;
		dbus_message_iter_append_basic(&array_iter,
				DBUS_TYPE_STRING, &buddy);
	}

	dbus_message_iter_close_container(&iter, &array_iter);

	return reply;
}

static DBusMessage *list_carman_buddies(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	DBusMessageIter iter, array_iter;
	char *buddy;
	GSList *l;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.Unknown",
				"Error retrieving online buddies list");

	if (NULL == get_buddies()) {
		/* FIXME: Free reply message */
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.BuddyListEmpty",
				"No Buddies added to the Buddy List");
	}

	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
				DBUS_TYPE_STRING_AS_STRING, &array_iter);

	for (l = get_carman_buddies_online(); l; l = l->next) {
		buddy = (char *) l->data;
		dbus_message_iter_append_basic(&array_iter,
				DBUS_TYPE_STRING, &buddy);
	}

	dbus_message_iter_close_container(&iter, &array_iter);

	return reply;
}

static DBusMessage *buddy_get_presence(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username;
	gboolean online;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		return reply;
	}

	online = get_buddy_presence(username);

        if (-ENOENT == online) {
            return g_dbus_create_error(msg,
                        "org.indt.carmanplugin.Error.AccountNotExists",
                        "Infosharing account not configured");
        }
        else
            if (-EINVAL == online) {
                return g_dbus_create_error(msg,
                            "org.indt.carmanplugin.Error.BuddyNotExists",
                            "Buddy not found");
            }
            else
                return g_dbus_create_reply(msg,
                            DBUS_TYPE_BOOLEAN, &online,
                            DBUS_TYPE_INVALID);
}

static DBusMessage *buddy_set_alias(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *username, *alias;
	int ret;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_STRING, &alias,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	ret = set_buddy_alias(username, alias);
	if (ret == -ENOENT)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.AccountNotExists",
				"Infosharing account not configured");
	if (ret == -EINVAL)
		return g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.BuddyNotExists",
				"Buddy not found");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *buddy_get_alias(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	const char *alias;
	char *username;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &username,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		return reply;
	}

	alias = get_buddy_alias(username);

	if (NULL == alias)
		alias = "";

	return g_dbus_create_reply(msg, DBUS_TYPE_STRING, &alias,
			DBUS_TYPE_INVALID);
}

static DBusMessage *set_client_busname(DBusConnection *conn, DBusMessage *msg,
		void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *client_name;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
				DBUS_TYPE_STRING, &client_name,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.carmanplugin.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	/* Monitoring client lifetime */
	g_dbus_add_disconnect_watch(connection, client_name,
			infosharingd_finalize, connection, NULL);

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

/*
 * Methods and signals that will be registered in Infosharingd's
 * DBUS Service Interface
 */
static GDBusMethodTable dbus_iface_service_methods[] = {
	{ "SendMsg",		"ss",		"",		send_msg	},
	{ "AuthorizeBuddy",	"sb",		"",		authorize_buddy },
	{ "RemoveAccount",	"",		"",		remove_account	},
	{ "SetAccountAlias",	"s",		"",		set_acc_alias},
	{ "SetAccountUsername",	"s",		"",		set_acc_username},
	{ "SetAccountPassword",	"s",		"",		set_acc_password},
	{ "SetAccountRemPwd",	"b",		"",		set_acc_rem_pwd},
	{ "SetAccountProtocolID", "s",		"",		set_acc_protocol_id},
	{ "GetAccountUsername",	"",		"s",		get_acc_username},
	{ "GetAccountProtocolID", "",		"s",		get_acc_protocol_id},
	{ "GetAccountAlias",	"",		"s",		get_acc_alias},
	{ "GetAccountPassword",	"",		"s",		get_acc_pwd},
	{ "GetAccountRemPwd",	"",		"b",		get_acc_rem_pwd},
	{ "NewAccount",		"ss",		"",		new_account },
	{ "ConnectAccount",	"",		"",		connect_account	},
	{ "DisconnectAccount",	"",		"",		disconnect_account},
	{ "AccountExists",	"",		"b",		account_exist},
	{ "AccountIsEnabled",	"",		"b",		account_enabled },
	{ "AccountIsConnected",	"",		"b",		account_connected},
	{ "AccountIsConnecting","",		"b",		account_connecting},
	{ "AccountIsDisconnected", "",		"b",		account_disconnected},
	{ "GetAccountProtocolOptions",	"",	"a{sv}",	account_get_prot_opts},
	{ "SetAccountProtocolOptions",	"a{sv}",	"",	account_set_prot_opts},
	{ "GetProtocolOptions",	"s",		"a{sv}",	protocol_options_get},
	{ "SetAccountStatus",	"ss",		"i",		set_account_status},
	{ "AddBuddy",		"sss",		"",		buddy_add	},
	{ "BuddyExists",	"s",		"b",		check_buddy_exists},
	{ "RemoveBuddy",	"s",		"",		buddy_remove	},
	{ "GetCarmanBuddies",	"",		"as",		list_carman_buddies},
	{ "GetBuddies",		"",		"as",		list_buddies},
	{ "GetBuddyPresence",	"s",		"b",		buddy_get_presence},
	{ "SetBuddyAlias",	"ss",		"",		buddy_set_alias},
	{ "GetBuddyAlias",	"s",		"s",		buddy_get_alias},
	{ "SetClientBusName",	"s",		"",		set_client_busname},
	{ NULL, NULL, NULL, NULL }
};

static GDBusSignalTable dbus_iface_service_signals[] = {
	{ "RequestAddBuddy",	"sss"	},
	{ "RequestAuthorizeBuddy", "ssss" },
	{ "BuddySignedOn",	"ss"	},
	{ "BuddySignedOff",	"ss"	},
	{ "SignedOn",		"ss"	},
	{ "SignedOff",		"ss"	},
	{ "ConnectionError",	"ssss"	},
	{ "ReceivingImMessage",	"sss"	},
	{ NULL, NULL }
};

gboolean emit_request_add_buddy(char *remote_user,
	char *alias, char *message)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == remote_user)
		return FALSE;

	if (NULL == alias)
		alias = g_strdup("");

	if (NULL == message)
		message = g_strdup("");

	return g_dbus_emit_signal(connection,
		CARMAND_DBUS_PATH,
		SERVICE_IFACE,
		"RequestAddBuddy",
		DBUS_TYPE_STRING, &remote_user,
		DBUS_TYPE_STRING, &alias,
		DBUS_TYPE_STRING, &message,
		DBUS_TYPE_INVALID);
}

gboolean emit_request_authorize_buddy(char *remote_user,
	char *alias, char *message, gboolean on_list)
{
	char *on_list_str;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == remote_user)
		return FALSE;

	if (NULL == alias)
		alias = g_strdup("");

	if (NULL == message)
		message = g_strdup("");

	/* REVISIT: AFAIK, dbus signals do not transmit different
	 * data types
	 */
	if (FALSE == on_list)
		on_list_str = "False";
	else
		on_list_str = "True";

	return g_dbus_emit_signal(connection,
		CARMAND_DBUS_PATH,
		SERVICE_IFACE,
		"RequestAuthorizeBuddy",
		DBUS_TYPE_STRING, &remote_user,
		DBUS_TYPE_STRING, &alias,
		DBUS_TYPE_STRING, &message,
		DBUS_TYPE_STRING, &on_list_str,
		DBUS_TYPE_INVALID);
}

gboolean emit_buddy_signed_on(char *username, char *alias)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == username)
		return FALSE;

	if (NULL == alias)
		alias = g_strdup("");

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"BuddySignedOn",
			DBUS_TYPE_STRING, &username,
			DBUS_TYPE_STRING, &alias,
			DBUS_TYPE_INVALID);
}

gboolean emit_buddy_signed_off(char *username, char *alias)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == username)
		return FALSE;

	if (NULL == alias)
		alias = g_strdup("");

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"BuddySignedOff",
			DBUS_TYPE_STRING, &username,
			DBUS_TYPE_STRING, &alias,
			DBUS_TYPE_INVALID);
}

gboolean emit_signed_on(char *username, char *protocol_id)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == username) || (NULL == protocol_id))
		return FALSE;

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"SignedOn",
			DBUS_TYPE_STRING, &username,
			DBUS_TYPE_STRING, &protocol_id,
			DBUS_TYPE_INVALID);
}

gboolean emit_signed_off(char *username, char *protocol_id)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == username) || (NULL == protocol_id))
		return FALSE;

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"SignedOff",
			DBUS_TYPE_STRING, &username,
			DBUS_TYPE_STRING, &protocol_id,
			DBUS_TYPE_INVALID);
}

gboolean emit_connection_error(char *username, char *protocol_id,
	char *sdesc, char *desc)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == username) || (NULL == protocol_id))
		return FALSE;

	if (NULL == sdesc)
		sdesc = g_strdup("");

	if (NULL == desc)
		desc = g_strdup("");

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"ConnectionError",
			DBUS_TYPE_STRING, &username,
			DBUS_TYPE_STRING, &protocol_id,
			DBUS_TYPE_STRING, &sdesc,
			DBUS_TYPE_STRING, &desc,
			DBUS_TYPE_INVALID);
}

gboolean emit_receiving_im_message(char *sender, char *alias, char *stripped)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == sender) || (NULL == stripped))
		return FALSE;

	if (NULL == alias)
		alias = g_strdup("");

	return g_dbus_emit_signal(connection,
			CARMAND_DBUS_PATH,
			SERVICE_IFACE,
			"ReceivingImMessage",
			DBUS_TYPE_STRING, &sender,
			DBUS_TYPE_STRING, &alias,
			DBUS_TYPE_STRING, &stripped,
			DBUS_TYPE_INVALID);
}

int dbus_init_events(struct infosharingd_event *event)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (!event)
		return -EINVAL;

	event->request_add = emit_request_add_buddy;
	event->request_authorize = emit_request_authorize_buddy;
	event->buddy_signed_on = emit_buddy_signed_on;
	event->buddy_signed_off = emit_buddy_signed_off;
	event->signed_on = emit_signed_on;
	event->signed_off = emit_signed_off;
	event->connection_error = emit_connection_error;
	event->receiving_im_message = emit_receiving_im_message;

	return 0;
}

gboolean dbus_init(DBusConnection *conn)
{
	DBusError derr;
	gboolean ret = FALSE;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_error_init(&derr);
	connection = dbus_connection_ref(conn);

	dbus_bus_register(connection, &derr);
	if (dbus_error_is_set(&derr)) {
		printf("ERROR bus register: %s\n", derr.message);
		dbus_error_free(&derr);
		goto out;
	}

	if (dbus_bus_request_name(connection, INFOSHARING,
				DBUS_NAME_FLAG_DO_NOT_QUEUE, &derr) !=
			DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER ) {
		printf("ERROR request name: %s\n", derr.message);
		dbus_error_free(&derr);
		goto out;
	}

	g_dbus_register_interface(connection, CARMAND_DBUS_PATH, SERVICE_IFACE,
			dbus_iface_service_methods,
			dbus_iface_service_signals, NULL, NULL, NULL);

	return TRUE;
out:
	if (ret == FALSE)
		dbus_connection_unref(connection);
	return ret;
}

void dbus_exit(void)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (!connection)
		return;

	g_dbus_unregister_interface(connection,
			CARMAND_DBUS_PATH, SERVICE_IFACE);

	dbus_bus_release_name(connection,
			INFOSHARING,
			NULL);

	dbus_connection_unref(connection);
}
