# -*- coding: utf-8 -*-
'''
Advanced Power
2008-2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
Connects to bunch of dbus signals, monitors values change and emmits signal in case of update.
'''

import os
import gtk, gobject, hildon

import dbus
from dbus.mainloop.glib import DBusGMainLoop

from advpowlib.ui.misc import Banner, Dialog
from advpowlib.ui.theme import Theme
from advpowlib.util.config import Config
from advpowlib.util.locale import Locale
from advpowlib.util.misc import Format, Shell, Sound

class AbstractAdvancedPower():

  app_name= 'Advanced Power'
  version = '0.4.1-6'
  dev_author = 'Kirill Plyashkevich'
  dev_email = 'ru.spirit@gmail.com'
  dev_years = '2008-2010'
  apm_iface = 'ru.spirit.advanced_power_monitor'
  tmp_file = '/tmp/advpow.tmp'

  def __init__(self):

    def _property_modified_handler(keys):
      for x in keys:
        try:
          self.update_upd_item(x, False)
        except:
          pass

    def _name_owner_handler(name, old, new):
      if str(name) == self.apm_iface:
        if str(new):
          self.monitor = dbus.Interface(self.bus.get_object(self.apm_iface, '/ru/spirit/advanced_power_monitor'), 'ru.spirit.advanced_power_monitor.request')
          self.monitor_uid = self.monitor.register()
          if not self.upd_items:
            self.recreate_menu()
          self.full_subscribe()
          self.load_wrappers()
          Banner.show_information(self, Locale.get('apmd_connected'))
        else:
          self.monitor = None
          self.monitor_uid = 0
          self.update_button_image(self.thm.get_button_image('0%'))
          self.config['wrappers'].clear()
          Banner.show_information(self, Locale.get('apmd_disconnected'))
        if self.button.get_active():
          self.menu_done(self.menu)
        self.update_button_state()
        self.queue_recreate_menu()

    self.__super_init__()
    self.signals = {}
    self.sounds = {}
    self.images = {}
    self.config = {'settings': {}, 'plugins': {}, 'translators': {}, 'wrappers': {}}
    self.upd_items = {}
    self.monitor = None
    self.recreate_menu_flag = False
    self.monitor_uid = 0
    self.working = False
    self.info_banner = None
    self.cfg = Config()
    self.thm = Theme(self.cfg.get_settings_value('use-gtk-theme'), self.cfg.get_settings_value('current-theme'))
    self.snd = Sound(self.cfg.get_settings_value('play-sounds'))
    self.loop = gobject.MainLoop()
    self.upd_items = {}
    self.create_statusbar_button()
    self.update_button_image(self.thm.get_button_image('0%'))
    self.m_loop = DBusGMainLoop()
    self.bus = dbus.SystemBus(mainloop = self.m_loop, private = True)
    try:
      self.monitor = dbus.Interface(self.bus.get_object(self.apm_iface, '/ru/spirit/advanced_power_monitor'), 'ru.spirit.advanced_power_monitor.request')
      self.monitor_uid = self.monitor.register()
    except Exception, e:
      self.monitor = None
      self.monitor_uid = 0
      self.queue_recreate_menu()
    self.load_wrappers()
    self.signals = {'NameOwnerChanged': {'cb':_name_owner_handler, 'signal_name':'NameOwnerChanged', 'dbus_interface':'org.freedesktop.DBus', 'path':'/org/freedesktop/DBus'}, 'PropertyModified': {'cb':_property_modified_handler, 'signal_name':'PropertyModified', 'dbus_interface':'ru.spirit.advanced_power_monitor.signal', 'path':'/ru/spirit/advanced_power_monitor'}}
    self.init_signals()
    self.menu_cb_map = {'os_version': 'auto',
                        'battery_status': 'auto',
                        'timeleft_active': 'auto',
                        'timeleft_idle': 'auto',
                        'ram': 'auto',
                        'swap': 'auto',
                        'memory': 'auto',
                        'diskspace': 'none',
                        'temperature': 'auto',
                        'light_sensor': 'auto',
                        'uptime': 'auto',
                        'cpu_frequency': 'auto',
                        'cpu_governor': 'manual',
                        'device_mode': 'none',
                        'bluetooth_mode': 'manual',
                        'buttons_emulation': 'none',
                        'usb_mode': 'manual',
                        'run_command': 'none'
                        }
    self.settings_cb_map = {'play-sounds': self.snd.set, 'temperature-metric': self.queue_recreate_menu, 'use-gtk-theme': self.set_gtk_theme_callback, 'current-theme': self.thm.load, 'use-custom-low-battery-level': self.update_button_state, 'low-battery-level': self.update_button_state}
    self.settings_range_map = {'low-battery-level': (1, 25), 'sounds-volume': (0, 100)}
    self.settings_order = ['use-custom-low-battery-level', 'low-battery-level', 'play-sounds', 'use-custom-sounds-volume', 'sounds-volume', 'temperature-metric', 'use-gtk-theme', 'current-theme']
    self.connect('destroy', self._cleanup)
    self.connect('style-set', self.gtk_theme_changed)
    self.create_settings_dialog()
    self.working = True
    self.create_menu()
    self.full_subscribe()
    if self.monitor:
      Banner.show_information(self, Locale.get('apmd_connected'))
    else:
      Banner.show_information(self, Locale.get('apmd_disconnected'))
    self.show_all()

  def __super_init__(self):
    raise NotImplementedError('AbstractAdvancedPower.__super_init__() must be implemented')

  def init_signals(self):
    for key in self.signals:
      self.signals[key]['receiver_match'] = self.bus.add_signal_receiver(self.signals[key]['cb'], signal_name=self.signals[key]['signal_name'], dbus_interface=self.signals[key]['dbus_interface'], path=self.signals[key]['path'])

  def _cleanup(self, widget=None):
    self.monitor.unregister(self.monitor_uid)
    self.working = False
    for key in self.signals:
      self.bus._clean_up_signal_match(self.signals[key]['receiver_match'])
      self.bus.remove_signal_receiver(self.signals[key]['receiver_match'])
      self.signals[key]['receiver_match'].remove()
    self.destroy_menu()
    self.settings_dialog.destroy()
    try:
      os.unlink(self.tmp_file)
    except:
      pass
    self.destroy()

  def full_subscribe(self):
    if self.monitor:
      self.monitor.subscribe(self.monitor_uid, self.upd_items.keys())
      self.update_upd_items()

  def create_statusbar_button(self):
    raise NotImplementedError('AbstractAdvancedPower.create_statusbar_button() must be implemented')

#Statusbar button section
  def redraw_button(self, state):
    use_custom_low_batt_lvl = self.cfg.get_settings_value('use-custom-low-battery-level')
    if state == 'full':
      Banner.show_information(self, Locale.get('battery_state_full'))
    elif state == 'charging':
      Banner.show_information(self, Locale.get('battery_state_charging'))
      self.snd.play(self.thm.get_sound('charging'), self.cfg.get_settings_value('use-custom-sounds-volume'), self.cfg.get_settings_value('sounds-volume'))
    elif state == 'charger_disconn':
      Banner.show_information(self, Locale.get('battery_state_charger_disconn'))
    else:
      percent = float(state.rsplit('%')[0])
      if (use_custom_low_batt_lvl and percent <= self.cfg.get_settings_value('low-battery-level')) or percent < 5:
        Banner.show_information(self, Locale.get('battery_state_very_low'))
        self.snd.play(self.thm.get_sound('verylow'), self.cfg.get_settings_value('use-custom-sounds-volume'), self.cfg.get_settings_value('sounds-volume'))
      elif percent < 12.5:
        self.snd.play(self.thm.get_sound('low'), self.cfg.get_settings_value('use-custom-sounds-volume'), self.cfg.get_settings_value('sounds-volume'))
    percent = 5
    if use_custom_low_batt_lvl:
      percent = self.cfg.get_settings_value('low-battery-level')
    self.update_button_image(self.thm.get_button_image(state, percent))

  def update_button_image(self, image):
    raise NotImplementedError('AbstractAdvancedPower.update_button_image(image) must be implemented')

  def update_button_state(self, param=None):
    if self.monitor:
      self.redraw_button(self.monitor.get_battery_state())
    else:
      self.redraw_button('0')

  def gtk_theme_changed(self, widget, prev_style):
    self.update_button_image(self.thm.gtk_theme_changed())
    self.recreate_menu_flag = True

  def update_apm_version(self):
    apm_version = Locale.get('unknown')
    if self.monitor:
      apm_version = self.monitor.get_version()
    self.apm_version_label.hide()
    self.apm_version_label.set_text('Advanced Power Monitor: %s\n' % apm_version)
    self.apm_version_label.show()

  def set_wrapper(self, key, wrapper):
    self.monitor.change_wrapper(key, wrapper)
    if key == 'battery':
      self.update_upd_item('battery-status', False)

#Menu items callbacks section
  def handle_battery_status(self, key):
    state = self.monitor.get_battery_state()
    self.redraw_button(state)
    if state == 'full':
      return Locale.get('battery_state_full')
    elif state == 'charging':
      return Locale.get('battery_state_charging')
    else:
      return '%s: %s' % (Locale.get_item(key), state)
    return Locale.get('battery_state_unknown')

  def handle_timeleft_active(self, key):
    return '%s: %s' % (Locale.get_item(key), Format.get_time(self.monitor.get_value(key)))

  def handle_timeleft_idle(self, key):
    return '%s: %s' % (Locale.get_item(key), Format.get_time(self.monitor.get_value(key)))

  def handle_os_version(self, key):
    return '%s: %s' % (Locale.get_item(key), self.monitor.get_value(key))

  def handle_temperature(self, key):
    temp = self.monitor.get_value(key)
    if self.cfg.get_settings_value('temperature-metric') == 'C':
      return '%s: %sC' % (Locale.get_item(key), temp)
    else:
      return '%s: %sF' % (Locale.get_item(key), (9.0/5.0 * temp + 32))

  def handle_light_sensor(self, key):
    return '%s: %s%s' % (Locale.get_item(key), str(self.monitor.get_value(key)), Locale.get('lux'))

  def handle_uptime(self, key):
    return '%s: %s' % (Locale.get_item(key), Format.get_time(self.monitor.get_value(key)))

  def handle_cpu_frequency(self, key):
    return '%s: %s%s' % (Locale.get_item(key), str(self.monitor.get_value(key)), Locale.get('MHz'))

  def handle_ram(self, key):
    ram_info = self.monitor.get_value(key)
    return '%s: %s/%s' % (Locale.get_item(key), Format.get_filesize(ram_info[1]), Format.get_filesize(ram_info[2]))

  def handle_swap(self, key):
    swap_info = self.monitor.get_value(key)
    swap_total = swap_info[2]
    if swap_total == 0:
      return Locale.get_item('swap_disabled')
    else:
      return '%s: %s/%s' % (Locale.get_item(key), Format.get_filesize(swap_info[1]), Format.get_filesize(swap_total))

  def handle_memory(self, key):
    memory_info = self.monitor.get_value(key)
    return '%s: %s/%s' % (Locale.get_item(key), Format.get_filesize(memory_info[1]), Format.get_filesize(memory_info[2]))

  def handle_diskspace(self, key):
    tmp = self.monitor.get_value(key)
    result = Locale.get_item('%s_title' % (key))
    for x in tmp:
      result = '%s\n%s: %s/%s' % (result, x[0], Format.get_filesize(x[2]), Format.get_filesize(x[3]))
    return result

  def handle_device_mode(self, key):
    dev_mode = self.monitor.get_value(key)
    next_mode = 'unknown'
    if dev_mode == 'normal':
      next_mode = 'flight'
    elif dev_mode == 'flight':
      next_mode = 'normal'
    return Locale.get_item('%s_%s' % (key, next_mode))

  def handle_cpu_governor(self, key):
    self.update_menu_mode(key)

  def handle_bluetooth_mode(self, key):
    self.update_menu_mode(key)

  def handle_usb_mode(self, key):
    self.update_menu_mode(key)

  def update_menu_mode(self, key):
    raise NotImplementedError('AbstractAdvancedPower.update_menu_mode(key) must be implemented')

  def set_cpu_governor(self, widget=None, governor=''):
    self.monitor.set_cpu_governor(governor)

  def set_device_mode(self, widget=None, mode=''):
    self.monitor.set_device_mode(mode)

  def set_bluetooth_mode(self, widget=None, mode=''):
    self.monitor.set_bt_mode(mode)

  def set_usb_mode(self, widget=None, mode=''):
    self.monitor.set_usb_mode(mode)
#end section

#MCE section
  def switch_device_mode(self, widget=None):
    self.monitor.switch_device_mode()

  def call_tklock_device(self, widget=None):
    self.monitor.tklock_device()

  def call_reboot(self, widget=None):
    Dialog.show_confirm(self.app_name, Locale.get('confirm_reboot'), self.monitor.reboot)

  def call_shutdown(self, widget=None):
    Dialog.show_confirm(self.app_name, Locale.get('confirm_shutdown'), self.monitor.shutdown)

  def home_key_pl(self, widget=None):
    self.monitor.home_key_pressed_long()

  def home_key_p(self, widget=None):
    self.monitor.home_key_pressed()
#end section

#Commands section
  def start_apm_daemon(self, widget=None):
    if widget:
      widget.set_sensitive(False)
    self.info_banner = Banner.show_animation(self, Locale.get('apmd_starting'), self.info_banner)
    Shell.run_command_async('/etc/init.d/apmonitord start', self.info_banner)
#    Shell.run_command('run-standalone.sh /etc/init.d/apmonitord start')
#end section

#Updatable items section
  def _add_upd_item(self, key, update, parent=None, connect=None):
    item = None
    if update == 'auto':
      text = getattr(self, 'handle_%s' % (key))(key)
      if not parent:
        parent = self.menu
      if connect:
        item = self.add_plain_menu_item(text, parent, True)
        for x in connect:
          item.connect(x, connect[x])
      else:
        item = self.add_plain_menu_item(text, parent, False, False)
    elif update == 'manual':
      if not parent:
        parent = self.menu
      getattr(self, 'add_%s_group' % (key))()
    self.upd_items[key] = {'update': update, 'item': item, 'parent': parent}

  def update_upd_item(self, key, skip_auto=True):
    try:
      item = self.upd_items[key]
      callback = getattr(self, 'handle_%s' % (key))
      if item['update'] == 'auto' and item['item'] and not skip_auto:
        self.set_menu_item_text(item['item'], callback(key))
      elif item['update'] == 'manual':
        callback(key)
    except KeyError:
      pass

  def update_upd_items(self):
    for x in self.upd_items:
      self.update_upd_item(x, False)
#end section

#Menu items section
  def set_menu_item_text(self, item, text):
    raise NotImplementedError('AbstractAdvancedPower.set_menu_item_text(self, item, text) must be implemented')

  def set_menu_item_text_from_callback(self, item, callback):
    self.set_menu_item_text(item, callback())

  def add_plain_menu_item(self, text, parent = None, selectable = False, greyed = True):
    raise NotImplementedError('AbstractAdvancedPower.add_plain_menu_item(text[, parent][, selectable][, greyed]) must be implemented')
#end section

#Menu section
  def queue_recreate_menu(self, key = None):
    self.recreate_menu_flag = True

  def create_menu(self):
    raise NotImplementedError('AbstractAdvancedPower.create_menu() must be implemented')

  def destroy_menu(self):
    raise NotImplementedError('AbstractAdvancedPower.create_statusbar_button() must be implemented')

  def recreate_menu(self):
    self.destroy_menu()
    self.upd_items.clear()
    self.create_menu()

  def popup_menu(self, widget, data=None):
    raise NotImplementedError('AbstractAdvancedPower.popup_menu(widget[, data]) must be implemented')

  def menu_done(self, widget, data=None):
    raise NotImplementedError('AbstractAdvancedPower.menu_done(widget[, data]) must be implemented')

  def menu_position(self,data=None):
    raise NotImplementedError('AbstractAdvancedPower.menu_position([data]) must be implemented')
#end section

#Special menu items section
  def add_device_mode(self, key):
    def flight_and_lock(widget=None):
      self.set_device_mode(widget, 'flight')
      self.call_tklock_device(widget)
    self.add_custom_group(key, [{'type': 'plain', 'key': 'device_lock', 'action': self.call_tklock_device}, {'type': 'upd_item', 'key': 'device_mode', 'actions': {'activate': self.switch_device_mode}}, {'type': 'plain', 'key': 'device_mode_flight_and_lock', 'action': flight_and_lock}, {'type': 'plain', 'key': 'reboot', 'action': self.call_reboot}, {'type': 'plain', 'key': 'shutdown', 'action': self.call_shutdown}])

  def add_buttons_emulation(self, key):
    self.add_custom_group(key, [{'type': 'plain', 'key': 'buttons-emu-home-p', 'action': self.home_key_p}, {'type': 'plain', 'key': 'buttons-emu-home-lp', 'action': self.home_key_pl}])

  def add_diskspace(self, key):
    self.add_custom_group(key, [{'type': 'upd_item', 'key': 'diskspace', 'actions': {}}])

  def add_custom_group(self, menu_text, items):
    raise NotImplementedError('AbstractAdvancedPower.add_custom_group(menu_text, items) must be implemented')

  def add_cpu_governor_group(self):
    menu = self.add_mode_group('cpu_governor', self.monitor.get_value('cpu_governors'))

  def add_bluetooth_mode_group(self):
    self.add_mode_group('bluetooth_mode', self.monitor.get_value('bluetooth_modes'))

  def add_usb_mode_group(self):
    self.add_mode_group('usb_mode', self.monitor.get_value('usb_modes'))

  def add_mode_group(self, key, modes):
    raise NotImplementedError('AbstractAdvancedPower.add_mode_group(key, modes) must be implemented')

  def add_run_command(self, key):
    self.add_plain_menu_item(text='%s...' % Locale.get_item(key), selectable=True).connect('activate', self.show_run_command, key)

  def show_run_command(self, widget=None, key='run_command'):
    commandline = ''
    su_prefix = ''
    dialog = gtk.Dialog(self.app_name, None, gtk.DIALOG_MODAL | gtk.DIALOG_NO_SEPARATOR, (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_NONE))
    input_field = gtk.Entry()
    in_terminal = gtk.CheckButton()
    as_root = gtk.CheckButton()
    dialog.vbox.pack_start(hildon.Caption(value=('%s:' % Locale.get_item(key)), control=input_field, flag=1, icon=gtk.Image(), group=gtk.SizeGroup(0)), fill=True)
    dialog.vbox.pack_start(hildon.Caption(value=('%s:' % Locale.get_item('%s_in_terminal' % (key))), control=in_terminal, flag=1, icon=gtk.Image(), group=gtk.SizeGroup(0)), fill=True)
    dialog.vbox.pack_start(hildon.Caption(value=('%s:' % Locale.get_item('%s_as_root' % (key))), control=as_root, flag=1, icon=gtk.Image(), group=gtk.SizeGroup(0)), fill=True)
    dialog.show_all()
    response = dialog.run()

    commandline = input_field.get_text()
    banner_text = commandline
    script = '#!/bin/sh\n%s' % (commandline)

    if as_root.get_active():
      su_prefix = 'sudo '
    commandline = '%s%s' % (su_prefix, self.tmp_file)

    if in_terminal.get_active():
      script = '%s\nread x' % (script)
      commandline = '/usr/bin/osso-xterm \'%s\'' % (commandline)

    dialog.hide()
    dialog.destroy()
    if response == gtk.RESPONSE_OK and commandline != '':
      self.info_banner = Banner.show_animation(self, '%s %s' % (Locale.get('starting'), banner_text), self.info_banner)
      Shell.run_command('echo "%s" > %s && chmod 777 %s' % (script, self.tmp_file, self.tmp_file))
      Shell.run_command_async(commandline, self.info_banner)
#      Shell.run_command(commandline)
#end section

#GConf settings section
  def save_config(self):
    self.save_settings_config()
    self.save_wrappers()
    self.save_plugins_config()

  def save_settings_config(self):
    settings_keys = self.cfg.get_settings_keys()
    for key in settings_keys:
      new_value = None
      cb = None
      s_type, s_value = self.cfg.get_settings_key(key)
      if (s_type == 'str'):
        new_value =  str(self.config['settings'][key]['control'].get_active_text())
      elif (s_type == 'bool'):
        new_value = bool(self.config['settings'][key]['control'].get_active())
      elif (s_type == 'int'):
        new_value = int(self.config['settings'][key]['control'].get_value())
      if new_value != None and new_value != s_value:
        self.cfg.set_settings_value(key, new_value)
        self.cfg.save_settings_key(key)
        if key in self.settings_cb_map:
          self.settings_cb_map[key](new_value)

  def save_wrappers(self):
    iter = self.wrappers_list.get_iter_first()
    while iter:
      wr_type = str(self.wrappers_list.get_value(iter, 0))
      new_value = str(self.wrappers_list.get_value(iter, 2))
      if self.config['wrappers'][wr_type]['value'] != new_value:
        self.set_wrapper(wr_type, new_value)
        self.config['wrappers'][wr_type]['value'] = new_value
        self.recreate_menu_flag = True
      iter = self.wrappers_list.iter_next(iter)

  def save_plugins_config(self):
    iter = self.plugins_list.get_iter_first()
    items_unsubscribe = []
    items_subscribe = []
    while iter:
      item = str(self.plugins_list.get_value(iter, 0))
      position = int(self.plugins_list.get_path(iter)[0])
      value = bool(self.plugins_list.get_value(iter, 2))
      changed = False
      s_text, s_value, s_position = self.cfg.get_plugins_key(item)
      if not s_value == value:
        s_value = value
        changed = True
        self.recreate_menu_flag = True
        if value:
          items_subscribe.append(item)
        else:
          items_unsubscribe.append(item)
      if not s_position == position:
        s_position = position
        changed = True
        self.recreate_menu_flag = True
      if changed:
        self.cfg.set_plugins_key(item, s_text, s_value, s_position)
        self.cfg.save_plugins_key(item)
      iter = self.plugins_list.iter_next(iter)
    if len(items_unsubscribe):
      self.monitor.unsubscribe(self.monitor_uid, items_unsubscribe)
    if len(items_subscribe):
      self.monitor.subscribe(self.monitor_uid, items_subscribe)

  def load_wrappers(self):
    if 'wrappers' in self.config:
      self.config['wrappers'].clear()
    else:
      self.config['wrappers'] = {}
    if 'wrappers' in self.config and self.monitor:
      wr_types = self.monitor.get_wrappers_types()
      for wr_type in wr_types:
        wr_type = str(wr_type)
        self.config['wrappers'][wr_type] = {}
        self.config['wrappers'][wr_type]['value'] = str(self.monitor.get_current_wrapper(wr_type))
        available_wrappers = self.monitor.get_available_wrappers(wr_type)
        self.config['wrappers'][wr_type]['list'] = []
        for av_wr in available_wrappers:
          self.config['wrappers'][wr_type]['list'].append(str(av_wr))

  def settings_changes_commit(self, widget):
    self.save_config()
    self.settings_changes_cancel(widget)

  def settings_changes_cancel(self, widget):
    self.plugins_list.clear()
    self.wrappers_list.clear()
    self.settings_dialog.hide_all()

  def set_gtk_theme_callback(self, use_gtk_theme):
    self.thm.set_use_gtk_theme(use_gtk_theme)
    self.update_button_image(self.thm.load())
#end section

#Settings GUI section
  def show_settings_dialog(self, widget=None):
    self.get_wrappers()
    self.get_temp_metric()
    self.get_themes(self.thm.themes_path)
    self.get_plugins()
    self.buttonUp.set_sensitive(False)
    self.buttonDown.set_sensitive(False)
    self.update_apm_version()
    self.settings_dialog.show_all()
    try:
      self.settings_bool_switch(self.config['settings']['use-gtk-theme']['control'], [self.config['settings']['current-theme']['caption']], inverse = True)
    except KeyError:
      pass
    try:
      self.settings_bool_switch(self.config['settings']['use-custom-low-battery-level']['control'], [self.config['settings']['low-battery-level']['caption']])
    except KeyError:
      pass
    try:
      self.settings_bool_switch(self.config['settings']['use-custom-sounds-volume']['control'], [self.config['settings']['sounds-volume']['caption']])
    except KeyError:
      pass

  def get_plugins(self):
    s_settings = self.cfg.sort_plugins_by_pos()
    s_list = self.monitor.get_property_list()
    s_list = [str(item) for item in s_list]
    for item in s_settings:
      try:
        if self.menu_cb_map[item[0]] == 'auto' and not item[0] in s_list:
          continue
      except KeyError:
        pass
      self.plugins_list.append([item[0], item[1]['text'], item[1]['value']])

  def get_themes(self, themes_path):
    try:
      control = self.config['settings']['current-theme']['control']
    except KeyError:
      return
    control.get_model().clear()
    dirs = sorted([filename for filename in os.listdir(themes_path) if os.path.isdir(os.path.join(themes_path, filename))])
    for sdir in dirs:
      control.append_text(sdir)
      if str(sdir) == self.cfg.get_settings_value('current-theme'):
        control.set_active(len(control.get_model()) - 1)

  def get_wrappers(self):
    wrappers_types = []
    try:
      wrappers_types = self.config['wrappers'].keys()
    except:
      self.config['wrappers'] = {}
      return
    self.current_combo_model = ''
    for wr_type in wrappers_types:
      self.config['wrappers'][wr_type]['value'] = str(self.monitor.get_current_wrapper(wr_type))
      list = gtk.ListStore(str)
      for wr in self.config['wrappers'][wr_type]['list']:
        list.append([wr])
      self.wrappers_list.append([wr_type, Locale.get('wrappers-%s' % wr_type), self.config['wrappers'][wr_type]['value'], list])

  def get_temp_metric(self):
    try:
      control = self.config['settings']['temperature-metric']['control']
    except KeyError:
      return
    control.get_model().clear()
    metric = ['F','C']
    for m in metric:
      control.append_text(m)
      if str(m) == self.cfg.get_settings_value('temperature-metric'):
        control.set_active(len(control.get_model()) - 1)

  def tab_switch_page(self, widget, tab, page):
    if tab.get_n_pages() > page:
      tab.set_current_page(page)

  def close_settings_dialog(self, widget):
    Banner.show_information(self, 'Close Settings dialog: %s' % widget)
    self.settings_changes_cancel(widget)

  def create_settings_dialog(self):
    self.settings_dialog = gtk.Dialog(Locale.get('title_settings'), None, gtk.DIALOG_NO_SEPARATOR)#, gtk.DIALOG_MODAL)
    self.settings_dialog.connect('close', self.close_settings_dialog)

    tabs = gtk.Notebook()
    tabs.set_show_border(True)
    tabs.set_scrollable(False)

    self.settings_dialog.connect('show', self.tab_switch_page, tabs, 0)
    settings_tab = gtk.ScrolledWindow()
    settings_tab.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    settings_view = gtk.VBox(True)

    caption_icon = gtk.Image()
    caption_group = gtk.SizeGroup(0)
    settings_keys = self.cfg.get_settings_keys()
    for item in settings_keys:
      self.config['settings'][item] = {}
      s_type, s_value = self.cfg.get_settings_key(item)
      if s_type == 'bool':
        self.config['settings'][item]['control'] =  gtk.CheckButton()
        self.config['settings'][item]['control'].set_active(s_value)
      elif s_type == 'int':
        self.config['settings'][item]['control'] = hildon.NumberEditor(self.settings_range_map[item][0], self.settings_range_map[item][1])
        self.config['settings'][item]['control'].set_value(s_value)
      elif s_type == 'str':
        self.config['settings'][item]['control'] = gtk.combo_box_new_text()
      self.config['settings'][item]['caption'] = hildon.Caption(value=Locale.get('settings-%s' % item), control=self.config['settings'][item]['control'], flag=1, icon=caption_icon, group=caption_group)

    try:
      self.config['settings']['use-gtk-theme']['control'].connect('toggled', self.settings_bool_switch, [self.config['settings']['current-theme']['caption']], [], True)
    except KeyError:
      pass
    try:
      self.config['settings']['use-custom-low-battery-level']['control'].connect('toggled', self.settings_bool_switch, [self.config['settings']['low-battery-level']['caption']])
    except KeyError:
      pass
    try:
      self.settings_bool_switch(self.config['settings']['use-custom-sounds-volume']['control'], [self.config['settings']['sounds-volume']['caption']])
      self.config['settings']['use-custom-sounds-volume']['control'].connect('toggled', self.settings_bool_switch, [self.config['settings']['sounds-volume']['caption']])
    except KeyError:
      pass

    for x in self.settings_order:
      try:
        settings_view.pack_start(self.config['settings'][x]['caption'])
      except KeyError:
        pass
    settings_tab.add_with_viewport(settings_view)

    items_tab = gtk.ScrolledWindow()
    items_tab.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)

    self.plugins_list = gtk.ListStore(str, str, bool)
    items_view = gtk.TreeView(self.plugins_list)
    text_renderer = gtk.CellRendererText()
    c_renderer = gtk.CellRendererToggle()
    c_renderer.set_property('activatable', True)
    c_renderer.connect('toggled', self.settings_col_toggled, 2)
    column = gtk.TreeViewColumn('item', text_renderer)
    column.set_attributes(text_renderer, text=1)
    column.set_expand(True)
    c_column = gtk.TreeViewColumn('button', c_renderer)
    c_column.add_attribute(c_renderer, 'active', 2)
    c_column.set_expand(False)
    items_view.append_column(column)
    items_view.append_column(c_column)
    items_view.set_reorderable(True)
    items_tab.add(items_view)

    wrappers_tab = gtk.ScrolledWindow()
    wrappers_tab.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.wrappers_list = gtk.ListStore(str, str, str, gobject.TYPE_OBJECT)
    wrappers_view = gtk.TreeView(self.wrappers_list)
    w_text_renderer = gtk.CellRendererText()
    w_combo_renderer = gtk.CellRendererCombo()
    
    w_combo_renderer.set_property('editable', True)
    w_combo_renderer.set_property('model', gtk.ListStore(str))
    w_combo_renderer.set_property('text-column', 0)
    w_combo_renderer.set_property('has-entry', False)
    w_combo_renderer.connect('edited', self.cellcombo_edited)

    w_text_column = gtk.TreeViewColumn('item', w_text_renderer)
    w_text_column.set_attributes(w_text_renderer, text=1)
    w_text_column.set_expand(True)

    w_combo_column = gtk.TreeViewColumn('list', w_combo_renderer)
    w_combo_column.set_expand(True)
    w_combo_column.set_attributes(w_combo_renderer, text=2)

    wrappers_view.append_column(w_text_column)
    wrappers_view.append_column(w_combo_column)
    wrappers_view.set_reorderable(False)
    wrappers_view.connect('cursor-changed', self.cellcombo_changed)
    wrappers_tab.add(wrappers_view)

    events_tab = gtk.VBox()

    translators_tab = gtk.ScrolledWindow()
    translators_tab.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    translators_list = gtk.ListStore(str, str)
    translators_view = gtk.TreeView(translators_list)
    t_text_renderer = gtk.CellRendererText()
    column1 = gtk.TreeViewColumn('language', t_text_renderer)
    column1.set_attributes(t_text_renderer, text=0)
    column2 = gtk.TreeViewColumn('author', t_text_renderer)
    column2.set_attributes(t_text_renderer, text=1)
    column1.set_expand(True)
    column2.set_expand(True)
    translators_view.append_column(column1)
    translators_view.append_column(column2)
    translators_values = self.cfg.get_translators_values()
    for lang in translators_values:
      translators_list.append([lang['text'], lang['authors']])
    translators_tab.add(translators_view)

    about_tab = gtk.VBox()
    about_tab.pack_start(gtk.Label('\n%s %s\n' % (self.app_name, self.version)))
    self.apm_version_label = gtk.Label()
    self.update_apm_version()
    about_tab.pack_start(self.apm_version_label)
    about_tab.pack_start(gtk.Label(u'\N{COPYRIGHT SIGN}%s %s <%s>\n' % (self.dev_years, self.dev_author, self.dev_email)))

    tabs.append_page(settings_tab, gtk.Label(Locale.get('title_main')))
    tabs.append_page(items_tab, gtk.Label(Locale.get('title_items')))
    tabs.append_page(wrappers_tab, gtk.Label(Locale.get('title_wrappers')))
#    tabs.append_page(events_tab, gtk.Label(Locale.get('title_events')))
    tabs.append_page(translators_tab, gtk.Label(Locale.get('title_translation')))
    tabs.append_page(about_tab, gtk.Label(Locale.get('title_about')))
    self.settings_dialog.vbox.pack_start(tabs)
    buttonOk = gtk.Button(stock=gtk.STOCK_OK, use_underline=False)
    buttonOk.connect('clicked', self.settings_changes_commit)
    buttonCancel = gtk.Button(stock=gtk.STOCK_CANCEL, use_underline=False)
    buttonCancel.connect('clicked', self.settings_changes_cancel)
    self.buttonUp = gtk.Button()
    self.buttonUp.set_image(gtk.image_new_from_file('/usr/share/icons/hicolor/26x26/hildon/qgn_indi_arrow_up.png'))
    self.buttonDown = gtk.Button()
    self.buttonDown.set_image(gtk.image_new_from_file('/usr/share/icons/hicolor/26x26/hildon/qgn_indi_arrow_down.png'))
    self.settings_dialog.action_area.pack_start(buttonOk)
    self.settings_dialog.action_area.pack_start(self.buttonUp)
    self.settings_dialog.action_area.pack_start(self.buttonDown)
    self.settings_dialog.action_area.pack_start(buttonCancel)
    tabs.connect('switch-page', self.settings_page_switched, items_view)
    items_view.connect('cursor-changed', self.settings_row_changed)
    items_view.connect('drag-data-received', self.settings_ddr_cb)
    self.buttonUp.connect('clicked', self.settings_move_item, items_view, False)
    self.buttonDown.connect('clicked', self.settings_move_item, items_view, True)

  def settings_page_switched(self, widget, page, page_num, view):
    if page_num == 1:
      self.buttonUp.show()
      self.buttonDown.show()
      self.settings_row_changed(view)
      view.set_cursor(0)
    else:
      self.buttonUp.hide()
      self.buttonDown.hide()

  def settings_col_toggled(self, cell, path, column=0):
    self.plugins_list[path][column] = not self.plugins_list[path][column]

  def settings_ddr_cb(self, view, context, x, y, selection, info, timestamp):
    drop_info = view.get_dest_row_at_pos(x, y)
    self.buttonUp.set_sensitive(False)
    self.buttonDown.set_sensitive(False)
  
  def settings_row_changed(self, view):
    (model, iter) = view.get_selection().get_selected()
    if not iter:
      self.buttonUp.set_sensitive(False)
      self.buttonDown.set_sensitive(False)
      return
    path = model.get_path(iter)
    if path[0] == 0:
      self.buttonUp.set_sensitive(False)
      self.buttonDown.set_sensitive(True)
    elif (path[0] + 1) == len(model):
      self.buttonUp.set_sensitive(True)
      self.buttonDown.set_sensitive(False)
    else:
      self.buttonUp.set_sensitive(True)
      self.buttonDown.set_sensitive(True)

  def settings_move_item(self, widget, view, down):
    (model, iter) = view.get_selection().get_selected()
    if not iter:
      return
    path = model.get_path(iter)
    n_iter = None
    try:
      if down and (path[0] + 1) < len(model):
        n_iter = model.get_iter(path[0] + 1)
      elif path[0] > 0:
        n_iter = model.get_iter(path[0] - 1)
    except:
      pass
    if n_iter:
      model.swap(iter, n_iter)
      view.scroll_to_cell(model.get_path(iter))
      self.settings_row_changed(view)

  def cellcombo_changed(self, view):
    (model, iter) = view.get_selection().get_selected()
    if not iter:
      return
    path = self.wrappers_list.get_path(iter)
    new_combo_model = str(self.wrappers_list.get_value(self.wrappers_list.get_iter(path), 0))
    if new_combo_model == self.current_combo_model:
      return
    self.current_combo_model = new_combo_model
    wrapper_list = self.wrappers_list.get_value(self.wrappers_list.get_iter(path), 3)
    combo_renderer =  view.get_column(1).get_cell_renderers()[0]
    combo_model = combo_renderer.get_property('model')
    combo_model.clear()
    iter = wrapper_list.get_iter_first()
    while True:
      try:
        item = str(wrapper_list.get_value(iter, 0))
        combo_model.append([item])
        iter = wrapper_list.iter_next(iter)
      except Exception, e:
        break

  def cellcombo_edited(self, cellrenderercombo, path, new_text):
    iter = self.wrappers_list.get_iter(path)
    self.wrappers_list.set_value(iter, 2, new_text)

  def settings_bool_switch(self, widget, affected_on = [], affected_off = [], inverse = False):
    flag = widget.get_active()
    if inverse:
      flag = not flag
    for x in affected_on:
      self.set_visible(x, flag)
    for x in affected_off:
      self.set_visible(x, not flag)

  def set_visible(self, widget, visible = True):
    if visible:
      widget.show()
    else:
      widget.hide()
#end section
