/*
  This file is part of "WhoAmI" - A "Guess whats on the photo" riddle game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "DownloadLists.h"

#include <cerrno>
#include <cstdio>
#include <cstring>
#include <sys/stat.h>
#include <fcntl.h>

#include <iostream>
#include <list>

#include <curl/curl.h>

#include <Lum/Base/String.h>

#include <Lum/Images/Loader/JPEG.h>

const char* tags[] = {
                      "animal",
                      "animals",
                      "animalkingdomelite",
                      "birds",
                      "pet",
                      "pets",
                      "specanimal",
                      "wildlife",
                      "zoo",
                      NULL,
                    };

static size_t WriteFunction(void* ptr, size_t size, size_t nmemb, void* stream)
{
  DownloadListsTask* task=static_cast<DownloadListsTask*>(stream);

  task->AppendToBuffer((const char*)ptr,size*nmemb);

  return size*nmemb;
}

static void Dequote(std::string& text)
{
  size_t pos=0;

  while (pos<text.length()) {
    if (text[pos]=='&') {
      if (text.substr(pos,4)=="&lt;") {
        text.erase(pos,4);
        text.insert(pos,"<");
        pos+=1;
      }
      else if (text.substr(pos,4)=="&gt;") {
        text.erase(pos,4);
        text.insert(pos,">");
        pos+=1;
      }
      else if (text.substr(pos,6)=="&quot;") {
        text.erase(pos,6);
        text.insert(pos,"\"");
        pos+=1;
      }
      else {
        pos++;
      }
    }
    else {
      pos++;
    }
  }
}

DownloadListsTask::DownloadListsTask()
{
  // no code
}

void DownloadListsTask::SetImageList(const std::list<Image*>& list)
{
  images=list;
}

void DownloadListsTask::GetImageList(std::list<Image*>& list)
{
  list=images;
}

void DownloadListsTask::Run()
{
  CURL        *curl;
  CURLcode    result;
  std::string url;
  size_t      steps=0;
  size_t      currentStep=0;
  size_t      tag;

  tag=0;
  while (tags[tag]!=NULL) {
    steps++;
    tag++;
  }

  steps*=2;

  error.clear();

  curl=curl_easy_init();

  if (curl==NULL) {
    error="Cannot initiale curl handle!";
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_NOSIGNAL,1);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_COOKIEFILE,"");
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEFUNCTION,WriteFunction);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEDATA,this);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  SetProgress(0);
  SetCaption(L"Scanning RSS feeds...");

  tag=0;
  while (tags[tag]!=NULL) {
    if (IsAborted()) {
      curl_easy_cleanup(curl);
      return;
    }

    url="http://api.flickr.com/services/feeds/photos_public.gne?tags=";
    url+=tags[tag];
    url+="&format=rss_200";

    result=curl_easy_setopt(curl,CURLOPT_URL,url.c_str());
    if (result!=CURLE_OK) {
      error=curl_easy_strerror(result);
      curl_easy_cleanup(curl);
      return;
    }

    buffer.clear();

    SetAction(L"Downloading RSS feed '"+Lum::Base::StringToWString(tags[tag])+L"'...");

    if (curl_easy_perform(curl)!=CURLE_OK) {
      std::cerr << "Cannot download file" << std::endl;
      std::cerr << buffer << std::endl;
      error=curl_easy_strerror(result);
      curl_easy_cleanup(curl);
      return;
    }

    currentStep++;
    SetProgress((currentStep*100)/steps);

    SetAction(L"Parsing RSS feed '"+Lum::Base::StringToWString(tags[tag])+L"'...");

    size_t a=0;
    size_t b=0;

    while ((a=buffer.find("<item>",a))!=std::string::npos) {
      std::string author;
      std::string title;
      std::string description;
      std::string link;
      Image       *image;
      bool        found=false;

      if (IsAborted()) {
        curl_easy_cleanup(curl);
        return;
      }

      a=buffer.find("<title>",a);
      if (a==std::string::npos) {
        error="Error while parsing rss feed content, cannot find title";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen("<title>");
      b=buffer.find("</title>",a);

      if (b==std::string::npos) {
        error="Error while parsing rss feed content, cannot find title end";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      title=buffer.substr(a,b-a);
      Dequote(title);

      a=b+strlen("</title>");

      a=buffer.find("<description>",a);
      if (a==std::string::npos) {
        error="Error while parsing rss feed content, cannot find description";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen("<description>");
      b=buffer.find("</description>",a);

      if (b==std::string::npos) {
        error="Error while parsing rss feed content, cannot find description end";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      description=buffer.substr(a,b-a);
      Dequote(description);

      a=b+strlen("</description>");

      a=buffer.find("<author",a);
      if (a==std::string::npos) {
        error="Error while parsing rss feed content, cannot find author";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen("<author");

      a=buffer.find(">",a);
      if (a==std::string::npos) {
        error="Error while parsing rss feed content, cannot find author";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen(">");

      b=buffer.find("</author>",a);

      if (b==std::string::npos) {
        error="Error while parsing rss feed content, cannot find author end";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      author=buffer.substr(a,b-a);
      Dequote(author);

      a=b+strlen("</author>");

      a=buffer.find("<media:content url=\"",a);
      if (a==std::string::npos) {
        error="Error while parsing rss feed content, cannot find url";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen("<media:content url=\"");
      b=buffer.find("\"",a);

      if (b==std::string::npos) {
        error="Error while parsing rss feed content, cannot find url end";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      link=buffer.substr(a,b-a);

      a=b+strlen("\"");

      b=buffer.find("</item>",a);
      if (b==std::string::npos) {
        error="Error while parsing rss feed content, cannot find item end";
        std::cerr << error << "\n" << buffer << std::endl;
        curl_easy_cleanup(curl);
        return;
      }

      a=b+strlen("</item>");

      for (std::list<Image*>::const_iterator iter=images.begin(); iter!=images.end(); ++iter) {
        if ((*iter)->GetImageLink()==link) {
          found=true;
          break;
        }
      }

      if (found) {
        continue;
      }

      image=new Image(author,title,description,link);
      images.push_back(image);
    }

    tag++;
    currentStep++;
    SetProgress((currentStep*100)/steps);
  }

  SetAction(L"Done.");
  SetProgress(100);

  curl_easy_cleanup(curl);
}

void DownloadListsTask::AppendToBuffer(const char* text, size_t size)
{
  buffer.append(text,size);
}

bool DownloadListsTask::HasError() const
{
  return !error.empty();
}

std::wstring DownloadListsTask::GetError() const
{
  return Lum::Base::StringToWString(error);
}

