/*
 * This file is part of vncviewer.
 *
 * Copyright (C) 2005, 2006 Aaron Levinson.
 * Copyright (C) 2006, Detlef Schmicker
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rfb.h"
#include "deshash.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <byteswap.h>
#include <gtk/gtk.h>

#define G_MALLOC0 g_malloc

guint32 ReadBytes=0;

//#define GUINT16_FROM_BE(A) (bswap_16(A),A)
//#define GUINT16_TO_BE(A) (bswap_16(A),A)

static gboolean fullWrite(gint fd, gchar *buf, guint n) {
  guint i = 0;
  int j;
  
  while (i < n) {
    j = write(fd, buf + i, (n - i));
    if (j <= 0) {
      if (j < 0) {
	g_warning("write failed: %s", g_strerror(errno));
      } else {
	g_warning("write failed");
      }
      return FALSE;
    }
    i += j;
  }
  return TRUE;
}

static gboolean fullRead(gint fd, gchar *buf, gint n) {
  guint i = 0;
  int j;
  
  while (i < n) {
    j = read(fd, buf + i, (n - i));
    if (j <= 0) {
      if (j < 0) {
	g_warning("read failed: %s", g_strerror(errno));
      } else {
	g_warning("read failed");
      }
      return FALSE;
    }
    i += j;
  }
  /*if (debug)
    PrintInHex(buf,n);*/
  ReadBytes+=n;
  return TRUE;
}

gboolean sameMachine(int fd) {
  struct sockaddr_in peeraddr, myaddr;
  int addrlen = sizeof(struct sockaddr_in);
  
  getpeername(fd, (struct sockaddr *)&peeraddr, &addrlen);
  getsockname(fd, (struct sockaddr *)&myaddr, &addrlen);
  
  return (peeraddr.sin_addr.s_addr == myaddr.sin_addr.s_addr);
}

static int StringToIPAddr(const char *str, unsigned int *addr) {
  struct hostent *hp;
  
  if ((*addr = inet_addr(str)) == -1) {
    if (!(hp = gethostbyname(str)))
      return 0;
    
    *addr = *(unsigned int *)hp->h_addr;
  }

  return 1;
}

static int parseDisplayStr(const char *display, char **host, int *port) {
  gchar *tmp = g_strdup(display);
  gchar *striped = g_strstrip(tmp);
  gchar **parts;

  if (striped[0] == '\0') {
    g_free(tmp);
    return -1;
  }
  parts = g_strsplit(striped, ":", 2);
  g_free(tmp);

  *host = NULL;
  *port = 0;
  if (parts[0]) {
    *host = parts[0];
    if ((*host)[0] == '\0')
      *host = "localhost";
  } else {
    g_strfreev(parts);
    return -1;
  }

  if (parts[1]) {
    *port = strtol(parts[1], &striped, 0);
    if (parts[1] == striped) {
      g_strfreev(parts);
      return -1; /* conversion failed */
    }
  }

  *host = g_strdup(*host);
  if (*port < 100)
    *port += SERVERPORT;
  g_strfreev(parts);

  return 0;
}

int connectToServer(char *display) {
  char *str;
  unsigned int host;
  int port, sock;
  struct sockaddr_in addr;
  int one = 1;
  struct timeval timeout;

  if (parseDisplayStr(display, &str, &port) < 0) {
    g_warning("couldn't parse display string");
    return -1;
  }

  if (!StringToIPAddr(str, &host)) {
    g_warning("couldn't get IP address of host %s", str);
    return -1;
  }
  g_free(str);

  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);
  addr.sin_addr.s_addr = host;
  
  sock = socket(AF_INET, SOCK_STREAM, 0);
  timeout.tv_sec = 16;
  timeout.tv_usec = 0;
  setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout));
  setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout));
  if (sock < 0) {
    g_warning("socket() error: %s", g_strerror(errno));
    return -1;
  }

  if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
    g_warning("connect() error: %s", g_strerror(errno));
    close(sock);
    return -1;
  }

  if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY,
		 (char *)&one, sizeof(one)) < 0) {
    g_warning("setsockopt() error: %s", g_strerror(errno));
    close(sock);
    return -1;
  }
  
  return sock;
}

gint32 doVersionHandshake(gint fd) {
  rfbProtocolVersionMsg msg;
  gint major, minor;
  guint32 authScheme;

  msg[12] = '\0';

  if (!fullRead(fd, msg, sz_rfbProtocolVersionMsg))
    return -1;
  if (sscanf(msg, rfbProtocolVersionFormat, &major, &minor) != 2) {
    g_error("Invalid protocol version message: %s", msg);
    return -1;
  }
  /*if (major < rfbProtocolMajorVersion) {
    g_error("major protocol version too low");
    return -1;
    }*/
  major = rfbProtocolMajorVersion;
  minor = rfbProtocolMinorVersion;
  g_snprintf(msg, 13, rfbProtocolVersionFormat, major, minor);
  if (!fullWrite(fd, msg, sz_rfbProtocolVersionMsg))
    return -1;
  if (!fullRead(fd, (gchar *)&authScheme, 4))
    return -1;
  return GUINT32_FROM_BE(authScheme);
}

/* only call this if rfbConnFailed was returned by doVersionHandshake */
gchar *getFailReason(gint fd) {
  guint32 length;
  gchar *msg;

  if (!fullRead(fd, (gchar *)&length, 4))
    return NULL;
  msg = g_new(gchar, length + 1);
  if (!fullRead(fd, msg, length)) {
    g_free(msg);
    return NULL;
  }
  msg[length] = '\0';
  return msg;
}

/* only call this if rfbVncAuth was returned by doVersionHandshake */
gint32 performAuth(gint fd, gchar *passwd) {
  guchar challenge[16];
  guint32 response;

  if (!fullRead(fd, challenge, 16))
    return -1;
  vncEncryptBytes(challenge, passwd);
  if (!fullWrite(fd, challenge, 16))
    return -1;
  if (!fullRead(fd, (gchar *)&response, 4))
    return -1;
  return GUINT32_FROM_BE(response);
}

gint sendClientInitMsg(gint fd, gboolean shared) {
  rfbClientInitMsg msg;

  msg.shared = shared;
  if (!fullWrite(fd, (gchar *)&msg, sz_rfbClientInitMsg))
    return -1;
  return 0;
}

gint getServerInitMsg(gint fd, guint16 *width, guint16 *height,
		      rfbPixelFormat *format, gchar **name) {
  rfbServerInitMsg msg;
  gchar *desk_name;

  if (!fullRead(fd, (gchar *)&msg, sz_rfbServerInitMsg))
    return -1;

  /* byte swapping */
  msg.framebufferWidth  = GUINT16_FROM_BE(msg.framebufferWidth);
  msg.framebufferHeight = GUINT16_FROM_BE(msg.framebufferHeight);
  msg.format.redMax     = GUINT16_FROM_BE(msg.format.redMax);
  msg.format.greenMax   = GUINT16_FROM_BE(msg.format.greenMax);
  msg.format.blueMax    = GUINT16_FROM_BE(msg.format.blueMax);
  msg.nameLength        = GUINT32_FROM_BE(msg.nameLength);

  desk_name = g_new(gchar, msg.nameLength + 1);
  desk_name[msg.nameLength] = '\0';
  if (!fullRead(fd, desk_name, msg.nameLength)) {
    g_free(desk_name);
    return -1;
  }

  if (width)  *width  = msg.framebufferWidth;
  if (height) *height = msg.framebufferHeight;
  if (format) *format = msg.format;
  if (name)   *name   = desk_name;
  else        g_free(desk_name);

  return 0;
}

void rfb_send_and_destroy_message(gint fd, rfbClientToServerMsg *msg) {
  gint num, i;
  guint16 *ptr16;
  guint32 *ptr32;

  switch (msg->type) {
  case rfbSetPixelFormat:
    msg->spf.format.redMax = GUINT16_TO_BE(msg->spf.format.redMax);
    msg->spf.format.greenMax = GUINT16_TO_BE(msg->spf.format.greenMax);
    msg->spf.format.blueMax = GUINT16_TO_BE(msg->spf.format.blueMax);
    fullWrite(fd, (gchar *)msg, sz_rfbSetPixelFormatMsg);
    break;
  case rfbFixColourMapEntries:
    num = msg->fcme.nColours;
    msg->fcme.firstColour = GUINT16_TO_BE(msg->fcme.firstColour);
    msg->fcme.nColours = GUINT16_TO_BE(msg->fcme.nColours);
    ptr16 = (guint16 *)((gchar *)msg + sz_rfbFixColourMapEntriesMsg);
    for (i = 0; i < num * 3; i++)
      ptr16[i] = GUINT16_TO_BE(ptr16[i]);
    fullWrite(fd, (gchar *)msg, sz_rfbFixColourMapEntriesMsg + 6 * num);
    break;
  case rfbSetEncodings:
    num = msg->se.nEncodings;
    msg->se.nEncodings = GUINT16_TO_BE(msg->se.nEncodings);
    ptr32 = (guint32 *)((gchar *)msg + sz_rfbSetEncodingsMsg);
    for (i = 0; i < num; i++)
      ptr32[i] = GUINT32_TO_BE(ptr32[i]);
    fullWrite(fd, (gchar *)msg, sz_rfbSetEncodingsMsg + 4 * num);
    break;
  case rfbFramebufferUpdateRequest:
    msg->fur.x = GUINT16_TO_BE(msg->fur.x);
    msg->fur.y = GUINT16_TO_BE(msg->fur.y);
    msg->fur.w = GUINT16_TO_BE(msg->fur.w);
    msg->fur.h = GUINT16_TO_BE(msg->fur.h);
    fullWrite(fd, (gchar *)msg, sz_rfbFramebufferUpdateRequestMsg);
    break;
  case rfbKeyEvent:
    msg->ke.key = GUINT32_TO_BE(msg->ke.key);
    fullWrite(fd, (gchar *)msg, sz_rfbKeyEventMsg);
    break;
  case rfbPointerEvent:
    msg->pe.x = GUINT16_TO_BE(msg->pe.x);
    msg->pe.y = GUINT16_TO_BE(msg->pe.y);
    fullWrite(fd, (gchar *)msg, sz_rfbPointerEventMsg);
    break;
  case rfbClientCutText:
    num = msg->cct.length;
    msg->cct.length = GUINT32_TO_BE(msg->cct.length);
    fullWrite(fd, (gchar *)msg, sz_rfbClientCutTextMsg + num);
    break;
  default:
    g_warning("unknown client->server message type %d", (gint)msg->type);
    g_assert_not_reached();
  }
  g_free(msg);
}

rfbClientToServerMsg *newSetPixelFormatMsg(rfbPixelFormat *format) {
  rfbSetPixelFormatMsg *msg;

  msg = G_MALLOC0(sz_rfbSetPixelFormatMsg);
  msg->type = rfbSetPixelFormat;
  msg->format = *format;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newFixColourMapEntriesMsg(guint16 firstColour,
						guint16 nColours,
						guint16 *entries) {
  rfbFixColourMapEntriesMsg *msg;
  guint16 *ents;
  gint i;

  msg = G_MALLOC0(sz_rfbFixColourMapEntriesMsg + 3 * 2 * nColours);
  msg->type = rfbFixColourMapEntries;
  msg->firstColour = firstColour;
  msg->nColours = nColours;
  ents = (guint16 *)((gchar *)msg + sz_rfbFixColourMapEntriesMsg);
  for (i = 0; i < 3 * nColours; i++)
    *ents++ = *entries++;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newSetEncodingsMsg(guint16 nEncodings,
					 guint32 *encodings) {
  rfbSetEncodingsMsg *msg;
  guint32 *encs;
  gint i;

  msg = G_MALLOC0(sz_rfbSetEncodingsMsg + 4 * nEncodings);
  msg->type = rfbSetEncodings;
  msg->nEncodings = nEncodings;
  encs = (guint32 *)((gchar *)msg + sz_rfbSetEncodingsMsg);
  for (i = 0; i < nEncodings; i++)
    *encs++ = *encodings++;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newFramebufferUpdateRequestMsg(guint8 incremental,
						     guint16 x, guint16 y,
						     guint16 w, guint16 h) {
  rfbFramebufferUpdateRequestMsg *msg;

  msg = G_MALLOC0(sz_rfbFramebufferUpdateRequestMsg);
  msg->type = rfbFramebufferUpdateRequest;
  msg->incremental = incremental;
  msg->x = x; msg->y = y;
  msg->w = w; msg->h = h;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newKeyEventMsg(guint8 down, guint32 key) {
  rfbKeyEventMsg *msg;

  msg = G_MALLOC0(sz_rfbKeyEventMsg);
  msg->type = rfbKeyEvent;
  msg->down = down;
  msg->key = key;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newPointerEventMsg(guint8 buttonMask,
					 guint16 x, guint16 y) {
  rfbPointerEventMsg *msg;

  msg = G_MALLOC0(sz_rfbPointerEventMsg);
  msg->type = rfbPointerEvent;
  msg->buttonMask = buttonMask;
  msg->x = x; msg->y = y;
  return (rfbClientToServerMsg *)msg;
}

rfbClientToServerMsg *newClientCutTextMsg(guint32 length, gchar *text) {
  rfbClientCutTextMsg *msg;
  gchar *txt;
  gint i;

  msg = g_malloc(sz_rfbClientCutTextMsg + length);
  msg->type = rfbClientCutText;
  msg->length = length;
  txt = (gchar *)((gchar *)msg + sz_rfbClientCutTextMsg);
  for (i = 0; i < length; i++)
    *txt++ = *text++;
  return (rfbClientToServerMsg *)msg;
}

rfbServerToClientMsg *getServerMsg(gint fd) {
  rfbServerToClientMsg *msg;

  msg = g_new0(rfbServerToClientMsg, 1);
  if (!fullRead(fd, (char *) msg, 1))
  {
    g_free(msg);
    return NULL;
  }
  switch (msg->type) {
  case rfbFramebufferUpdate:
    if (!fullRead(fd, ((char *)msg) + 1, sz_rfbFramebufferUpdateMsg - 1))
    {
      g_free(msg);
      return NULL;
    }
    msg->fu.nRects = GUINT16_FROM_BE(msg->fu.nRects);
    
#ifdef DDEBUG
    g_message("read pad    %hu", msg->fu.pad);
    g_message("read Rects  %u", msg->fu.nRects);
	
    if (msg->fu.nRects == 0) 
    {
      char aa; 
      int aai;
      for (aai = 0; aai < 30; aai++)
      {
	fullRead(fd, &aa, 1);
	g_message("read %d %hu", aai, aa);
      }
    }
#endif

    /* handle rect structures */
    break;
  case rfbSetColourMapEntries:
    if (!fullRead(fd, ((char *) msg) + 1, sz_rfbSetColourMapEntriesMsg - 1))
    {
      g_free(msg);
      return NULL;
    }
    msg->scme.firstColour = GUINT16_FROM_BE(msg->scme.firstColour);
    msg->scme.nColours = GUINT16_FROM_BE(msg->scme.nColours);
    break;
  case rfbBell:
    /* bell has no extra args */
    break;
  case rfbServerCutText:
    if (!fullRead(fd, ((char *) msg) + 1, sz_rfbServerCutTextMsg - 1))
    {
      g_free(msg);
      return NULL;
    }
    msg->sct.length = GUINT32_FROM_BE(msg->sct.length);
    break;
  default:
    g_assert_not_reached();
    g_free(msg);
    return NULL;
  }
  return msg;
}

static inline guint32 readPixel(gint fd, gint bpp) {
  guint32 pix32;
  guint16 pix16;
  guint8 pix8;

  switch (bpp) {
  case 1:
    if (!fullRead(fd, (gchar *)&pix8, 1))
      return 0;
    return pix8;
  case 2:
    if (!fullRead(fd, (gchar *)&pix16, 2))
      return 0;
    return pix16;
    // not necessary as little endian ask from server
    //    return GUINT16_FROM_BE(pix16);
  case 4:
    if (!fullRead(fd, (gchar *)&pix32, 4))
      return 0;
    return pix32;
    // not necessary as little endian ask from server
    //    return GUINT32_FROM_BE(pix32);
  default:
    g_assert_not_reached();
  }
  return 0;
}

// Must be kept between subrects because not guaranteed to be sent for each tile
// if same as for previous tile
guint32 bgcolour = 0, fgcolour = 0;
// rfbRectangleDescription *rdesc_max = NULL;
	  
/* bpp is *bytes* per pixel */
rfbRectangleDescription *getNextRectangle(gint fd, gint bpp,
					  gboolean bMessages)
{
  rfbFramebufferUpdateRectHeader header;
  rfbRectangleDescription *rdesc = NULL;
  guint32 nSubrects, i;

  if (!fullRead(fd, (gchar *)&header, sz_rfbFramebufferUpdateRectHeader))
    return NULL;
  header.r.x      = GUINT16_FROM_BE(header.r.x);
  header.r.y      = GUINT16_FROM_BE(header.r.y);
  header.r.w      = GUINT16_FROM_BE(header.r.w);
  header.r.h      = GUINT16_FROM_BE(header.r.h);
  header.encoding = GUINT32_FROM_BE(header.encoding);

  if (bMessages == TRUE)
    g_message("Rect encoding = %u.", header.encoding);

  switch (header.encoding) {
  case rfbEncodingRaw:
    rdesc = G_MALLOC0(sz_rfbFramebufferUpdateRectHeader +
		     header.r.w * header.r.h * bpp);
    rdesc->header = header;
    if (!fullRead(fd, (gchar *)rdesc->data.raw, header.r.w*header.r.h*bpp)) {
      g_free(rdesc);
      return NULL;
    }
    //Not necessary as little endian is ask from server
    /*
    if (bpp == 2) {
      guint16 *data = (guint16 *)rdesc->data.raw;
      for (i = 0; i < header.r.w*header.r.h; i++)
	data[i] = GUINT16_FROM_BE(data[i]);
    } else if (bpp == 4) {
      guint32 *data = (guint32 *)rdesc->data.raw;
      for (i = 0; i < header.r.w*header.r.h; i++)
	data[i] = GUINT32_FROM_BE(data[i]);
    }
    */

    break;
  case rfbEncodingCopyRect:
    rdesc = G_MALLOC0(sz_rfbFramebufferUpdateRectHeader + sz_rfbCopyRect);
    rdesc->header = header;
    if (!fullRead(fd, (gchar *)&(rdesc->data.cr), sz_rfbCopyRect)) {
      g_free(rdesc);
      return NULL;
    }
    rdesc->data.cr.srcX = GUINT16_FROM_BE(rdesc->data.cr.srcX);
    rdesc->data.cr.srcY = GUINT16_FROM_BE(rdesc->data.cr.srcY);
    break;
  case rfbEncodingRRE:
    if (!fullRead(fd, (gchar *)&nSubrects, 4))
      return NULL;
    nSubrects = GUINT32_FROM_BE(nSubrects);
    rdesc = G_MALLOC0(sz_rfbFramebufferUpdateRectHeader + 8 +
		      nSubrects * (4 + sz_rfbRectangle));
    rdesc->header = header;
    rdesc->data.rre.nSubrects = nSubrects;
    rdesc->data.rre.bgPixel = readPixel(fd, bpp);
    for (i = 0; i < nSubrects; i++) {
      rdesc->data.rre.rects[i].pixel = readPixel(fd, bpp);
      if (!fullRead(fd, (gchar *)&(rdesc->data.rre.rects[i].rect),
		    sz_rfbRectangle)) {
	g_free(rdesc);
	return NULL;
      }
      rdesc->data.rre.rects[i].rect.x =
	GUINT16_FROM_BE(rdesc->data.rre.rects[i].rect.x);
      rdesc->data.rre.rects[i].rect.y =
	GUINT16_FROM_BE(rdesc->data.rre.rects[i].rect.y);
      rdesc->data.rre.rects[i].rect.w =
	GUINT16_FROM_BE(rdesc->data.rre.rects[i].rect.w);
      rdesc->data.rre.rects[i].rect.h =
	GUINT16_FROM_BE(rdesc->data.rre.rects[i].rect.h);
    }
    break;
  case rfbEncodingCoRRE:
    if (!fullRead(fd, (gchar *)&nSubrects, 4))
      return NULL;
    nSubrects = GUINT32_FROM_BE(nSubrects);
    rdesc = G_MALLOC0(sz_rfbFramebufferUpdateRectHeader + 8 +
		      nSubrects * (4 + sz_rfbCoRRERectangle));
    rdesc->header = header;
    rdesc->data.corre.nSubrects = nSubrects;
    rdesc->data.corre.bgPixel = readPixel(fd, bpp);
    for (i = 0; i < nSubrects; i++) {
      rdesc->data.corre.rects[i].pixel = readPixel(fd, bpp);
      if (!fullRead(fd, (gchar *)&(rdesc->data.corre.rects[i].rect),
		    sz_rfbCoRRERectangle)) {
	g_free(rdesc);
	return NULL;
      }
    }
    break;
  case rfbEncodingHextile:
    {
      guint16 width = header.r.w, height = header.r.h;
      guint16 tilesPerRow, tilesPerCol;
      guint32 nTiles, tile, tile2;

      if (width  & 0x0f) width  += 0x10 - (width  & 0x0f);
      if (height & 0x0f) height += 0x10 - (height & 0x0f);
      tilesPerRow = width  >> 4;
      tilesPerCol = height >> 4;
      nTiles = tilesPerRow * tilesPerCol;
      //      if (rdesc_max==NULL) {
      //	  rdesc_max=G_MALLOC0(sz_rfbFramebufferUpdateRectHeader + 4 +
      //			nTiles * sizeof(rfbHextile *));
      //	  for (tile=0;tile < nTiles;tile++) {
      //		rdesc_max->data.hextile.tiles[tile] = G_MALLOC0(4+16*16*3*2); //*2 for security
      //		}
      //	}
      rdesc = G_MALLOC0(sz_rfbFramebufferUpdateRectHeader + 4 + nTiles * sizeof(rfbHextile *));
      rdesc->header = header;
      rdesc->data.hextile.nTiles = nTiles;
      GTimer * Trespond=g_timer_new();;
      for (tile = 0; tile < nTiles; tile++)
      {
	guint8 tWidth = 16, tHeight = 16;
	guint8 type;
	rfbHextile *ht;

	// increase responsiveness
	// Network transfer may take very long, if its more than 0.3 seconds
	// in between events of GUI are handled
	if (g_timer_elapsed(Trespond,NULL) > 0.3)
	{
	  gtk_main_iteration_do(FALSE);
	  g_timer_start(Trespond);
	}

	if (tile % tilesPerRow == tilesPerRow - 1) tWidth  = header.r.w & 0x0f;
	if (tile / tilesPerRow == tilesPerCol - 1) tHeight = header.r.h & 0x0f;
	//Detlef	
	if (tWidth==0) tWidth=16;
	if (tHeight==0) tHeight=16;

	if (!fullRead(fd, (gchar *)&type, 1)) {
	  for (tile2 = 0; tile2 < tile; tile2++)
	    g_free(rdesc->data.hextile.tiles[tile2]);
	  g_free(rdesc);
	  g_timer_destroy(Trespond);
	  return NULL;
	}
	if (type & rfbHextileRaw) {
	  // handle the raw type separate from the other types
	  ht=rdesc->data.hextile.tiles[tile] = G_MALLOC0(4+tWidth*tHeight*bpp);
	  ht->width = tWidth;
	  ht->height = tHeight;
	  ht->type = type;
          if (!fullRead(fd, ht->data.raw, tWidth * tHeight * bpp)) {
	    for (tile2 = 0; tile2 <= tile; tile2++)
	      g_free(rdesc->data.hextile.tiles[tile2]);
	    g_free(rdesc);
	    g_timer_destroy(Trespond);
	    return NULL;
	  }
/* Not necessary, as it is optimized (look raw encoding above)
	  if (bpp == 2) {
	    guint16 *data = (guint16 *)ht->data.raw;
	    for (i = 0; i < tWidth*tHeight; i++)
	      data[i] = GUINT16_FROM_BE(data[i]);
	  } else if (bpp == 4) {
	    guint32 *data = (guint32 *)ht->data.raw;
	    for (i = 0; i < tWidth*tHeight; i++)
	      data[i] = GUINT32_FROM_BE(data[i]);
	  }
*/
	}
	else
	{
	  guint8 nRects = 0, i;

	  /* rather than reading each item separately, do all at once (more efficient)
	  if (type & rfbHextileBackgroundSpecified)
	    bgcolour = readPixel(fd, bpp);
	  if (type & rfbHextileForegroundSpecified)
	    fgcolour = readPixel(fd, bpp);
	  if (type & rfbHextileAnySubrects)
	  {
	    if (!fullRead(fd, (gchar *)&nRects, 1)) {
	      for (tile2 = 0; tile2 < tile; tile2++)
		g_free(rdesc->data.hextile.tiles[tile2]);
	      g_free(rdesc);
	      return NULL;
	    }
	  }
	  */

	  guint8 bytes=0, bb[5];

	  // determine how many bytes to read, depending on which types
	  // are specified
	  if (type & rfbHextileBackgroundSpecified) bytes+=bpp;
	  if (type & rfbHextileForegroundSpecified) bytes+=bpp;
	  if (type & rfbHextileAnySubrects) bytes++;

	  // if only supporting a maximum of 2 byte pixels, then the largest
	  // size that could be read is five bytes (2 bytes for foreground and
	  // background each, 1 byte for number of sub-rectangles)
	  // only 8 and 16 bit support in hextile
	  
	  // read the proper number of bytes to get the information
	  // for each type that was specified
	  if (bytes>0 && !fullRead(fd, (gchar *)bb, bytes) )
	  {
	    for (tile2 = 0; tile2 < tile; tile2++)
	      g_free(rdesc->data.hextile.tiles[tile2]);
	    g_free(rdesc);
	    g_timer_destroy(Trespond);
	    return NULL;
	  }

	  bytes=0;

	  if (bpp==1)
	  {
	    // 1 byte per pixel
	    if (type & rfbHextileBackgroundSpecified)
	    {
	      bgcolour = *(guint8*)&bb[0];
	      bytes++;
	    }

	    if (type & rfbHextileForegroundSpecified)
	    {
	      fgcolour = *(guint8*)&bb[bytes];
	      bytes++;
	    }
	  }
	  else
	  {
	    // 2 bytes per pixel
	    if (type & rfbHextileBackgroundSpecified)
	    {
	      bgcolour = *(guint16*)&bb[0];
	      bytes+=2;
	    }

	    if (type & rfbHextileForegroundSpecified)
	    {
	      fgcolour = *(guint16*)&bb[bytes];
	      bytes+=2;
	    }

	  }

	  if (type & rfbHextileAnySubrects)
	  {
	    // number of rectangles stored as 8-bit quantity
	    nRects = *(guint8*)&bb[bytes];
	  }

	  // g_message(" bytes %d nrects %d",bytes,nRects);

	  // even if the number of subrectangles is 0, still want to allocate
	  // the memory

	  guint32 nbytes = 0;
	  if (nRects)
	  {
	    // then will need to read at least 2 bytes (the x-and-y position and
	    // the width-and-height--each are 8-bit quantities)
	    bytes=2;

	    if (type & rfbHextileSubrectsColoured)
	      // then also need to read pixel value giving color of the sub-rectangle
	      bytes+=bpp;

	    // number of bytes to read in total is number of subrectangles times
	    // number of bytes per subrectangle
	    nbytes = nRects * bytes; 
	  }
	  else
	  {
	    // only setting bytes for the g_message below
	    // bytes = 0;
	  }

	  //g_message("bpp %d bytes %d nRects %d",bpp,nbytes,nRects);

	  // Aaron:  Seems like there should be a better way than having to
	  // always allocate each time.  As a single rectangle description will
	  // only be in use at any one time, it should be possible to reuse it.
	  // It looks like you may have started implementing this with rdesc_max.
	  // However, that could be difficult to implement properly for all
	  // encoding types.
	  // Rather, you might have a different rdesk_max for each encoding type
	  // (but only create it as needed).

	  // Detlef: The malloc was tested to be of no performance effect, but there
	  // may be problems in case of reentrance. (should not happen)
	  // Therefore I felt safe to do it like this and not having a global variable

	  // The size of the rfbHextile is determined as follows (one rfbHextile * per tile):
	  // 4 bytes for the header (width and height, 1 byte each, type is 2 bytes)
	  // 12 bytes for second header (foreground and background, 4 bytes each,
	  // number of rectangles is 1 byte)
	  // And nbytes for all of the subrectangles' data, if any
	  // Aaron:  Detlef, if we are only supporting 2-byte pixels, then
	  // bgcolour and fgcolour only need to be of type guint16s, instead of guint32.
	  // However, we probably want the size to add up such that it is a multiple of
	  // 8 for better performance, so it might be better to leave at 32 bits.

	  ht=rdesc->data.hextile.tiles[tile] = G_MALLOC0(4 + 12 + nbytes);
	  ht->width = tWidth;
	  ht->height = tHeight;
	  ht->type = type;
	  ht->data.rects.bgcolour = bgcolour;
	  ht->data.rects.fgcolour = fgcolour;
	  ht->data.rects.nRects = nRects;
	  if (!fullRead(fd, (gchar*)ht->data.rects.rects, nbytes)) {
	    for (tile2 = 0; tile2 <= tile; tile2++)
	      g_free(rdesc->data.hextile.tiles[tile2]);
	    g_free(rdesc);
	    g_timer_destroy(Trespond);
	    return NULL;
	  }

	  /*
	  for (i = 0; i < nRects; i++) {
	    guint8 pack[2];
	    if (type & rfbHextileSubrectsColoured)
	      ht->data.rects.rects[i].colour = readPixel(fd, bpp);
	    else
	      ht->data.rects.rects[i].colour = ht->data.rects.fgcolour;
	    if (!fullRead(fd, (gchar *)pack, 2)) {
	      for (tile2 = 0; tile2 <= tile; tile2++)
		g_free(rdesc->data.hextile.tiles[tile2]);
	      g_free(rdesc);
	      return NULL;
	    }
	    ht->data.rects.rects[i].x = rfbHextileExtractX(pack[0]);
	    ht->data.rects.rects[i].y = rfbHextileExtractY(pack[0]);
	    ht->data.rects.rects[i].w = rfbHextileExtractW(pack[1]);
	    ht->data.rects.rects[i].h = rfbHextileExtractH(pack[1]);
	  }
	  */
	}
      }
      g_timer_destroy(Trespond);
    }

    break;
  default:
    if (bMessages == TRUE)
      g_message("Got unrecognized encoding type %u!", header.encoding);

    //g_assert_not_reached();
    return NULL;
  }

  return rdesc;
}

void freeRectangleDescription(rfbRectangleDescription *rd) {
  if (rd->header.encoding == rfbEncodingHextile) {
    guint32 i;

    for (i = 0; i < rd->data.hextile.nTiles; i++)
      g_free(rd->data.hextile.tiles[i]);
  }

  g_free(rd);
}

guint16 *getColourMapEntries(gint fd, guint16 nColours) {
  guint16 *colours;
  guint i;

  colours = g_malloc(nColours * 3 * 2);
  if (!fullRead(fd, (gchar *)colours, nColours * 3 * 2)) {
    g_free(colours);
    return NULL;
  }
  for (i = 0; i < nColours * 3; i++)
    colours[i] = GUINT16_FROM_BE(colours[i]);
  return colours;
}

gchar *getServerCutText(gint fd, guint32 length) {
  gchar *text;

  text = g_malloc(length + 1);
  text[length] = '\0';
  if (!fullRead(fd, text, length)) {
    g_free(text);
    return NULL;
  }
  return text;
}
