/*
  tiltstick_mass.c

  simple sdl tiltstick demo
*/

#include <stdlib.h>
#include <SDL/SDL.h>
#include "SDL_gfxPrimitives.h"

#undef FULLSCREEN

#ifdef EVENT_INTERFACE
#include <tiltstick.h>
#endif

#ifdef OSSO
#include <libosso.h>      /* required for screen saver timeout */
osso_context_t *osso_context = NULL;
#endif

#ifdef FULLSCREEN
#define WIDTH  800
#define HEIGHT 480
#define MODE   SDL_SWSURFACE|SDL_FULLSCREEN
#else
#define WIDTH  720
#define HEIGHT 420
#define MODE   SDL_SWSURFACE
#endif

#define SCALE  HEIGHT   /* whatever is smaller ... */

/* points where the rubber bands are attached */
float rubber_band[][2] = {
  { 0.0, -0.5 }, { 0.0, 0.5 }
};

float mass_pos[2] = { 0.0, 0.0 };
float mass_move[2] = { 0.0, 0.0 };

#define MASS        0.2
#define DAMP        0.1
#define RUBBER_BAND 1.0

void add_forces(float x, float y) {
  int i;

  /* each force makes the mass accelerate */

  for(i=0;i<sizeof(rubber_band)/sizeof(rubber_band[0]);i++) {
    mass_move[0] += RUBBER_BAND * MASS * (rubber_band[i][0] - mass_pos[0]);
    mass_move[1] += RUBBER_BAND * MASS * (rubber_band[i][1] - mass_pos[1]);
  }

  mass_move[0] += MASS * x;
  mass_move[1] += MASS * y;

  mass_pos[0] += mass_move[0];
  mass_pos[1] += mass_move[1];

  mass_move[0] *= (1-DAMP);
  mass_move[1] *= (1-DAMP);
}

void update_screen(SDL_Surface *screen) {
  int i;

  SDL_FillRect(screen, NULL, 0); 

  for(i=0;i<sizeof(rubber_band)/sizeof(rubber_band[0]);i++) {
    lineRGBA(screen, 
	     WIDTH/2+SCALE*rubber_band[i][0], HEIGHT/2+SCALE*rubber_band[i][1], 
	     WIDTH/2+SCALE*mass_pos[0], HEIGHT/2+SCALE*mass_pos[1], 
	     255, 255, 0, 255);
  }
    
  filledCircleRGBA(screen,
		   WIDTH/2+SCALE*mass_pos[0], HEIGHT/2+SCALE*mass_pos[1], 
		   0.05*SCALE, 255, 0, 0, 255);

  stringRGBA(screen, 10, 10, "Using device:", 0, 0, 255, 255);
#ifndef EVENT_INTERFACE
  stringRGBA(screen, 10, 18, SDL_JoystickName(0), 0, 0, 255, 255);
#else
  stringRGBA(screen, 10, 18, tiltstick_get_name(), 0, 0, 255, 255);
#endif

  SDL_UpdateRect(screen, 0, 0, 0, 0);
}

main(int argc, char *argv[]) {
  SDL_Surface *screen;
  int quit = 0;

#ifndef EVENT_INTERFACE
  SDL_Joystick *stick;
#else
  int stick, i;
#endif

#ifdef OSSO
  putenv("SDL_VIDEO_X11_WMCLASS=" PROG);
#endif

  /* init SDL library */
  if ( SDL_Init(SDL_INIT_AUDIO|SDL_INIT_VIDEO) < 0 ) {
    fprintf(stderr, "Unable to initialize SDL: %s\n", 
	    SDL_GetError());
    exit(1);
  }
  atexit(SDL_Quit);
  
  screen = SDL_SetVideoMode(WIDTH, HEIGHT, 16, MODE);
  if ( screen == NULL ) {
    fprintf(stderr, "Error opening window: %s\n", SDL_GetError());
    exit(1);
  }

  SDL_WM_SetCaption(PROG, NULL);

#ifdef OSSO
  SDL_ShowCursor(0);  // hide mouse cursor on maemo

  osso_context = osso_initialize (PROG, "1.0", 0, NULL);
  if(osso_context == NULL) {
      fprintf(stderr, "error initiating osso context\n");
  }
#endif

#ifndef EVENT_INTERFACE
  SDL_InitSubSystem (SDL_INIT_JOYSTICK);
  printf("Found %d joysticks\n", SDL_NumJoysticks());

  /* get access to joystick which is hopefully a tiltstick */
  /* but we actually don't care */
  stick = SDL_JoystickOpen(0);
  if(stick == NULL) {
    fprintf(stderr, "Unable to initialize joystick: %s\n", 
	    SDL_GetError());
    exit(1);
  }

  printf("Found joystick \"%s\":\n"
	 "  %i axis, %i buttons, %i hats, %i balls\n",
	 SDL_JoystickName(0),
	 SDL_JoystickNumAxes(stick),
	 SDL_JoystickNumButtons(stick),
	 SDL_JoystickNumHats(stick),
	 SDL_JoystickNumBalls(stick));
#else
  if((tiltstick_evif_open()) < 0) {
    fprintf(stderr, "Unable to access joystick via event interface\n");
    exit(1);
  }
#endif
  
  while(!quit) {
    float x, y;

    SDL_Event event;

#ifdef OSSO
    if (osso_display_blanking_pause(osso_context) != OSSO_OK) {
      fprintf(stderr, "error with display blank\n");
    }
#endif

#ifdef EVENT_INTERFACE
    int num=2, axis[2], b;
    tiltstick_evif_poll(axis, &num, &b);
    /* no need to check for num as 2 axes are always supported */
    x = (float)axis[0]/512.0;
    y = (float)axis[1]/512.0;

    /* exit on any button */
    if(b) quit = 1;
#endif

    while(SDL_PollEvent(&event)) {

      switch(event.type) {

#ifndef EVENT_INTERFACE
      case SDL_JOYAXISMOTION:
	if(event.jaxis.axis == 0)
	  x = (float)event.jaxis.value / 32768.0;

	if(event.jaxis.axis == 1)
	  y = (float)event.jaxis.value / 32768.0;

	break;
#endif

      case SDL_JOYBUTTONDOWN:
      case SDL_KEYDOWN:
      case SDL_QUIT:
	quit = 1;
	break;
      }
    }

    add_forces(x,y);
    update_screen(screen);
    SDL_Delay(1000/25);  // 25 frames per second
  }

#ifdef OSSO

  if (osso_context)
    osso_deinitialize(osso_context);
#endif

#ifdef EVENT_INTERFACE
  close(stick);
#endif
}
