/* roadgps_main.c - The main function of the RoadGPS application.
 *  
 *     -Split off from roadmap_main.c at 1.1.0f to make dbus integration possible for maemo...
 *      Eventually, will use dbus instead of spawn to launch roadgps?
 *
 * Copyright 2008 Charles Werbick
 * This file is part of Roadmap for Maemo (among others, see below ;-)
 *
 * LICENSE:
 *
 *   Copyright 2002 Pascal F. Martin
 *
 *   This file is part of RoadMap.
 *
 *   RoadMap is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   RoadMap is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with RoadMap; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * SYNOPSYS:
 *
 *   int main (int argc, char **argv);
 */

#include <stdlib.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <locale.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkmain.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-window.h>
#include <hildon/hildon-banner.h>
#include <libosso.h>

#include <glib.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "roadmap.h"
#include "roadmap_path.h"
#include "roadmap_start.h"
#include "roadmap_config.h"
#include "roadmap_history.h"
#include "roadmap_gtkcanvas.h"
#include "roadmap_gtkmain.h"

#include "roadmap_main.h"
#include "roadmap_time.h"
#include "appdata.h"
#include "state.h"

struct AppData appdata;
struct StateData statedata;

#define OSSO_NAME    "roadgps"
#define OSSO_VERSION    "1.1.0"

void topmost_event_handler(const char *args, gpointer data)
{

};


void hw_event_handler(osso_hw_state_t *state, gpointer data)
{

    if (state->shutdown_ind)
    {
            state_save();
            roadmap_start_exit();
    
    }
    if (state->memory_low_ind)
    {
        hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Memory low event!");
    }
    if (state->save_unsaved_data_ind)
    {
        hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Must save unsaved data event!");
    }
    if (state->system_inactivity_ind)
    {
        hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Minimize application inactivity event!");
    }
}


osso_context_t * osso_init ()
{

    osso_context_t *osso = osso_initialize ( OSSO_NAME, OSSO_VERSION, FALSE, NULL );

    if (OSSO_OK != osso_application_set_top_cb(osso, topmost_event_handler, NULL))
        return;

    if (OSSO_OK != osso_hw_set_event_cb(osso, NULL, hw_event_handler, NULL))
        return;

    return osso;
}
/* Callback for normal D-BUS messages */
gint dbus_callback (const gchar *interface, const gchar *method,
	       GArray *arguments, gpointer data,
	       osso_rpc_t *retval)
{
//  printf ("roadmap dbus: %s, %s\n", interface, method);

  if (!strcmp (method, "top_application"))
      gtk_window_present (GTK_WINDOW (data));

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}



/* Callback for exit D-BUS event */
void exit_event_handler(gboolean die_now)
{

    /* Do whatever application needs to do before exiting */ 
    roadmap_start_exit();
};

static void ui_main_topmost_cb(GObject *self, GParamSpec *property_param, gpointer null)
{
       HildonProgram *program = HILDON_PROGRAM(self);

       if (program == NULL) return;

       if (hildon_program_get_is_topmost(program)) {
               hildon_program_set_can_hibernate(program, FALSE);
       }/* else {
               if (state.game == 1 && global.game!=NULL) {
                       game_1_player_save(global.game);
                       state.loadmap=1;
               }
               state_save();
               hildon_program_set_can_hibernate(program, TRUE);
       }*/
}
//end of osso/hildon functions

struct roadmap_main_io {
   int id;
   RoadMapIO io;
   RoadMapInput callback;
};

#define ROADMAP_MAX_IO 16
static struct roadmap_main_io RoadMapMainIo[ROADMAP_MAX_IO];


struct roadmap_main_timer {
   guint id;
   RoadMapCallback callback;
};

#define ROADMAP_MAX_TIMER 16
static struct roadmap_main_timer RoadMapMainPeriodicTimer[ROADMAP_MAX_TIMER];


static char *RoadMapMainTitle = NULL;

static RoadMapKeyInput RoadMapMainInput = NULL;
static GtkWidget      *RoadMapMainWindow  = NULL;
static GtkWidget      *RoadMapMainBox     = NULL;
static GtkWidget      *RoadMapCanvasBox   = NULL;
static GtkWidget      *RoadMapMainMenuBar = NULL;
static GtkWidget      *RoadMapMainToolbar = NULL;
static GtkWidget      *RoadMapMainStatus  = NULL;


static int GtkIconsInitialized = 0;


#ifdef ROADMAP_USES_GPE

static struct gpe_icon RoadMapGpeIcons[] = {
   {"rm_destination", NULL},
   {"rm_location", NULL},
   {"rm_gps", NULL},
   {"rm_hold", NULL},
   {"rm_counterclockwise", NULL},
   {"rm_clockwise", NULL},
   {"rm_zoomin", NULL},
   {"rm_zoomout", NULL},
   {"rm_zoom1", NULL},
   {"rm_up", NULL},
   {"rm_left", NULL},
   {"rm_right", NULL},
   {"rm_down", NULL},
   {"rm_full", NULL},
   {NULL, NULL}
};


static GtkWidget *roadmap_main_toolbar_icon (const char *icon) {

   if (icon == NULL) return NULL;

   if (! GtkIconsInitialized) {
      if (gpe_load_icons (RoadMapGpeIcons) == FALSE) return NULL;
      GtkIconsInitialized = 1;
   }
   return gpe_render_icon (RoadMapMainWindow->style, gpe_find_icon (icon));
}

#else // Not GPE, i.e. standard GTK.

static GtkWidget *roadmap_main_toolbar_icon (const char *icon) {

   if (icon != NULL) {

      const char *icon_file = roadmap_path_search_icon (icon);

      if (icon_file != NULL) {
         GtkIconsInitialized = 1;
         return gtk_image_new_from_file (icon_file);
      }
   }
   return NULL;
}
#endif // ROADMAP_USES_GPE


static void roadmap_main_close (GtkWidget *widget,
                                GdkEvent *event,
                                gpointer data) {

   roadmap_main_exit ();
}


static void roadmap_main_activate (GtkWidget *widget, gpointer data) {

   if (data != NULL) {
      (* (RoadMapCallback) data) ();
   }
}


static gint roadmap_main_key_pressed (GtkWidget *w, GdkEventKey *event) {

   char *key = NULL;
   char regular_key[2];


   switch (event->keyval) {

      case HILDON_HARDKEY_LEFT:   key = "Button-Left";           break;
      case HILDON_HARDKEY_RIGHT:  key = "Button-Right";          break;
      case HILDON_HARDKEY_UP:     key = "Button-Up";             break;
      case HILDON_HARDKEY_DOWN:   key = "Button-Down";           break;
      case GDK_F6:                key = "F6";                    break;
      case GDK_Return:            key = "Enter";                 break;
      case GDK_F7:                key = "F7";                    break;
      case GDK_F8:                key = "F8";                    break;

      /* These binding are for the iPAQ buttons: */
      case 0x1008ff1a: key = "Button-Menu";           break;
      case 0x1008ff20: key = "Button-Calendar";       break;
      case 0xaf9:      key = "Button-Contact";        break;
      case 0xff67:     key = "Button-Start";          break;

      /* Regular keyboard keys: */
      default:

         if (event->keyval > 0 && event->keyval < 128) {

            regular_key[0] = event->keyval;
            regular_key[1] = 0;
            key = regular_key;
         }
         break;
   }

   if (key != NULL && RoadMapMainInput != NULL) {
      (*RoadMapMainInput) (key);
   }

   return FALSE;
}


void roadmap_main_set_window_size (GtkWidget *w, int width, int height) {

   int screen_width  = gdk_screen_width();
   int screen_height = gdk_screen_height();

   if (screen_width <= width - 10 || screen_height <= height - 40) {

      /* Small screen: take it all (almost: keep room for the wm). */

      gtk_window_resize (GTK_WINDOW(w), screen_width-10, screen_height-40);

   } else {

      gtk_window_resize (GTK_WINDOW(w), width, height);
   }  
}


void roadmap_main_toggle_full_screen (void) {

   static int RoadMapIsFullScreen = 0;

   if (RoadMapIsFullScreen) {
      gtk_window_unfullscreen (GTK_WINDOW(RoadMapMainWindow));
      RoadMapIsFullScreen = 0;
   } else {
      gtk_window_fullscreen (GTK_WINDOW(RoadMapMainWindow));
      RoadMapIsFullScreen = 1;
   }
}

void roadmap_main_new (const char *title, int width, int height) {
   
		
   if (RoadMapMainBox == NULL) {


      gtk_widget_set_events (RoadMapMainWindow, GDK_KEY_PRESS_MASK);

      roadmap_main_set_window_size (RoadMapMainWindow, width, height);

      g_signal_connect (RoadMapMainWindow, "destroy_event",
                        G_CALLBACK(roadmap_main_close),
                        RoadMapMainWindow);

      g_signal_connect (RoadMapMainWindow, "delete_event",
                        G_CALLBACK(roadmap_main_close),
                        NULL);

      g_signal_connect (RoadMapMainWindow, "key_press_event",
                        G_CALLBACK(roadmap_main_key_pressed),
                        NULL);

      gtk_widget_set_events (RoadMapMainWindow, GDK_KEY_PRESS_MASK);

      RoadMapCanvasBox = RoadMapMainBox = gtk_vbox_new (FALSE, 0);
      gtk_container_add (GTK_CONTAINER(RoadMapMainWindow), RoadMapMainBox);
   }
   //g_set_application_name(title);


   if (RoadMapMainTitle != NULL) {
      free(RoadMapMainTitle);
   }
   RoadMapMainTitle = strdup (title);
}

void roadmap_main_title(char *fmt, ...) {

   char newtitle[200];
   va_list ap;
   int n;

   n = snprintf(newtitle, 200, "%s", RoadMapMainTitle);
   va_start(ap, fmt);
   vsnprintf(&newtitle[n], 200 - n, fmt, ap);
   va_end(ap);

   gtk_window_set_title (GTK_WINDOW(RoadMapMainWindow), newtitle);
}


void roadmap_main_set_keyboard (RoadMapKeyInput callback) {
   RoadMapMainInput = callback;
}


RoadMapMenu roadmap_main_new_menu (const char *title) {

   return (RoadMapMenu) gtk_menu_new ();
}


void roadmap_main_free_menu (RoadMapMenu menu) {

   gtk_widget_destroy ((GtkWidget *)menu);
}


void roadmap_main_add_menu (RoadMapMenu menu, const char *label) {

   GtkWidget *menu_item;

   if (RoadMapMainMenuBar == NULL) {

      RoadMapMainMenuBar = gtk_menu_new();

   }

   menu_item = gtk_menu_item_new_with_label (label);
   gtk_menu_shell_append (GTK_MENU_SHELL(RoadMapMainMenuBar), menu_item);

   gtk_menu_item_set_submenu (GTK_MENU_ITEM(menu_item), (GtkWidget *) menu);
}


void roadmap_main_popup_menu (RoadMapMenu menu,
                              const RoadMapGuiPoint *position) {

   if (menu != NULL) {
      gtk_menu_popup (GTK_MENU(menu),
                      NULL,
                      NULL,
                      NULL,
                      NULL,
                      0,
                      gtk_get_current_event_time());
   }
}


void roadmap_main_add_menu_item (RoadMapMenu menu,
                                 const char *label,
                                 const char *tip,
                                 RoadMapCallback callback) {

   GtkWidget *menu_item;

   if (label != NULL) {

      menu_item = gtk_menu_item_new_with_label (label);
      g_signal_connect (menu_item, "activate",
                        GTK_SIGNAL_FUNC(roadmap_main_activate),
                        callback);
   } else {
      menu_item = gtk_menu_item_new ();
   }
   gtk_menu_shell_append (GTK_MENU_SHELL(menu), menu_item);
   gtk_widget_show(menu_item);

   if (tip != NULL) {
      gtk_tooltips_set_tip (gtk_tooltips_new (), menu_item, tip, NULL);
   }
}


void roadmap_main_add_separator (RoadMapMenu menu) {

   roadmap_main_add_menu_item (menu, NULL, NULL, NULL);
}


void roadmap_main_add_toolbar () {

   if (RoadMapMainToolbar == NULL) {

      RoadMapMainToolbar = gtk_toolbar_new ();
      gtk_toolbar_set_orientation( GTK_TOOLBAR(RoadMapMainToolbar), GTK_ORIENTATION_HORIZONTAL);
      gtk_toolbar_set_style( GTK_TOOLBAR(RoadMapMainToolbar), GTK_TOOLBAR_BOTH_HORIZ);
      
   }

}

void roadmap_main_add_tool (const char *label,
                            const char *icon,
                            const char *tip,
                            RoadMapCallback callback) {

   if (RoadMapMainToolbar == NULL) {
      roadmap_main_add_toolbar ("");
   }

   gtk_toolbar_append_item (GTK_TOOLBAR(RoadMapMainToolbar),
                            label, tip, NULL,
                            roadmap_main_toolbar_icon (icon),
                            (GtkSignalFunc) roadmap_main_activate, callback);

   if (gdk_screen_height() < 550)
   {
      /* When using a small screen, we want either the labels or the icons,
       * but not both (small screens are typical with PDAs).
       */
      gtk_toolbar_set_style
         (GTK_TOOLBAR(RoadMapMainToolbar),
          GtkIconsInitialized?GTK_TOOLBAR_ICONS:GTK_TOOLBAR_TEXT);
   }
}


void roadmap_main_add_tool_space (void) {

   if (RoadMapMainToolbar == NULL) {
      roadmap_log (ROADMAP_FATAL, "Invalid toolbar space: no toolbar yet");
   }

   gtk_toolbar_append_space (GTK_TOOLBAR(RoadMapMainToolbar));
}

static unsigned long roadmap_main_busy_start;

void roadmap_main_set_cursor (RoadMapCursor newcursor) {
   GdkCursor *cursor = NULL;
   static int lastcursor;

   roadmap_main_busy_start = 0;

   if (newcursor == ROADMAP_CURSOR_WAIT_WITH_DELAY) {
      roadmap_main_busy_start = roadmap_time_get_millis();
      return;
   }

   if (newcursor == lastcursor)
      return;

   lastcursor = newcursor;

   switch (newcursor) {

   case ROADMAP_CURSOR_NORMAL:
      cursor = NULL;
      break;

   case ROADMAP_CURSOR_WAIT:
      cursor = gdk_cursor_new(GDK_WATCH);
      break;

   case ROADMAP_CURSOR_CROSS:
      cursor = gdk_cursor_new(GDK_CROSSHAIR);
      break;
   }

   gdk_window_set_cursor(RoadMapMainWindow->window, cursor);

   if (cursor) {
      gdk_cursor_destroy(cursor);
   }
   gdk_flush();
}

void roadmap_main_busy_check(void) {

   if (roadmap_main_busy_start == 0)
      return;

   if (roadmap_time_get_millis() - roadmap_main_busy_start > 1000) {
      roadmap_main_set_cursor (ROADMAP_CURSOR_WAIT);
   }
}

void roadmap_main_add_canvas (void) {

   gtk_box_pack_start (GTK_BOX(RoadMapCanvasBox),
                       roadmap_canvas_new (), TRUE, TRUE, 2);
}


void roadmap_main_add_status (void) {

   RoadMapMainStatus = gtk_entry_new ();

   gtk_editable_set_editable (GTK_EDITABLE(RoadMapMainStatus), FALSE);
   gtk_entry_set_text (GTK_ENTRY(RoadMapMainStatus), "Initializing..");

   gtk_box_pack_end (GTK_BOX(RoadMapMainBox),
                     RoadMapMainStatus, FALSE, FALSE, 0);
}


void roadmap_main_show (void) {

   if (RoadMapMainWindow != NULL) {
      hildon_window_set_menu(HILDON_WINDOW(RoadMapMainWindow), GTK_MENU(RoadMapMainMenuBar));
      if (roadmap_factory_use_toolbar()){
          hildon_window_add_toolbar(HILDON_WINDOW(RoadMapMainWindow), GTK_TOOLBAR(RoadMapMainToolbar));
      };
      gtk_widget_show_all (RoadMapMainWindow);

   }
}


static void roadmap_main_input
               (gpointer data, gint source, GdkInputCondition conditions) {

   if (data != NULL) {
      struct roadmap_main_io *context = (struct roadmap_main_io *) data;
      (* context->callback) (&context->io);
   }
}


void roadmap_main_set_input (RoadMapIO *io, RoadMapInput callback) {

   int i;
   int fd = io->os.file; /* All the same on UNIX. */

   for (i = 0; i < ROADMAP_MAX_IO; ++i) {
      if (RoadMapMainIo[i].io.subsystem == ROADMAP_IO_INVALID) {
         RoadMapMainIo[i].io = *io;
         RoadMapMainIo[i].callback = callback;
         RoadMapMainIo[i].id =
            gtk_input_add_full (fd, GDK_INPUT_READ, roadmap_main_input,
                                NULL, &RoadMapMainIo[i], NULL);
         break;
      }
   }
}


void roadmap_main_remove_input (RoadMapIO *io) {

   int i;
   int fd = io->os.file; /* All the same on UNIX. */

   for (i = 0; i < ROADMAP_MAX_IO; ++i) {
      if (RoadMapMainIo[i].io.os.file == fd) {
         gtk_input_remove (RoadMapMainIo[i].id);
         RoadMapMainIo[i].io.os.file = -1;
         RoadMapMainIo[i].io.subsystem = ROADMAP_IO_INVALID;
         break;
      }
   }
}


static gboolean roadmap_main_timeout (gpointer data) {

   RoadMapCallback callback = (RoadMapCallback) data;

   if (callback != NULL) {
      (*callback) ();
   }
   return TRUE;
}

void roadmap_main_set_periodic (int interval, RoadMapCallback callback) {

   int index;
   struct roadmap_main_timer *timer = NULL;

   for (index = 0; index < ROADMAP_MAX_TIMER; ++index) {

      if (RoadMapMainPeriodicTimer[index].callback == callback) {
         return;
      }
      if (timer == NULL) {
         if (RoadMapMainPeriodicTimer[index].callback == NULL) {
            timer = RoadMapMainPeriodicTimer + index;
         }
      }
   }

   if (timer == NULL) {
      roadmap_log (ROADMAP_FATAL, "Timer table saturated");
   }

   timer->id = gtk_timeout_add (interval, roadmap_main_timeout, callback);
   timer->callback = callback;
}


void roadmap_main_remove_periodic (RoadMapCallback callback) {

   int index;

   for (index = 0; index < ROADMAP_MAX_TIMER; ++index) {

      if (RoadMapMainPeriodicTimer[index].callback == callback) {

         RoadMapMainPeriodicTimer[index].callback = NULL;
         gtk_timeout_remove (RoadMapMainPeriodicTimer[index].id);

         return;
      }
   }
}


void roadmap_main_set_status (const char *text) {

   if (RoadMapMainStatus != NULL) {
      gtk_entry_set_text (GTK_ENTRY(RoadMapMainStatus), text);
   }
}

roadmap_main_flush (void) {

   while (gtk_events_pending ()) {
      if (gtk_main_iteration ()) {
         exit(0);  /* gtk_main_quit() called */
      }
   }
}


int roadmap_main_flush_synchronous (int deadline) {

   long start_time, duration;

   start_time = roadmap_time_get_millis();

   while (gtk_events_pending ()) {
      if (gtk_main_iteration ()) {
         exit(0);  /* gtk_main_quit() called */
      }
   }
   gdk_flush();

   duration = roadmap_time_get_millis() - start_time;

   if (duration > deadline) {

      roadmap_log (ROADMAP_DEBUG, "processing flush took %d", duration);

      return 0; /* Busy. */
   }

   return 1; /* Not so busy. */
}


void roadmap_main_exit (void) {

   static int exit_done;

   if (!exit_done++) {
      roadmap_start_exit ();
      osso_deinitialize(appdata.osso_context);
      gtk_main_quit();
   }
}

void roadmap_signals_init(void);

int main (int argc, char **argv) {

   int i;
   HildonProgram* program;
   osso_return_t ret;
 
   roadmap_option (argc, argv, 0, NULL);

   appdata.osso_context = osso_init();

   // fix for locale issues
   gtk_disable_setlocale();
   setlocale(LC_ALL, "");
   setlocale(LC_NUMERIC, "C");
   gtk_init (&argc, &argv);

   state_load();

   program = HILDON_PROGRAM(hildon_program_get_instance());
   g_set_application_name( "RoadGPS" );

   appdata.program = program;

   RoadMapMainWindow = hildon_window_new();
      
   hildon_program_add_window(appdata.program, HILDON_WINDOW(RoadMapMainWindow));
   hildon_banner_show_information(GTK_WIDGET(RoadMapMainWindow), NULL, "RoadGPS Starting...");

   for (i = 0; i < ROADMAP_MAX_IO; ++i) {
      RoadMapMainIo[i].io.os.file = -1;
      RoadMapMainIo[i].io.subsystem = ROADMAP_IO_INVALID;
   }

   roadmap_start (argc, argv);
      
   g_signal_connect(G_OBJECT(program), "notify::is-topmost",
                    G_CALLBACK(ui_main_topmost_cb), NULL);

   ret = osso_rpc_set_default_cb_f (appdata.osso_context, dbus_callback, RoadMapMainWindow);
   if (ret != OSSO_OK){

       fprintf (stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret);
       exit (1);
   }

   atexit(roadmap_main_exit);

   roadmap_signals_init();

   gtk_main();

   return 0;
}
