
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <locale.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-window.h>
#include <libosso.h>

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>

#include <coniciap.h>
#include <gconf/gconf-client.h>
#include <device_symbols.h>
#include <conicconnection.h>
#include <conicconnectionevent.h>

#include "roadmap.h"

#include "roadmap_io.h"
#include "roadmap_config.h"
#include "roadmap_gps.h"
#include "roadmap_math.h"
#include "roadmap_messagebox.h"
#include "roadmap_gpsmgr.h"
//#include "roadmap_input.h"
#include "appdata.h"

#include "getmaps_osm_start.h"
#include "getmaps_osm_main.h"
//options
gboolean had_error;
static gboolean from_dbus;

gchar *error_list, *getosm_dir;
gboolean config_output = FALSE, config_remember = TRUE, config_errorlist = TRUE, config_log = FALSE;

GIOChannel *chin;
GTimeVal startime, pausedtime;

struct GetMapsData getmapsdata;
FILE *log_file = NULL;

static gboolean has_location;
static RoadMapGpsPosition GetMapsPosition;
static RoadMapGpsPrecision GetMapsPrecision;
static time_t RoadMapGpsReceivedTime = 0;

static int getmaps_pid;

static ConIcConnection *_conic_conn = NULL;
static gboolean _conic_is_connecting = FALSE;
static gboolean _conic_is_connected = FALSE;
static gboolean _conic_conn_failed = FALSE;
static GMutex *_conic_connection_mutex = NULL;
static GCond *_conic_connection_cond = NULL;
static volatile GThread *_conic_thread = NULL;

GtkWidget *window;
  GtkWidget *label3;
  GtkWidget *label4;
  GtkWidget *lat1_spinbutton;
  GtkWidget *lon1_spinbutton;
  GtkWidget *lat2_spinbutton;
  GtkWidget *lon2_spinbutton;
  GtkWidget *bits_spinbutton;
  GtkWidget *radius_spinbutton;
  GtkWidget *map_chooser_button;
  GtkWidget *classfile_chooser_button;
  GtkWidget *osm_proxy_text;
  GtkWidget *mode_radio1;
  GtkWidget *mode_radio2;
  GSList *mode_radio2_group = NULL;
  GtkWidget *hseparator4;
  GtkWidget *hseparator5;
  GtkWidget *label5;
  GtkWidget *radius_spinbutton;
  GtkWidget *hseparator8;
  GtkWidget *hseparator9;
  GtkWidget *go_button;
  GtkWidget *textview1;

struct AppData appdata;

#define OSSO_NAME    "getmaps"
#define OSSO_VERSION    "1.1.0"

void getmaps_osm_main_download_complete(){
   if(from_dbus){ //only notify if started from the main app. otherwise Roadmap will start if not running...

     gint ret = osso_rpc_run(appdata.osso_context, "org.maemo.roadmap", "/org/maemo/roadmap",
                      "org.maemo.roadmap", "download_complete", NULL, DBUS_TYPE_INVALID);
   }

} 
void topmost_event_handler(const char *args, gpointer data)
{
  //do stuff
};


void hw_event_handler(osso_hw_state_t *state, gpointer data)
{

    if (state->shutdown_ind)
    {
        getmaps_osm_close();
    }
    if (state->memory_low_ind)
    {
      hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Caution, Memory Low!");
    }
    if (state->save_unsaved_data_ind)
    {
      //  hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Must save unsaved data event!");
    }
    if (state->system_inactivity_ind)
    {
      //  hildon_banner_show_information(GTK_WIDGET(appdata.window), NULL, "Minimize application inactivity event!");
    }
}


osso_context_t * osso_init ()
{

    osso_context_t *osso = osso_initialize ( OSSO_NAME, OSSO_VERSION, FALSE, NULL );

    if (OSSO_OK != osso_application_set_top_cb (osso, topmost_event_handler, NULL))
        return;

    if (OSSO_OK != osso_hw_set_event_cb (osso, NULL, hw_event_handler, NULL))
        return;

    return osso;
}
/* Callback for normal D-BUS messages */

gint dbus_callback (const gchar *interface, const gchar *method,
	       GArray *arguments, gpointer data,
	       osso_rpc_t *retval)
{


    printf ("getmaps dbus: %s, %s\n", interface, method);   

    if (!strcmp (method, "bbox_open")){
           gint lat1, lon1, lat2, lon2;
           from_dbus = TRUE;

           lat1 = (gint)g_array_index(arguments, osso_rpc_t, 0).value.i;
           lon1 = (gint)g_array_index(arguments, osso_rpc_t, 1).value.i;
           lat2 = (gint)g_array_index(arguments, osso_rpc_t, 2).value.i;
           lon2 = (gint)g_array_index(arguments, osso_rpc_t, 3).value.i;

           gtk_toggle_button_set_active    (GTK_TOGGLE_BUTTON(mode_radio1), TRUE);

           getmapsdata.lat1 = (gdouble)lat1 / 1000000.0;
           getmapsdata.lon1 = (gdouble)lon1 / 1000000.0;
           getmapsdata.lat2 = (gdouble)lat2 / 1000000.0;
           getmapsdata.lon2 = (gdouble)lon2 / 1000000.0;

           gtk_spin_button_set_value(GTK_SPIN_BUTTON(lat1_spinbutton), getmapsdata.lat1);
           gtk_spin_button_set_value(GTK_SPIN_BUTTON(lon1_spinbutton), getmapsdata.lon1);
           gtk_spin_button_set_value(GTK_SPIN_BUTTON(lat2_spinbutton), getmapsdata.lat2);
           gtk_spin_button_set_value(GTK_SPIN_BUTTON(lon2_spinbutton), getmapsdata.lon2);

           if(_conic_is_connected){

              gtk_button_clicked(GTK_BUTTON(go_button));

           }
    }

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}

/* Callback for exit D-BUS event */
void exit_event_handler(gboolean die_now)
{
    /* Do whatever application needs to do before exiting */ 
    getmaps_osm_close();

};

static void ui_main_topmost_cb(GObject *self, GParamSpec *property_param, gpointer null)
{

       if (appdata.program == NULL) return;

       if (hildon_program_get_is_topmost(appdata.program)) {
               hildon_program_set_can_hibernate(appdata.program, FALSE);
       }
}
//end of osso/hildon functions

static void
conic_conn_event(ConIcConnection *connection, ConIcConnectionEvent *event)
{
    ConIcConnectionStatus status;

    g_mutex_lock(_conic_connection_mutex);

    status = con_ic_connection_event_get_status(event);

    if((_conic_is_connected = (status == CON_IC_CONNECTION_ERROR_NONE)))
    {
        /* We're connected. */
        _conic_conn_failed = FALSE;
        fprintf(stdout, "Connected...\n");
    }
    else
    {
        /* We're not connected. */
        /* Mark as a failed connection, if we had been trying to connect. */
        _conic_conn_failed = _conic_is_connecting;
            fprintf(stdout, "Could not connect to the Internet!\n");
    }

    _conic_is_connecting = FALSE; /* No longer trying to connect. */
    g_cond_broadcast(_conic_connection_cond);
    g_mutex_unlock(_conic_connection_mutex);

}

void
conic_recommend_connected()
{

#ifndef DEBUG
    g_mutex_lock(_conic_connection_mutex);
    if(!_conic_is_connecting)
    {
        /* Fire up a connection request. */
        con_ic_connection_connect(_conic_conn, CON_IC_CONNECT_FLAG_NONE);
        _conic_is_connecting = TRUE;
    }
    g_mutex_unlock(_conic_connection_mutex);
#endif

}

void
conic_ensure_connected()
{

#ifndef DEBUG
    GTimeVal *time;
    time = g_new (GTimeVal, sizeof(GTimeVal));
    g_get_current_time (time);
    g_time_val_add (time, 5000000);

    while(!_conic_is_connected)
    {   
        g_mutex_lock(_conic_connection_mutex);
        /* If we're not connected, and if we're not connecting, and if we're
         * not in the wake of a connection failure, then try to connect. */
        if(!_conic_is_connected && !_conic_is_connecting &&!_conic_conn_failed)
        {
            /* Fire up a connection request. */
            con_ic_connection_connect(_conic_conn, CON_IC_CONNECT_FLAG_NONE);
            _conic_is_connecting = TRUE;
        }
        
        g_cond_timed_wait(_conic_connection_cond, _conic_connection_mutex, time);
        g_mutex_unlock(_conic_connection_mutex);
    }
#endif
    g_free(time);
}



struct getmaps_osm_main_io {
   int id;
   RoadMapIO io;
   RoadMapInput callback;
};

#define GETMAPS_MAX_IO 16
static struct getmaps_osm_main_io GetMapsOsmMainIo[GETMAPS_MAX_IO];

struct getmaps_osm_main_timer {
   guint id;
   RoadMapCallback callback;
};

#define GETMAPS_MAX_TIMER 16
static struct getmaps_osm_main_timer GetMapsOsmMainPeriodicTimer[GETMAPS_MAX_TIMER];

static void getmaps_osm_main_input
               (gpointer data, gint source, GdkInputCondition conditions) {

   if (data != NULL) {
      struct getmaps_osm_main_io *context = (struct getmaps_osm_main_io *) data;
      (* context->callback) (&context->io);
   }
}

void getmaps_osm_main_set_input (RoadMapIO *io, RoadMapInput callback) {

   int i;
   int fd = io->os.file; /* All the same on UNIX. */

   for (i = 0; i < GETMAPS_MAX_IO; ++i) {
      if (GetMapsOsmMainIo[i].io.subsystem == ROADMAP_IO_INVALID) {
         GetMapsOsmMainIo[i].io = *io;
         GetMapsOsmMainIo[i].callback = callback;
         GetMapsOsmMainIo[i].id =
            gtk_input_add_full (fd, GDK_INPUT_READ, getmaps_osm_main_input,
                                NULL, &GetMapsOsmMainIo[i], NULL);
         break;
      }
   }
}

void getmaps_osm_main_remove_input (RoadMapIO *io) {

   int i;
   int fd = io->os.file; /* All the same on UNIX. */

   for (i = 0; i < GETMAPS_MAX_IO; ++i) {
      if (GetMapsOsmMainIo[i].io.os.file == fd) {
         gtk_input_remove (GetMapsOsmMainIo[i].id);
         GetMapsOsmMainIo[i].io.os.file = -1;
         GetMapsOsmMainIo[i].io.subsystem = ROADMAP_IO_INVALID;
         break;
      }
   }
}

static void getmaps_osm_main_add_gps (RoadMapIO *io) {

   getmaps_osm_main_set_input (io, roadmap_gps_input);
}

static void getmaps_osm_main_remove_gps (RoadMapIO *io) {

   getmaps_osm_main_remove_input(io);
}

void getmaps_osm_to_coord(char data[], int islatitude, int value) {

  data[0] = "EWNS"[(islatitude * 2)  + (value < 0)];
  
  roadmap_math_to_floatstring(data + 1, value, MILLIONTHS);
}

static void getmaps_osm_listener
                  (int reception,
                   int gps_time,
                   const RoadMapGpsPrecision *dilution,
                   const RoadMapGpsPosition *position) {

   GetMapsPosition.latitude = position->latitude;
   GetMapsPosition.longitude = position->longitude;
   GetMapsPosition.latitude = position->latitude;
   GetMapsPosition.longitude = position->longitude;
   GetMapsPosition.altitude = position->altitude;    // meters
   GetMapsPosition.speed = position->speed;          // knots
   GetMapsPosition.steering = position->steering;

   GetMapsPrecision.dimension = dilution->dimension;
   GetMapsPrecision.dilution_position = dilution->dilution_position;
   GetMapsPrecision.dilution_horizontal = dilution->dilution_horizontal;
   GetMapsPrecision.dilution_vertical = dilution->dilution_vertical;

  RoadMapGpsReceivedTime = gps_time;

   if (reception > 1 ){

     has_location = TRUE;

   }
   else has_location = FALSE;

}
static void getmaps_osm_main_set_unit (void) {

   static RoadMapConfigDescriptor RoadMapConfigGeneralUnit =
                            ROADMAP_CONFIG_ITEM("General", "Unit");

   const char *unit = roadmap_config_get (&RoadMapConfigGeneralUnit);

   if (strcmp (unit, "imperial") == 0) {

      roadmap_math_use_imperial();

   } else if (strcmp (unit, "metric") == 0) {

      roadmap_math_use_metric();

   } else {
      roadmap_log (ROADMAP_ERROR, "%s is not a supported unit", unit);
      roadmap_math_use_imperial();
   }
}

static gboolean getmaps_osm_main_timeout (gpointer data) {

   RoadMapCallback callback = (RoadMapCallback) data;

   if (callback != NULL) {
      (*callback) ();
   }
   return TRUE;
}

void getmaps_osm_main_set_periodic (int interval, RoadMapCallback callback) {

   int index;
   struct getmaps_osm_main_timer *timer = NULL;

   for (index = 0; index < GETMAPS_MAX_TIMER; ++index) {

      if (GetMapsOsmMainPeriodicTimer[index].callback == callback) {
         return;
      }
      if (timer == NULL) {
         if (GetMapsOsmMainPeriodicTimer[index].callback == NULL) {
            timer = GetMapsOsmMainPeriodicTimer + index;
         }
      }
   }

   timer->id = gtk_timeout_add (interval, getmaps_osm_main_timeout, callback);
   timer->callback = callback;
}


void getmaps_osm_main_remove_periodic (RoadMapCallback callback) {

   int index;

   for (index = 0; index < GETMAPS_MAX_TIMER; ++index) {

      if (GetMapsOsmMainPeriodicTimer[index].callback == callback) {

         GetMapsOsmMainPeriodicTimer[index].callback = NULL;
         gtk_timeout_remove (GetMapsOsmMainPeriodicTimer[index].id);

         return;
      }
   }
}

static void getmaps_osm_main_set_timeout (RoadMapCallback callback) {

   getmaps_osm_main_set_periodic (3000, callback);
}

void
on_main_create                         (GtkWidget       *widget,
                                        gpointer         user_data)
{
   roadmap_config_initialize ();

   roadmap_gps_initialize    ();
   roadmap_gps_register_listener (getmaps_osm_listener);
//   roadgps_logger_initialize ();

   roadmap_config_load ();

 //  roadmap_option (argc, argv, 1, roadgps_start_usage);


   getmaps_osm_main_set_unit ();

   // last GPS stuff
   roadmap_gps_register_link_control
      (getmaps_osm_main_add_gps, getmaps_osm_main_remove_gps);

   roadmap_gps_register_periodic_control
      (getmaps_osm_main_set_timeout, getmaps_osm_main_remove_periodic);
   
   roadmap_gpsmgr_initialize     ();
   
   roadmap_gps_open ();

//end new
    // get lat1 and lon1 from session file?
   getmapsdata.radius = 10;
   getmapsdata.osm_bits = 21;
   getmapsdata.radius_mode = TRUE;
   getosm_dir = g_strconcat(g_get_home_dir(), "/.getosm", NULL);

   getmapsdata.source = g_new(gchar, 255);
   getmapsdata.maps = g_new(gchar, 255);
   getmapsdata.class = g_new(gchar, 255);


   getmapsdata.source = "http://www.symbianos.org/~adam/osmgetbmap/osmgetbmap.php";
   getmapsdata.maps = "/media/mmc1/roadmaps";
   getmapsdata.class = "/usr/share/roadmap/default/All";

   gtk_toggle_button_set_active    (GTK_TOGGLE_BUTTON(mode_radio1), TRUE);
   gtk_toggle_button_set_active    (GTK_TOGGLE_BUTTON(mode_radio2), TRUE);

    //load default values here
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lat1_spinbutton), getmapsdata.lat1);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lon1_spinbutton), getmapsdata.lon1);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lat2_spinbutton), getmapsdata.lat2);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lon2_spinbutton), getmapsdata.lon2);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(bits_spinbutton), getmapsdata.osm_bits);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(radius_spinbutton), getmapsdata.radius);
   gtk_file_chooser_set_filename (GTK_FILE_CHOOSER(classfile_chooser_button), getmapsdata.class);
   gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(map_chooser_button), getmapsdata.maps);
   gtk_entry_set_text(GTK_ENTRY(osm_proxy_text), getmapsdata.source);



}
void
on_main_destroy                        (GtkObject       *object,
                                        gpointer         user_data)
{
   getmaps_osm_close();
}


void
on_lat1_spinbutton_value_changed       (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.lat1 = gtk_spin_button_get_value ( spinbutton );
}


void
on_lon1_spinbutton_value_changed       (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.lon1 = gtk_spin_button_get_value ( spinbutton );
}


void
on_lat2_spinbutton_value_changed       (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.lat2 = gtk_spin_button_get_value ( spinbutton );
}


void
on_lon2_spinbutton_value_changed       (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.lon2 = gtk_spin_button_get_value ( spinbutton );
}


void
on_radius_spinbutton_value_changed     (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.radius = gtk_spin_button_get_value_as_int ( spinbutton );
}


void
on_bits_spinbutton_value_changed       (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
    getmapsdata.osm_bits = gtk_spin_button_get_value_as_int ( spinbutton );
}

void
on_mode_radius_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  gtk_widget_hide(label3);
  gtk_widget_hide(label4);
  gtk_widget_hide(lat2_spinbutton);
  gtk_widget_hide(lon2_spinbutton);
  gtk_widget_hide(hseparator4);
  gtk_widget_hide(hseparator5);

  gtk_widget_show(label5);
  gtk_widget_show(radius_spinbutton);
  gtk_widget_show(hseparator8);
  gtk_widget_show(hseparator9);

  getmapsdata.radius_mode = TRUE;
}


void
on_mode_boundary_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  gtk_widget_hide(label5);
  gtk_widget_hide(radius_spinbutton);
  gtk_widget_hide(hseparator8);
  gtk_widget_hide(hseparator9);
  gtk_widget_show(label3);
  gtk_widget_show(label4);
  gtk_widget_show(lat2_spinbutton);
  gtk_widget_show(lon2_spinbutton);
  gtk_widget_show(hseparator4);
  gtk_widget_show(hseparator5);

  getmapsdata.radius_mode = FALSE;

}

void
on_replace_checkbutton_toggled         (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
   getmapsdata.replace_maps = gtk_toggle_button_get_active (togglebutton);
}


void
on_map_chooser_button_selection_changed
                                        (GtkFileChooser  *filechooser,
                                        gpointer         user_data)
{
   getmapsdata.maps = gtk_file_chooser_get_current_folder (filechooser);
}

void
on_classfile_chooser_button_selection_changed
                                        (GtkFileChooser  *filechooser,
                                        gpointer         user_data)
{
   getmapsdata.class = gtk_file_chooser_get_filename (filechooser);
}


void
on_osm_proxy_text_changed         (GtkCellEditable *celleditable,
                                        gpointer         user_data)
{

   getmapsdata.source = g_strdup(gtk_entry_get_text (GTK_ENTRY(celleditable)));

}

void on_getmaps (struct GetMapsData *data)
{
   gchar *latlonarg;
   latlonarg = g_new(gchar, 255);
   gint tmp_lat, tmp_lon, tmp_lat2, tmp_lon2;

   tmp_lat = data->lat1 * 1000000;
   tmp_lon = data->lon1 * 1000000;

   if (data->radius_mode){
      g_sprintf (latlonarg, "%i,%i:%ikm", tmp_lat, tmp_lon, data->radius);
   }else{
      tmp_lat2 = data->lat2 * 1000000;
      tmp_lon2 = data->lon2 * 1000000;
      g_sprintf (latlonarg, "%i,%i:%i,%i", tmp_lat, tmp_lon, tmp_lat2, tmp_lon2);
   }
   if (getmaps(data->osm_bits, data->replace_maps, data->maps, data->source, data->class, latlonarg)){
      fprintf(stdout, "\nDownload Finished with Errors!\n");
   }else{
      fprintf(stdout, "\nMaps Downloaded Successfully!\n");
   }
   g_free(latlonarg);
   return;
   
}

void getosm_cleanup(gpointer data) {
   getmaps_pid = 0;
}

void scroll_to_end(GtkTextView* view) {
	GtkTextBuffer *buffer;
	GtkTextIter iter;
	GtkTextMark *mark;

	buffer = gtk_text_view_get_buffer(view);
	gtk_text_buffer_get_end_iter(buffer, &iter);
	mark = gtk_text_buffer_create_mark(buffer, "end_mark", &iter, TRUE);
	gtk_text_view_scroll_mark_onscreen(view, mark);
	gtk_text_buffer_delete_mark(buffer, mark);
}

gboolean out_watch(GIOChannel *source, GIOCondition condition, gpointer data) {
	GtkTextBuffer *buffer;
	GtkTextIter iter, iter2;
	GString *str;
	GtkProgressBar *tmpp;
	static gchar *carriage = NULL, *oldfile = NULL;
	gchar *file, tmps[50];
	gdouble fraction, done, total;

	if (condition & G_IO_IN) {
		buffer = gtk_text_view_get_buffer((GtkTextView*)data);

		if (carriage != NULL) {
			gtk_text_buffer_get_iter_at_line(buffer, &iter, gtk_text_buffer_get_line_count(buffer) - 2);
			gtk_text_buffer_get_end_iter(buffer, &iter2);
			gtk_text_buffer_delete(buffer, &iter, &iter2);
		}

		str = g_string_new("");
		g_io_channel_read_line_string(source, str, NULL, NULL);
		carriage = strchr(str->str, '\r');

		gtk_text_buffer_insert_at_cursor(buffer, (gchar*)str->str, -1);
		scroll_to_end((GtkTextView*)data);
		if (config_log) fputs((gchar*)str->str, log_file);

		g_string_free(str, TRUE);
	} else {
		g_io_channel_shutdown (source, FALSE, NULL);
		g_io_channel_unref (source);
		getosm_cleanup(data);

		if (oldfile != NULL) g_free(oldfile);
		oldfile = NULL;
		return FALSE;
	}
	return TRUE;
}
/*
gboolean err_watch(GIOChannel *source, GIOCondition condition, gpointer data) {
	GtkTextBuffer *buffer;
	GtkTextIter iter;
	GString *str;
	gchar *tmpc;

	if (condition & G_IO_IN) {
		buffer = gtk_text_view_get_buffer((GtkTextView*)data);
		str = g_string_new("");

		g_io_channel_read_line_string(source, str, NULL, NULL);
		if (str->len > 2) had_error = TRUE;

		gtk_text_buffer_get_end_iter(buffer, &iter);
		gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, (gchar*)str->str, -1, "fore-red", NULL);
		scroll_to_end((GtkTextView*)data);

		tmpc = g_strconcat(error_list, str->str, NULL);
		g_free(error_list);
		error_list = tmpc;
		g_string_free(str, TRUE);
	} else {
//	if (condition & G_IO_HUP || condition & G_IO_ERR) {
		g_io_channel_shutdown (source, FALSE, NULL);
		g_io_channel_unref (source);
		getosm_cleanup(data);
		return FALSE;
	}
	return TRUE;
}
*/
void
on_go_button_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkTextBuffer *buffer;
   GtkTextIter iter;
   GtkTextTag *tag;
   GtkWidget *error;
   gboolean channel_closed;
   int i, ret, fork_return, out[2];
   GIOChannel *chout;

    if (pipe (out))
    {
      fprintf (stderr, "Pipe failed.\n");
    }

// get maps
   fork_return = fork();
   if (fork_return > 0){  
        //parent- show output
        getmaps_pid = fork_return;
        buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview1));
        tag = gtk_text_buffer_create_tag(buffer, "fore-red", "foreground", "red", NULL);
        gtk_text_buffer_get_end_iter(buffer, &iter);
        gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, "Connecting, Please Wait...\n", -1, "fore-red", NULL);

        close(out[1]);
        chout = g_io_channel_unix_new (out[0]);
        g_io_channel_set_flags(chout, G_IO_FLAG_NONBLOCK, NULL);
        channel_closed = FALSE;
        g_io_add_watch_full(chout, G_PRIORITY_DEFAULT_IDLE, G_IO_IN | G_IO_HUP | G_IO_ERR, out_watch, textview1, NULL);
        return;
   }else{ 
        //child- get maps
        close(out[0]);
        dup2(out[1], 1);

        on_getmaps(&getmapsdata);
        channel_closed = TRUE;
        close (out[1]);
        getmaps_osm_main_download_complete();
        exit(0);
   }
  return;
}

void on_get_gps_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
   gchar *tmp;
   gchar data[100];
   tmp = g_new(char, 255);
   static gdouble gps_lat, gps_lon;

   gps_lat = (gdouble)GetMapsPosition.latitude  / 1000000.0 ;
   gps_lon = (gdouble)GetMapsPosition.longitude / 1000000.0 ;
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lat1_spinbutton), gps_lat);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(lon1_spinbutton), gps_lon);

   getmapsdata.lat1 = gps_lat;
   getmapsdata.lon1 = gps_lon;

   if ( has_location == TRUE ){

      g_sprintf(tmp, "GPS fix is lat- %.3f, lon- %.3f", gps_lat, gps_lon );
      roadmap_messagebox("GPS Fix Acquired", tmp);
      
   }else{

      g_sprintf(tmp, "Unable to acquire GPS fix. Last fix was lat- %.3f, lon- %.3f", gps_lat, gps_lon);
      roadmap_messagebox("Error", tmp);

   }

   g_free(tmp);
}

void on_cancel_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkTextBuffer *buffer;
   GtkTextIter iter;

   if(getmaps_pid > 0){
      GError *err;
      gchar *com;
      com = g_new(gchar, 255);
      err = g_new(GError, sizeof(GError));
      g_sprintf(com, "kill %i", getmaps_pid);
      g_spawn_command_line_async(com, &err);
      usleep(500000);
      buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview1));
      gtk_text_buffer_get_end_iter(buffer, &iter);
      gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, "Download Canceled by User!\n", -1, "fore-red", NULL);

      g_free(com);
      g_free(err);
      getmaps_pid = 0;
   }else getmaps_osm_close();
}
GtkWidget* create_window (void)
{

  GtkWidget *vbox4;
  GtkWidget *hbox3;
  GtkWidget *table1;
  GtkWidget *label1;
  GtkWidget *label2;

  GtkObject *lat1_spinbutton_adj;

  GtkObject *lon1_spinbutton_adj;

  GtkObject *lat2_spinbutton_adj;

  GtkObject *lon2_spinbutton_adj;


  GtkObject *radius_spinbutton_adj;

  GtkWidget *label6;
  GtkObject *bits_spinbutton_adj;

  GtkWidget *hseparator2;
  GtkWidget *hseparator3;

  GtkWidget *getgps_button;

  GtkWidget *label18;
  GtkWidget *hseparator6;
  GtkWidget *hseparator7;

  GtkWidget *vseparator1;
  GtkWidget *table2;

  GtkWidget *cancel_button;
  GtkWidget *hseparator11;
  GtkWidget *replace_checkbutton;
  GtkWidget *label19;
  GtkWidget *label7;
  GtkWidget *label10;
  GtkWidget *hseparator12;
  GtkWidget *hseparator13;


  GtkWidget *hseparator1;
  GtkWidget *frame1;
  GtkWidget *alignment1;
  GtkWidget *scrolledwindow1;

  GtkWidget *label16;



  window = hildon_window_new();
      
  hildon_program_add_window(appdata.program, HILDON_WINDOW(window));
  //window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Get OSM Maps");

  vbox4 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox4);
  gtk_container_add (GTK_CONTAINER (window), vbox4);

  hbox3 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox3);
  gtk_box_pack_start (GTK_BOX (vbox4), hbox3, TRUE, TRUE, 0);

  table1 = gtk_table_new (13, 2, FALSE);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (hbox3), table1, TRUE, TRUE, 0);

  label1 = gtk_label_new ("1st Lat.-");
  gtk_widget_show (label1);
  gtk_table_attach (GTK_TABLE (table1), label1, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_FILL);
  gtk_misc_set_alignment (GTK_MISC (label1), 0, 0.5);

  label2 = gtk_label_new ("1st Lon.-");
  gtk_widget_show (label2);
  gtk_table_attach (GTK_TABLE (table1), label2, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label2), 0, 0.5);

  label3 = gtk_label_new ("2nd Lat.-");
  gtk_widget_show (label3);
  gtk_table_attach (GTK_TABLE (table1), label3, 0, 1, 7, 8,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label3), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label3), 0, 0.5);

  label4 = gtk_label_new ("2nd Lon.-");
  gtk_widget_show (label4);
  gtk_table_attach (GTK_TABLE (table1), label4, 0, 1, 8, 9,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label4), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label4), 0, 0.5);

  lat1_spinbutton_adj = gtk_adjustment_new (0, -90, 90, 0.00499999988824, 0.0500000007451, 0.10000000149);
  lat1_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (lat1_spinbutton_adj), 1, 3);
  gtk_widget_show (lat1_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), lat1_spinbutton, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  lon1_spinbutton_adj = gtk_adjustment_new (0, -179.994995117, 180, 0.00499999988824, 10, 10);
  lon1_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (lon1_spinbutton_adj), 1, 3);
  gtk_widget_show (lon1_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), lon1_spinbutton, 1, 2, 4, 5,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  lat2_spinbutton_adj = gtk_adjustment_new (0, -90, 90, 0.00499999988824, 0.0500000007451, 0.10000000149);
  lat2_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (lat2_spinbutton_adj), 1, 3);
  gtk_widget_show (lat2_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), lat2_spinbutton, 1, 2, 7, 8,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  lon2_spinbutton_adj = gtk_adjustment_new (0, -179.994995117, 180, 0.00499999988824, 0.0500000007451, 0.10000000149);
  lon2_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (lon2_spinbutton_adj), 1, 3);
  gtk_widget_show (lon2_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), lon2_spinbutton, 1, 2, 8, 9,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  label5 = gtk_label_new ("Radius (km.)");
  gtk_widget_show (label5);
  gtk_table_attach (GTK_TABLE (table1), label5, 0, 1, 10, 11,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label5), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label5), 0, 0.5);

  radius_spinbutton_adj = gtk_adjustment_new (10, 1, 100, 1, 10, 10);
  radius_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (radius_spinbutton_adj), 1, 0);
  gtk_widget_show (radius_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), radius_spinbutton, 1, 2, 10, 11,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  label6 = gtk_label_new ("Address Bits");
  gtk_widget_show (label6);
  gtk_table_attach (GTK_TABLE (table1), label6, 0, 1, 12, 13,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label6), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label6), 0, 0.5);

  bits_spinbutton_adj = gtk_adjustment_new (19, 12, 27, 1, 10, 10);
  bits_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (bits_spinbutton_adj), 1, 0);
  gtk_widget_show (bits_spinbutton);
  gtk_table_attach (GTK_TABLE (table1), bits_spinbutton, 1, 2, 12, 13,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  hseparator2 = gtk_hseparator_new ();
  gtk_widget_show (hseparator2);
  gtk_table_attach (GTK_TABLE (table1), hseparator2, 0, 1, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hseparator3 = gtk_hseparator_new ();
  gtk_widget_show (hseparator3);
  gtk_table_attach (GTK_TABLE (table1), hseparator3, 1, 2, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  hseparator4 = gtk_hseparator_new ();
  gtk_widget_show (hseparator4);
  gtk_table_attach (GTK_TABLE (table1), hseparator4, 0, 1, 9, 10,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hseparator5 = gtk_hseparator_new ();
  gtk_widget_show (hseparator5);
  gtk_table_attach (GTK_TABLE (table1), hseparator5, 1, 2, 9, 10,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  getgps_button = gtk_button_new_with_mnemonic ("GPS Position");
  gtk_widget_show (getgps_button);
  gtk_table_attach (GTK_TABLE (table1), getgps_button, 1, 2, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  mode_radio2 = gtk_radio_button_new_with_mnemonic (NULL, "Radius");
  gtk_widget_show (mode_radio2);
  gtk_table_attach (GTK_TABLE (table1), mode_radio2, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_radio_button_set_group (GTK_RADIO_BUTTON (mode_radio2), mode_radio2_group);
  mode_radio2_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (mode_radio2));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mode_radio2), TRUE);

  label18 = gtk_label_new ("Mode-");
  gtk_widget_show (label18);
  gtk_table_attach (GTK_TABLE (table1), label18, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label18), 0, 0.5);

  hseparator6 = gtk_hseparator_new ();
  gtk_widget_show (hseparator6);
  gtk_table_attach (GTK_TABLE (table1), hseparator6, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hseparator7 = gtk_hseparator_new ();
  gtk_widget_show (hseparator7);
  gtk_table_attach (GTK_TABLE (table1), hseparator7, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  hseparator8 = gtk_hseparator_new ();
  gtk_widget_show (hseparator8);
  gtk_table_attach (GTK_TABLE (table1), hseparator8, 0, 1, 11, 12,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hseparator9 = gtk_hseparator_new ();
  gtk_widget_show (hseparator9);
  gtk_table_attach (GTK_TABLE (table1), hseparator9, 1, 2, 11, 12,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  mode_radio1 = gtk_radio_button_new_with_mnemonic (NULL, "Bounding Box");
  gtk_widget_show (mode_radio1);
  gtk_table_attach (GTK_TABLE (table1), mode_radio1, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_radio_button_set_group (GTK_RADIO_BUTTON (mode_radio1), mode_radio2_group);
  mode_radio2_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (mode_radio1));

  vseparator1 = gtk_vseparator_new ();
  gtk_widget_show (vseparator1);
  gtk_box_pack_start (GTK_BOX (hbox3), vseparator1, TRUE, TRUE, 0);

  table2 = gtk_table_new (12, 1, FALSE);
  gtk_widget_show (table2);
  gtk_box_pack_start (GTK_BOX (hbox3), table2, TRUE, TRUE, 0);

  go_button = gtk_button_new_with_mnemonic ("Get OSM Maps");
  gtk_widget_show (go_button);
  gtk_table_attach (GTK_TABLE (table2), go_button, 0, 1, 10, 11,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  cancel_button = gtk_button_new_from_stock ("gtk-cancel");
  gtk_widget_show (cancel_button);
  gtk_table_attach (GTK_TABLE (table2), cancel_button, 0, 1, 11, 12,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  hseparator11 = gtk_hseparator_new ();
  gtk_widget_show (hseparator11);
  gtk_table_attach (GTK_TABLE (table2), hseparator11, 0, 1, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  replace_checkbutton = gtk_check_button_new_with_mnemonic ("Overwrite Existing Maps");
  gtk_widget_show (replace_checkbutton);
  gtk_table_attach (GTK_TABLE (table2), replace_checkbutton, 0, 1, 8, 9,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  label19 = gtk_label_new ("Destination for Maps-");
  gtk_widget_show (label19);
  gtk_table_attach (GTK_TABLE (table2), label19, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label19), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment (GTK_MISC (label19), 0, 0.5);

  label7 = gtk_label_new ("Map Proxy-");
  gtk_widget_show (label7);
  gtk_table_attach (GTK_TABLE (table2), label7, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label7), 0, 0.5);

  label10 = gtk_label_new ("Class File");
  gtk_widget_show (label10);
  gtk_table_attach (GTK_TABLE (table2), label10, 0, 1, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label10), 0, 0.5);

  hseparator12 = gtk_hseparator_new ();
  gtk_widget_show (hseparator12);
  gtk_table_attach (GTK_TABLE (table2), hseparator12, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hseparator13 = gtk_hseparator_new ();
  gtk_widget_show (hseparator13);
  gtk_table_attach (GTK_TABLE (table2), hseparator13, 0, 1, 9, 10,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  map_chooser_button = gtk_file_chooser_button_new ("Select A Directory", GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
  gtk_widget_show (map_chooser_button);
  gtk_table_attach (GTK_TABLE (table2), map_chooser_button, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  classfile_chooser_button = gtk_file_chooser_button_new ("Select A File", GTK_FILE_CHOOSER_ACTION_OPEN);
  gtk_widget_show (classfile_chooser_button);
  gtk_table_attach (GTK_TABLE (table2), classfile_chooser_button, 0, 1, 7, 8,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  osm_proxy_text = gtk_entry_new ();
  gtk_entry_set_max_length(GTK_ENTRY(osm_proxy_text), 233);

  gtk_widget_show (osm_proxy_text);
  gtk_table_attach (GTK_TABLE (table2), osm_proxy_text, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_entry_set_text (GTK_ENTRY(osm_proxy_text), "osmgetbmap");
/*
  hseparator1 = gtk_hseparator_new ();
  gtk_widget_show (hseparator1);
  gtk_box_pack_start (GTK_BOX (vbox4), hseparator1, TRUE, TRUE, 0);
*/
  frame1 = gtk_frame_new (NULL);
  gtk_widget_show (frame1);
  gtk_box_pack_start (GTK_BOX (vbox4), frame1, TRUE, TRUE, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame1), GTK_SHADOW_NONE);

  alignment1 = gtk_alignment_new (0.5, 0.5, 1, 1);
  gtk_widget_show (alignment1);
  gtk_container_add (GTK_CONTAINER (frame1), alignment1);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment1), 0, 0, 12, 0);

  scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow1);
  gtk_container_add (GTK_CONTAINER (alignment1), scrolledwindow1);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow1), GTK_SHADOW_IN);

  textview1 = gtk_text_view_new ();
  gtk_widget_show (textview1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow1), textview1);
  gtk_text_view_set_editable (GTK_TEXT_VIEW (textview1), FALSE);
  gtk_text_view_set_accepts_tab (GTK_TEXT_VIEW (textview1), FALSE);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (textview1), FALSE);

  label16 = gtk_label_new ("Output-");
  gtk_widget_show (label16);
  gtk_frame_set_label_widget (GTK_FRAME (frame1), label16);
  gtk_label_set_use_markup (GTK_LABEL (label16), TRUE);

  g_signal_connect ((gpointer) window, "show",
                    G_CALLBACK (on_main_create),
                    NULL);
  g_signal_connect ((gpointer) window, "destroy",
                    G_CALLBACK (on_main_destroy),
                    NULL);
  g_signal_connect ((gpointer) lat1_spinbutton, "value_changed",
                    G_CALLBACK (on_lat1_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) lon1_spinbutton, "value_changed",
                    G_CALLBACK (on_lon1_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) lat2_spinbutton, "value_changed",
                    G_CALLBACK (on_lat2_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) lon2_spinbutton, "value_changed",
                    G_CALLBACK (on_lon2_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) radius_spinbutton, "value_changed",
                    G_CALLBACK (on_radius_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) bits_spinbutton, "value_changed",
                    G_CALLBACK (on_bits_spinbutton_value_changed),
                    NULL);
  g_signal_connect ((gpointer) mode_radio2, "toggled",
                    G_CALLBACK (on_mode_radius_toggled),
                    NULL);
  g_signal_connect ((gpointer) mode_radio1, "toggled",
                    G_CALLBACK (on_mode_boundary_toggled),
                    NULL);
  g_signal_connect ((gpointer) go_button, "clicked",
                    G_CALLBACK (on_go_button_clicked),
                    NULL);
  g_signal_connect ((gpointer) cancel_button, "clicked",
                    G_CALLBACK (on_cancel_button_clicked),
                    NULL);
  g_signal_connect ((gpointer) getgps_button, "clicked",
                    G_CALLBACK (on_get_gps_button_clicked),
                    NULL);
  g_signal_connect ((gpointer) replace_checkbutton, "toggled",
                    G_CALLBACK (on_replace_checkbutton_toggled),
                    NULL);
  g_signal_connect ((gpointer) map_chooser_button, "selection_changed",
                    G_CALLBACK (on_map_chooser_button_selection_changed),
                    NULL);
  g_signal_connect ((gpointer) classfile_chooser_button, "selection_changed",
                    G_CALLBACK (on_classfile_chooser_button_selection_changed),
                    NULL);
  g_signal_connect ((gpointer) osm_proxy_text, "changed",
                    G_CALLBACK (on_osm_proxy_text_changed),
                    NULL);

  g_signal_connect(G_OBJECT(appdata.program), "notify::is-topmost",
                    G_CALLBACK(ui_main_topmost_cb), NULL);

  _conic_connection_mutex = g_mutex_new();
  _conic_connection_cond = g_cond_new();

  _conic_conn = con_ic_connection_new();
  g_object_set(_conic_conn, "automatic-connection-events", TRUE, NULL);
  g_signal_connect(G_OBJECT(_conic_conn), "connection-event",
                    G_CALLBACK(conic_conn_event), NULL);

  g_set_application_name("GetMaps OSM");
  return window;
}

getmaps_osm_close()
{
   if( getmaps_pid > 0 ){   
      GError *err;
      gchar *com;
      com = g_new(gchar, 255);
      err = g_new(GError, sizeof(GError));
      g_sprintf(com, "kill -9 %i", getmaps_pid);
      g_spawn_command_line_async(com, &err);
      g_free(com);
      g_free(err);
   }
   wait();
   roadmap_gps_shutdown ();
   roadmap_gpsmgr_shutdown();

   g_free( _conic_connection_mutex );
   g_free( _conic_connection_cond );
   g_free(  _conic_conn );

   g_free(getmapsdata.source);
   g_free(getmapsdata.maps);
   g_free(getmapsdata.class);

   gtk_main_quit();

}

int
main (int argc, char *argv[])
{
  GtkWidget *window;
  gint ret;
  gchar infostr[512] = "";

  if (!g_thread_supported ()) g_thread_init (NULL);

   // fix for locale issues
   gtk_disable_setlocale();
   setlocale(LC_ALL, "");
   setlocale(LC_NUMERIC, "C");

  gtk_init (&argc, &argv);
  g_type_init();
  
  appdata.osso_context = osso_init();
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  
  window = create_window ();

  ret = osso_rpc_set_cb_f(appdata.osso_context, 
                               "org.maemo.getmaps", 
                               "/org/maemo/getmaps", 
                               "org.maemo.getmaps",
                               dbus_callback, &appdata);
   if (ret != OSSO_OK) {
       sprintf(infostr, "Error setting D-BUS callback (%d)\n", ret);
       hildon_banner_show_information(GTK_WIDGET(window), NULL, infostr);
       exit (1);
   }
  gtk_widget_show (window);
  conic_recommend_connected();
//  enable next line to figure out screwups

 // buildmap_message_adjust_level (1);
  gtk_main ();

  osso_deinitialize(appdata.osso_context);
  return 0;
}

