/**
 * python-launcher.c
 * Daemon application to start Python and fork it when receiving
 * information from python-launcher (client).
 *
 * Copyright (C) 2005-2008 INdT - Instituto Nokia de Tecnologia
 *
 * Contact: Luciano Miguel Wolf <luciano.wolf@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "Python.h"
#include "python-launcher.h"

#include "debug.h"

#include <stdio.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>

static int start_daemon(void)
{
    int fds[2], pid;
    char *args[3], arg[10], status = ' ';

    if (pipe(fds) < 0) {
        perror("Create start pipe");
        return -1;
    }

    snprintf(arg, 10, "-f%d", fds[1]);
    args[0] = (char *)PL_DEFAULT_PATH;
    args[1] = (char *)arg;
    args[2] = NULL;

    pid = fork();
    if (pid) {
        close(fds[0]);
        execv(args[0], args);
        perror("Execv");
        _exit(-1);
    }
    close(fds[1]);
    if (read(fds[0], &status, sizeof(status)) < 0)
        perror("Read pipe from daemon");
    close(fds[0]);

    if (strncmp(&status, PL_SOCK_READY, 1) == 0)
        return 0;

    return -1;
}

/*
 * Send a message to the daemon.
 *
 * Returns 0 on success or -1 on failure.
 */
static int send_msg(const int c_sock, const char *msg, const int elems)
{
    if (write(c_sock, &elems, sizeof(int)) < 0)
        return -1;
    if (write(c_sock, msg, sizeof(char) * elems) < 0)
        return -1;

    return 0;
}

/*
 * Gets the script's full path.
 *
 * Returns a new string on success or NULL on error.
 */
static char *file_to_exec(const char *fname)
{
    char *script, *path;
    int fname_len = strlen(fname) + 1;
    int path_len;
    FILE *filename;

    path = get_current_dir_name();
    if (!path) {
        perror("Get cur dir");
        return path;
    }
    path_len = strlen(path) + 1;

    if (fname[0] == '/') {
        script = malloc(sizeof(char) * fname_len);
        if (script)
            memcpy(script, fname, fname_len);
        else
            perror("Malloc script name");
    } else if (fname[0] == '.') {
        int cnt = 0;
        if (fname[1] == '.')
            cnt = 2;
        script = malloc(sizeof(char) * (fname_len + path_len - 2 + cnt));
        if (script)
            snprintf(script, fname_len + path_len - 2 + cnt, "%s/%s",
                     path, fname + 2 - cnt);
        else
            perror("Malloc script name");
    } else {
        char buffer[BUFSIZ];
        int read;
        int err;

        script = malloc(sizeof(char) * (strlen("which ") + fname_len));
        if (script)
            snprintf(script, strlen("which ")+fname_len, "which %s", fname);
        else
            perror("Malloc script name");

        filename = popen(script, "r");
        if (filename) {
            read = fread(buffer, sizeof(char), BUFSIZ, filename);
        } else {
            perror("Running which");
            free(script);
            return NULL;
        }
        err = pclose(filename);

        if (err != 0)
            snprintf(buffer, strlen(path) + strlen(fname) + 2, "%s/%s", path, fname);

        script = realloc(script , sizeof(char) * (strlen(buffer) + 1));
        if (!script) {
            perror("Realloc script");
        } else {
            snprintf(script, strlen(buffer) + 1, "%s", buffer);
            if (script[strlen(buffer)-1] == '\n')
                script[strlen(buffer)-1] = '\0';
        }
    }
    free(path);
    return script;
}

/*
 * Fill in buf with a serialized version of argv.
 *
 * Returns a pointer to buf.
 */
static char *marshal_string_array(int *buflen, char *buf, int *reqlen,
                                  int argc, const char * const *argv)
{
    char **offsets, *p, *script;
    int *sizes;
    int i, header_size;

    DEBUG ("argc: %d", argc);
    for (i = 0; i < argc; i++)
        DEBUG ("argv[%d]: '%s'", i, argv[i]);
    DEBUG ("buf: '%s'", buf);

    if (argc < 1) {
        *reqlen = 0;
        return buf;
    }

    script = file_to_exec(argv[1]);
    DEBUG ("script: '%s'", script);
    if (!script)
        return NULL;

    header_size = sizeof(char *) * (argc - 1);
    *reqlen = header_size;
    sizes = alloca(sizeof(int) * (argc - 1));
    for (i = 0; i < (argc - 1); i++) {
        if (i)
            sizes[i] = strlen(argv[i + 1]) + 1;
        else
            sizes[i] = strlen(script) + 1;
        *reqlen += sizes[i];
    }

    DEBUG ("buflen: %d, reqlen: %d", *buflen, *reqlen);

    if (*buflen < *reqlen) {
        char *tmp;

        tmp = realloc(buf, *reqlen);
        if (!tmp) {
            perror("realloc: failed to enlarge buffer");
            *reqlen = 0;
            return buf;
        }

        *buflen = *reqlen;
        buf = tmp;
    }

    offsets = (char **)buf;
    offsets[0] = (char *)0;
    for (i = 0; i < (argc - 2); i++)
        offsets[i + 1] = offsets[i] + sizes[i];

    offsets[0] = (char *)(argc - 1);/*first is always zero, store useful info*/

    p = buf + header_size;

    DEBUG ("p: '%s'", p);

    for (i = 0; i < (argc - 1); p += sizes[i], i++) {
        if (i)
            memcpy(p, argv[i + 1], sizes[i]);
        else
            memcpy(p, script, sizes[i]);
    }
    free(script);

    DEBUG ("buf: '%s'", buf);

    return buf;
}

/*
 * Try to open a new connection with daemon.
 *
 * Returns sock number on success or -1 on failure.
 */
static int client_connect(const char *sock_path)
{
    int c_sock, addrlen, start = 1;
    struct sockaddr_un c_sockad = {
            AF_LOCAL, PL_SOCK_PATH
    };

    c_sock = socket(PF_LOCAL, SOCK_STREAM, 0);
    if (c_sock < 0) {
        perror("Create socket");
        return -1;
    }

    addrlen = SUN_LEN(&c_sockad);

    while (connect(c_sock, (struct sockaddr_un *)&c_sockad, addrlen) < 0) {
        if ((errno == ENOENT || errno == ECONNREFUSED) && start) {
            start = 0;
            if (start_daemon() == 0)
                continue;
        }

        perror("Client connect");
        close(c_sock);
        return -1;
    }

    return c_sock;
}

int main(int argc, const char *argv[])
{
    int i;
    char *msg = NULL;
    int c_sock, reqlen, buflen = 0;
    uid_t uid;
    load_type opt;

    uid = getuid();
    opt = pl_get_load_opt();

    /* No arguments, "-something" parameter or invoked by root?
     * Just call python2.5 */
    if (argc == 1)
        opt = PL_DISABLED;
    else if (argv[1][0] == '-')
        opt = PL_DISABLED;
    else if (uid == 0)
        opt = PL_DISABLED;

    if (opt == PL_DISABLED) {
        execv(PYTHON_DEFAULT, (char **)argv);
        exit(EXIT_FAILURE);
        return -1;
    }

    DEBUG ("argc: %d", argc);
    for (i = 0; i < argc; i++)
        DEBUG ("argv[%d]: '%s'", i, argv[i]);

    msg = marshal_string_array(&buflen, msg, &reqlen, argc, argv);

    DEBUG ("msg: '%s'", msg);

    if (!msg)
        fprintf(stderr, "Possible incorrect script name.\n");
    else {
        c_sock = client_connect(PL_SOCK_PATH);
        if (c_sock < 0) {
            free(msg);
            return -1;
        }
        if (send_msg(c_sock, msg, reqlen) < 0)
            fprintf(stderr, "Problem sending message to the daemon.\n");
        free(msg);
        close(c_sock);
    }

    return 0;
}

/* vim:et:sw=4:ts=4:co=80
 */
