#
# This file is part of Python Download Manager
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Author: Kenneth Christiansen <kenneth.christiansen@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import urllib2
import urlparse
import os
import stat
import shutil
import gzip
import zlib
import threading
import gobject
import dbus.service
import logging
import constants
import mimetypes
from constants import DownloadState


USER_AGENT = "DownloadManager"


from dbus import DBusException

log = logging.getLogger("downloadmanager.client")


class DownloadSession(dbus.service.Object):
    """
    Class to be used by clients to create and manage sessions.
    """
    DBUS_SERVICE_NAME = constants.Session.DBUS_SERVICE_NAME
    DBUS_OBJ_PATH     = constants.Session.DBUS_OBJ_PATH
    DBUS_IFACE        = constants.Session.DBUS_IFACE
    BUF_SIZE          = 4096

    # indicating if dbus service name was already registered
    dbus_registered = False
    bus_name = None

    def __init__(self, session_bus, sender, http_uri, local_path):
        self.dbus_id = sender
        self.session_bus = session_bus
        self.session_path = None

        self.http_uri = http_uri
        self.local_path = local_path
        self.mime_type = None

        self.id = id(self)
        self._dbus_register()

        self._worker = None
        self._outfile = None
        self._infile = None
        self._mime_type_filter = None

        self._clean_vars()
        self.is_working = False
        self._state = DownloadState.NOT_STARTED

        self.compression = None
        self.compress = False

    def _clean_vars(self):
        self._cancel = False
        self._pause = False
        self._filter = False
        self.total_size = -1
        self.done_size = 0
        self.__excpt = None

    def _dbus_register(self):
        try:
            if not self.dbus_registered:
                self.bus_name = dbus.service.BusName(self.DBUS_SERVICE_NAME,
                                                     bus=self.session_bus)
                self.dbus_registered = True

            dbus.service.Object.__init__(self, self.bus_name,
                "%s/%s" % (self.DBUS_OBJ_PATH, self.id))
            log.debug("Dbus session registered")
        except DBusException, e:
            raise DBusRegisterError(e)

    @dbus.service.method(DBUS_IFACE)
    def SetAcceptedMimeTypes(self, accept_list):
        self._mime_type_filter = accept_list

    @dbus.service.method(DBUS_IFACE)
    def GetExceptionInfo(self):
        e = self.__excpt
        if e is None:
            return None

        if isinstance(e, urllib2.HTTPError):
            etype = constants.ExceptionType.HTTP
            emsg = e.msg
            errno = e.code
        elif isinstance(e, urllib2.URLError):
            etype = constants.ExceptionType.URL
            try: # URLError only wraps socket.error like this
                emsg = e.args[0][1]
                errno = e.args[0][0]
            except:
                emsg = e.args
                errno = -1
        elif isinstance(e, IOError):
            etype = constants.ExceptionType.IO
            emsg = e.strerror
            errno = e.errno
        elif isinstance(e, OSError):
            etype = constants.ExceptionType.IO
            emsg = e.strerror
            errno = e.errno
        else:
            etype = constants.ExceptionType.UNKNOWN
            errno = -1

            if hasattr(e, "message") and e.message:
                return etype, errno, e.message

            if hasattr(e, "msg") and e.msg:
                 return etype, errno, e.msg

            if hasattr(e, "strerror") and e.strerror:
                return etype, errno, e.strerror

            emsg = str(e)

        return etype, errno, emsg

    def __set_state(self, value):
        self.StateChanged(value)

    @dbus.service.signal(DBUS_IFACE)
    def StateChanged(self, value):
        self._state = value
        log.debug("Changed State: %s", value)
        return True

    def main_thread_emit_state_changed(self, state):
        def f():
            self.state = state
            return False
        gobject.idle_add(f)

    def __get_state(self):
        return self._state

    @dbus.service.method(DBUS_IFACE)
    def SetState(self, value):
        self.state = value

    @dbus.service.method(DBUS_IFACE)
    def GetState(self):
        return self.state

    state = property(__get_state, __set_state)

    @dbus.service.method(DBUS_IFACE)
    def MarkAsQueued(self):
        self.state = DownloadState.QUEUED

    @dbus.service.method(DBUS_IFACE)
    def GetProgress(self):
        return self.done_size, self.total_size

    @dbus.service.method(DBUS_IFACE)
    def GetTotalSize(self):
        return self.total_size

    @dbus.service.method(DBUS_IFACE)
    def GetMimeType(self):
        return self.mime_type

    @dbus.service.method(DBUS_IFACE)
    def Pause(self):
        self._pause = True

        if not self.is_working:
            self.state = DownloadState.PAUSED
        else:
            self.is_working = False

    def _do_cancel(self):
        try:
            os.remove(self.local_path)
        except Exception, e:
            log.warning("Could not remove cancelled file %r",
                        self.local_path, exc_info=True)
        self.state = DownloadState.CANCELLED
        self.state = DownloadState.NOT_STARTED # XXX remove me? test canola

    @dbus.service.method(DBUS_IFACE)
    def Cancel(self):
        self._cancel = True
        if not self.is_working:
            self._do_cancel()
        else:
            self.is_working = False

    def main_thread_emit_cancel(self):
        def f():
            self._do_cancel()
            return False
        gobject.idle_add(f)

    @staticmethod
    def _get_compression(info):
        compression = info.getheader('content-encoding', None)
        if compression not in ['gzip', 'deflate']:
            return None
        else:
            return compression

    def _get_mimetype(self, info):
        mimetype_and_charset = info.getheader("content-type", None)

        if mimetype_and_charset is None:
            redirected_uri = self._infile.geturl()
            return mimetypes.guess_type(redirected_uri)[0]
        else:
            return mimetype_and_charset.split(";")[0]

    @staticmethod
    def _get_content_range(info):
        spec = info.getheader("content-range")
        if spec is None:
            return None
        try:
            unit, trange = spec.split(' ')
            if unit != "bytes":
                return None
            range, total = trange.split('/')
            start, stop = range.split('-')
            return (long(start), long(stop), long(total))
        except Exception, e:
            log.warning("Unable to parse Content-Range: %r", spec)
            return None

    def _download_worker_process(self):
        # Runs from thread! May raise exception.
        infile = self._infile
        outfile = self._outfile
        bufsize = self.BUF_SIZE
        while self.is_working:
            buf = infile.read(bufsize)
            if not buf:
                return
            outfile.write(buf)
            self.done_size += len(buf)

    def _download_worker_int(self, start):
        # Runs from thread! May raise exception.
        uri_info = urlparse.urlparse(self.http_uri)

        quest_index = uri_info.geturl().find("?")
        if quest_index >= 0:
            base_url = uri_info.geturl()[:quest_index]
            params = uri_info.geturl()[quest_index:]
        else:
            base_url = uri_info.geturl()
            params = ""

        address = base_url.split("@")[-1].split("://")[-1]

        password = uri_info.password
        username = uri_info.username

        uri = uri_info.scheme + "://" + address + params
        server = uri_info.netloc.split("@")[-1]

        if password is not None and username is not None:
            passman = urllib2.HTTPPasswordMgrWithDefaultRealm()
            passman.add_password(None, server, username, password)
            authhandler = urllib2.HTTPBasicAuthHandler(passman)
            opener = urllib2.build_opener(authhandler)
            urllib2.install_opener(opener)

        req = urllib2.Request(uri)

        # Let's accept zlib, gzip compressed data (quicker download)
        if self.compress:
            req.add_header('Accept-encoding', 'gzip, deflate')

        if start > 0:
            req.add_header("Range", "bytes=%d-" % start)

        try:
            self._infile = urllib2.urlopen(req)
        except urllib2.HTTPError, e:
            if start > 0 and e.code == 416: # Requested Range Not Satisfiable
                self.total_size = self.done_size = start
                return
            raise

        info = self._infile.info()

        self.mime_type = self._get_mimetype(info)

        if self._mime_type_filter is not None:
            if not self.mime_type in self._mime_type_filter:
                self._filter = True
                return

        self.compression = self._get_compression(info)

        self.done_size = start
        if start <= 0:
            self.total_size = int(info.getheader("content-length", 0))
        else:
            content_range = self._get_content_range(info)
            if content_range is not None:
                rstart, rstop, rtotal = content_range
                if rstart == start:
                    self.total_size = rtotal
                else:
                    # Oops! File changed? Restart!
                    log.warning("File changed? Restart! Requested %r, Got %r",
                                start, rstart)
                    self._outfile.truncate(0)
                    self._infile.close()
                    return self._download_worker_int(0)
            else:
                # When we set range with a number that is equal to the size
                # of the file or larger, the returned contents-range is empty
                # and we land here.
                self.total_size = int(info.getheader("content-length", 0))
                if start == self.total_size:
                    self._close_files()
                    self.main_thread_emit_state_changed(DownloadState.COMPLETED)
                    return
                else:
                    # Oops! File changed? Restart!
                    log.warning("File changed? Restart! Requested %r, "
                                "But total size is %r", start, self.total_size)
                    self._outfile.truncate(0)
                    self._infile.close()
                    return self._download_worker_int(0)

        self.main_thread_emit_state_changed(DownloadState.IN_PROGRESS)
        self._download_worker_process()

    def _main_thread_collect_worker(self):
        def f():
            self._worker.join(None)
            alive = self._worker.isAlive()
            if not alive:
                self._worker = None
            return alive
        gobject.timeout_add(500, f)

    def _download_worker(self, start):
        # Runs from thread! Must not raise exceptions.
        try:
            self._download_worker_int(start)
            self._close_files()
            if self._cancel:
                self.main_thread_emit_cancel()
            elif self._pause:
                self.main_thread_emit_state_changed(DownloadState.PAUSED)
            elif self._filter:
                self.main_thread_emit_state_changed(DownloadState.FILTERED)
            else:
                self._uncompress()
                self.main_thread_emit_state_changed(DownloadState.COMPLETED)
        except Exception, e:
            self.__excpt = e
            log.error("Error downloading: %s", e, exc_info=True)
            self.main_thread_emit_state_changed(DownloadState.EXCEPTION)

        self._main_thread_collect_worker()

    def _uncompress(self):
        log.info("Compression: %s", str(self.compression))
        compressed_path = self.local_path + ".gz"

        if self.compression is not None:
            shutil.move(self.local_path, compressed_path)
            outfile = open(self.local_path, 'wb')
            bufsize = self.BUF_SIZE
        else:
            return

        if self.compression == 'gzip':
            infile = gzip.GzipFile(compressed_path, 'rb')

            while True:
                buf = infile.read(bufsize)
                if not buf:
                    break
                outfile.write(buf)

            outfile.close()
            infile.close()

        if self.compression == 'zlib':
            infile = open(compressed_path, 'rb')
            cdata = infile.read()
            data = zlib.decompress(cdata, -zlib.MAX_WBITS)
            outfile.write(data)
            outfile.close()
            infile.close()

        try:
            if os.path.exists(compressed_path):
                os.unlink(compressed_path)
        except IOError, e:
            log.debug("Couldn't remove no longer needed "
                      "compressed data file: %s" % compressed_path)

    def _close_files(self):
        def close_or_log(f):
            try:
                f.close()
            except Exception, e:
                log.error("Could not close: %s", e)

        if self._infile:
            close_or_log(self._infile)
            self._infile = None
        if self._outfile:
            close_or_log(self._outfile)
            self._outfile = None

    @dbus.service.method(DBUS_IFACE)
    def Start(self, resume=True, compress=False):
        if self._worker is not None:
            return

        self.state = DownloadState.CONNECTING
        self._clean_vars()
        self.compress = compress

        try:
            dir_path = os.path.dirname(self.local_path)
            if not os.path.exists(dir_path):
                os.makedirs(dir_path)

            if not resume:
                self._outfile = open(self.local_path, "wb")
                args = (0,)
            else:
                self._outfile = open(self.local_path, "ab")
                args = (os.fstat(self._outfile.fileno())[stat.ST_SIZE],)

            self.is_working = True
            self._worker = threading.Thread(target=self._download_worker, args=args)
            self._worker.start()
        except Exception, e:
            self.__excpt = e
            self.state = DownloadState.EXCEPTION
