#include <signal.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <string.h>

/* config.h */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef ENABLE_NLS
/* internationalisation headers */
#include <glib/gi18n-lib.h>
#endif

#ifdef _RUN_STANDALONE_
#	define _(s) s
#	define PIDGIN_HIG_CAT_SPACE 5
#	define PIDGIN_HIG_BORDER 2
#	define PIDGIN_HIG_BOX_SPACE 5
#else /* !_RUN_STANDALONE_ */
#	include "pidgin.h"
#endif /* _RUN_STANDALONE_ */

#include "window-ipc.h"

#define OVAL_EXTRA 10
#define PIGEON_SPACING 30

#define CX_PIGEON 20
#define CY_PIGEON 30

#define X_OVALS 5
#define Y_OVALS 1
#define N_STEPS 10

#define ANIM_TIMEOUT 50 /* ms */
#define ACTION_TIMEOUT_MIN 1000 /* ms */
#define ACTION_TIMEOUT_MAX 2000 /* ms */
#define ENTROPY_UPDATE_TIMEOUT 1000 /* ms */

/* standing-bird-pidgin-logo.png file */
static const guchar pidgin_pigeon[] = 
{
    0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d, 0x49, 0x48, 0x44, 0x52, 
    0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x28, 0x08, 0x06, 0x00, 0x00, 0x00, 0x15, 0x06, 0x61, 
    0xd8, 0x00, 0x00, 0x00, 0x06, 0x62, 0x4b, 0x47, 0x44, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xa0, 
    0xbd, 0xa7, 0x93, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x01, 0x05, 0x00, 
    0x00, 0x01, 0x05, 0x01, 0x60, 0x6a, 0x12, 0xd1, 0x00, 0x00, 0x00, 0x07, 0x74, 0x49, 0x4d, 0x45, 
    0x07, 0xd7, 0x07, 0x13, 0x07, 0x21, 0x3a, 0x7c, 0xbf, 0x78, 0x82, 0x00, 0x00, 0x07, 0xe5, 0x49, 
    0x44, 0x41, 0x54, 0x48, 0xc7, 0xa5, 0x97, 0x79, 0x8c, 0x55, 0xd5, 0x1d, 0xc7, 0x3f, 0x67, 0xbb, 
    0xf7, 0xcd, 0xdb, 0x66, 0xde, 0x38, 0xe3, 0x2c, 0x0c, 0xc8, 0x30, 0x44, 0xa0, 0x80, 0x54, 0x18, 
    0x51, 0x61, 0xb4, 0x8a, 0x82, 0xc6, 0x2d, 0x2a, 0xa2, 0x68, 0xa9, 0x6d, 0x53, 0xd4, 0xd6, 0x46, 
    0x23, 0xb6, 0x36, 0x6d, 0xd4, 0xfe, 0xd1, 0x9a, 0x74, 0x49, 0xb4, 0x69, 0x6c, 0x2c, 0xda, 0x16, 
    0xb5, 0xb5, 0xb6, 0xe2, 0xd2, 0x2a, 0x76, 0x31, 0xb6, 0x6a, 0x17, 0x6a, 0x9a, 0x4c, 0x55, 0xdc, 
    0x10, 0xab, 0xc8, 0x26, 0x0c, 0x33, 0xcc, 0xcc, 0x9b, 0x79, 0xf3, 0x96, 0xbb, 0x9d, 0xd3, 0x3f, 
    0x66, 0x4a, 0xc1, 0x05, 0x29, 0x9e, 0xe4, 0xe4, 0x26, 0xf7, 0xdc, 0xdf, 0xe7, 0x7e, 0xf3, 0xbb, 
    0xbf, 0xed, 0x0a, 0xe7, 0x1c, 0x1f, 0xb5, 0xce, 0x99, 0x79, 0xe1, 0x79, 0x85, 0x4c, 0xd3, 0x6f, 
    0x9d, 0x73, 0x61, 0xa9, 0x3c, 0xba, 0x7a, 0xc3, 0x96, 0x87, 0x1f, 0xe4, 0x30, 0x96, 0x3c, 0x9c, 
    0x87, 0x3c, 0x95, 0x5a, 0x25, 0x84, 0x52, 0x4a, 0x9a, 0xba, 0x86, 0x5c, 0xe1, 0x97, 0x17, 0xcc,
    0xbe, 0xf4, 0xe6, 0xc3, 0xb1, 0x13, 0x1f, 0xa5, 0xfc, 0x92, 0xb9, 0x57, 0xfe, 0x4d, 0x6a, 0xd9, 
    0x83, 0x05, 0x29, 0x24, 0xd6, 0x59, 0xa4, 0x04, 0x97, 0x88, 0x5d, 0x0f, 0xbd, 0x7c, 0xef, 0xe4, 
    0x23, 0x56, 0x7e, 0xc1, 0xec, 0x95, 0x57, 0x2b, 0x25, 0x7b, 0xa4, 0x93, 0x28, 0xa9, 0x90, 0x52, 
    0xa2, 0x94, 0x06, 0x14, 0x42, 0x89, 0x8e, 0x8b, 0x67, 0x5f, 0xf1, 0xeb, 0x23, 0x86, 0x37, 0x35, 
    0x14, 0xbe, 0x95, 0x49, 0xe5, 0xe9, 0x3e, 0x69, 0x3e, 0x0b, 0xe6, 0x2d, 0x20, 0x9f, 0xce, 0xe3, 
    0x1b, 0x0f, 0xdf, 0xf8, 0x74, 0xb5, 0x4d, 0x67, 0xcd, 0xf7, 0xbe, 0x7c, 0xd9, 0x55, 0x67, 0x5d, 
    0x77, 0xf5, 0x11, 0xc1, 0x33, 0x85, 0x6c, 0xae, 0x7d, 0xea, 0xd1, 0x64, 0xba, 0x3c, 0x4c, 0xa7, 
    0xa3, 0xb9, 0xad, 0x09, 0xdf, 0xa4, 0xf0, 0x8d, 0xc7, 0xb4, 0x53, 0x27, 0x31, 0x3c, 0x3a, 0xc2, 
    0x51, 0xcd, 0x85, 0xcb, 0x3f, 0xcc, 0x5e, 0x1f, 0x0a, 0x9e, 0x6f, 0xc8, 0x69, 0x3b, 0x0a, 0xf9, 
    0x54, 0x3d, 0x91, 0x8d, 0xd9, 0x36, 0xb4, 0x87, 0xb6, 0x0e, 0x0f, 0x2d, 0x61, 0xe0, 0xad, 0x61, 
    0xf6, 0xbc, 0xb3, 0x97, 0xd7, 0x5e, 0x78, 0xe3, 0xd6, 0x23, 0x82, 0xa7, 0x7c, 0xcf, 0x0e, 0x8e, 
    0x95, 0xd8, 0xf2, 0xec, 0x36, 0x08, 0xf7, 0x71, 0x74, 0x43, 0x89, 0xe3, 0x26, 0x95, 0x08, 0x8a, 
    0x83, 0x6c, 0xed, 0x6b, 0x64, 0xc0, 0xef, 0xec, 0xdb, 0xf0, 0xfa, 0xfa, 0xbf, 0x1f, 0x91, 0x5b, 
    0x4c, 0xda, 0x77, 0x52, 0x4a, 0x6a, 0xe5, 0x1a, 0x46, 0x69, 0xf2, 0xc5, 0x5e, 0x4e, 0xea, 0x69, 
    0xe1, 0x93, 0x73, 0x32, 0x78, 0xd1, 0x5e, 0xa2, 0xa8, 0xb6, 0xfd, 0x50, 0xf6, 0x87, 0x54, 0x6e, 
    0xb4, 0x91, 0x5a, 0x69, 0x8c, 0x36, 0xd4, 0x62, 0xc3, 0x0e, 0x73, 0x01, 0x4f, 0x3c, 0xb1, 0x99,
    0x94, 0x74, 0xf4, 0xe9, 0x25, 0x44, 0xd1, 0xc0, 0xf6, 0x23, 0x8a, 0x96, 0xe3, 0x0b, 0xa7, 0x37, 
    0x14, 0xf7, 0x8d, 0x78, 0x4a, 0xa9, 0xff, 0x29, 0x51, 0x9a, 0xb7, 0xfa, 0xe7, 0xf2, 0x6a, 0x5f, 
    0x37, 0x02, 0x41, 0xae, 0x3e, 0xbb, 0xe0, 0xff, 0x86, 0x77, 0x4f, 0xea, 0x4e, 0x7f, 0x6a, 0xf1, 
    0x89, 0x03, 0x7b, 0x77, 0xf4, 0x8b, 0x43, 0x19, 0xb7, 0xb4, 0xb6, 0x75, 0x5d, 0xb5, 0xe4, 0xfa, 
    0xd7, 0x0f, 0x3b, 0x43, 0x17, 0x4d, 0x5e, 0xd4, 0x38, 0xa7, 0xeb, 0x84, 0x81, 0xa8, 0x16, 0x4b, 
    0x4f, 0x7b, 0x14, 0xb2, 0x85, 0x89, 0xc4, 0x79, 0xff, 0x2a, 0x55, 0x4b, 0x94, 0xab, 0x65, 0x12, 
    0x19, 0x17, 0xef, 0xdb, 0x78, 0x57, 0xe1, 0x90, 0xca, 0x2f, 0x9e, 0x73, 0xf9, 0x4f, 0x5a, 0x1a, 
    0xa7, 0x0c, 0x0e, 0x0f, 0x17, 0xa5, 0x73, 0x16, 0x6b, 0x1d, 0xe5, 0x5a, 0x99, 0x0f, 0x2b, 0x11, 
    0x71, 0x1c, 0x01, 0x60, 0x23, 0xdb, 0x70, 0xc5, 0xc9, 0x9f, 0x77, 0xe7, 0xcf, 0x58, 0x79, 0xfb, 
    0x07, 0xc2, 0x2f, 0x9d, 0x7b, 0xe5, 0xdb, 0xb5, 0x28, 0x5c, 0x1d, 0x47, 0x8e, 0x30, 0x0a, 0x48, 
    0xac, 0x23, 0xb6, 0x31, 0x51, 0x12, 0x93, 0x24, 0xf1, 0xfb, 0xc0, 0x51, 0x1c, 0x91, 0x58, 0x8b, 
    0x75, 0x09, 0x95, 0xda, 0x18, 0xa5, 0x91, 0x32, 0x91, 0x0b, 0x6f, 0x5c, 0x3e, 0x67, 0xd5, 0xeb, 
    0x07, 0xc1, 0xcf, 0x9f, 0xb5, 0x62, 0xdd, 0x48, 0x30, 0x3a, 0x8d, 0x09, 0x0f, 0x27, 0xce, 0x52, 
    0x0d, 0x2b, 0x24, 0x36, 0x26, 0x8c, 0x43, 0x46, 0x2a, 0xa3, 0x04, 0x61, 0x40, 0x14, 0x47, 0x44, 
    0x71, 0x44, 0x10, 0x05, 0x54, 0x6a, 0x65, 0x6a, 0x61, 0x8d, 0x7d, 0x23, 0x03, 0x04, 0x71, 0x04, 
    0x08, 0xa4, 0x10, 0x94, 0xc3, 0xf2, 0xac, 0xf3, 0x67, 0xad, 0x58, 0x07, 0x20, 0x16, 0xb4, 0x2f,
    0x48, 0x37, 0x66, 0xda, 0xcb, 0x4a, 0x1a, 0x04, 0xa0, 0xa4, 0x46, 0x48, 0x81, 0x16, 0x0a, 0xa3, 
    0x0c, 0xbe, 0x49, 0xa1, 0x94, 0x42, 0x6b, 0x43, 0xc6, 0x4b, 0xe3, 0x80, 0x24, 0x89, 0x19, 0xab, 
    0x95, 0x19, 0x19, 0x1b, 0x46, 0x28, 0x49, 0x9c, 0xc4, 0x38, 0xe7, 0x48, 0x6c, 0x42, 0x62, 0x63, 
    0x1c, 0x96, 0xa1, 0xf2, 0xee, 0x8c, 0x3a, 0xbe, 0x6d, 0xd1, 0x6d, 0xd5, 0xa8, 0xba, 0x38, 0x65, 
    0xea, 0xd0, 0xca, 0xa0, 0xb5, 0x46, 0x4b, 0x8d, 0x94, 0x8a, 0x30, 0x09, 0x71, 0x76, 0xdc, 0xdf, 
    0xe3, 0x86, 0x16, 0xe7, 0x2c, 0xa5, 0x5a, 0x89, 0x6a, 0x58, 0xc5, 0x62, 0x91, 0x52, 0x22, 0x85, 
    0x42, 0x49, 0x89, 0x14, 0x02, 0x81, 0x20, 0x88, 0x02, 0x1a, 0xea, 0x9a, 0x8d, 0xae, 0xcf, 0xd7, 
    0x5f, 0x14, 0x0c, 0x07, 0xfc, 0x37, 0x59, 0xb4, 0xd2, 0x48, 0x31, 0xfe, 0x29, 0x84, 0x10, 0x2c, 
    0x2b, 0x34, 0xf3, 0xa7, 0xc1, 0x7e, 0x62, 0x29, 0x88, 0xe3, 0x08, 0x07, 0x38, 0xe7, 0xe8, 0xce, 
    0xd7, 0xd3, 0x3b, 0x5a, 0xc4, 0x48, 0x83, 0x13, 0x00, 0x8e, 0x24, 0x49, 0x10, 0x42, 0x62, 0x6c, 
    0x4c, 0xa1, 0xbe, 0xfe, 0x22, 0x99, 0xad, 0xcf, 0x4c, 0x32, 0xda, 0xc3, 0xe1, 0x48, 0x79, 0x29, 
    0x94, 0xd4, 0xf8, 0x26, 0x85, 0xa7, 0x7d, 0xba, 0xea, 0xd2, 0x2c, 0xeb, 0xe8, 0xe4, 0xfb, 0xf3, 
    0x4f, 0xc6, 0x25, 0x09, 0xb5, 0x28, 0x24, 0x88, 0x02, 0x7c, 0x91, 0x70, 0x79, 0xd7, 0x4c, 0x9a, 
    0xb5, 0x46, 0x1b, 0x43, 0xca, 0xa4, 0xd0, 0xca, 0x60, 0xb4, 0x87, 0x10, 0x02, 0x4f, 0x7b, 0xd4, 
    0xe5, 0xd2, 0x93, 0x64, 0xbe, 0x90, 0xf7, 0x8d, 0xd2, 0x68, 0xa5, 0xf0, 0xb4, 0x87, 0x6f, 0x3c, 
    0x3c, 0xe3, 0x21, 0x04, 0x7c, 0xb6, 0x6b, 0x26, 0x32, 0x65, 0x50, 0x75, 0x1e, 0x41, 0x1c, 0x63,
    0x6d, 0x84, 0x06, 0x6e, 0x3d, 0xb1, 0x07, 0x55, 0xe7, 0x73, 0xd5, 0xcc, 0xb9, 0x48, 0xc0, 0x18, 
    0x83, 0xa7, 0x0d, 0xbe, 0xf1, 0xc6, 0x3d, 0xa0, 0x0c, 0xf5, 0x0d, 0x79, 0x5f, 0xe7, 0x1a, 0x72, 
    0x52, 0x2b, 0x8d, 0x56, 0x06, 0x25, 0x15, 0x42, 0x48, 0xb4, 0x52, 0xe4, 0x62, 0x47, 0x36, 0x93, 
    0x45, 0x7a, 0x1a, 0xb4, 0xa2, 0xbb, 0xbd, 0x95, 0xc5, 0x1d, 0x53, 0xf9, 0x44, 0x4b, 0x1b, 0xce, 
    0x25, 0xb8, 0xd8, 0xd2, 0xa2, 0x0a, 0x64, 0x04, 0x08, 0x6d, 0x08, 0x01, 0x9c, 0xc3, 0x9b, 0x50, 
    0x9f, 0x6b, 0xcc, 0x4a, 0xed, 0x92, 0x24, 0x92, 0x52, 0x79, 0x4a, 0x2a, 0xa2, 0x24, 0xc6, 0x68, 
    0x03, 0xc0, 0xa7, 0xda, 0xda, 0x91, 0x9e, 0x46, 0xf8, 0x06, 0xe9, 0x19, 0xae, 0x59, 0x74, 0x0a, 
    0x28, 0x05, 0x38, 0x48, 0x2c, 0x2e, 0x4e, 0xb0, 0x52, 0xd0, 0xd3, 0xd6, 0xc6, 0xc6, 0xe2, 0xd8, 
    0x78, 0xec, 0x27, 0x31, 0x42, 0x08, 0x94, 0x54, 0x90, 0xb8, 0x48, 0x56, 0x2a, 0xd5, 0x51, 0x81, 
    0x40, 0x0a, 0x49, 0x9c, 0x44, 0x58, 0x6b, 0xa9, 0x06, 0x55, 0xe6, 0xb7, 0x4f, 0x42, 0x78, 0x1a, 
    0xe9, 0x19, 0xa4, 0x6f, 0x90, 0x75, 0x29, 0x54, 0x9d, 0x8f, 0xaa, 0x4b, 0x21, 0x53, 0x3e, 0x22, 
    0xe5, 0x21, 0x7d, 0x8f, 0xee, 0x8e, 0x29, 0x58, 0xeb, 0xb0, 0xd6, 0x12, 0x27, 0xd1, 0xfe, 0x40, 
    0x28, 0x97, 0x6b, 0xa3, 0xba, 0x52, 0xaa, 0xec, 0x72, 0xb8, 0x26, 0xeb, 0x2c, 0x4a, 0x28, 0x9c, 
    0xb3, 0xf8, 0x00, 0x4a, 0x21, 0x8d, 0x46, 0x4c, 0xc0, 0x85, 0xd1, 0x30, 0x91, 0x65, 0x42, 0x4b, 
    0x08, 0x04, 0xd6, 0x3a, 0x72, 0xe9, 0x34, 0x51, 0x54, 0x23, 0xb6, 0xc9, 0xfe, 0xec, 0x75, 0xce, 
    0x51, 0x29, 0x8d, 0xbd, 0x2b, 0x4b, 0xc5, 0xe1, 0x67, 0x12, 0x9b, 0x90, 0x24, 0x09, 0xd6, 0x59,
    0xf2, 0xcd, 0x39, 0x3a, 0xe7, 0x4c, 0x06, 0x29, 0x41, 0x4a, 0x84, 0x96, 0x08, 0xad, 0x41, 0x88, 
    0x89, 0x7b, 0x02, 0x84, 0x40, 0x1a, 0x85, 0x50, 0x12, 0xa1, 0x04, 0x1d, 0x9d, 0xcd, 0xf8, 0x19, 
    0x7f, 0xbc, 0xce, 0x38, 0x4b, 0x9c, 0xc4, 0x94, 0x46, 0x8a, 0xcf, 0xe8, 0xbe, 0xb7, 0xdf, 0xfd, 
    0xe6, 0x17, 0xbf, 0x7d, 0xdd, 0x1a, 0x93, 0x36, 0x62, 0xda, 0xcc, 0x63, 0x98, 0x3a, 0xe3, 0x18, 
    0xee, 0xb9, 0x79, 0x2d, 0x2e, 0x8a, 0x21, 0x9d, 0x1a, 0x17, 0x2b, 0x00, 0x04, 0x62, 0xe2, 0xea, 
    0x9c, 0xdb, 0xff, 0x12, 0x97, 0x58, 0xf2, 0x4d, 0x39, 0x6e, 0xba, 0xfb, 0x26, 0xb6, 0x6d, 0xd9, 
    0xc1, 0x6b, 0xbd, 0x9b, 0xd1, 0x4a, 0xbb, 0xb5, 0xb7, 0xdc, 0x79, 0xab, 0x70, 0xce, 0x71, 0xcf, 
    0x6d, 0xf7, 0xed, 0xcc, 0x15, 0x32, 0x1d, 0x03, 0x7b, 0x06, 0xc9, 0xe6, 0x33, 0x74, 0x1e, 0x3b, 
    0x85, 0x69, 0x3f, 0x7f, 0x09, 0xbf, 0xab, 0x05, 0x55, 0x9f, 0x46, 0xa5, 0x7c, 0x50, 0x72, 0xbf, 
    0x5b, 0x70, 0x0e, 0x92, 0x84, 0x64, 0xac, 0x4a, 0xb0, 0xad, 0x9f, 0x2d, 0x17, 0xcd, 0xa0, 0xf7, 
    0xb9, 0x17, 0x69, 0x6e, 0x6f, 0x22, 0x9d, 0x4d, 0x33, 0x32, 0x54, 0xda, 0x75, 0xcd, 0x2d, 0x9f, 
    0x9b, 0x2c, 0x01, 0x9e, 0x7c, 0xe0, 0xb1, 0xcb, 0xa4, 0x55, 0xac, 0xbc, 0x76, 0x39, 0x47, 0xb5, 
    0x36, 0x72, 0xff, 0x9d, 0xbf, 0xc2, 0x06, 0x21, 0xe1, 0x8e, 0x41, 0x92, 0xa1, 0x31, 0x6c, 0x10, 
    0x81, 0x75, 0xe3, 0x50, 0xe7, 0xc0, 0x5a, 0xa2, 0xe1, 0x0a, 0xc1, 0xf6, 0x7d, 0xec, 0x1d, 0x1a, 
    0xe6, 0x81, 0x3b, 0x1f, 0xa2, 0xfb, 0xb4, 0xe3, 0x39, 0x6f, 0xd5, 0xd9, 0xa4, 0xb2, 0x3e, 0xbf, 
    0x7b, 0xe0, 0xb1, 0xcb, 0x0e, 0x6a, 0x16, 0x5f, 0x59, 0xf1, 0xf5, 0x3f, 0x57, 0x4b, 0xc9, 0x92,
    0x55, 0x5f, 0xbd, 0x84, 0x8d, 0x4f, 0x3f, 0x4f, 0xe1, 0xd9, 0xb7, 0x39, 0xb3, 0xad, 0x73, 0x22, 
    0x3a, 0x3c, 0x64, 0xbe, 0x0e, 0x31, 0xd1, 0xf2, 0x6c, 0x35, 0xc0, 0xd5, 0x42, 0x6c, 0x35, 0x64, 
    0xfd, 0xd6, 0x57, 0xa9, 0xbb, 0x70, 0x3e, 0xd3, 0x67, 0x4d, 0xe3, 0xf1, 0xb5, 0x7f, 0xc0, 0xcf, 
    0x9b, 0x67, 0xee, 0x78, 0xe4, 0x3b, 0x67, 0x1c, 0xd4, 0xa0, 0x77, 0x3e, 0xb2, 0x75, 0x59, 0xfe, 
    0xd4, 0xd6, 0x91, 0x17, 0x9e, 0x7f, 0x29, 0xf3, 0xe4, 0xfa, 0x3f, 0x92, 0xd7, 0x1e, 0x67, 0x34, 
    0x76, 0x90, 0x08, 0x81, 0x73, 0x16, 0x1b, 0xc5, 0x08, 0x39, 0x5e, 0x73, 0x9c, 0x4d, 0x70, 0x51, 
    0x82, 0x0d, 0x42, 0x9e, 0x1e, 0xde, 0x85, 0x5a, 0x3f, 0xc0, 0xb9, 0xe7, 0x9d, 0x4d, 0x10, 0x04, 
    0xe5, 0x81, 0x47, 0x77, 0x2c, 0xfb, 0xc0, 0x36, 0xd7, 0xdd, 0xd8, 0x5d, 0x3f, 0x7d, 0xca, 0xdc, 
    0xdd, 0x46, 0x7b, 0xe9, 0x28, 0x0a, 0x83, 0xa5, 0xb9, 0xb6, 0xbb, 0x1b, 0xd2, 0xe2, 0xfa, 0x05, 
    0x5e, 0x01, 0xb4, 0x3a, 0x08, 0xfe, 0xa6, 0xad, 0xd0, 0x7f, 0x42, 0x3b, 0x3f, 0x5d, 0xf7, 0x30, 
    0x0e, 0x47, 0x5b, 0x7d, 0x5b, 0xe5, 0xad, 0x1d, 0xaf, 0xb4, 0xf7, 0x0e, 0xf5, 0x8e, 0x7c, 0x60, 
    0x9b, 0xeb, 0x1d, 0xea, 0x1d, 0xd9, 0xbb, 0x7b, 0x5b, 0xcb, 0xd0, 0xe8, 0xe0, 0x53, 0x89, 0x8d, 
    0xab, 0x0f, 0x6e, 0xdd, 0x74, 0xc7, 0xbf, 0x76, 0x0f, 0x7e, 0x2d, 0xa9, 0xd4, 0xb0, 0xe5, 0xea, 
    0x01, 0xbb, 0x46, 0x71, 0xd8, 0x6e, 0xb8, 0xf7, 0xc7, 0x8f, 0x4e, 0x6d, 0xce, 0x36, 0x15, 0xb3, 
    0x26, 0xfb, 0xd4, 0xde, 0xdd, 0xdb, 0x5a, 0x0e, 0x04, 0x1f, 0xd6, 0x08, 0xbd, 0xa4, 0xe5, 0xec, 
    0x79, 0x9f, 0xe9, 0x98, 0xf6, 0x52, 0x8f, 0xd7, 0xc4, 0xce, 0x99, 0x8d, 0x34, 0xff, 0x7b, 0x90,
    0x81, 0x6a, 0x99, 0x7f, 0xc6, 0xe1, 0xa7, 0xbf, 0xb1, 0xe9, 0x47, 0x0f, 0x7e, 0xac, 0xe1, 0xff, 
    0x4b, 0x67, 0x96, 0x56, 0x2e, 0x5c, 0xba, 0x25, 0xf9, 0x7d, 0xd2, 0xf7, 0xf8, 0x92, 0x7b, 0x6f, 
    0x10, 0x1b, 0x46, 0x07, 0xd6, 0xe6, 0x7b, 0x5e, 0x09, 0xbb, 0x17, 0x6e, 0x5a, 0xfe, 0xb1, 0xff, 
    0x2c, 0xb6, 0x0d, 0xc8, 0x62, 0xd7, 0x34, 0xab, 0xce, 0x59, 0xba, 0x79, 0x29, 0xc0, 0x29, 0x8b, 
    0x5e, 0x3e, 0x77, 0xce, 0xec, 0xd8, 0xab, 0xd6, 0xec, 0x47, 0xda, 0x1e, 0x34, 0x90, 0xac, 0x98, 
    0x71, 0x5a, 0xcf, 0xea, 0x85, 0xf6, 0xf6, 0x4a, 0xe8, 0x86, 0x2a, 0xa1, 0x18, 0xec, 0x68, 0x70, 
    0xf3, 0x2e, 0x3e, 0x4e, 0x4d, 0x97, 0x45, 0x89, 0xad, 0x08, 0x0b, 0x20, 0x22, 0x61, 0x65, 0x51, 
    0x32, 0xbf, 0x43, 0x9f, 0xbb, 0xf1, 0x0b, 0xa7, 0x6e, 0xd9, 0x35, 0xcc, 0x26, 0x29, 0x5d, 0xaa, 
    0x31, 0x23, 0x26, 0x3d, 0xfa, 0x32, 0xdf, 0xbd, 0xeb, 0xc5, 0xbf, 0x3e, 0xfc, 0x3e, 0xf8, 0x9a, 
    0xb9, 0x8b, 0x4f, 0xba, 0x67, 0xb9, 0xf9, 0x4b, 0x16, 0x29, 0x47, 0x03, 0x8b, 0x91, 0x90, 0xf6, 
    0x14, 0x0e, 0x49, 0xdc, 0x1f, 0x12, 0x8e, 0xd8, 0xf1, 0xb2, 0x3a, 0xe6, 0x54, 0xdc, 0x5f, 0xa3, 
    0x35, 0xe5, 0x99, 0xd6, 0x56, 0x77, 0xec, 0xbc, 0xa3, 0xdc, 0xb1, 0xb5, 0x18, 0xf2, 0x75, 0x82, 
    0x13, 0x27, 0xcb, 0xf5, 0x3f, 0x38, 0xed, 0xe4, 0x1b, 0xd6, 0x3c, 0xf7, 0xfc, 0x0f, 0x0f, 0x72, 
    0x4b, 0x26, 0x95, 0x1c, 0xdd, 0xbf, 0xaf, 0x32, 0xb6, 0x79, 0x7b, 0xa9, 0x92, 0x9d, 0xae, 0xf0, 
    0x64, 0x85, 0xa0, 0x56, 0xe1, 0x8d, 0x77, 0xab, 0xef, 0xac, 0xfa, 0x8d, 0x5d, 0x38, 0x77, 0xdd, 
    0xc6, 0x1c, 0xc0, 0xe9, 0x0f, 0xfd, 0x63, 0xf2, 0x25, 0xf7, 0xc9, 0x19, 0x9b, 0xde, 0xa9, 0xbd,
    0x18, 0x54, 0xab, 0x18, 0xca, 0x64, 0xa7, 0xc2, 0x8e, 0xbd, 0xe5, 0x60, 0x4f, 0x5f, 0xb9, 0xac, 
    0x13, 0x32, 0x07, 0x95, 0xc7, 0x03, 0xf7, 0x1d, 0x3d, 0xdd, 0xab, 0xb7, 0xaf, 0x59, 0x58, 0x7e, 
    0xf3, 0x9a, 0x05, 0xc3, 0xf7, 0x9f, 0xb5, 0xe8, 0xda, 0xf7, 0x9e, 0x1f, 0xb8, 0x7f, 0xb6, 0xb4, 
    0xfb, 0x9c, 0xd7, 0x56, 0xcf, 0xef, 0xdb, 0x79, 0xe3, 0x09, 0xb5, 0x5f, 0x9c, 0xb5, 0xf0, 0xf6, 
    0xf7, 0x9e, 0xff, 0x07, 0x09, 0x2d, 0xa0, 0x30, 0x79, 0xd8, 0xae, 0xa6, 0x00, 0x00, 0x00, 0x00, 
    0x49, 0x45, 0x4e, 0x44, 0xae, 0x42, 0x60, 0x82
};

static gsize pidgin_pigeon_size = 2136;

typedef struct
{
    GdkRectangle rc;
    gint state; /* -N-STEPS <-> N_STEPS */
    gboolean is_dirty;
} WhackaPigeon;

typedef struct
{
    GRand *rnd;
    WhackaPigeon pigeons[X_OVALS][Y_OVALS];
    GtkWidget *dialog;
    GtkWidget *close_button;
    GtkWidget *notebook;
    GtkWidget *label;
    GtkWidget *entropy_label;
    GtkWidget *da;
    GThread *thread;
    guint score;

    GdkPixbuf *pigeon;
    guint cx_pigeon, cy_pigeon;

    guint entropy_update_timeout;
    guint next_action_timeout;
    guint anim_timeout;
} PleaseWaitDialogApp;

static GdkColor
    clr_black  = {.pixel = 0, .red = 0x0000, .green = 0x0000, .blue = 0x0000},
    clr_pigeon = {.pixel = 0, .red = 0xffff, .green = 0x0000, .blue = 0xffff};

static gboolean da_expose(GtkWidget *da, GdkEventExpose *event, PleaseWaitDialogApp *dialog);
static gboolean da_button_press(GtkWidget *da, GdkEventButton *event, PleaseWaitDialogApp *dialog);
static gboolean next_action(PleaseWaitDialogApp *dialog);
static gboolean anim_timeout(PleaseWaitDialogApp *dialog);
static gboolean entropy_update_timeout(GtkWidget *entropy_label);

static gboolean draw_scene(PleaseWaitDialogApp *dialog);
static void finish_game(PleaseWaitDialogApp *dialog, gboolean never_really_ran);

static GdkPixbuf *
load_pigeon()
{
    GdkPixbuf *pixbuf = NULL;
    GdkPixbufLoader *loader = NULL;

    if ((loader = gdk_pixbuf_loader_new()) != NULL) {
	if (gdk_pixbuf_loader_write(loader, pidgin_pigeon, pidgin_pigeon_size, NULL)) {
	    if (gdk_pixbuf_loader_close(loader, NULL)) {
		if ((pixbuf = gdk_pixbuf_loader_get_pixbuf(loader)) != NULL) {
		    /* So it may survive the destruction of the loader */
		    g_object_ref(pixbuf);
		}
	    }
	}
	g_object_unref(loader);
    }

    return pixbuf;
}

static void
dialog_show(GtkWidget *dialog, GdkWindow *parent)
{
    if (parent)
    	if (dialog->window)
	    gdk_window_set_transient_for(dialog->window, parent);
}

static PleaseWaitDialogApp *
please_wait_dialog_new(GdkWindow *parent, const char *account, const char *protocol)
{
    int Nix, Nix1;
    PleaseWaitDialogApp *ret = NULL;
    GtkWidget *game_vbox = NULL;
    char *str1 = NULL, *str2 = NULL;

    if (!(clr_black.pixel))
	    gdk_rgb_find_color(gdk_colormap_get_system(), &clr_black);
    if (!(clr_pigeon.pixel))
	    gdk_rgb_find_color(gdk_colormap_get_system(), &clr_pigeon);

    if ((ret = g_new0(PleaseWaitDialogApp, 1)) == NULL) return NULL;

    ret->rnd = g_rand_new();

    if ((ret->pigeon = load_pigeon()) != NULL) {
	    ret->cx_pigeon = gdk_pixbuf_get_width(ret->pigeon);
	    ret->cy_pigeon = gdk_pixbuf_get_height(ret->pigeon);
    } else {
	    ret->cx_pigeon = CX_PIGEON;
	    ret->cy_pigeon = CY_PIGEON;
    }

    /* Dialog ***************/
    ret->dialog = g_object_new(GTK_TYPE_DIALOG, "modal", TRUE, "title", _("Key Generation"), "has-separator", FALSE, NULL);
		g_signal_connect(G_OBJECT(ret->dialog), "show", (GCallback)dialog_show, parent);

    ret->close_button = gtk_dialog_add_button(GTK_DIALOG(ret->dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
    g_object_set(G_OBJECT(ret->close_button), "sensitive", FALSE, NULL);
    g_signal_connect_swapped(G_OBJECT(ret->close_button), "clicked", (GCallback)gtk_widget_hide, ret->dialog);
    g_object_set(G_OBJECT(GTK_DIALOG(ret->dialog)->vbox), "spacing", PIDGIN_HIG_CAT_SPACE, NULL);

	/* Status label *********/
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(ret->dialog)->vbox),
	    g_object_new(GTK_TYPE_LABEL, "visible", TRUE,
		"label", 
		    str2 = g_strdup_printf("<b>%s</b>\n",
		    	str1 = g_strdup_printf(_("Generating private key for %s (%s)..."), account, protocol)),
		"justify", GTK_JUSTIFY_LEFT, "xalign", 0.0, "yalign", 0.0, "use-markup", TRUE, "wrap", TRUE, NULL));
    	g_free(str1);
    	g_free(str2);
	/* Status label *********/

	/* Notebook *************/
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(ret->dialog)->vbox),
	    ret->notebook = g_object_new(GTK_TYPE_NOTEBOOK, "visible", TRUE, "show-border", FALSE, "show-tabs", FALSE, "border-width", PIDGIN_HIG_BORDER, NULL));

	    /* First tab - game box */
	    gtk_container_add(GTK_CONTAINER(ret->notebook),
		game_vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, "spacing", PIDGIN_HIG_BOX_SPACE, NULL));

		/* Explanation label ****/
		gtk_container_add(GTK_CONTAINER(game_vbox),
		    g_object_new(GTK_TYPE_LABEL, "visible", TRUE,
			"label", str1 = g_strdup_printf(_("<small>This means that %s is currently hung. "
    	    	    	    	    	"Please don't kill it, and enjoy a game of "
					"whack-a-pigeon while your key is being generated. "
					"Your whacks will provide valuable random data for the key generator.</small>"),
					_("Pidgin")),
			"wrap", TRUE, "justify", GTK_JUSTIFY_LEFT, "xalign", 0.0, "yalign", 0.0, "use-markup", TRUE, NULL));
		    g_free(str1);
		/* Explanation label ****/

		/* Drawing area *********/
		gtk_container_add(GTK_CONTAINER(game_vbox),
		    g_object_new(GTK_TYPE_ALIGNMENT, "visible", TRUE,
		    	"xalign", 0.5, "yalign", 0.5, "xscale", 0.0, "yscale", 0.0,
    	    	    	"child", ret->da = g_object_new(GTK_TYPE_DRAWING_AREA, "visible", TRUE, NULL), NULL));
		gtk_widget_add_events(ret->da, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);
		gtk_widget_set_size_request(ret->da,
		    X_OVALS * (ret->cx_pigeon + 2 * OVAL_EXTRA) + (X_OVALS - 1) * PIGEON_SPACING,
		    Y_OVALS * (ret->cy_pigeon + OVAL_EXTRA) + (Y_OVALS - 1) * PIGEON_SPACING);
		/* Drawing area *********/

		/* Entropy count ********/
		gtk_container_add(GTK_CONTAINER(game_vbox),
		    ret->entropy_label = g_object_new(GTK_TYPE_LABEL, "visible", TRUE,
			"xalign", 0.0, "yalign", 0.5, "justify", GTK_JUSTIFY_LEFT, "use-markup", TRUE, 
			"label", "Data available (more is better): <b></b>", NULL));
		/* Entropy count ********/

	    /* First tab - game box */

	    /* Second tab - score ***/
	    gtk_container_add(GTK_CONTAINER(ret->notebook),
		    ret->label = g_object_new(GTK_TYPE_LABEL, "visible", TRUE,
			    "xalign", 0.5, "yalign", 0.5, "justify", GTK_JUSTIFY_CENTER, "wrap", TRUE, "use-markup", TRUE, NULL));
	    /* Second tab - score ***/
	/* Notebook *************/
    /* Dialog ***************/

    g_signal_connect(G_OBJECT(ret->da), "expose-event", (GCallback)da_expose, ret);
    g_signal_connect(G_OBJECT(ret->da), "button-press-event", (GCallback)da_button_press, ret);

    ret->pigeons[0][0].rc.x = 0;
    ret->pigeons[0][0].rc.y = 0;
    ret->pigeons[0][0].rc.width = ret->cx_pigeon + 2 * OVAL_EXTRA ;
    ret->pigeons[0][0].rc.height = ret->cy_pigeon + OVAL_EXTRA ;
    ret->pigeons[0][0].is_dirty = TRUE;
    for (Nix = 1 ; Nix < X_OVALS ; Nix++) {
	ret->pigeons[Nix][0].rc.x = Nix * (ret->cx_pigeon + 2 * OVAL_EXTRA + PIGEON_SPACING);
	ret->pigeons[Nix][0].rc.y = 0;
	ret->pigeons[Nix][0].rc.width = ret->cx_pigeon + 2 * OVAL_EXTRA ;
	ret->pigeons[Nix][0].rc.height = ret->cy_pigeon + OVAL_EXTRA ;
	ret->pigeons[Nix][0].is_dirty = TRUE;
    }
    for (Nix1 = 1 ; Nix1 < Y_OVALS ; Nix1++) {
	ret->pigeons[0][Nix1].rc.x = 0;
	ret->pigeons[0][Nix1].rc.y = Nix1 * (ret->cy_pigeon + OVAL_EXTRA + PIGEON_SPACING);
	ret->pigeons[0][Nix1].rc.width = ret->cx_pigeon + 2 * OVAL_EXTRA ;
	ret->pigeons[0][Nix1].rc.height = ret->cy_pigeon + OVAL_EXTRA ;
	ret->pigeons[0][Nix1].is_dirty = TRUE;
	for (Nix = 1 ; Nix < X_OVALS ; Nix++) {
	    ret->pigeons[Nix][Nix1].rc.x = Nix * (ret->cx_pigeon + 2 * OVAL_EXTRA + PIGEON_SPACING);
	    ret->pigeons[Nix][Nix1].rc.y = Nix1 * (ret->cy_pigeon + OVAL_EXTRA + PIGEON_SPACING);
	    ret->pigeons[Nix][Nix1].rc.width = ret->cx_pigeon + 2 * OVAL_EXTRA ;
	    ret->pigeons[Nix][Nix1].rc.height = ret->cy_pigeon + OVAL_EXTRA ;
	    ret->pigeons[Nix][Nix1].is_dirty = TRUE;
	}
    }

    gtk_widget_show(ret->dialog);
    /* Make sure dialog is visible */
    while (g_main_context_iteration(NULL, FALSE));

    ret->next_action_timeout = g_timeout_add(ACTION_TIMEOUT_MIN, (GSourceFunc)next_action, ret);
    ret->entropy_update_timeout = g_timeout_add(ENTROPY_UPDATE_TIMEOUT, (GSourceFunc)entropy_update_timeout, ret->entropy_label);
    /* Call timeout once */
    entropy_update_timeout(ret->entropy_label);

    return ret;
}

static void
finish_game(PleaseWaitDialogApp *dialog, gboolean never_really_ran)
{
    char *label = NULL;
    int Nix, Nix1;

    /* No more pigeons :o( */
    if (dialog->next_action_timeout)
	g_source_remove(dialog->next_action_timeout);

    /* Make all pigeons retreat */
    for (Nix1 = 0 ; Nix1 < Y_OVALS ; Nix1++)
	for (Nix = 0 ; Nix < X_OVALS ; Nix++)
	    if (dialog->pigeons[Nix][Nix1].state > 0) {
		dialog->pigeons[Nix][Nix1].state = -(dialog->pigeons[Nix][Nix1].state);
		dialog->pigeons[Nix][Nix1].is_dirty = TRUE;
	    }

    /* Display their retreat to the user */
    if (0 == dialog->anim_timeout)
	dialog->anim_timeout = g_timeout_add(ANIM_TIMEOUT, (GSourceFunc)anim_timeout, dialog);
    while (dialog->anim_timeout)
	g_main_context_iteration(NULL, TRUE);

    g_object_set(G_OBJECT(dialog->label), "wrap", TRUE, "justify", GTK_JUSTIFY_LEFT,
    	"label", label = g_strdup_printf(_("Congratulations! Your key is now ready.\nYou have whacked <b>%d</b> pigeons!"), dialog->score), NULL);
    g_free(label);
    g_object_set(G_OBJECT(dialog->notebook), "page", 1, NULL);
}

static PleaseWaitDialogApp *
please_wait_dialog_destroy(PleaseWaitDialogApp *please_wait_dialog, gboolean never_really_ran)
{
    gboolean visible = FALSE;

		if (!please_wait_dialog) return NULL;

    finish_game(please_wait_dialog, never_really_ran);

    g_object_set(G_OBJECT(please_wait_dialog->close_button), "sensitive", TRUE, NULL);
    /* Wait for the dialog to be hidden (if not already) */
    while (TRUE) {
	g_object_get(G_OBJECT(please_wait_dialog->dialog), "visible", &visible, NULL);
	if (!visible) break;
	g_main_context_iteration(NULL, TRUE);
    }
    if (please_wait_dialog->rnd)
	g_rand_free(please_wait_dialog->rnd);
    if (please_wait_dialog->entropy_update_timeout)
	g_source_remove(please_wait_dialog->entropy_update_timeout);
    gtk_widget_destroy(please_wait_dialog->dialog);
    g_free(please_wait_dialog);

    return NULL;
}

static gboolean
entropy_update_timeout(GtkWidget *entropy_label)
{
    gboolean have_available_text = TRUE;
    char *avail_text = NULL;
    char *label = NULL;

    g_file_get_contents("/proc/sys/kernel/random/entropy_avail", &avail_text, NULL, NULL);

    if (avail_text) {
	int Nix;

	/* Strip trailing newlines */
	for (Nix = strlen(avail_text) - 1 ; Nix >= 0 ; Nix--)
	    if (!(avail_text[Nix] == '\r' || avail_text[Nix] == '\n'))
		break;
	    else
		avail_text[Nix] = 0;

	if (avail_text[0] != 0) {
	    if ((label = g_strdup_printf("Data available (more is better): <b>%s</b>", avail_text))) {
		g_object_set(G_OBJECT(entropy_label), "label", label, NULL);
		g_free(label);
		have_available_text = TRUE;
	    }
	}
	g_free(avail_text);
    }

    g_object_set(G_OBJECT(entropy_label), "visible", have_available_text, NULL);
    return TRUE;
}

static gboolean
anim_timeout(PleaseWaitDialogApp *dialog)
{
    gboolean b_quit = TRUE;
    int Nix, Nix1;

    for (Nix1 = 0 ; Nix1 < Y_OVALS ; Nix1++)
	for (Nix = 0 ; Nix < X_OVALS ; Nix++)
	    if (dialog->pigeons[Nix][Nix1].state != N_STEPS && dialog->pigeons[Nix][Nix1].state != 0) {
		b_quit = FALSE;
		if (dialog->pigeons[Nix][Nix1].state < 0) {
		    dialog->pigeons[Nix][Nix1].state += 2;
		    dialog->pigeons[Nix][Nix1].state = MIN(dialog->pigeons[Nix][Nix1].state, 0);
		}
		else
		    dialog->pigeons[Nix][Nix1].state++;
		dialog->pigeons[Nix][Nix1].is_dirty = TRUE;
	    }

    if (b_quit) {
	dialog->anim_timeout = 0;
	return FALSE;
    }
    draw_scene(dialog);
    return TRUE;
}

static gboolean
next_action(PleaseWaitDialogApp *dialog)
{
    guint x, y;
    guint next_timeout = ACTION_TIMEOUT_MIN;

    if (dialog->rnd)
	next_timeout = g_rand_int_range(dialog->rnd, ACTION_TIMEOUT_MIN, ACTION_TIMEOUT_MAX);

    x = g_rand_int_range(dialog->rnd, 0, X_OVALS);
    y = g_rand_int_range(dialog->rnd, 0, Y_OVALS);

    dialog->pigeons[x][y].state = (0 == dialog->pigeons[x][y].state) ? 1 : -(dialog->pigeons[x][y].state);
    dialog->pigeons[x][y].is_dirty = TRUE;
    if (!(dialog->anim_timeout))
	dialog->anim_timeout = g_timeout_add(ANIM_TIMEOUT, (GSourceFunc)anim_timeout, dialog);

    dialog->next_action_timeout = g_timeout_add(next_timeout, (GSourceFunc)next_action, dialog);

    return FALSE;
}

static gboolean
da_expose(GtkWidget *da, GdkEventExpose *event, PleaseWaitDialogApp *dialog)
{
    int Nix, Nix1;

    for (Nix1 = 0 ; Nix1 < Y_OVALS ; Nix1++)
	for (Nix = 0 ; Nix < X_OVALS ; Nix++)
	    dialog->pigeons[Nix][Nix1].is_dirty = TRUE;

    return draw_scene(dialog);
}

static gboolean
da_button_press(GtkWidget *da, GdkEventButton *event, PleaseWaitDialogApp *dialog)
{
    int x1, y1, x2, y2;
    int Nix, Nix1;

    for (Nix1 = 0 ; Nix1 < Y_OVALS ; Nix1++) {
	for (Nix = 0 ; Nix < X_OVALS ; Nix++) {
	    if (ABS(dialog->pigeons[Nix][Nix1].state) > 2) {
		x1 = dialog->pigeons[Nix][Nix1].rc.x + OVAL_EXTRA;
		y1 = dialog->pigeons[Nix][Nix1].rc.y + dialog->pigeons[Nix][Nix1].rc.height - OVAL_EXTRA - ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS);
		x2 = x1 + dialog->cx_pigeon;
		y2 = y1 + ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS);
		if (event->x >= x1 && event->x <= x2 && event->y >= y1 && event->y <= y2) {
		    dialog->pigeons[Nix][Nix1].state = -(dialog->pigeons[Nix][Nix1].state);
		    dialog->pigeons[Nix][Nix1].is_dirty = TRUE;
		    (dialog->score)++;
		    if (!(dialog->anim_timeout))
			dialog->anim_timeout = g_timeout_add(ANIM_TIMEOUT, (GSourceFunc)anim_timeout, dialog);
		    return TRUE;
		}
	    }
	}
    }
    return FALSE;
}

static gboolean
draw_scene(PleaseWaitDialogApp *dialog)
{
    int Nix, Nix1;
    GdkGC *gc = NULL;

    if ((gc = gdk_gc_new(dialog->da->window)) == NULL) return FALSE;

    for (Nix1 = 0 ; Nix1 < Y_OVALS ; Nix1++) {
	for (Nix = 0 ; Nix < X_OVALS ; Nix++) {
	    if (dialog->pigeons[Nix][Nix1].is_dirty) {
		gdk_window_clear_area(dialog->da->window, dialog->pigeons[Nix][Nix1].rc.x, dialog->pigeons[Nix][Nix1].rc.y, dialog->pigeons[Nix][Nix1].rc.width, dialog->pigeons[Nix][Nix1].rc.height);
		gdk_gc_set_foreground(gc, &clr_black);
		gdk_gc_set_background(gc, &clr_black);
		gdk_draw_arc(dialog->da->window, gc, TRUE, 
		    dialog->pigeons[Nix][Nix1].rc.x,
		    dialog->pigeons[Nix][Nix1].rc.y + dialog->pigeons[Nix][Nix1].rc.height - 2 * OVAL_EXTRA,
		    dialog->pigeons[Nix][Nix1].rc.width,
		    2 * OVAL_EXTRA, 0, 360 * 64);
		if (dialog->pigeon) {
		    gdk_draw_pixbuf(dialog->da->window, gc, dialog->pigeon, 0, 0,
			dialog->pigeons[Nix][Nix1].rc.x + OVAL_EXTRA,
			dialog->pigeons[Nix][Nix1].rc.y + dialog->pigeons[Nix][Nix1].rc.height - OVAL_EXTRA - ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS),
			dialog->cx_pigeon,
			ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS),
			GDK_RGB_DITHER_NONE, 0, 0);
		} else {
		    gdk_gc_set_foreground(gc, &clr_pigeon);
		    gdk_gc_set_background(gc, &clr_pigeon);
		    gdk_draw_rectangle(dialog->da->window, gc, TRUE,
			dialog->pigeons[Nix][Nix1].rc.x + OVAL_EXTRA,
			dialog->pigeons[Nix][Nix1].rc.y + dialog->pigeons[Nix][Nix1].rc.height - OVAL_EXTRA - ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS),
			dialog->cx_pigeon,
			ABS(dialog->pigeons[Nix][Nix1].state) * (dialog->cy_pigeon / N_STEPS));
		}
		dialog->pigeons[Nix][Nix1].is_dirty = FALSE;
	    }
	}
    }

    g_object_unref(gc);

    return TRUE;
}

static gboolean
stdin_has_data(GIOChannel *gio_stdin, GIOCondition gio_condition, PleaseWaitDialogApp *please_wait_dialog)
{
    please_wait_dialog_destroy(please_wait_dialog, FALSE);
    gtk_main_quit();
    return FALSE;
}

int
main(int argc, char **argv)
{
    guint window_id = 0;
    char *account = NULL;
    char *protocol = NULL;
    PleaseWaitDialogApp *please_wait_dialog = NULL;
    GIOChannel *gio_stdin = NULL;
    GdkWindow *wnd = NULL;

    window_id = read_window_id(0);

    if ((account = read_string(0)) == NULL)
    	account = g_strdup(_("Unknown"));
    else
    if (0 == (*account)) {
    	g_free(account);
	account = g_strdup(_("Unknown"));
    }
    if ((protocol = read_string(0)) == NULL)
    	protocol = g_strdup(_("Unknown"));
    else
    if (0 == (*protocol)) {
    	g_free(protocol);
	protocol = g_strdup(_("Unknown"));
    }

    gtk_init(&argc, &argv);

    if (window_id)
    	wnd = gdk_window_foreign_new(window_id);

    please_wait_dialog = please_wait_dialog_new(wnd, account, protocol);

    if ((gio_stdin = g_io_channel_new(0)) != NULL) {
    	g_io_add_watch(gio_stdin, G_IO_IN, (GIOFunc)stdin_has_data, please_wait_dialog);
    } else {
        please_wait_dialog_destroy(please_wait_dialog, TRUE);
    }

    g_free(account);
    g_free(protocol);

    gtk_main();

    return 0;
}
