/*
 *  Off-the-Record Messaging plugin for pidgin
 *  Copyright (C) 2004-2007  Ian Goldberg, Chris Alexander, Nikita Borisov
 *                           <otr@cypherpunks.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* config.h */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* system headers */
#include <gtk/gtk.h>

/* libgcrypt headers */
#include <gcrypt.h>

/* libotr headers */
#include <libotr/privkey.h>

/* purple headers */
#include "util.h"
#include "account.h"
#include "notify.h"
#include "gtkutils.h"

/* pidgin headers */
#include "gtkblist.h"
#include "hildon-hacks.h"

#ifdef ENABLE_NLS
/* internationalisation header */
#include <glib/gi18n-lib.h>
#endif

/* purple-otr headers */
#include "dialogs.h"
#include "ui.h"
#include "otr-plugin.h"

/* The OTR icons */

static const char * not_private_xpm[] = {
"20 20 100 2",
"  	c None",
". 	c #555555",
"+ 	c #5A5A5A",
"@ 	c #404040",
"# 	c #515151",
"$ 	c #919191",
"% 	c #9C9C9C",
"& 	c #949494",
"* 	c #848484",
"= 	c #646464",
"- 	c #161616",
"; 	c #959595",
"> 	c #B7B7B7",
", 	c #C2C2C2",
"' 	c #AFAFAF",
") 	c #8F8F8F",
"! 	c #7B7B7B",
"~ 	c #4F4F4F",
"{ 	c #5C5C5C",
"] 	c #A8A8A8",
"^ 	c #CECECE",
"/ 	c #D4D4D4",
"( 	c #B9B9B9",
"_ 	c #7A7A7A",
": 	c #686868",
"< 	c #101010",
"[ 	c #636363",
"} 	c #A3A3A3",
"| 	c #C4C4C4",
"1 	c #888888",
"2 	c #757575",
"3 	c #6B6B6B",
"4 	c #141414",
"5 	c #9E9E9E",
"6 	c #9D9D9D",
"7 	c #8C8C8C",
"8 	c #6D6D6D",
"9 	c #0C0C0C",
"0 	c #777777",
"a 	c #808080",
"b 	c #7E7E7E",
"c 	c #767676",
"d 	c #6C6C6C",
"e 	c #373737",
"f 	c #000000",
"g 	c #313131",
"h 	c #696969",
"i 	c #606060",
"j 	c #3D3D3D",
"k 	c #707070",
"l 	c #676767",
"m 	c #626262",
"n 	c #0E0E0E",
"o 	c #020202",
"p 	c #DADADA",
"q 	c #B2B2B2",
"r 	c #969696",
"s 	c #898989",
"t 	c #5E5E5E",
"u 	c #5B5B5B",
"v 	c #727272",
"w 	c #303030",
"x 	c #CFCFCF",
"y 	c #A2A2A2",
"z 	c #828282",
"A 	c #7C7C7C",
"B 	c #797979",
"C 	c #CBCBCB",
"D 	c #9F9F9F",
"E 	c #747474",
"F 	c #6E6E6E",
"G 	c #9A9A9A",
"H 	c #868686",
"I 	c #272727",
"J 	c #BFBFBF",
"K 	c #909090",
"L 	c #818181",
"M 	c #7D7D7D",
"N 	c #151515",
"O 	c #878787",
"P 	c #717171",
"Q 	c #1A1A1A",
"R 	c #8B8B8B",
"S 	c #656565",
"T 	c #292929",
"U 	c #4D4D4D",
"V 	c #1D1D1D",
"W 	c #616161",
"X 	c #3A3A3A",
"Y 	c #525252",
"Z 	c #464646",
"` 	c #080808",
" .	c #565656",
"..	c #2E2E2E",
"+.	c #262626",
"@.	c #2F2F2F",
"#.	c #535353",
"$.	c #4B4B4B",
"%.	c #111111",
"&.	c #2C2C2C",
"      . + @                             ",
"  # $ % & * = -             . + @       ",
"  ; > , ' ) ! ~         # $ % & * = -   ",
"{ ] ^ / ( $ _ : <       ; > , ' ) ! ~   ",
"[ } , | ] 1 2 3 4     { ] ^ / ( $ _ : < ",
"# ) 5 6 7 _ 8 { 9     [ } , | ] 1 2 3 4 ",
"  0 a b c d = e f     # ) 5 6 7 _ 8 { 9 ",
"  g h d h i j . + @     0 a b c d = e f ",
"  & k l m # $ % & * = - g h d h i j n o ",
"  p q r s ; > , ' ) ! ~ & k l m t u v w ",
"  x y 1 { ] ^ / ( $ _ : < q r s z A B m ",
"  C D 1 [ } , | ] 1 2 3 4 y 1 b _ E F u ",
"  | G H # ) 5 6 7 _ 8 { 9 D 1 b _ 2 8 I ",
"  J K L _ 0 a b c d = e f G H M B E 3 N ",
"  } O _ 2 g h d h i j n o K L _ c P : Q ",
"  R M v 3 & k l m t u v w O _ 2 P d S T ",
"  3 c h U p q r s z A B m V v 3 : l W X ",
"  Y P Z ` x y 1 b _ E F u  ...U +.@.#.$.",
"    i <   C D 1 b _ 2 8 I %.@.&.    n Z ",
"          | G H M B E 3 N i <           "};

static const char * unverified_xpm[] = {
"20 20 103 2",
"  	c None",
". 	c #555555",
"+ 	c #5A5A5A",
"@ 	c #404040",
"# 	c #000000",
"$ 	c #515151",
"% 	c #919191",
"& 	c #9C9C9C",
"* 	c #949494",
"= 	c #848484",
"- 	c #646464",
"; 	c #161616",
"> 	c #FFFF00",
", 	c #959595",
"' 	c #B7B7B7",
") 	c #C2C2C2",
"! 	c #AFAFAF",
"~ 	c #8F8F8F",
"{ 	c #7B7B7B",
"] 	c #4F4F4F",
"^ 	c #5C5C5C",
"/ 	c #A8A8A8",
"( 	c #CECECE",
"_ 	c #D4D4D4",
": 	c #B9B9B9",
"< 	c #7A7A7A",
"[ 	c #686868",
"} 	c #101010",
"| 	c #636363",
"1 	c #A3A3A3",
"2 	c #C4C4C4",
"3 	c #888888",
"4 	c #757575",
"5 	c #6B6B6B",
"6 	c #141414",
"7 	c #9E9E9E",
"8 	c #9D9D9D",
"9 	c #8C8C8C",
"0 	c #6D6D6D",
"a 	c #0C0C0C",
"b 	c #777777",
"c 	c #808080",
"d 	c #7E7E7E",
"e 	c #767676",
"f 	c #6C6C6C",
"g 	c #373737",
"h 	c #313131",
"i 	c #696969",
"j 	c #606060",
"k 	c #3D3D3D",
"l 	c #0E0E0E",
"m 	c #020202",
"n 	c #707070",
"o 	c #676767",
"p 	c #626262",
"q 	c #5E5E5E",
"r 	c #5B5B5B",
"s 	c #727272",
"t 	c #303030",
"u 	c #DADADA",
"v 	c #B2B2B2",
"w 	c #969696",
"x 	c #898989",
"y 	c #828282",
"z 	c #7C7C7C",
"A 	c #797979",
"B 	c #1D1D1D",
"C 	c #CFCFCF",
"D 	c #A2A2A2",
"E 	c #747474",
"F 	c #6E6E6E",
"G 	c #565656",
"H 	c #2E2E2E",
"I 	c #CBCBCB",
"J 	c #9F9F9F",
"K 	c #272727",
"L 	c #111111",
"M 	c #2F2F2F",
"N 	c #2C2C2C",
"O 	c #9A9A9A",
"P 	c #868686",
"Q 	c #7D7D7D",
"R 	c #151515",
"S 	c #BFBFBF",
"T 	c #909090",
"U 	c #818181",
"V 	c #717171",
"W 	c #1A1A1A",
"X 	c #878787",
"Y 	c #656565",
"Z 	c #292929",
"` 	c #8B8B8B",
" .	c #616161",
"..	c #3A3A3A",
"+.	c #4D4D4D",
"@.	c #262626",
"#.	c #535353",
"$.	c #4B4B4B",
"%.	c #525252",
"&.	c #464646",
"*.	c #080808",
"=.	c #121212",
"-.	c #242424",
"      . + @               # # # # #     ",
"  $ % & * = - ;         # > > > > > #   ",
"  , ' ) ! ~ { ]       # > > > # > > > # ",
"^ / ( _ : % < [ }     # > > #   # > > # ",
"| 1 ) 2 / 3 4 5 6     # > > #   # > > # ",
"$ ~ 7 8 9 < 0 ^ a     # > > #   # > > # ",
"  b c d e f - g #       # #     # > > # ",
"  h i f i j k l m             # > > > # ",
"  * n o p q r s t             # > > #   ",
"  u v w x y z A p B         # > > #     ",
"  C D 3 d < E F r G H       # > #       ",
"  I J 3 d < 4 0 K L M N     # > #       ",
"  2 O P Q A E 5 R             #         ",
"  S T U < e V [ W           # # #       ",
"  1 X < 4 V f Y Z         # > > > #     ",
"  ` Q s 5 [ o  ...        # > > > #     ",
"  5 e i +.@.M #.$.m       # > > > #     ",
"  %.V &.*.    l &.=.        # # #       ",
"    j }           -.                    ",
"                                        "};

static const char * private_xpm[] = {
"20 20 148 2",
"  	c None",
". 	c #978214",
"+ 	c #A58A10",
"@ 	c #77620A",
"# 	c #85781D",
"$ 	c #EBD437",
"% 	c #F4DE44",
"& 	c #F3D936",
"* 	c #EFC819",
"= 	c #C19207",
"- 	c #2C1E01",
"; 	c #EAD641",
"> 	c #F6E978",
", 	c #F7EB8D",
"' 	c #F5E569",
") 	c #F2D42C",
"! 	c #EBB50C",
"~ 	c #9C6302",
"{ 	c #99891F",
"] 	c #F5E45C",
"^ 	c #F8EFA4",
"/ 	c #F8F0B0",
"( 	c #F6E97D",
"_ 	c #F1D531",
": 	c #E9B20C",
"< 	c #CE7C02",
"[ 	c #201000",
"} 	c #AB961B",
"| 	c #F4E252",
"1 	c #F6EB8F",
"2 	c #F6EC93",
"3 	c #F4E25C",
"4 	c #EECB22",
"5 	c #E4A407",
"6 	c #D67401",
"7 	c #291400",
"8 	c #8F7A13",
"9 	c #F0D32E",
"0 	c #F2DC4B",
"a 	c #F2DB49",
"b 	c #EECE2B",
"c 	c #E6B10F",
"d 	c #D88503",
"e 	c #B95800",
"f 	c #190A00",
"g 	c #D9AE15",
"h 	c #E7BC1A",
"i 	c #E6B817",
"j 	c #E0A60D",
"k 	c #D58404",
"l 	c #C76301",
"m 	c #6E2E00",
"n 	c #010000",
"o 	c #58440B",
"p 	c #C68E0C",
"q 	c #D28A07",
"r 	c #CE7904",
"s 	c #BE5F02",
"t 	c #793601",
"u 	c #1A0F02",
"v 	c #040300",
"w 	c #A9A480",
"x 	c #8A8156",
"y 	c #9C8032",
"z 	c #A27A23",
"A 	c #9D7E1F",
"B 	c #9C801A",
"C 	c #C39D21",
"D 	c #5B3B05",
"E 	c #F5EEC0",
"F 	c #F0E074",
"G 	c #EAD243",
"H 	c #E7C72C",
"I 	c #E5BE20",
"J 	c #E1B218",
"K 	c #DDA615",
"L 	c #C27003",
"M 	c #3B1C00",
"N 	c #F5ECA9",
"O 	c #F1DE53",
"P 	c #ECCD25",
"Q 	c #E9C013",
"R 	c #E8B50C",
"S 	c #E2A307",
"T 	c #D98B03",
"U 	c #B66501",
"V 	c #AC5201",
"W 	c #5D2B00",
"X 	c #F6ECA1",
"Y 	c #F3DF4B",
"Z 	c #F0CF21",
"` 	c #EDC30F",
" .	c #EBB709",
"..	c #E5A405",
"+.	c #D88603",
"@.	c #4E2600",
"#.	c #231000",
"$.	c #5F2900",
"%.	c #582300",
"&.	c #F6EA93",
"*.	c #F4DD40",
"=.	c #F1CF1B",
"-.	c #EEC10C",
";.	c #EBB307",
">.	c #E49D04",
",.	c #D47A02",
"'.	c #2B1500",
").	c #F6E889",
"!.	c #F2D72F",
"~.	c #EFC614",
"{.	c #ECB508",
"].	c #E8A604",
"^.	c #E09002",
"/.	c #CF6E01",
"(.	c #351800",
"_.	c #EEDB59",
":.	c #F0CB1E",
"<.	c #EAB40B",
"[.	c #E5A205",
"}.	c #D87B01",
"|.	c #C96101",
"1.	c #532100",
"2.	c #DCC33B",
"3.	c #EAB811",
"4.	c #E09804",
"5.	c #D57D02",
"6.	c #CD6601",
"7.	c #C35600",
"8.	c #742D00",
"9.	c #BD9D19",
"0.	c #E3A109",
"a.	c #D17402",
"b.	c #9A4801",
"c.	c #4C2400",
"d.	c #5E2900",
"e.	c #A64300",
"f.	c #963500",
"g.	c #050200",
"h.	c #957610",
"i.	c #DC9107",
"j.	c #8B4101",
"k.	c #100600",
"l.	c #1C0A00",
"m.	c #8D3000",
"n.	c #240B00",
"o.	c #BB7B06",
"p.	c #210E00",
"q.	c #491800",
"            . + @                       ",
"        # $ % & * = -                   ",
"        ; > , ' ) ! ~                   ",
"      { ] ^ / ( _ : < [                 ",
"      } | 1 2 3 4 5 6 7                 ",
"      8 9 0 a b c d e f                 ",
"        g h i j k l m n                 ",
"        o p q r s t u v                 ",
"        w x y z A B C D                 ",
"        E F G H I J K L M               ",
"        N O P Q R S T U V W             ",
"        X Y Z `  ...+.@.#.$.%.          ",
"        &.*.=.-.;.>.,.'.                ",
"        ).!.~.{.].^./.(.                ",
"        _.:.<.[.^.}.|.1.                ",
"        2.3.4.5./.6.7.8.                ",
"        9.0.a.b.c.d.e.f.g.              ",
"        h.i.j.k.    l.m.n.              ",
"          o.p.          q.              ",
"                                        "};

static const char * finished_xpm[] = {
"20 20 101 2",
"  	c None",
". 	c #555555",
"+ 	c #FF0000",
"@ 	c #F31111",
"# 	c #C94949",
"$ 	c #C14242",
"% 	c #B13232",
"& 	c #8A0A0A",
"* 	c #FE0000",
"= 	c #DC5656",
"- 	c #C2BFBF",
"; 	c #AFAFAF",
"> 	c #8F8F8F",
", 	c #7B7B7B",
"' 	c #4F4F4F",
") 	c #C66D6D",
"! 	c #CECECE",
"~ 	c #D4D4D4",
"{ 	c #B9B9B9",
"] 	c #919191",
"^ 	c #7A7A7A",
"/ 	c #686868",
"( 	c #101010",
"_ 	c #9E3D3D",
": 	c #A3A3A3",
"< 	c #C2C2C2",
"[ 	c #C4C4C4",
"} 	c #A8A8A8",
"| 	c #888888",
"1 	c #757575",
"2 	c #6B6B6B",
"3 	c #141414",
"4 	c #515151",
"5 	c #9E9E9E",
"6 	c #9D9D9D",
"7 	c #8C8C8C",
"8 	c #6D6D6D",
"9 	c #5C5C5C",
"0 	c #0C0C0C",
"a 	c #777777",
"b 	c #808080",
"c 	c #7E7E7E",
"d 	c #767676",
"e 	c #6C6C6C",
"f 	c #646464",
"g 	c #373737",
"h 	c #000000",
"i 	c #313131",
"j 	c #696969",
"k 	c #606060",
"l 	c #3D3D3D",
"m 	c #0E0E0E",
"n 	c #949494",
"o 	c #707070",
"p 	c #676767",
"q 	c #626262",
"r 	c #5E5E5E",
"s 	c #5B5B5B",
"t 	c #DADADA",
"u 	c #B2B2B2",
"v 	c #969696",
"w 	c #898989",
"x 	c #828282",
"y 	c #1D1D1D",
"z 	c #CFCFCF",
"A 	c #A2A2A2",
"B 	c #6E6E6E",
"C 	c #565656",
"D 	c #2E2E2E",
"E 	c #CBCBCB",
"F 	c #9F9F9F",
"G 	c #272727",
"H 	c #111111",
"I 	c #2F2F2F",
"J 	c #2C2C2C",
"K 	c #9A9A9A",
"L 	c #797979",
"M 	c #747474",
"N 	c #151515",
"O 	c #BFBFBF",
"P 	c #717171",
"Q 	c #1A1A1A",
"R 	c #656565",
"S 	c #292929",
"T 	c #7D7D7D",
"U 	c #727272",
"V 	c #616161",
"W 	c #3A3A3A",
"X 	c #9B4848",
"Y 	c #4D4D4D",
"Z 	c #262626",
"` 	c #535353",
" .	c #4B4B4B",
"..	c #020202",
"+.	c #C13030",
"@.	c #4F4242",
"#.	c #080808",
"$.	c #464646",
"%.	c #121212",
"&.	c #FC0000",
"*.	c #DE0505",
"            . + + + + +                 ",
"        + + @ # $ % & + + + +           ",
"      + * = - ; > , '       + +         ",
"    + * ) ! ~ { ] ^ / (       + +       ",
"    + _ : < [ } | 1 2 3     + + +       ",
"  + + 4 > 5 6 7 ^ 8 9 0   + +   + +     ",
"  +     a b c d e f g h + +       +     ",
"+ +     i j e j k l m + +         + +   ",
"+ +     n o p q r s + +           + +   ",
"+       t u v w x + + q y           +   ",
"+       z A | c + + B s C D       + +   ",
"+ +     E F | + + 1 8 G H I J     + +   ",
"+ +     [ K + + L M 2 N           + +   ",
"  +     O + + ^ d P / Q           +     ",
"  + +   + + ^ 1 P e R S         + +     ",
"    + + + T U 2 / p V W         +       ",
"    + + X d j Y Z I `  ...    + +       ",
"      + + +.@.#.    m $.%.  + +         ",
"        + + &.+       + *.+ +           ",
"              + + + + +                 "};
enum
{
    STATUS_COLUMN = 0,
    VERIFIED_COLUMN,
    SCREEN_NAME_COLUMN,
    FP_STRING_COLUMN,
    ACCOUNT_COLUMN,
    FINGERPRINT_DATA_COLUMN
} ;

struct otroptionsdata {
    GtkWidget *enablebox;
    GtkWidget *automaticbox;
    GtkWidget *onlyprivatebox;
    GtkWidget *avoidloggingotrbox;
};

static struct {
    GtkWidget *accountmenu;
    GtkWidget *fprint_label;
    GtkWidget *generate_button;
    GtkWidget *scrollwin;
    GtkTreeView *keylist;
    gint sortcol, sortdir;
    Fingerprint *selected_fprint;
    GtkWidget *connect_button;
    GtkWidget *disconnect_button;
    GtkWidget *forget_button;
    GtkWidget *verify_button;
    struct otroptionsdata oo;
} ui_layout;

GdkPixbuf *
otr_trust_level_icon(TrustLevel level)
{
    const char **data = NULL;

    switch(level) {
	case TRUST_NOT_PRIVATE:
	    data = not_private_xpm;
	    break;
	case TRUST_UNVERIFIED:
	    data = unverified_xpm;
	    break;
	case TRUST_PRIVATE:
	    data = private_xpm;
	    break;
	case TRUST_FINISHED:
	    data = finished_xpm;
	    break;
    }

    if (!data) return NULL;

    return gdk_pixbuf_new_from_xpm_data(data);
}

static void account_menu_changed_cb(GtkWidget *item, PurpleAccount *account,
	void *data)
{
    const char *accountname;
    const char *protocol;
    GtkWidget *fprint = ui_layout.fprint_label;
    char s[100];
    char *fingerprint;
    
    if (account) {
	char fingerprint_buf[45];
	accountname = purple_account_get_username(account);
	protocol = purple_account_get_protocol_id(account);
	fingerprint = otrl_privkey_fingerprint(otrg_plugin_userstate,
		fingerprint_buf, accountname, protocol);

	if (fingerprint) {
	    sprintf(s, _("Fingerprint: %.80s"), fingerprint);
	    if (ui_layout.generate_button)
		gtk_widget_set_sensitive(ui_layout.generate_button, 0);
	} else {
	    sprintf(s, _("No key present"));
	    if (ui_layout.generate_button)
		gtk_widget_set_sensitive(ui_layout.generate_button, 1);
	}
    } else {
	sprintf(s, _("No account available"));
	if (ui_layout.generate_button)
	    gtk_widget_set_sensitive(ui_layout.generate_button, 0);
    }
    if (fprint) {
	gtk_label_set_text(GTK_LABEL(fprint), s);
	gtk_widget_show(fprint);
    }
}

/* Call this function when the DSA key is updated; it will redraw the
 * UI, if visible. */
static void otrg_gtk_ui_update_fingerprint(void)
{
    g_signal_emit_by_name(G_OBJECT(ui_layout.accountmenu), "changed");
}

static void account_menu_added_removed_cb(PurpleAccount *account, void *data)
{
    otrg_gtk_ui_update_fingerprint();
}

/* Update the keylist, if it's visible */
static void otrg_gtk_ui_update_keylist(void)
{
    GtkListStore *ls = NULL;
    GtkTreeIter itr;
    GtkTreePath *tp = NULL;
    GtkTreeRowReference *trr = NULL;
    gboolean is_verified;
    char *username = NULL, *account = NULL;
    char hash[45];
    ConnContext * context;
    Fingerprint * fingerprint;
    GList *ll_columns = NULL, *ll_itr = NULL;

    if (ui_layout.keylist == NULL)
	return;

    if ((ls = gtk_list_store_new(6,
	G_TYPE_INT,
	G_TYPE_BOOLEAN, 
	G_TYPE_STRING, 
	G_TYPE_STRING, 
	G_TYPE_STRING, 
	G_TYPE_POINTER)) == NULL)
	return;

    for (context = otrg_plugin_userstate->context_root; context != NULL;
	    context = context->next) {
	PurplePlugin *p;
	char *proto_name;
	fingerprint = context->fingerprint_root.next;
	/* If there's no fingerprint, don't add it to the known
	 * fingerprints list */
	while(fingerprint) {
	    username = context->username;

	    is_verified = (fingerprint->trust && fingerprint->trust[0]);
	    otrl_privkey_hash_to_human(hash, fingerprint->fingerprint);
	    p = purple_find_prpl(context->protocol);
	    proto_name = (p && p->info->name) ? p->info->name : "Unknown";
	    account = g_strdup_printf("%s (%s)", context->accountname, proto_name);

	    gtk_list_store_append(ls, &itr);
	    gtk_list_store_set(ls, &itr,
		STATUS_COLUMN, otrg_plugin_context_to_trust(context),
		VERIFIED_COLUMN, is_verified, 
		SCREEN_NAME_COLUMN, username, 
		FP_STRING_COLUMN, hash, 
		ACCOUNT_COLUMN, account,
		FINGERPRINT_DATA_COLUMN, fingerprint,
		-1);

	    g_free(account);

	    if (ui_layout.selected_fprint == fingerprint && NULL == trr) {
	    	tp = gtk_tree_model_get_path(GTK_TREE_MODEL(ls), &itr);
		if (tp) {
		    trr = gtk_tree_row_reference_new(GTK_TREE_MODEL(ls), tp);
		    gtk_tree_path_free(tp);
		}
	    }
	    fingerprint = fingerprint->next;
	}
    }

    g_object_set(G_OBJECT(ui_layout.keylist), "model", ls, NULL);
    if (trr) {
	if ((tp = gtk_tree_row_reference_get_path(trr)) != NULL) {
	    GtkTreeSelection *sel =
	    	gtk_tree_view_get_selection(ui_layout.keylist);

	    if (sel)
		gtk_tree_selection_select_path(sel, tp);
	    gtk_tree_path_free(tp);
	}
	gtk_tree_row_reference_free(trr);
    }

    /* Reset the sorting */
    for (ll_columns = ll_itr = gtk_tree_view_get_columns(ui_layout.keylist);
         ll_itr != NULL ;
	 ll_itr = ll_itr->next) {
	if (ll_itr->data)
	    g_object_set_data(G_OBJECT(ll_itr->data), "sort-type",
	    	(gpointer)GTK_SORT_ASCENDING);
    }
    g_list_free(ll_columns);
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(ls),
    	STATUS_COLUMN, GTK_SORT_ASCENDING);
}

static void generate(GtkWidget *widget, gpointer data)
{
    PurpleAccount *account;
    account = pidgin_account_option_menu_get_selected(ui_layout.accountmenu);
	
    if (account == NULL) return;
	
    otrg_plugin_create_privkey(purple_account_get_username(account),
	    purple_account_get_protocol_id(account),
	    gtk_widget_get_toplevel(widget));
}

static void ui_destroyed(GtkObject *object)
{
    /* If this is called, we need to invalidate the stored pointers in
     * the ui_layout struct. */
    ui_layout.accountmenu = NULL;
    ui_layout.fprint_label = NULL;
    ui_layout.generate_button = NULL;
    ui_layout.scrollwin = NULL;
    ui_layout.keylist = NULL;
    ui_layout.selected_fprint = NULL;
    ui_layout.connect_button = NULL;
    ui_layout.disconnect_button = NULL;
    ui_layout.forget_button = NULL;
    ui_layout.verify_button = NULL;
    ui_layout.oo.enablebox = NULL;
    ui_layout.oo.automaticbox = NULL;
    ui_layout.oo.onlyprivatebox = NULL;
}

static void connect_connection(GtkWidget *widget, gpointer data)
{
    /* Send an OTR Query to the other side. */
    ConnContext *context;

    if (ui_layout.selected_fprint == NULL) return;

    context = ui_layout.selected_fprint->context;
    otrg_ui_connect_connection(context);
}

static void disconnect_connection(GtkWidget *widget, gpointer data)
{
    /* Forget whatever state we've got with this context */
    ConnContext *context;

    if (ui_layout.selected_fprint == NULL) return;

    context = ui_layout.selected_fprint->context;
    if (context == NULL) return;
	
    /* Don't do anything with fingerprints other than the active one
     * if we're in the ENCRYPTED state */
    if (context->msgstate == OTRL_MSGSTATE_ENCRYPTED &&
	    context->active_fingerprint != ui_layout.selected_fprint) {
	return;
    }
	
    otrg_ui_disconnect_connection(context);
}

static void forget_fingerprint(GtkWidget *widget, gpointer data)
{
    Fingerprint *fingerprint = ui_layout.selected_fprint;

    otrg_ui_forget_fingerprint(fingerprint);
}

static void verify_fingerprint(GtkWidget *widget, gpointer data)
{
    Fingerprint *fingerprint = ui_layout.selected_fprint;

    otrg_dialog_verify_fingerprint(fingerprint, gtk_widget_get_toplevel(widget));
}

static void otroptions_clicked_cb(GtkButton *button, struct otroptionsdata *oo)
{
    gtk_widget_set_sensitive(oo->enablebox, TRUE);
    if (gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(oo->enablebox))) {
	gtk_widget_set_sensitive(oo->automaticbox, TRUE);
	if (gtk_toggle_button_get_active(
		    GTK_TOGGLE_BUTTON(oo->automaticbox))) {
	    gtk_widget_set_sensitive(oo->onlyprivatebox, TRUE);
	} else {
	    gtk_widget_set_sensitive(oo->onlyprivatebox, FALSE);
	}
	gtk_widget_set_sensitive(oo->avoidloggingotrbox, TRUE);
    } else {
	gtk_widget_set_sensitive(oo->automaticbox, FALSE);
	gtk_widget_set_sensitive(oo->onlyprivatebox, FALSE);
	gtk_widget_set_sensitive(oo->avoidloggingotrbox, FALSE);
    }
}

static void create_otroption_buttons(struct otroptionsdata *oo,
	GtkWidget *vbox)
{
    GtkWidget *hbox;

    oo->enablebox = gtk_check_button_new_with_label(_("Enable private "
	    "messaging"));
    oo->automaticbox = gtk_check_button_new_with_label(_("Automatically "
	    "initiate private messaging"));
    oo->onlyprivatebox = gtk_check_button_new_with_label(_("Require private "
	    "messaging"));
    oo->avoidloggingotrbox = gtk_check_button_new_with_label(
	    _("Don't log OTR conversations"));

    /* Pack the first checkbox straight */
    gtk_box_pack_start(GTK_BOX(vbox), oo->enablebox, FALSE, TRUE, 0);

    /* Indent the second checkbox by one label */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(N_(" ")), FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), oo->automaticbox, FALSE, TRUE, 0);

    /* Indent the third checkbox by two labels */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(N_(" ")), FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(N_(" ")), FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), oo->onlyprivatebox, FALSE, TRUE, 0);

    /* Pack the fourth checkbox straight */
    gtk_box_pack_start(GTK_BOX(vbox), oo->avoidloggingotrbox, FALSE, TRUE, 0);

    g_signal_connect(G_OBJECT(oo->enablebox), "clicked",
		     G_CALLBACK(otroptions_clicked_cb), oo);
    g_signal_connect(G_OBJECT(oo->automaticbox), "clicked",
		     G_CALLBACK(otroptions_clicked_cb), oo);
    g_signal_connect(G_OBJECT(oo->onlyprivatebox), "clicked",
		     G_CALLBACK(otroptions_clicked_cb), oo);
    g_signal_connect(G_OBJECT(oo->avoidloggingotrbox), "clicked",
		     G_CALLBACK(otroptions_clicked_cb), oo);
}

/* Load the global OTR prefs */
static void otrg_gtk_ui_global_prefs_load(gboolean *enabledp,
	gboolean *automaticp, gboolean *onlyprivatep,
	gboolean *avoidloggingotrp)
{
    if (purple_prefs_exists("/OTR/enabled")) {
	*enabledp = purple_prefs_get_bool("/OTR/enabled");
	*automaticp = purple_prefs_get_bool("/OTR/automatic");
	*onlyprivatep = purple_prefs_get_bool("/OTR/onlyprivate");
	*avoidloggingotrp = purple_prefs_get_bool("/OTR/avoidloggingotr");
    } else {
	*enabledp = TRUE;
	*automaticp = TRUE;
	*onlyprivatep = FALSE;
	*avoidloggingotrp = FALSE;
    }
}

/* Save the global OTR prefs */
static void otrg_gtk_ui_global_prefs_save(gboolean enabled,
	gboolean automatic, gboolean onlyprivate, gboolean avoidloggingotr)
{
    if (! purple_prefs_exists("/OTR")) {
	purple_prefs_add_none("/OTR");
    }
    purple_prefs_set_bool("/OTR/enabled", enabled);
    purple_prefs_set_bool("/OTR/automatic", automatic);
    purple_prefs_set_bool("/OTR/onlyprivate", onlyprivate);
    purple_prefs_set_bool("/OTR/avoidloggingotr", avoidloggingotr);
}

/* Load the OTR prefs for a particular buddy */
static void otrg_gtk_ui_buddy_prefs_load(PurpleBuddy *buddy,
	gboolean *usedefaultp, gboolean *enabledp, gboolean *automaticp,
	gboolean *onlyprivatep, gboolean *avoidloggingotrp)
{
    PurpleBlistNode *node = &(buddy->node);

    *usedefaultp = ! purple_blist_node_get_bool(node, "OTR/overridedefault");

    if (*usedefaultp) {
	otrg_gtk_ui_global_prefs_load(enabledp, automaticp, onlyprivatep,
		avoidloggingotrp);
    } else {
	*enabledp = purple_blist_node_get_bool(node, "OTR/enabled");
	*automaticp = purple_blist_node_get_bool(node, "OTR/automatic");
	*onlyprivatep = purple_blist_node_get_bool(node, "OTR/onlyprivate");
	*avoidloggingotrp =
	    purple_blist_node_get_bool(node, "OTR/avoidloggingotr");
    }
}

/* Save the OTR prefs for a particular buddy */
static void otrg_gtk_ui_buddy_prefs_save(PurpleBuddy *buddy,
	gboolean usedefault, gboolean enabled, gboolean automatic,
	gboolean onlyprivate, gboolean avoidloggingotr)
{
    PurpleBlistNode *node = &(buddy->node);

    purple_blist_node_set_bool(node, "OTR/overridedefault", !usedefault);
    purple_blist_node_set_bool(node, "OTR/enabled", enabled);
    purple_blist_node_set_bool(node, "OTR/automatic", automatic);
    purple_blist_node_set_bool(node, "OTR/onlyprivate", onlyprivate);
    purple_blist_node_set_bool(node, "OTR/avoidloggingotr", avoidloggingotr);
}

static void load_otroptions(struct otroptionsdata *oo)
{
    gboolean otrenabled;
    gboolean otrautomatic;
    gboolean otronlyprivate;
    gboolean otravoidloggingotr;

    otrg_gtk_ui_global_prefs_load(&otrenabled, &otrautomatic, &otronlyprivate,
	    &otravoidloggingotr);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oo->enablebox),
	    otrenabled);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oo->automaticbox),
	    otrautomatic);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oo->onlyprivatebox),
	    otronlyprivate);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oo->avoidloggingotrbox),
	    otravoidloggingotr);

    otroptions_clicked_cb(GTK_BUTTON(oo->enablebox), oo);
}

/* Create the privkeys UI, and pack it into the vbox */
static void make_privkeys_ui(GtkWidget *vbox)
{
    GtkWidget *fbox;
    GtkWidget *hbox;
    GtkWidget *label;

    fbox = pidgin_make_frame(vbox, _("My private keys"));

    hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_BOX_SPACE);
    gtk_box_pack_start(GTK_BOX(fbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new(_("Key for account:"));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    ui_layout.accountmenu = pidgin_account_option_menu_new(NULL, 1,
	    G_CALLBACK(account_menu_changed_cb), NULL, NULL);
    gtk_box_pack_start(GTK_BOX(hbox), ui_layout.accountmenu, TRUE, TRUE, 0);

    /* Make sure we notice if the menu changes because an account has
     * been added or removed */
    purple_signal_connect(purple_accounts_get_handle(), "account-added",
	    ui_layout.accountmenu,
	    PURPLE_CALLBACK(account_menu_added_removed_cb), NULL);
    purple_signal_connect(purple_accounts_get_handle(), "account-removed",
	    ui_layout.accountmenu,
	    PURPLE_CALLBACK(account_menu_added_removed_cb), NULL);

    ui_layout.fprint_label = gtk_label_new("");
    gtk_label_set_selectable(GTK_LABEL(ui_layout.fprint_label), 1);
    gtk_box_pack_start(GTK_BOX(fbox), ui_layout.fprint_label,
	    FALSE, FALSE, 0);

    ui_layout.generate_button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(ui_layout.generate_button), "clicked",
	    GTK_SIGNAL_FUNC(generate), NULL);

    label = gtk_label_new(_("Generate"));
    gtk_container_add(GTK_CONTAINER(ui_layout.generate_button), label);

    otrg_gtk_ui_update_fingerprint();

    gtk_box_pack_start(GTK_BOX(fbox), ui_layout.generate_button,
	    FALSE, FALSE, 0);
}

/* Save the global OTR options whenever they're clicked */
static void otroptions_save_cb(GtkButton *button, struct otroptionsdata *oo)
{
    otrg_gtk_ui_global_prefs_save(
	    gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(oo->enablebox)),
	    gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(oo->automaticbox)),
	    gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(oo->onlyprivatebox)),
	    gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(oo->avoidloggingotrbox)));

    otrg_dialog_resensitize_all();
}

/* Make the options UI, and pack it into the vbox */
static void make_options_ui(GtkWidget *vbox)
{
    GtkWidget *fbox;

    fbox = pidgin_make_frame(vbox, _("Default OTR Settings"));

    create_otroption_buttons(&(ui_layout.oo), fbox);

    load_otroptions(&(ui_layout.oo));

    g_signal_connect(G_OBJECT(ui_layout.oo.enablebox), "clicked",
		     G_CALLBACK(otroptions_save_cb), &(ui_layout.oo));
    g_signal_connect(G_OBJECT(ui_layout.oo.automaticbox), "clicked",
		     G_CALLBACK(otroptions_save_cb), &(ui_layout.oo));
    g_signal_connect(G_OBJECT(ui_layout.oo.onlyprivatebox), "clicked",
		     G_CALLBACK(otroptions_save_cb), &(ui_layout.oo));
    g_signal_connect(G_OBJECT(ui_layout.oo.avoidloggingotrbox), "clicked",
		     G_CALLBACK(otroptions_save_cb), &(ui_layout.oo));
}

static void
fingerprint_list_selection_changed(GtkTreeSelection *sel, gpointer null)
{
    GtkTreeModel *model = NULL;
    GtkTreeIter itr;
    gboolean connect_sensitive = FALSE;
    gboolean disconnect_sensitive = FALSE;
    gboolean forget_sensitive = FALSE;
    gboolean verify_sensitive = FALSE;
    Fingerprint *f = NULL;

    if (gtk_tree_selection_get_selected(sel, &model, &itr)) {
	if (model)
	    gtk_tree_model_get(model, &itr, FINGERPRINT_DATA_COLUMN, &f, -1);
    }

    if (f) {
	verify_sensitive = TRUE;

	disconnect_sensitive =
	    (f->context->msgstate == OTRL_MSGSTATE_ENCRYPTED && 
	     f->context->active_fingerprint == f) ||
	    (f->context->msgstate == OTRL_MSGSTATE_FINISHED);

	forget_sensitive =
	    (f->context->msgstate != OTRL_MSGSTATE_ENCRYPTED || 
	     f->context->active_fingerprint != f);

	connect_sensitive = 
	    (f->context->msgstate == OTRL_MSGSTATE_PLAINTEXT) ||
	    (f->context->msgstate == OTRL_MSGSTATE_FINISHED);
    }

    gtk_widget_set_sensitive(ui_layout.connect_button, connect_sensitive);
    gtk_widget_set_sensitive(ui_layout.disconnect_button, disconnect_sensitive);
    gtk_widget_set_sensitive(ui_layout.forget_button, forget_sensitive);
    gtk_widget_set_sensitive(ui_layout.verify_button, verify_sensitive);

    ui_layout.selected_fprint = f;
}

/* Convert a trust level inside the tree view to its corresponding icon */
static void
cr_trust_to_pixbuf(GtkTreeViewColumn *col, GtkCellRenderer *cr,
    GtkTreeModel *tm, GtkTreeIter *itr, gpointer null)
{
    TrustLevel trust_level;

    gtk_tree_model_get(tm, itr, STATUS_COLUMN, &trust_level, -1);
    g_object_set(G_OBJECT(cr), "pixbuf", otr_trust_level_icon(trust_level),
    	NULL);
}

static void
fp_tv_column_clicked(GObject *col, GObject *tree_view)
{
    GtkSortType sort_type;
    GtkTreeSortable *sortable = NULL;

    g_object_get(tree_view, "model", &sortable, NULL);
    if (!sortable) return;

    sort_type = (GtkSortType)g_object_get_data(col, "sort-type");
    sort_type = (sort_type == GTK_SORT_ASCENDING)
    	? GTK_SORT_DESCENDING : GTK_SORT_ASCENDING;

    gtk_tree_sortable_set_sort_column_id(sortable,
	    (gint)g_object_get_data(col, "column-idx"), sort_type);

    g_object_set_data(col, "sort-type", (gpointer)sort_type);
}

/* Create the fingerprint UI, and pack it into the vbox */
static void make_fingerprints_ui(GtkWidget *vbox)
{
    GtkWidget *alignment;
    GtkWidget *table;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *col;
    GtkCellRenderer *cr;

    ui_layout.scrollwin = gtk_scrolled_window_new(0, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ui_layout.scrollwin), 
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    ui_layout.keylist = GTK_TREE_VIEW(gtk_tree_view_new());
		if (g_object_class_find_property(g_type_class_peek(GTK_TYPE_TREE_VIEW), "allow-checkbox-mode"))
			g_object_set(G_OBJECT(ui_layout.keylist), "allow-checkbox-mode", FALSE, NULL);
    gtk_tree_view_set_headers_clickable(ui_layout.keylist, TRUE);
    gtk_tree_view_set_headers_visible(ui_layout.keylist, TRUE);
    selection = gtk_tree_view_get_selection(ui_layout.keylist);
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    cr = gtk_cell_renderer_pixbuf_new();
    col = gtk_tree_view_column_new_with_attributes(_("Status"), cr, NULL);
    g_object_set_data(G_OBJECT(col), "column-idx", (gpointer)STATUS_COLUMN);
    g_object_set_data(G_OBJECT(col), "sort-type", (gpointer)GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_clickable(col, TRUE);
    gtk_tree_view_column_set_cell_data_func(col, cr,
	    (GtkTreeCellDataFunc)cr_trust_to_pixbuf, NULL, NULL);
    gtk_tree_view_append_column(ui_layout.keylist, col);
    g_signal_connect(G_OBJECT(col), "clicked", (GCallback)fp_tv_column_clicked,
	    ui_layout.keylist);

    col = gtk_tree_view_column_new_with_attributes(_("Verified"),
	    gtk_cell_renderer_toggle_new(), "active", VERIFIED_COLUMN, NULL);
    g_object_set_data(G_OBJECT(col), "column-idx", (gpointer)VERIFIED_COLUMN);
    g_object_set_data(G_OBJECT(col), "sort-type", (gpointer)GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_clickable(col, TRUE);
    gtk_tree_view_append_column(ui_layout.keylist, col);
    g_signal_connect(G_OBJECT(col), "clicked", (GCallback)fp_tv_column_clicked,
	    ui_layout.keylist);

    col = gtk_tree_view_column_new_with_attributes(_("Screen name"),
	    gtk_cell_renderer_text_new(), "text", SCREEN_NAME_COLUMN, NULL);
    g_object_set_data(G_OBJECT(col), "column-idx", (gpointer)SCREEN_NAME_COLUMN);
    g_object_set_data(G_OBJECT(col), "sort-type", (gpointer)GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_clickable(col, TRUE);
    gtk_tree_view_append_column(ui_layout.keylist, col);
    g_signal_connect(G_OBJECT(col), "clicked", (GCallback)fp_tv_column_clicked,
	    ui_layout.keylist);

    col = gtk_tree_view_column_new_with_attributes(_("Fingerprint"),
	    gtk_cell_renderer_text_new(), "text", FP_STRING_COLUMN, NULL);
    g_object_set_data(G_OBJECT(col), "column-idx", (gpointer)FP_STRING_COLUMN);
    g_object_set_data(G_OBJECT(col), "sort-type", (gpointer)GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_clickable(col, TRUE);
    gtk_tree_view_append_column(ui_layout.keylist, col);
    g_signal_connect(G_OBJECT(col), "clicked", (GCallback)fp_tv_column_clicked,
	    ui_layout.keylist);

    col = gtk_tree_view_column_new_with_attributes(_("Account"),
	    gtk_cell_renderer_text_new(), "text", ACCOUNT_COLUMN, NULL);
    g_object_set_data(G_OBJECT(col), "column-idx", (gpointer)ACCOUNT_COLUMN);
    g_object_set_data(G_OBJECT(col), "sort-type", (gpointer)GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_clickable(col, TRUE);
    gtk_tree_view_append_column(ui_layout.keylist, col);
    g_signal_connect(G_OBJECT(col), "clicked", (GCallback)fp_tv_column_clicked,
	    ui_layout.keylist);

    gtk_container_add(GTK_CONTAINER(ui_layout.scrollwin),
			GTK_WIDGET(ui_layout.keylist));
    gtk_box_pack_start(GTK_BOX(vbox), ui_layout.scrollwin,
	    TRUE, TRUE, 0);

    otrg_gtk_ui_update_keylist();

    alignment = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), alignment, FALSE, TRUE, 0);

    table = gtk_table_new(2, 2, TRUE);
    gtk_container_add(GTK_CONTAINER(alignment), table);
    gtk_table_set_row_spacings(GTK_TABLE(table), PIDGIN_HIG_BOX_SPACE);
    gtk_table_set_col_spacings(GTK_TABLE(table), PIDGIN_HIG_CAT_SPACE);

    ui_layout.connect_button = g_object_new(GTK_TYPE_BUTTON,
	"label", _("Start private connection"), "sensitive", FALSE,
	"image", g_object_new(GTK_TYPE_IMAGE,
	    "pixbuf", otr_trust_level_icon(TRUST_PRIVATE), NULL), NULL);
    gtk_signal_connect(GTK_OBJECT(ui_layout.connect_button), "clicked",
	    GTK_SIGNAL_FUNC(connect_connection), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), ui_layout.connect_button,
	    0, 1, 0, 1);

    ui_layout.disconnect_button = g_object_new(GTK_TYPE_BUTTON,
	"label", _("End private connection"), "sensitive", FALSE,
	"image", g_object_new(GTK_TYPE_IMAGE,
	    "pixbuf", otr_trust_level_icon(TRUST_FINISHED), NULL), NULL);
    gtk_signal_connect(GTK_OBJECT(ui_layout.disconnect_button), "clicked",
	    GTK_SIGNAL_FUNC(disconnect_connection), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), ui_layout.disconnect_button,
	    0, 1, 1, 2);

    ui_layout.verify_button = g_object_new(GTK_TYPE_BUTTON,
	"label", _("Verify fingerprint"), "sensitive", FALSE,
	"image", g_object_new(GTK_TYPE_IMAGE, "stock", GTK_STOCK_YES,
    	    "icon-size", GTK_ICON_SIZE_BUTTON, NULL), NULL);
    gtk_signal_connect(GTK_OBJECT(ui_layout.verify_button), "clicked",
	    GTK_SIGNAL_FUNC(verify_fingerprint), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), ui_layout.verify_button,
	    1, 2, 0, 1);

    ui_layout.forget_button = g_object_new(GTK_TYPE_BUTTON,
	"label", _("Forget fingerprint"), "sensitive", FALSE,
	"image", g_object_new(GTK_TYPE_IMAGE, "stock", GTK_STOCK_DELETE,
	    "icon-size", GTK_ICON_SIZE_BUTTON, NULL), NULL);
    gtk_signal_connect(GTK_OBJECT(ui_layout.forget_button), "clicked",
	    GTK_SIGNAL_FUNC(forget_fingerprint), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), ui_layout.forget_button,
	    1, 2, 1, 2);

    gtk_signal_connect(GTK_OBJECT(vbox), "destroy",
	    GTK_SIGNAL_FUNC(ui_destroyed), NULL);

    g_signal_connect(G_OBJECT(selection), "changed",
	    (GCallback)fingerprint_list_selection_changed, NULL);
}

/* Construct the OTR UI widget */
GtkWidget* otrg_gtk_ui_make_widget(PurplePlugin *plugin)
{
    GtkWidget *vbox = gtk_vbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    GtkWidget *fingerprintbox = gtk_vbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    GtkWidget *configbox = gtk_vbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    GtkWidget *notebook = gtk_notebook_new();

    gtk_container_set_border_width(GTK_CONTAINER(vbox), PIDGIN_HIG_BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(fingerprintbox), PIDGIN_HIG_BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(configbox), PIDGIN_HIG_BORDER);

    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

    make_privkeys_ui(configbox);

    make_options_ui(configbox);

    /*
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
    gtk_container_add(GTK_CONTAINER(confwindow), vbox);
    */

    make_fingerprints_ui(fingerprintbox);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
	    _pidgin_make_container_scrollable(configbox, NULL),
	    gtk_label_new(_("Config")));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
	    _pidgin_make_container_scrollable(fingerprintbox, NULL),
	    gtk_label_new(_("Known fingerprints")));

    gtk_widget_show_all(vbox);

    return vbox;
}

struct cbdata {
    GtkWidget *dialog;
    PurpleBuddy *buddy;
    GtkWidget *defaultbox;
    struct otroptionsdata oo;
};

static void default_clicked_cb(GtkButton *button, struct cbdata *data)
{
    gboolean defaultset =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->defaultbox));
    if (defaultset) {
	gtk_widget_set_sensitive(data->oo.enablebox, FALSE);
	gtk_widget_set_sensitive(data->oo.automaticbox, FALSE);
	gtk_widget_set_sensitive(data->oo.onlyprivatebox, FALSE);
	gtk_widget_set_sensitive(data->oo.avoidloggingotrbox, FALSE);
    } else {
	otroptions_clicked_cb(button, &(data->oo));
    }
}

static void load_buddyprefs(struct cbdata *data)
{
    gboolean usedefault, enabled, automatic, onlyprivate, avoidloggingotr;

    otrg_gtk_ui_buddy_prefs_load(data->buddy, &usedefault, &enabled,
	    &automatic, &onlyprivate, &avoidloggingotr);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->defaultbox),
	    usedefault);

    if (usedefault) {
	/* Load the global defaults */
	load_otroptions(&(data->oo));
    } else {
	/* We've got buddy-specific prefs */
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(data->oo.enablebox), enabled);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(data->oo.automaticbox), automatic);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(data->oo.onlyprivatebox), onlyprivate);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(data->oo.avoidloggingotrbox),
		avoidloggingotr);
    }

    default_clicked_cb(GTK_BUTTON(data->defaultbox), data);
}

static void config_buddy_destroy_cb(GtkWidget *w, struct cbdata *data)
{
    free(data);
}

static void config_buddy_clicked_cb(GtkButton *button, struct cbdata *data)
{
    gboolean enabled = gtk_toggle_button_get_active(
			     GTK_TOGGLE_BUTTON(data->oo.enablebox));
    
    /* Apply the changes */
    otrg_gtk_ui_buddy_prefs_save(data->buddy,
	 gtk_toggle_button_get_active(
	     GTK_TOGGLE_BUTTON(data->defaultbox)),
	 enabled,
	 gtk_toggle_button_get_active(
	     GTK_TOGGLE_BUTTON(data->oo.automaticbox)),
	 gtk_toggle_button_get_active(
	     GTK_TOGGLE_BUTTON(data->oo.onlyprivatebox)),
	 gtk_toggle_button_get_active(
	     GTK_TOGGLE_BUTTON(data->oo.avoidloggingotrbox)));

    otrg_dialog_resensitize_all();
}

static void config_buddy_response_cb(GtkDialog *dialog, gint resp,
	struct cbdata *data)
{
    gtk_widget_destroy(data->dialog);
}

static void otrg_gtk_ui_config_buddy(PurpleBuddy *buddy)
{
    GtkWidget *dialog;
    GtkWidget *label;
    char *label_text;
    char *label_markup;
    struct cbdata *data = malloc(sizeof(struct cbdata));

    if (!data) return;

    dialog = gtk_dialog_new_with_buttons(_("OTR Settings"),
					 NULL, 0,
					 GTK_STOCK_OK, GTK_RESPONSE_OK,
					 NULL);
    gtk_window_set_accept_focus(GTK_WINDOW(dialog), FALSE);
    gtk_window_set_role(GTK_WINDOW(dialog), "otr_options");

    gtk_container_set_border_width(GTK_CONTAINER(dialog), PIDGIN_HIG_BORDER);
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_BOX_SPACE);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 0);

    data->dialog = dialog;
    data->buddy = buddy;

    /* Set the title */

    label_text = g_strdup_printf(_("OTR Settings for %s"),
	    purple_buddy_get_contact_alias(buddy));
    label_markup = g_strdup_printf("<span weight=\"bold\" size=\"larger\">"
	    "%s</span>", label_text);

    label = gtk_label_new(NULL);

    gtk_label_set_markup(GTK_LABEL(label), label_markup);
    g_free(label_markup);
    g_free(label_text);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label,
	    FALSE, FALSE, 0);

    /* Make the cascaded checkboxes */

    data->defaultbox = gtk_check_button_new_with_label(_("Use default "
	    "OTR settings for this buddy"));

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), data->defaultbox,
	    FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), gtk_hseparator_new(),
	    FALSE, FALSE, 0);

    create_otroption_buttons(&(data->oo), GTK_DIALOG(dialog)->vbox);

    g_signal_connect(G_OBJECT(data->defaultbox), "clicked",
		     G_CALLBACK(default_clicked_cb), data);
    g_signal_connect(G_OBJECT(data->defaultbox), "clicked",
		     G_CALLBACK(config_buddy_clicked_cb), data);
    g_signal_connect(G_OBJECT(data->oo.enablebox), "clicked",
		     G_CALLBACK(config_buddy_clicked_cb), data);
    g_signal_connect(G_OBJECT(data->oo.automaticbox), "clicked",
		     G_CALLBACK(config_buddy_clicked_cb), data);
    g_signal_connect(G_OBJECT(data->oo.onlyprivatebox), "clicked",
		     G_CALLBACK(config_buddy_clicked_cb), data);
    g_signal_connect(G_OBJECT(data->oo.avoidloggingotrbox), "clicked",
		     G_CALLBACK(config_buddy_clicked_cb), data);

    /* Set the inital states of the buttons */
    load_buddyprefs(data);

    g_signal_connect(G_OBJECT(dialog), "destroy",
		     G_CALLBACK(config_buddy_destroy_cb), data);
    g_signal_connect(G_OBJECT(dialog), "response",
		     G_CALLBACK(config_buddy_response_cb), data);

    gtk_widget_show_all(dialog);
}

/* Load the preferences for a particular account / username */
static void otrg_gtk_ui_get_prefs(OtrgUiPrefs *prefsp, PurpleAccount *account,
	const char *name)
{
    PurpleBuddy *buddy;
    gboolean otrenabled, otrautomatic, otronlyprivate, otravoidloggingotr;
    gboolean buddyusedefault, buddyenabled, buddyautomatic, buddyonlyprivate,
	     buddyavoidloggingotr;

    prefsp->policy = OTRL_POLICY_DEFAULT;
    prefsp->avoid_logging_otr = FALSE;
    
    /* Get the default policy */
    otrg_gtk_ui_global_prefs_load(&otrenabled, &otrautomatic, &otronlyprivate,
	    &otravoidloggingotr);

    if (otrenabled) {
	if (otrautomatic) {
	    if (otronlyprivate) {
		prefsp->policy = OTRL_POLICY_ALWAYS;
	    } else {
		prefsp->policy = OTRL_POLICY_OPPORTUNISTIC;
	    }
	} else {
	    prefsp->policy = OTRL_POLICY_MANUAL;
	}
	prefsp->avoid_logging_otr = otravoidloggingotr;
    } else {
	prefsp->policy = OTRL_POLICY_NEVER;
    }

    buddy = purple_find_buddy(account, name);
    if (!buddy) return;

    /* Get the buddy-specific policy, if present */
    otrg_gtk_ui_buddy_prefs_load(buddy, &buddyusedefault, &buddyenabled,
	    &buddyautomatic, &buddyonlyprivate, &buddyavoidloggingotr);

    if (buddyusedefault) return;

    if (buddyenabled) {
	if (buddyautomatic) {
	    if (buddyonlyprivate) {
		prefsp->policy = OTRL_POLICY_ALWAYS;
	    } else {
		prefsp->policy = OTRL_POLICY_OPPORTUNISTIC;
	    }
	} else {
	    prefsp->policy = OTRL_POLICY_MANUAL;
	}
	prefsp->avoid_logging_otr = buddyavoidloggingotr;
    } else {
	prefsp->policy = OTRL_POLICY_NEVER;
    }
}

static void *
otrg_gtk_ui_ui_data_from_conversation(PurpleConversation *convo)
{
    PidginBuddyList *blist = NULL;
    if (convo) {
/*
	if ((widget = purple_conversation_get_data(conv, "otr-button")) != NULL)
	    if ((widget = gtk_widget_get_toplevel(widget)) != NULL)
		return GTK_WINDOW(widget);
*/
	return PIDGIN_CONVERSATION(convo)->win->window;
    }

    if ((blist = pidgin_blist_get_default_gtk_blist()) != NULL)
    	return GTK_WINDOW(blist->window);

    return NULL;
}

/* Initialize the OTR UI subsystem */
static void otrg_gtk_ui_init(void)
{
    /* Nothing to do */
}

/* Deinitialize the OTR UI subsystem */
static void otrg_gtk_ui_cleanup(void)
{
    /* Nothing to do */
}

static const OtrgUiUiOps gtk_ui_ui_ops = {
    otrg_gtk_ui_init,
    otrg_gtk_ui_cleanup,
    otrg_gtk_ui_update_fingerprint,
    otrg_gtk_ui_update_keylist,
    otrg_gtk_ui_config_buddy,
    otrg_gtk_ui_get_prefs,
	otrg_gtk_ui_ui_data_from_conversation
};

/* Get the GTK UI ops */
const OtrgUiUiOps *otrg_gtk_ui_get_ui_ops(void)
{
    return &gtk_ui_ui_ops;
}
