/*
 *  Off-the-Record Messaging plugin for pidgin
 *  Copyright (C) 2004-2007  Ian Goldberg, Chris Alexander, Nikita Borisov
 *                           <otr@cypherpunks.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* config.h */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* system headers */
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>

/* gcrypt headers */
#include <gcrypt.h>

/* purple headers */
#include "version.h"
#include "pidginstock.h"
#include "plugin.h"
#include "notify.h"
#include "gtkconv.h"
#include "gtkutils.h"
#include "gtkimhtml.h"
#include "hildon-hacks.h"
#include "util.h"

/* pidgin headers */
#include "gtkblist.h"

#ifdef ENABLE_NLS
/* internationalisation headers */
#include <glib/gi18n-lib.h>
#endif

/* libotr headers */
#include <libotr/dh.h>
#include <libotr/privkey.h>
#include <libotr/proto.h>
#include <libotr/message.h>
#include <libotr/userstate.h>

/* purple-otr headers */
#include "otr-plugin.h"
#include "dialogs.h"
#include "gtk-dialog.h"
#include "gtk-ui.h"
#include "ui.h"
#include "pleasewaitdialog.h"

typedef struct {
    ConnContext *context;  /* The context used to fire library code */
    GtkEntry *entry;	   /* The text entry field containing the secret */
    gboolean responder;	   /* Whether or not this is the first side to give
			      their secret */
} SmpResponsePair;

/* The response code returned by pushing the "Advanced..." button on the
 * SMP dialog */
#define OTRG_RESPONSE_ADVANCED 1

/* Information used by the plugin that is specific to both the
 * application and connection. */
typedef struct dialog_context_data {
    GtkWidget       *smp_secret_dialog;
    SmpResponsePair *smp_secret_smppair;
    GtkWidget       *smp_progress_dialog;
    GtkWidget       *smp_progress_bar;
    GtkWidget       *smp_progress_label;
} SMPData;

static void close_progress_window(SMPData *smp_data)
{
    if (smp_data->smp_progress_dialog) {
	gtk_dialog_response(GTK_DIALOG(smp_data->smp_progress_dialog),
		GTK_RESPONSE_REJECT);
    }
    smp_data->smp_progress_dialog = NULL;
    smp_data->smp_progress_bar = NULL;
    smp_data->smp_progress_label = NULL;
}

static void otrg_gtk_dialog_free_smp_data(PurpleConversation *conv)
{
    SMPData *smp_data = purple_conversation_get_data(conv, "otr-smpdata");
    if (!smp_data) return;

    if (smp_data->smp_secret_dialog) {
	gtk_dialog_response(GTK_DIALOG(smp_data->smp_secret_dialog),
		GTK_RESPONSE_REJECT);
    }
    smp_data->smp_secret_dialog = NULL;
    smp_data->smp_secret_smppair = NULL;

    close_progress_window(smp_data);

    free(smp_data);

    g_hash_table_remove(conv->data, "otr-smpdata");
}

static void otrg_gtk_dialog_add_smp_data(PurpleConversation *conv)
{
    SMPData *smp_data = malloc(sizeof(SMPData));
    smp_data->smp_secret_dialog = NULL;
    smp_data->smp_secret_smppair = NULL;
    smp_data->smp_progress_dialog = NULL;
    smp_data->smp_progress_bar = NULL;
    smp_data->smp_progress_label = NULL;

    purple_conversation_set_data(conv, "otr-smpdata", smp_data);
}

static GtkWidget *otr_icon(GtkWidget *image, TrustLevel level)
{
    GdkPixbuf *pixbuf = otr_trust_level_icon(level);

    if (image) {
	gtk_image_set_from_pixbuf(GTK_IMAGE(image), pixbuf);
    } else {
	image = gtk_image_new_from_pixbuf(pixbuf);
    }
    gdk_pixbuf_unref(pixbuf);

    return image;
}

static void message_response_cb(GtkDialog *dialog, gint id, GtkWidget *widget)
{
    gtk_widget_destroy(GTK_WIDGET(widget));
}

/* Forward declarations for the benefit of smp_message_response_cb */
static void verify_fingerprint(GtkWindow *parent, Fingerprint *fprint);
static GtkWidget *create_smp_progress_dialog(GtkWindow *parent,
	ConnContext *context);

/* Called when a button is pressed on the "progress bar" smp dialog */
static void smp_progress_response_cb(GtkDialog *dialog, gint response,
	ConnContext *context)
{
    PurpleConversation *conv = otrg_plugin_context_to_conv(context, 0);
    SMPData *smp_data = NULL;
    
    if (conv) {
	gdouble frac;

	smp_data = purple_conversation_get_data(conv, "otr-smpdata");
	frac = gtk_progress_bar_get_fraction(
		GTK_PROGRESS_BAR(smp_data->smp_progress_bar));

	if (frac != 0.0 && frac != 1.0 && response == GTK_RESPONSE_REJECT) {
	    otrg_plugin_abort_smp(context);
	}
    }
    /* In all cases, destroy the current window */
    gtk_widget_destroy(GTK_WIDGET(dialog));

    /* Clean up variables pointing to the destroyed objects */

    if (smp_data) {
	smp_data->smp_progress_bar = NULL;
	smp_data->smp_progress_label = NULL;
	smp_data->smp_progress_dialog = NULL;
    }
}

/* Called when a button is pressed on the "enter the secret" smp dialog
 * The data passed contains a pointer to the text entry field containing
 * the entered secret as well as the current context.
 */
static void smp_secret_response_cb(GtkDialog *dialog, gint response,
	SmpResponsePair *smppair)
{
    ConnContext* context;
    PurpleConversation *conv;
    SMPData *smp_data;

    if (!smppair) return;

    context = smppair->context;
    if (response == GTK_RESPONSE_ACCEPT) {
	GtkEntry* entry = smppair->entry;
	char *secret;
	size_t secret_len;

	if (context == NULL || context->msgstate != OTRL_MSGSTATE_ENCRYPTED)
		return;

	secret = g_strdup(gtk_entry_get_text(entry));

	secret_len = strlen(secret);

	if (smppair->responder) {
	    otrg_plugin_continue_smp(context, (const unsigned char *)secret,
		    secret_len);
	} else {
	    otrg_plugin_start_smp(context, (const unsigned char *)secret,
		    secret_len);
	}
	g_free(secret);

	/* launch progress bar window */
	create_smp_progress_dialog(
			gtk_window_get_transient_for(GTK_WINDOW(dialog)),
			context);
    } else if (response == OTRG_RESPONSE_ADVANCED) {
	ConnContext* context = smppair->context;

	if (context == NULL || context->msgstate != OTRL_MSGSTATE_ENCRYPTED)
		return;

	verify_fingerprint(
	    gtk_window_get_transient_for(GTK_WINDOW(dialog)),
	    context->active_fingerprint);
    } else {
        otrg_plugin_abort_smp(context);
    }
    /* In all cases, destroy the current window */
    gtk_widget_destroy(GTK_WIDGET(dialog));
    
    /* Clean up references to this window */
    conv = otrg_plugin_context_to_conv(smppair->context, 0);
    smp_data = purple_conversation_get_data(conv, "otr-smpdata");
    if (smp_data) {
	smp_data->smp_secret_dialog = NULL;
	smp_data->smp_secret_smppair = NULL;
    }

    /* Free the smppair memory */
    free(smppair);
}

static void close_smp_window(PurpleConversation *conv)
{
    SMPData *smp_data = purple_conversation_get_data(conv, "otr-smpdata");
    if (smp_data && smp_data->smp_secret_dialog) {
	gtk_dialog_response(GTK_DIALOG(smp_data->smp_secret_dialog),
		GTK_RESPONSE_REJECT);
    }
}

static GtkWidget *create_dialog(GtkWindow *parent,
	PurpleNotifyMsgType type, const char *title,
	const char *primary, const char *secondary, int sensitive,
	GtkWidget **labelp, void (*add_custom)(GtkWidget *vbox, void *data),
	void *add_custom_data)
{
    GtkWidget *dialog;
    GtkWidget *hbox;
    GtkWidget *vbox;
    GtkWidget *label;
    GtkWidget *img = NULL;
    char *label_text;
    const char *icon_name = NULL;

    switch (type) {
	case PURPLE_NOTIFY_MSG_ERROR:
	    icon_name = PIDGIN_STOCK_DIALOG_ERROR;
	    break;

	case PURPLE_NOTIFY_MSG_WARNING:
	    icon_name = PIDGIN_STOCK_DIALOG_WARNING;
	    break;

	case PURPLE_NOTIFY_MSG_INFO:
	    icon_name = PIDGIN_STOCK_DIALOG_INFO;
	    break;

	default:
	    icon_name = NULL;
	    break;
    }

    if (icon_name != NULL) {
	img = gtk_image_new_from_stock(icon_name,
		gtk_icon_size_from_name(PIDGIN_ICON_SIZE_TANGO_HUGE));
	gtk_misc_set_alignment(GTK_MISC(img), 0, 0);
    }

    dialog = gtk_dialog_new_with_buttons(
	    title ? title : PIDGIN_ALERT_TITLE, parent, 0,
	    GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

    gtk_window_set_focus_on_map(GTK_WINDOW(dialog), FALSE);
    gtk_window_set_role(GTK_WINDOW(dialog), "notify_dialog");

    g_signal_connect(G_OBJECT(dialog), "response",
			 G_CALLBACK(message_response_cb), dialog);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT,
	    sensitive);

    gtk_container_set_border_width(GTK_CONTAINER(dialog), PIDGIN_HIG_BORDER);
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_CAT_SPACE);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_BORDER);

    hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    vbox = gtk_vbox_new(FALSE, PIDGIN_HIG_BOX_SPACE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);

    if (img != NULL) {
	gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);
    }

    label_text = g_strdup_printf(
		       "<span weight=\"bold\" size=\"larger\">%s</span>%s%s",
		       (primary ? primary : ""),
		       (primary ? "\n\n" : ""),
		       (secondary ? secondary : ""));

    label = gtk_label_new(NULL);

    gtk_label_set_markup(GTK_LABEL(label), label_text);
    gtk_label_set_selectable(GTK_LABEL(label), 1);
    g_free(label_text);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    if (add_custom) {
	add_custom(vbox, add_custom_data);
    }
    gtk_box_pack_start(GTK_BOX(hbox), _pidgin_make_container_scrollable(vbox, dialog), TRUE, TRUE, 0);

    gtk_widget_show_all(dialog);

    if (labelp) *labelp = label;
    return dialog;
}

#define SHOW_ME_MORE_KEY "ShowMeMore"
#define MORE_INFORMATION (_("+ Explain further"))
#define LESS_INFORMATION (_("- Hide explanation"))

static void
set_moremarkup_visible(GtkWidget *imhtml, gboolean is_visible)
{
    GtkTextBuffer *buffer = NULL;
    GtkTextIter itr_mark, itr_end;
    GtkTextMark *mark = NULL;

    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(imhtml));

    if (!buffer) return;
    if ((mark = gtk_text_buffer_get_mark(buffer, SHOW_ME_MORE_KEY)) == NULL) return;

    gtk_text_buffer_get_iter_at_mark(buffer, &itr_mark, mark);
    gtk_text_buffer_get_end_iter(buffer, &itr_end);
    gtk_text_buffer_delete(buffer, &itr_mark, &itr_end);

    if(is_visible) {
	char *moremarkup = g_object_get_data(G_OBJECT(imhtml), SHOW_ME_MORE_KEY);

	gtk_imhtml_insert_link(GTK_IMHTML(imhtml), mark, LESS_INFORMATION, LESS_INFORMATION);
	if (moremarkup) {
	    gtk_imhtml_append_text(GTK_IMHTML(imhtml), "\n", GTK_IMHTML_NO_SCROLL);
	    gtk_imhtml_append_text(GTK_IMHTML(imhtml), moremarkup, GTK_IMHTML_NO_SCROLL);
	    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(imhtml), mark, 0.0, TRUE, 0.0, 0.0);
	}
    } else {
	gtk_imhtml_insert_link(GTK_IMHTML(imhtml), mark, MORE_INFORMATION, MORE_INFORMATION);
	gtk_text_buffer_get_start_iter(buffer, &itr_mark);
	gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(imhtml), &itr_mark, 0.0, TRUE, 0.0, 0.0);
    }
}

static gboolean
whatsthis_url_clicked_idle(char *url)
{
    purple_notify_uri(NULL, url);
    g_free(url);
    return FALSE;
}

static void
whatsthis_url_clicked(GtkWidget *imhtml, const char *url)
{
    if (!strcmp(url, MORE_INFORMATION))
	set_moremarkup_visible(imhtml, TRUE);
    else
    if (!strcmp(url, LESS_INFORMATION))
	set_moremarkup_visible(imhtml, FALSE);
    else
	g_idle_add((GSourceFunc)whatsthis_url_clicked_idle, g_strdup(url));
}

/* Adds a "What's this?" expander to a vbox, containing some "whatsthis"
 * markup displayed in a GtkIMHTML. This is followed by a "+ Explain further"
 * link which, when toggled, appends some "more" markup and removes it when
 * toggled again. */
static void add_whatsthis_more(GtkWidget *vbox, const char *whatsthismarkup,
	const char *moremarkup)
{
    GtkWidget *expander;
    GtkWidget *imh;
    GtkWidget *sw;
    GtkTextIter itr;
    GtkTextBuffer *buffer = NULL;
    GtkTextMark *mark = NULL;

    expander = gtk_expander_new_with_mnemonic(_("_What's this?"));
    gtk_box_pack_start(GTK_BOX(vbox), expander, FALSE, FALSE, 0);

    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
    	GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_container_add(GTK_CONTAINER(expander), sw);
    gtk_widget_set_size_request(sw, -1, 100);

    imh = gtk_imhtml_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(sw), imh);

    gtk_imhtml_append_text(GTK_IMHTML(imh), whatsthismarkup,
    	GTK_IMHTML_NO_SCROLL);
    gtk_imhtml_append_text(GTK_IMHTML(imh), "\n",
    	GTK_IMHTML_NO_SCROLL);

    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(imh));
    gtk_text_buffer_get_end_iter(buffer, &itr);
    mark = gtk_text_buffer_create_mark(buffer, SHOW_ME_MORE_KEY, &itr, TRUE);
    g_object_set_data_full(G_OBJECT(imh),
    	SHOW_ME_MORE_KEY, g_strdup(moremarkup), (GDestroyNotify)g_free);
    set_moremarkup_visible(imh, FALSE);

    g_signal_connect(G_OBJECT(imh), "url-clicked",  
    	(GCallback)whatsthis_url_clicked, NULL);
}

static GtkWidget *create_smp_dialog(const char *title,
	const char *primary, const char *secondary, int sensitive,
	GtkWidget **labelp, ConnContext *context, gboolean responder)
{
    GtkWidget *dialog;

    PurpleConversation *conv = otrg_plugin_context_to_conv(context, 1);
    SMPData *smp_data = purple_conversation_get_data(conv, "otr-smpdata");

    close_progress_window(smp_data);
    if (!(smp_data->smp_secret_dialog)) {
	GtkWidget *advbutton;
	GtkWidget *buttonspacer;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *entry;
	GtkWidget *label;
	GtkWidget *label2;
	GtkWidget *img = NULL;
	char *label_text;
	const char *icon_name = NULL;
	SmpResponsePair* smppair;
	char *moremarkup;

	icon_name = PIDGIN_STOCK_DIALOG_INFO;
	img = gtk_image_new_from_stock(icon_name, gtk_icon_size_from_name(PIDGIN_ICON_SIZE_TANGO_HUGE));
	gtk_misc_set_alignment(GTK_MISC(img), 0, 0);

	dialog = gtk_dialog_new_with_buttons(title ? title : PIDGIN_ALERT_TITLE, 
				     otrg_ui_ui_data_from_conversation(conv), 0,
				     GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
	gtk_dialog_set_default_response(GTK_DIALOG(dialog),
		GTK_RESPONSE_ACCEPT);

	/* Create the Advanced... button, and left-justify it.  This
	 * involves adding the button, and a blank label as a spacer, and
	 * reordering them so that they're at the beginning. */
	advbutton = gtk_dialog_add_button(GTK_DIALOG(dialog), _("Advanced..."),
		OTRG_RESPONSE_ADVANCED);
	buttonspacer = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		buttonspacer, TRUE, TRUE, 0);
	gtk_box_reorder_child(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		advbutton, 0);
	gtk_box_reorder_child(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		buttonspacer, 1);

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
	        GTK_RESPONSE_ACCEPT, sensitive);

	gtk_window_set_focus_on_map(GTK_WINDOW(dialog), !responder);
	gtk_window_set_role(GTK_WINDOW(dialog), "notify_dialog");

	gtk_container_set_border_width(GTK_CONTAINER(dialog), PIDGIN_HIG_BORDER);
	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
	gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_CAT_SPACE);
	gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_BORDER);

	hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);

	gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);

	label_text = g_strdup_printf(
	       "<span weight=\"bold\" size=\"larger\">%s</span>%s%s",
	       (primary ? primary : ""),
	       (primary ? "\n\n" : ""),
	       (secondary ? secondary : ""));

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), label_text);
	gtk_label_set_selectable(GTK_LABEL(label), 1);
	g_free(label_text);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
       
	/* Create the text view where the user enters their secret */
	entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(entry), _("Enter secret here"));
	gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

	if (context->active_fingerprint->trust &&
		context->active_fingerprint->trust[0]) {
	    label2 = gtk_label_new(_("This buddy is already authenticated."));
	} else {
	    label2 = NULL;
	}

	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	
	if (label2) {
	    gtk_box_pack_start(GTK_BOX(vbox), label2, FALSE, FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(NULL), FALSE,
		    FALSE, 0);
	}

	moremarkup = g_strdup_printf(
		"%s\n\n%s\n\n<a href=\"%s%s\">%s</a>",
		_("To authenticate, pick a secret known "
		    "only to you and your buddy.  Enter this secret, then "
		    "wait for your buddy to enter it too.  If the secrets "
		    "don't match, then you may be talking to an imposter."),
		_("If your buddy uses multiple IM accounts or multiple "
		    "computers, you may have to authenticate multiple "
		    "times.  However, as long as they use an account and "
		    "computer that you've seen before, you don't need to "
		    "authenticate each individual conversation."),
		AUTHENTICATE_HELPURL, _("?lang=en"),
		_("Click here for more information about authentication "
		    "in OTR."));

	add_whatsthis_more(vbox,
		_("Authenticating a buddy helps ensure that the person "
		    "you are talking to is who they claim to be."),
		moremarkup);

	g_free(moremarkup);
	
	gtk_box_pack_start(GTK_BOX(hbox), _pidgin_make_container_scrollable(vbox, dialog), TRUE, TRUE, 0);

	smppair = malloc(sizeof(SmpResponsePair));
	smppair->context = context;
	smppair->entry = GTK_ENTRY(entry);
	smppair->responder = responder;
	g_signal_connect(G_OBJECT(dialog), "response",
			 G_CALLBACK(smp_secret_response_cb),
			 smppair);

	gtk_widget_show_all(dialog);
	smp_data->smp_secret_dialog = dialog;
	smp_data->smp_secret_smppair = smppair;

	if (labelp) *labelp = label;
    } else {
	/* Set the responder field to TRUE if we were passed that value,
	 * even if the window was already up. */
	if (responder) {
	    smp_data->smp_secret_smppair->responder = responder;
	}
    }

    return smp_data->smp_secret_dialog;
}

static GtkWidget *create_smp_progress_dialog(GtkWindow *parent,
	ConnContext *context)
{
    GtkWidget *dialog;
    GtkWidget *hbox;
    GtkWidget *vbox;
    GtkWidget *label;
    GtkWidget *proglabel;
    GtkWidget *bar;
    GtkWidget *img = NULL;
    char *label_text;
    const char *icon_name = NULL;
    PurpleConversation *conv;
    SMPData *smp_data;

    icon_name = PIDGIN_STOCK_DIALOG_INFO;
    img = gtk_image_new_from_stock(icon_name,
	    gtk_icon_size_from_name(PIDGIN_ICON_SIZE_TANGO_HUGE));
    gtk_misc_set_alignment(GTK_MISC(img), 0, 0);

    dialog = gtk_dialog_new_with_buttons(_("Authenticating Buddy"),
	    parent, 0, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	    GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog),
	    GTK_RESPONSE_ACCEPT);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
	    GTK_RESPONSE_REJECT, 1);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
	    GTK_RESPONSE_ACCEPT, 0);

    gtk_window_set_focus_on_map(GTK_WINDOW(dialog), FALSE);
    gtk_window_set_role(GTK_WINDOW(dialog), "notify_dialog");

    gtk_container_set_border_width(GTK_CONTAINER(dialog), PIDGIN_HIG_BORDER);
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_CAT_SPACE);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), PIDGIN_HIG_BORDER);

    hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);

    gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);

    label_text = g_strdup_printf(
	       "<span weight=\"bold\" size=\"larger\">%s %s</span>\n",
	       _("Authenticating"), context->username);

    label = gtk_label_new(NULL);

    gtk_label_set_markup(GTK_LABEL(label), label_text);
    gtk_label_set_selectable(GTK_LABEL(label), 1);
    g_free(label_text);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    proglabel = gtk_label_new(NULL);
    gtk_label_set_selectable(GTK_LABEL(proglabel), 1);
    gtk_label_set_line_wrap(GTK_LABEL(proglabel), TRUE);
    gtk_misc_set_alignment(GTK_MISC(proglabel), 0, 0);
    gtk_box_pack_start(GTK_BOX(vbox), proglabel, FALSE, FALSE, 0);
   
    /* Create the progress bar */
    bar = gtk_progress_bar_new();
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(bar), 0.1);
    gtk_box_pack_start(GTK_BOX(vbox), bar, FALSE, FALSE, 0);
    
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    conv = otrg_plugin_context_to_conv(context, 0);
    smp_data = purple_conversation_get_data(conv, "otr-smpdata");
    if (smp_data) {
	smp_data->smp_progress_dialog = dialog;
	smp_data->smp_progress_bar = bar;
	smp_data->smp_progress_label = proglabel;
    }

    g_signal_connect(G_OBJECT(dialog), "response",
		     G_CALLBACK(smp_progress_response_cb),
		     context);

    gtk_widget_show_all(dialog);

    return dialog;
}

/* This is just like purple_notify_message, except: (a) it doesn't grab
 * keyboard focus, (b) the button is "OK" instead of "Close", and (c)
 * the labels aren't limited to 2K. */
static void otrg_gtk_dialog_notify_message(PurpleNotifyMsgType type,
	const char *accountname, const char *protocol, const char *username,
	const char *title, const char *primary, const char *secondary)
{
    create_dialog(NULL, type, title, primary, secondary, 1, NULL, NULL, NULL);
}

/* Put up a Please Wait dialog, with the "OK" button desensitized.
 * Return a handle that must eventually be passed to
 * otrg_dialog_private_key_wait_done. */
static OtrgDialogWaitHandle otrg_gtk_dialog_private_key_wait_start(
	const char *account, const char *protocol, void *ui_data)
{
    PurplePlugin *p;
    const char *protocol_print;

    p = purple_find_prpl(protocol);
    protocol_print = (p ? p->info->name : _("Unknown"));

    return please_wait_dialog_new(ui_data ? GTK_WINDOW(ui_data) : otrg_ui_ui_data_from_conversation(NULL), account, protocol_print);
}

static int otrg_gtk_dialog_display_otr_message(const char *accountname,
	const char *protocol, const char *username, const char *msg)
{
    /* See if there's a conversation window we can put this in. */
    PurpleAccount *account;
    PurpleConversation *conv;

    account = purple_accounts_find(accountname, protocol);
    if (!account) return -1;

    conv = purple_find_conversation_with_account(PURPLE_CONV_TYPE_IM, username, account);
    if (!conv) return -1;

    purple_conversation_write(conv, NULL, msg, PURPLE_MESSAGE_SYSTEM, time(NULL));

    return 0;
}

/* End a Please Wait dialog. */
static void otrg_gtk_dialog_private_key_wait_done(OtrgDialogWaitHandle handle)
{
    please_wait_dialog_destroy(handle);
}

#if 0
static void add_unk_fingerprint_expander(GtkWidget *vbox, void *data)
{
    char *moremarkup = g_strdup_printf(
	    "%s\n\n%s\n\n<a href=\"%s\">%s%s</a>",
	    __("If your buddy has more than one IM account, or uses more than "
	    "one computer, he may have multiple fingerprints."),
	    __("However, the only way an imposter could duplicate one of your "
	    "buddy's fingerprints is by stealing information from his "
	    "computer."),
	    FINGERPRINT_HELPURL, __("?lang=en"),
	    __("Click here for more information about fingerprints."));

    add_whatsthis_more(vbox,
	    __("A <b>fingerprint</b> is a unique identifier that you should "
	    "use to authenticate your buddy.  Right-click on the OTR button "
	    "in your buddy's conversation window, and choose \"Verify "
	    "fingerprint\"."), moremarkup);

    g_free(moremarkup);
}
#endif

/* Inform the user that an unknown fingerprint was received. */
static void otrg_gtk_dialog_unknown_fingerprint(OtrlUserState us,
	const char *accountname, const char *protocol, const char *who,
	unsigned char fingerprint[20])
{
    PurpleConversation *conv;
    char *buf;
    ConnContext *context;
    int seenbefore = FALSE;

    /* Figure out if this is the first fingerprint we've seen for this
     * user. */
    context = otrl_context_find(us, who, accountname, protocol, FALSE,
	    NULL, NULL, NULL);
    if (context) {
	Fingerprint *fp = context->fingerprint_root.next;
	while(fp) {
	    if (memcmp(fingerprint, fp->fingerprint, 20)) {
		/* This is a previously seen fingerprint for this user,
		 * different from the one we were passed. */
		seenbefore = TRUE;
		break;
	    }
	    fp = fp->next;
	}
    }

    if (seenbefore) {
	buf = g_strdup_printf(_("%s is contacting you from an unrecognized "
		    "computer.  You should <a href=\"%s%s\">authenticate</a> "
		    "this buddy."), who, AUTHENTICATE_HELPURL, _("?lang=en"));
    } else {
	buf = g_strdup_printf(_("%s has not been authenticated yet.  You "
		    "should <a href=\"%s%s\">authenticate</a> this buddy."),
		who, AUTHENTICATE_HELPURL, _("?lang=en"));
    }

    conv = otrg_plugin_userinfo_to_conv(accountname, protocol, who, TRUE);

    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM,
	    time(NULL));
    g_free(buf);
}

static void otrg_gtk_dialog_clicked_connect(GtkWidget *widget, gpointer data);

static void dialog_update_label_conv(PurpleConversation *conv, TrustLevel level)
{
    GtkWidget *label;
    GtkWidget *icon;
    GtkWidget *icontext;
    GtkWidget *button;
    GtkWidget *menuquery;
    GtkWidget *menuend;
    GtkWidget *menuquerylabel;
    GtkWidget *menuview;
    GtkWidget *menuverf;
    GtkWidget *menusmp;
    PidginConversation *gtkconv = PIDGIN_CONVERSATION(conv);
    label = purple_conversation_get_data(conv, "otr-label");
    icon = purple_conversation_get_data(conv, "otr-icon");
    icontext = purple_conversation_get_data(conv, "otr-icontext");
    button = purple_conversation_get_data(conv, "otr-button");
    menuquery = purple_conversation_get_data(conv, "otr-menuquery");
    menuquerylabel = gtk_bin_get_child(GTK_BIN(menuquery));
    menuend = purple_conversation_get_data(conv, "otr-menuend");
    menuview = purple_conversation_get_data(conv, "otr-menuview");
    menuverf = purple_conversation_get_data(conv, "otr-menuverf");
    menusmp = purple_conversation_get_data(conv, "otr-menusmp");

    /* Set the button's icon, label and tooltip. */
    otr_icon(icon, level);
    gtk_label_set_text(GTK_LABEL(label),
	    level == TRUST_FINISHED ? _("Finished") :
	    level == TRUST_PRIVATE ? _("Private") :
	    level == TRUST_UNVERIFIED ? _("Unverified") :
	    _("Not private"));
    gtk_tooltips_set_tip(gtkconv->tooltips, button,
	    level == TRUST_NOT_PRIVATE ? _("Start a private conversation") :
		    _("Refresh the private conversation"), NULL);

    /* Set the menu item label for the OTR Query item. */
    gtk_label_set_markup_with_mnemonic(GTK_LABEL(menuquerylabel),
	    level == TRUST_NOT_PRIVATE ? _("Start _private conversation") :
		    _("Refresh _private conversation"));

    /* Sensitize the menu items as appropriate. */
    gtk_widget_set_sensitive(GTK_WIDGET(menuend), level != TRUST_NOT_PRIVATE);
    gtk_widget_set_sensitive(GTK_WIDGET(menuview), level != TRUST_NOT_PRIVATE);
    gtk_widget_set_sensitive(GTK_WIDGET(menuverf), level != TRUST_NOT_PRIVATE);
    gtk_widget_set_sensitive(GTK_WIDGET(menusmp), level != TRUST_NOT_PRIVATE);

    /* Use any non-NULL value for "private", NULL for "not private" */
    purple_conversation_set_data(conv, "otr-private",
	    level == TRUST_NOT_PRIVATE ? NULL : conv);

    /* Set the appropriate visibility */
    gtk_widget_show_all(button);
}

static void dialog_update_label(ConnContext *context)
{
    PurpleAccount *account;
    PurpleConversation *conv;
    TrustLevel level = otrg_plugin_context_to_trust(context);

    account = purple_accounts_find(context->accountname, context->protocol);
    if (!account) return;
    conv = purple_find_conversation_with_account(PURPLE_CONV_TYPE_IM, context->username, account);
    if (!conv) return;
    dialog_update_label_conv(conv, level);
}

#if 0
/* Add the help text for the "view session id" dialog. */
static void add_sessid_expander(GtkWidget *vbox, void *data)
{
    char *moremarkup = g_strdup_printf(
	    "%s\n\n%s\n\n%s\n\n<a href=\"%s%s\">%s</a>",
	    __("To verify the session id, contact your buddy via some "
	    "<i>other</i> authenticated channel, such as the telephone "
	    "or GPG-signed email.  Each of you should tell your bold "
	    "half of the above session id to the other "
	    "(your buddy will have the same session id as you, but with the "
	    "other half bold)."),
	    __("If everything matches up, then <i>the "
	    "current conversation</i> between your computer and your buddy's "
	    "computer is private."),
	    __("<b>Note:</b> You will probably never have to do this.  You "
	    "should normally use the \"Verify fingerprint\" functionality "
	    "instead."),
	    SESSIONID_HELPURL, _("?lang=en"),
	    __("Click here for more information about the secure session id."));

    add_whatsthis_more(vbox,
	    __("You can use this <b>secure session id</b> to double-check "
	    "the privacy of <i>this one conversation</i>."), moremarkup);

    g_free(moremarkup);
}

static GtkWidget* otrg_gtk_dialog_view_sessionid(ConnContext *context)
{
    GtkWidget *dialog;
    unsigned char *sessionid;
    char sess1[21], sess2[21];
    char *primary = g_strdup_printf(__("Private connection with %s "
	    "established."), context->username);
    char *secondary;
    int i;
    OtrlSessionIdHalf whichhalf = context->sessionid_half;
    size_t idhalflen = (context->sessionid_len) / 2;

    /* Make a human-readable version of the sessionid (in two parts) */
    sessionid = context->sessionid;
    for(i=0;i<idhalflen;++i) sprintf(sess1+(2*i), "%02x", sessionid[i]);
    for(i=0;i<idhalflen;++i) sprintf(sess2+(2*i), "%02x",
	    sessionid[i+idhalflen]);
    
    secondary = g_strdup_printf("%s\n"
	    "<span %s>%s</span> <span %s>%s</span>\n",
	    __("Secure session id:"),
	    whichhalf == OTRL_SESSIONID_FIRST_HALF_BOLD ?
		    "weight=\"bold\"" : "", sess1,
	    whichhalf == OTRL_SESSIONID_SECOND_HALF_BOLD ?
		    "weight=\"bold\"" : "", sess2);

    dialog = create_dialog(PURPLE_NOTIFY_MSG_INFO,
	    __("Private connection established"), primary, secondary, 1, NULL,
	    add_sessid_expander, NULL);

    g_free(primary);
    g_free(secondary);

    return dialog;
}
#endif

struct vrfy_fingerprint_data {
    Fingerprint *fprint;   /* You can use this pointer right away, but
			      you can't rely on it sticking around for a
			      while.  Use the copied pieces below
			      instead. */
    char *accountname, *username, *protocol;
    unsigned char fingerprint[20];
};

static void vrfy_fingerprint_data_free(struct vrfy_fingerprint_data *vfd)
{
    free(vfd->accountname);
    free(vfd->username);
    free(vfd->protocol);
    free(vfd);
}

static struct vrfy_fingerprint_data* vrfy_fingerprint_data_new(
	Fingerprint *fprint)
{
    struct vrfy_fingerprint_data *vfd;
    ConnContext *context = fprint->context;

    vfd = malloc(sizeof(*vfd));
    vfd->fprint = fprint;
    vfd->accountname = strdup(context->accountname);
    vfd->username = strdup(context->username);
    vfd->protocol = strdup(context->protocol);
    memmove(vfd->fingerprint, fprint->fingerprint, 20);

    return vfd;
}

static void vrfy_fingerprint_destroyed(GtkWidget *w,
	struct vrfy_fingerprint_data *vfd)
{
    vrfy_fingerprint_data_free(vfd);
}

static void vrfy_fingerprint_changed(GtkComboBox *combo, void *data)
{
    struct vrfy_fingerprint_data *vfd = data;
    ConnContext *context = otrl_context_find(otrg_plugin_userstate,
	    vfd->username, vfd->accountname, vfd->protocol, 0, NULL,
	    NULL, NULL);
    Fingerprint *fprint;
    int oldtrust, trust;

    if (context == NULL) return;

    fprint = otrl_context_find_fingerprint(context, vfd->fingerprint,
	    0, NULL);

    if (fprint == NULL) return;

    oldtrust = (fprint->trust && fprint->trust[0]);
    trust = gtk_combo_box_get_active(combo) == 1 ? 1 : 0;

    /* See if anything's changed */
    if (trust != oldtrust) {
	otrl_context_set_trust(fprint, trust ? "verified" : "");
	/* Write the new info to disk, redraw the ui, and redraw the
	 * OTR buttons. */
	otrg_plugin_write_fingerprints();
	otrg_ui_update_keylist();
	otrg_dialog_resensitize_all();
    }
}

/* Add the verify widget and the help text for the verify fingerprint box. */
static void add_vrfy_fingerprint(GtkWidget *vbox, void *data)
{
    GtkWidget *hbox;
    GtkWidget *combo, *label;
    struct vrfy_fingerprint_data *vfd = data;
    char *labelt;
    int verified = 0;
    char *moremarkup;

    if (vfd->fprint->trust && vfd->fprint->trust[0]) {
	verified = 1;
    }

    hbox = gtk_hbox_new(FALSE, 0);
    combo = gtk_combo_box_new_text();
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("I have not"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("I have"));
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), verified);
    label = gtk_label_new(_(" verified that this is in fact the correct"));
    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    g_signal_connect(G_OBJECT(combo), "changed",
	    G_CALLBACK(vrfy_fingerprint_changed), vfd);

    hbox = gtk_hbox_new(FALSE, 0);
    labelt = g_strdup_printf(_("fingerprint for %s."),
	    vfd->username);
    label = gtk_label_new(labelt);
    g_free(labelt);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    moremarkup = g_strdup_printf(
	    "%s\n\n%s\n\n%s\n\n%s\n\n<a href=\"%s%s\">%s</a>",
	    _("To verify the fingerprint, contact your buddy via some "
	    "<i>other</i> authenticated channel, such as the telephone "
	    "or GPG-signed email.  Each of you should tell your fingerprint "
	    "to the other."),
	    _("If everything matches up, you should indicate in the above "
	    "dialog that you <b>have</b> verified the fingerprint."),
	    _("If your buddy has more than one IM account, or uses more than "
	    "one computer, he may have multiple fingerprints."),
	    _("However, the only way an imposter could duplicate one of your "
	    "buddy's fingerprints is by stealing information from her/his "
	    "computer."),
	    FINGERPRINT_HELPURL, _("?lang=en"),
	    _("Click here for more information about fingerprints."));

    add_whatsthis_more(vbox,
	    _("A <b>fingerprint</b> is a unique identifier that you should "
	    "use to authenticate your buddy."), moremarkup);
    g_free(moremarkup);

}

/* Replace spaces with non-breaking spaces in the hash */
static GString *make_hash_nbsp(char hash[45]) {
	GString *str = g_string_new("");
	char nbsp[6] = "";
	int idx_start = 0, Nix;

	g_unichar_to_utf8(0xa0, nbsp);

	for (Nix = 0 ; Nix < 44 ; Nix++)
		if (' ' == hash[Nix]) {
			g_string_append_len(str, &hash[idx_start], Nix - idx_start);
			g_string_append(str, nbsp);
			idx_start = Nix + 1;
		}
	g_string_append(str, &hash[idx_start]);

	return str;
}

static void verify_fingerprint(GtkWindow *parent, Fingerprint *fprint)
{
    GtkWidget *dialog;
    char our_hash[45] = "", their_hash[45] = "";
    char *primary;
    char *secondary;
    struct vrfy_fingerprint_data *vfd;
    ConnContext *context;
    PurplePlugin *p;
    char *proto_name;
		GString *our_hash_nbsp, *their_hash_nbsp;

    if (fprint == NULL) return;
    if (fprint->fingerprint == NULL) return;
    context = fprint->context;
    if (context == NULL) return;

    primary = g_strdup_printf(_("Verify fingerprint for %s"),
	    context->username);
    vfd = vrfy_fingerprint_data_new(fprint);

    strcpy(our_hash, _("[none]"));
    otrl_privkey_fingerprint(otrg_plugin_userstate, our_hash,
	    context->accountname, context->protocol);
		our_hash_nbsp = make_hash_nbsp(our_hash);

    otrl_privkey_hash_to_human(their_hash, fprint->fingerprint);
		their_hash_nbsp = make_hash_nbsp(their_hash);

    p = purple_find_prpl(context->protocol);
    proto_name = (p && p->info->name) ? p->info->name : _("Unknown");
    secondary = g_strdup_printf(_("Fingerprint for you, %s (%s):\n%s\n\n"
	    "Purported fingerprint for %s:\n%s\n"), context->accountname,
	    proto_name, our_hash_nbsp->str, context->username, their_hash_nbsp->str);

		g_string_free(our_hash_nbsp, TRUE);
		g_string_free(their_hash_nbsp, TRUE);

    dialog = create_dialog(parent, PURPLE_NOTIFY_MSG_INFO,
	    _("Verify fingerprint"), primary, secondary, 1, NULL,
	    add_vrfy_fingerprint, vfd);
    g_signal_connect(G_OBJECT(dialog), "destroy",
	    G_CALLBACK(vrfy_fingerprint_destroyed), vfd);

    g_free(primary);
    g_free(secondary);
}

static void otrg_gtk_dialog_verify_fingerprint(Fingerprint *fprint, void *ui_data)
{
    verify_fingerprint(ui_data ? GTK_WINDOW(ui_data) : NULL, fprint);
}

/* Create the SMP dialog.  responder is true if this is called in
 * response to someone else's run of SMP. */
static void otrg_gtk_dialog_socialist_millionaires(ConnContext *context,
	gboolean responder)
{
    GtkWidget *dialog;
    char *primary;
    char *secondary;
    PurplePlugin *p;
    char *proto_name;

    if (context == NULL) return;

    primary = g_strdup_printf(_("Authenticate %s"),
	    context->username);

    p = purple_find_prpl(context->protocol);
    proto_name = (p && p->info->name) ? p->info->name : _("Unknown");
    secondary = g_strdup_printf(_("Enter a secret known only to %s and "
		"yourself.\n"), context->username);

    dialog = create_smp_dialog(_("Authenticate buddy"),
	    primary, secondary, 1, NULL, context, responder);

    g_free(primary);
    g_free(secondary);
}

/* Call this to update the status of an ongoing socialist millionaires
 * protocol.  Progress_level is a percentage, from 0.0 (aborted) to
 * 1.0 (complete).  Any other value represents an intermediate state. */
static void otrg_gtk_dialog_update_smp(ConnContext *context,
	double progress_level)
{
    PurpleConversation *conv = otrg_plugin_context_to_conv(context, 0);
    GtkProgressBar *bar;
    SMPData *smp_data = purple_conversation_get_data(conv, "otr-smpdata");

    if (!smp_data) return;

    bar = GTK_PROGRESS_BAR(smp_data->smp_progress_bar);
    gtk_progress_bar_set_fraction(bar, progress_level);

    /* If the counter is reset to absolute zero, the protocol has aborted */
    if (progress_level == 0.0) {
        GtkDialog *dialog = GTK_DIALOG(smp_data->smp_progress_dialog);

	gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, 1);
	gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_REJECT, 0);
	gtk_dialog_set_default_response(GTK_DIALOG(dialog),
		GTK_RESPONSE_ACCEPT);

	gtk_label_set_text(GTK_LABEL(smp_data->smp_progress_label),
		_("An error occurred during authentication."));
	return;
    }

    /* If the counter reaches 1.0, the protocol is complete */
    if (progress_level == 1.0) {
        GtkDialog *dialog = GTK_DIALOG(smp_data->smp_progress_dialog);

	gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, 1);
	gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_REJECT, 0);
	gtk_dialog_set_default_response(GTK_DIALOG(dialog),
		GTK_RESPONSE_ACCEPT);

        if (context->active_fingerprint->trust &&
		context->active_fingerprint->trust[0]) {
	    gtk_label_set_text(GTK_LABEL(smp_data->smp_progress_label),
		    _("Authentication successful."));
        } else {
	    gtk_label_set_text(GTK_LABEL(smp_data->smp_progress_label),
		    _("Authentication failed."));
	}
    }
}

/* Call this when a context transitions to ENCRYPTED. */
static void otrg_gtk_dialog_connected(ConnContext *context)
{
    PurpleConversation *conv;
    char *buf;
    char *format_buf;
    TrustLevel level;
    OtrgUiPrefs prefs;

    conv = otrg_plugin_context_to_conv(context, TRUE);
    level = otrg_plugin_context_to_trust(context);

    otrg_ui_get_prefs(&prefs, purple_conversation_get_account(conv),
	    context->username);
    if (prefs.avoid_logging_otr) {
	purple_conversation_set_logging(conv, FALSE);
    }

    switch(level) {
       case TRUST_PRIVATE:
           format_buf = g_strdup(_("Private conversation with %s started.%s"));
           break;

       case TRUST_UNVERIFIED:
           format_buf = g_strdup_printf(_("<a href=\"%s%s\">Unverified</a> "
                       "conversation with %%s started.%%s"),
                       UNVERIFIED_HELPURL, _("?lang=en"));
           break;

       default:
           /* This last case should never happen, since we know
            * we're in ENCRYPTED. */
           format_buf = g_strdup(_("Not private conversation with %s "
                       "started.%s"));
           break;
    }
    buf = g_strdup_printf(format_buf,
		purple_conversation_get_name(conv),
		context->protocol_version == 1 ? _("  Warning: using old "
		    "protocol version 1.") : "");

    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM,
	time(NULL));
    g_free(buf);
    g_free(format_buf);

    dialog_update_label(context);
}

/* Call this when a context transitions to PLAINTEXT. */
static void otrg_gtk_dialog_disconnected(ConnContext *context)
{
    PurpleConversation *conv;
    char *buf;
    OtrgUiPrefs prefs;

    conv = otrg_plugin_context_to_conv(context, 1);

    buf = g_strdup_printf(_("Private conversation with %s lost."),
	    purple_conversation_get_name(conv));
    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM, time(NULL));
    g_free(buf);

    otrg_ui_get_prefs(&prefs, purple_conversation_get_account(conv),
	    context->username);
    if (prefs.avoid_logging_otr) {
	if (purple_prefs_get_bool("/purple/logging/log_ims"))
	{
	    purple_conversation_set_logging(conv, TRUE);
	}
    }

    dialog_update_label(context);
    close_smp_window(conv);
}

/* Call this if the remote user terminates his end of an ENCRYPTED
 * connection, and lets us know. */
static void otrg_gtk_dialog_finished(const char *accountname,
	const char *protocol, const char *username)
{
    /* See if there's a conversation window we can put this in. */
    PurpleAccount *account;
    PurpleConversation *conv;
    char *buf;

    account = purple_accounts_find(accountname, protocol);
    if (!account) return;

    conv = purple_find_conversation_with_account(PURPLE_CONV_TYPE_IM,
	    username, account);
    if (!conv) return;

    buf = g_strdup_printf(_("%s has ended his/her private conversation with "
		"you; you should do the same."),
	    purple_conversation_get_name(conv));
    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM,
	    time(NULL));
    g_free(buf);

    dialog_update_label_conv(conv, TRUST_FINISHED);
    close_smp_window(conv);
}

/* Call this when we receive a Key Exchange message that doesn't cause
 * our state to change (because it was just the keys we knew already). */
static void otrg_gtk_dialog_stillconnected(ConnContext *context)
{
    PurpleConversation *conv;
    char *buf;
    char *format_buf;
    TrustLevel level;

    conv = otrg_plugin_context_to_conv(context, 1);
    level = otrg_plugin_context_to_trust(context);

    switch(level) {
       case TRUST_PRIVATE:
           format_buf = g_strdup(_("Successfully refreshed the private "
                       "conversation with %s.%s"));
           break;

       case TRUST_UNVERIFIED:
           format_buf = g_strdup_printf(_("Successfully refreshed the "
                       "<a href=\"%s%s\">unverified</a> conversation with "
                       "%%s.%%s"),
                       UNVERIFIED_HELPURL, _("?lang=en"));
           break;

       default:
           /* This last case should never happen, since we know
            * we're in ENCRYPTED. */
           format_buf = g_strdup(_("Successfully refreshed the not private "
                       "conversation with %s.%s"));
           break;
    }

    buf = g_strdup_printf(format_buf,
		purple_conversation_get_name(conv),
		context->protocol_version == 1 ? _("  Warning: using old "
		    "protocol version 1.") : "");

    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM,
	time(NULL));
    g_free(buf);
    g_free(format_buf);

    dialog_update_label(context);
}

/* This is called when the OTR button in the button box is clicked, or
 * when the appropriate context menu item is selected. */
static void otrg_gtk_dialog_clicked_connect(GtkWidget *widget, gpointer data)
{
    const char *format;
    char *buf;
    PurpleConversation *conv = data;

    if (purple_conversation_get_data(conv, "otr-private")) {
	format = _("Attempting to refresh the private conversation with %s...");
    } else {
	format = _("Attempting to start a private conversation with %s...");
    }
    buf = g_strdup_printf(format, purple_conversation_get_name(conv));
    purple_conversation_write(conv, NULL, buf, PURPLE_MESSAGE_SYSTEM,
	    time(NULL));
    g_free(buf);
	
    otrg_plugin_send_default_query_conv(conv);
}

#if 0
static void view_sessionid(GtkWidget *widget, gpointer data)
{
    PurpleConversation *conv = data;
    ConnContext *context = otrg_plugin_conv_to_context(conv);

    if (context == NULL || context->msgstate != OTRL_MSGSTATE_ENCRYPTED)
	return;

    otrg_gtk_dialog_view_sessionid(context);
}
#endif

/* Called when SMP verification option selected from menu */
static void socialist_millionaires(GtkWidget *widget, gpointer data)
{
    PurpleConversation *conv = data;
    ConnContext *context = otrg_plugin_conv_to_context(conv);

    if (context == NULL || context->msgstate != OTRL_MSGSTATE_ENCRYPTED)
	return;

    otrg_gtk_dialog_socialist_millionaires(context, FALSE);
}

#if 0
static void verify_fingerprint(GtkWidget *widget, gpointer data)
{
    PurpleConversation *conv = data;
    ConnContext *context = otrg_plugin_conv_to_context(conv);

    if (context == NULL || context->msgstate != OTRL_MSGSTATE_ENCRYPTED)
	return;

    otrg_gtk_dialog_verify_fingerprint(context->active_fingerprint);
}
#endif

static void menu_whatsthis(GtkWidget *widget, gpointer data)
{
    char *uri = g_strdup_printf("%s%s", BUTTON_HELPURL, _("?lang=en"));
    purple_notify_uri(otrg_plugin_handle, uri);
    g_free(uri);
}

static void menu_end_private_conversation(GtkWidget *widget, gpointer data)
{
    PurpleConversation *conv = data;
    ConnContext *context = otrg_plugin_conv_to_context(conv);

    otrg_ui_disconnect_connection(context);
}

static void dialog_resensitize(PurpleConversation *conv);

/* If the action button is clicked, show the context menu */

static void
place_action_menu(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, GtkWidget *action_button)
{
    GtkWidget *parent = NULL;

    (*push_in) = TRUE;

    (*x) = (*y) = 0;

    if (action_button->window)
	gdk_window_get_root_origin(action_button->window, x, y);

    if ((parent = gtk_widget_get_parent(action_button)) != NULL)
	if (parent->window == action_button->window) {
	    (*x) += action_button->allocation.x;
	    (*y) += action_button->allocation.y;
	}
}

static void
action_button_clicked(GtkWidget *action_button, GtkMenu *menu)
{
    gtk_menu_popup(menu, NULL, NULL, (GtkMenuPositionFunc)place_action_menu,
	action_button, 1, gtk_get_current_event_time());
}

/* If the conversation gets destroyed on us, clean up the data we stored
 * pointing to it. */
static void conversation_destroyed(PurpleConversation *conv, void *data)
{
    GtkWidget *menu = purple_conversation_get_data(conv, "otr-menu");
    if (menu) gtk_object_destroy(GTK_OBJECT(menu));
    g_hash_table_remove(conv->data, "otr-label");
    g_hash_table_remove(conv->data, "otr-button");
    g_hash_table_remove(conv->data, "otr-widget");
    g_hash_table_remove(conv->data, "otr-icon");
    g_hash_table_remove(conv->data, "otr-icontext");
    g_hash_table_remove(conv->data, "otr-private");
    g_hash_table_remove(conv->data, "otr-menu");
    g_hash_table_remove(conv->data, "otr-menuquery");
    g_hash_table_remove(conv->data, "otr-menuend");
    g_hash_table_remove(conv->data, "otr-menuview");
    g_hash_table_remove(conv->data, "otr-menuverf");
    g_hash_table_remove(conv->data, "otr-menusmp");
    otrg_gtk_dialog_free_smp_data(conv);
}

/* Set up the per-conversation information display */
static void otrg_gtk_dialog_new_conv(PurpleConversation *conv)
{
    PidginConversation *gtkconv = PIDGIN_CONVERSATION(conv);
    ConnContext *context;
    GtkWidget *widget;
    GtkWidget *bbox;
    GtkWidget *vbox;
    GtkWidget *button;
    GtkWidget *action_button;
    GtkWidget *label;
    GtkWidget *bvbox;
    GtkWidget *icon;
    GtkWidget *menu;
    GtkWidget *menuquery;
    GtkWidget *menuend;
    GtkWidget *menusep;
    GtkWidget *alignment;
    /*
    GtkWidget *menuview;
    GtkWidget *menuverf;
    */
    GtkWidget *menusmp;
    GtkWidget *whatsthis;

    /* Do nothing if this isn't an IM conversation */
    if (purple_conversation_get_type(conv) != PURPLE_CONV_TYPE_IM) return;
    if (!gtkconv) return;
    bbox = gtkconv->lower_hbox;

    context = otrg_plugin_conv_to_context(conv);

    /* See if we're already set up */
    widget = purple_conversation_get_data(conv, "otr-widget");
    if (widget) {
	/* Check if we've been removed from the conversation bbox; purple does this
	 * when the user changes her prefs for the style of buttons to
	 * display. */
	GList *children = gtk_container_get_children(GTK_CONTAINER(bbox));
	if (!g_list_find(children, widget)) {
	    gtk_box_pack_start(GTK_BOX(bbox), widget, FALSE, FALSE, 0);
	}
	g_list_free(children);
	dialog_update_label_conv(conv, otrg_plugin_context_to_trust(context));
	return;
    }

    widget = gtk_frame_new(_("OTR:"));
    gtk_box_pack_start(GTK_BOX(bbox), widget, FALSE, TRUE, 0);

    vbox = gtk_vbox_new(FALSE, PIDGIN_HIG_BOX_SPACE);
    gtk_container_add(GTK_CONTAINER(widget), vbox);

    /* Make the button */
    button = gtk_button_new();
    gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);

    alignment = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(button), alignment);

    bvbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(alignment), bvbox);

    icon = otr_icon(NULL, TRUST_NOT_PRIVATE);
    gtk_box_pack_start(GTK_BOX(bvbox), icon, FALSE, TRUE, 0);

    label = gtk_label_new(NULL);
    gtk_box_pack_start(GTK_BOX(bvbox), label, FALSE, TRUE, 0);

    action_button = gtk_button_new_with_label(_("Actions..."));
    gtk_box_pack_start(GTK_BOX(vbox), action_button, FALSE, TRUE, 0);

    /* Make the context menu */
    menu = gtk_menu_new();
    gtk_menu_set_title(GTK_MENU(menu), _("OTR Messaging"));

    g_signal_connect(G_OBJECT(action_button), "clicked",
    	(GCallback)action_button_clicked, menu);

    menuquery = gtk_menu_item_new_with_mnemonic("");
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuquery);
    gtk_widget_show(menuquery);

    menuend = gtk_menu_item_new_with_mnemonic(_("_End private conversation"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuend);
    gtk_widget_show(menuend);

    menusep = gtk_separator_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menusep);
    gtk_widget_show(menusep);

    /*
     * Don't show the Verify fingerprint menu option any more.  You can
     * still get to the dialog through Authenticate connection ->
     * Advanced...
     *
    menuverf = gtk_menu_item_new_with_mnemonic(_("_Verify fingerprint"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuverf);
    gtk_widget_show(menuverf);
    */

    menusmp = gtk_menu_item_new_with_mnemonic(_("_Authenticate buddy"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menusmp);
    gtk_widget_show(menusmp);

    /*
     * Don't show the View secure session id menu option any more.  It's
     * not really useful at all.
     *
    menuview = gtk_menu_item_new_with_mnemonic(_("View _secure session id"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuview);
    gtk_widget_show(menuview);
    */

    menusep = gtk_separator_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menusep);
    gtk_widget_show(menusep);

    whatsthis = gtk_menu_item_new_with_mnemonic(_("_What's this?"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), whatsthis);
    gtk_widget_show(whatsthis);

    purple_conversation_set_data(conv, "otr-widget", widget);
    purple_conversation_set_data(conv, "otr-label", label);
    purple_conversation_set_data(conv, "otr-button", button);
    purple_conversation_set_data(conv, "otr-icon", icon);
    purple_conversation_set_data(conv, "otr-menu", menu);
    purple_conversation_set_data(conv, "otr-menuquery", menuquery);
    purple_conversation_set_data(conv, "otr-menuend", menuend);
    /*
    purple_conversation_set_data(conv, "otr-menuview", menuview);
    purple_conversation_set_data(conv, "otr-menuverf", menuverf);
    */
    purple_conversation_set_data(conv, "otr-menusmp", menusmp);
    gtk_signal_connect(GTK_OBJECT(menuquery), "activate",
	    GTK_SIGNAL_FUNC(otrg_gtk_dialog_clicked_connect), conv);
    gtk_signal_connect(GTK_OBJECT(menuend), "activate",
	    GTK_SIGNAL_FUNC(menu_end_private_conversation), conv);
    /*
    gtk_signal_connect(GTK_OBJECT(menuverf), "activate",
	    GTK_SIGNAL_FUNC(verify_fingerprint), conv);
    */
    gtk_signal_connect(GTK_OBJECT(menusmp), "activate",
	    GTK_SIGNAL_FUNC(socialist_millionaires), conv);
    /*
    gtk_signal_connect(GTK_OBJECT(menuview), "activate",
	    GTK_SIGNAL_FUNC(view_sessionid), conv);
    */
    gtk_signal_connect(GTK_OBJECT(whatsthis), "activate",
	    GTK_SIGNAL_FUNC(menu_whatsthis), conv);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
	    GTK_SIGNAL_FUNC(otrg_gtk_dialog_clicked_connect), conv);
    dialog_update_label_conv(conv, otrg_plugin_context_to_trust(context));
    dialog_resensitize(conv);

    /* Finally, add the state for the socialist millionaires dialogs */
    otrg_gtk_dialog_add_smp_data(conv);

    gtk_widget_show_all(widget);
}

/* Remove the per-conversation information display */
static void otrg_gtk_dialog_remove_conv(PurpleConversation *conv)
{
    GtkWidget *widget;

    /* Do nothing if this isn't an IM conversation */
    if (purple_conversation_get_type(conv) != PURPLE_CONV_TYPE_IM) return;

    widget = purple_conversation_get_data(conv, "otr-widget");
    if (widget) gtk_object_destroy(GTK_OBJECT(widget));
    conversation_destroyed(conv, NULL);
}

/* Set the OTR button to "sensitive" or "insensitive" as appropriate. */
static void dialog_resensitize(PurpleConversation *conv)
{
    PurpleAccount *account;
    PurpleConnection *connection;
    GtkWidget *widget;
    const char *name;
    OtrgUiPrefs prefs;

    /* Do nothing if this isn't an IM conversation */
    if (purple_conversation_get_type(conv) != PURPLE_CONV_TYPE_IM) return;

    account = purple_conversation_get_account(conv);
    name = purple_conversation_get_name(conv);
    otrg_ui_get_prefs(&prefs, account, name);

    if (prefs.policy == OTRL_POLICY_NEVER) {
	otrg_gtk_dialog_remove_conv(conv);
    } else {
	otrg_gtk_dialog_new_conv(conv);
    }
    widget = purple_conversation_get_data(conv, "otr-widget");
    if (!widget) return;
    if (account) {
	connection = purple_account_get_connection(account);
	if (connection) {
	    /* Set the button to "sensitive" */
	    gtk_widget_set_sensitive(widget, TRUE);
	    return;
	}
    }
    /* Set the button to "insensitive" */
    gtk_widget_set_sensitive(widget, FALSE);
}

/* Set all OTR buttons to "sensitive" or "insensitive" as appropriate.
 * Call this when accounts are logged in or out. */
static void otrg_gtk_dialog_resensitize_all(void)
{
    purple_conversation_foreach(dialog_resensitize);
}

/* Initialize the OTR dialog subsystem */
static void otrg_gtk_dialog_init(void)
{
    purple_signal_connect(purple_conversations_get_handle(),
	    "deleting-conversation", otrg_plugin_handle,
	    PURPLE_CALLBACK(conversation_destroyed), NULL);
}

/* Deinitialize the OTR dialog subsystem */
static void otrg_gtk_dialog_cleanup(void)
{
    purple_signal_disconnect(purple_conversations_get_handle(),
	    "deleting-conversation", otrg_plugin_handle,
	    PURPLE_CALLBACK(conversation_destroyed));
}

static const OtrgDialogUiOps gtk_dialog_ui_ops = {
    otrg_gtk_dialog_init,
    otrg_gtk_dialog_cleanup,
    otrg_gtk_dialog_notify_message,
    otrg_gtk_dialog_display_otr_message,
    otrg_gtk_dialog_private_key_wait_start,
    otrg_gtk_dialog_private_key_wait_done,
    otrg_gtk_dialog_unknown_fingerprint,
    otrg_gtk_dialog_verify_fingerprint,
    otrg_gtk_dialog_socialist_millionaires,
    otrg_gtk_dialog_update_smp,
    otrg_gtk_dialog_connected,
    otrg_gtk_dialog_disconnected,
    otrg_gtk_dialog_stillconnected,
    otrg_gtk_dialog_finished,
    otrg_gtk_dialog_resensitize_all,
    otrg_gtk_dialog_new_conv,
    otrg_gtk_dialog_remove_conv
};

/* Get the GTK dialog UI ops */
const OtrgDialogUiOps *otrg_gtk_dialog_get_ui_ops(void)
{
    return &gtk_dialog_ui_ops;
}
