/*
 *  Copyright (c) 2008 Andrew Olmsted <andrew.olmsted@gmail.com>
 *
 *  Parts of this file are based on Command Navigator which is
 *  Copyright (c) 2007-2008 Jiri Benc <jbenc@upir.cz>
 *  Parts of this file are based on hello-world-app
 *  Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib/gmem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkmain.h>
#include <dbus/dbus.h>
#include <libgnomevfs/gnome-vfs.h>

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <X11/Xatom.h>

#include <libosso.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-helper.h>
#include <libhildondesktop/libhildondesktop.h>
#include <libhildondesktop/hildon-desktop-toggle-button.h>

#include "personal-menu.h"

#define WORKAREA_ATOM "_NET_WORKAREA"
#define HOME_DIR g_get_home_dir()

static void create_menu (PersonalMenuNavigatorPlugin *plugin);

HD_DEFINE_PLUGIN (PersonalMenuNavigatorPlugin, personal_menu_navigator_plugin, TASKNAVIGATOR_TYPE_ITEM);

GtkWidget *personal_menu_button_new(int padding)
{
  GtkIconTheme *icon_theme;
  GdkPixbuf *icon;
  GtkWidget *icon_image, *button;

  icon_theme = gtk_icon_theme_get_default ();
  icon = gtk_icon_theme_load_icon (icon_theme,"personal-menu",64,0,NULL);
  if (icon == NULL)
    icon = gtk_icon_theme_load_icon (icon_theme,"qgn_list_gene_default_app",64,0,NULL);
    
  icon_image = gtk_image_new_from_pixbuf (icon);
  gtk_misc_set_padding (GTK_MISC (icon_image), padding, padding);
  g_object_unref (G_OBJECT (icon));
  button = hildon_desktop_toggle_button_new ();
  gtk_container_add (GTK_CONTAINER (button), icon_image);

  gtk_widget_show_all (button);

  return button;
}

static void execute_command(GtkMenuItem *item, gint index)
{
	pid_t pid;
	gchar *appName;
	gchar *iconName;
	gchar *executable;
	gchar *service;
	gboolean termFlag,rootFlag;
	gint argc;
	gchar** argv;
	gchar entry[5];
	gchar* filename;
	gboolean fileExists;
	GKeyFile* keyFile;
	GError *error=NULL;
	gboolean result;

	if (index>=0)
	{
		snprintf(entry,5,"%d",index);
		keyFile=g_key_file_new();
		filename=g_strdup(HOME_DIR);
		filename=g_strconcat(filename,"/.personal_menu.rc",NULL);
		fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
		if (fileExists)
		{
			appName=g_key_file_get_string(keyFile,entry,"app name",NULL);
			iconName=g_key_file_get_string(keyFile,entry,"icon name",NULL);
			executable=g_key_file_get_string(keyFile,entry,"executable",NULL);
			termFlag=g_key_file_get_boolean(keyFile,entry,"run in term",NULL);
			rootFlag=g_key_file_get_boolean(keyFile,entry,"run as root",NULL);
			service=g_key_file_get_string(keyFile,entry,"service",NULL);
		}
		g_key_file_free(keyFile);
	}
	else
	{
		appName="Edit Menu";
		iconName="personal-menu";
		
		osso_context_t *osso=osso_initialize("personal-menu-launch","0.5",TRUE,NULL);
		osso_return_t retVal;

		retVal=osso_rpc_async_run(osso,"com.nokia.controlpanel","/com/nokia/controlpanel/rpc","com.nokia.controlpanel","run_applet",NULL,NULL,DBUS_TYPE_STRING,"libpersonalmenusetup.so",DBUS_TYPE_BOOLEAN,TRUE,DBUS_TYPE_INVALID);
		if (retVal==OSSO_OK)
		{
			char buf[64];
			g_snprintf(buf, 64, "%s started",appName);
			hildon_banner_show_information(NULL, iconName, buf);
		}
		else
		{
			hildon_banner_show_information(NULL, "qgn_note_gene_syserror", "Execution failed");
		}
		osso_deinitialize(osso);
		return;
	}

	if (rootFlag)
	{
		executable=g_strconcat("sudo ",executable,NULL);
	}

	if (termFlag)
	{
		osso_context_t *osso=osso_initialize("personal-menu-launch","0.5",TRUE,NULL);
		osso_return_t retVal;
	
		executable = g_strconcat("sh -c \"", g_strescape(executable, NULL), " ; read x\"", NULL);

		retVal=osso_rpc_async_run(osso,"com.nokia.xterm","/com/nokia/xterm","com.nokia.xterm","run_command",NULL,NULL,DBUS_TYPE_STRING,executable,DBUS_TYPE_INVALID);
		if (retVal==OSSO_OK)
		{
			char buf[64];
			g_snprintf(buf, 64, "%s started",appName);
			hildon_banner_show_information(NULL, iconName, buf);
		}
		else
		{
			hildon_banner_show_information(NULL, "qgn_note_gene_syserror", "Execution failed");
		}
		osso_deinitialize(osso);
	}
	else 
	{
		osso_context_t *osso=osso_initialize("personal-menu-launch","0.5",TRUE,NULL);
		if ((g_strcasecmp(service,"")==0) || !(result = osso_rpc_run_with_defaults(osso, service, "top_application", NULL, DBUS_TYPE_INVALID) == OSSO_OK))
		{
			if (g_shell_parse_argv(executable,&argc,&argv,NULL))
				g_spawn_async(NULL,argv,NULL,(GSpawnFlags)G_SPAWN_SEARCH_PATH,NULL,NULL,&pid,&error);
			if (!error)
			{
				char buf[64];
				g_snprintf(buf, 64, "%s started",appName);
				hildon_banner_show_information(NULL, iconName, buf);
				/*free(buf);*/
			}
			if (error)
			{
				hildon_banner_show_information(NULL, "qgn_note_gene_syserror", "Execution failed");
				g_error_free(error);
			}
		}
		osso_deinitialize(osso);
	}
}

static void get_workarea(GtkAllocation *allocation)
{
	unsigned long n;
	unsigned long extra;
	int format;
	int status;
	Atom property = XInternAtom (GDK_DISPLAY(),WORKAREA_ATOM,FALSE);
	Atom realType;

	union
	{
		unsigned char *char_value;
		int *int_value;
	} value;

	status=XGetWindowProperty(GDK_DISPLAY(),GDK_ROOT_WINDOW(),property,0L,4L,0,XA_CARDINAL,&realType,&format,&n,&extra,(unsigned char**) &value.char_value);

	if (status==Success && realType==XA_CARDINAL && format==32 && n==4 &&value.char_value!=NULL)
	{
		allocation->x=value.int_value[0];
		allocation->y=value.int_value[1];
		allocation->width=value.int_value[2];
		allocation->height=value.int_value[3];
	}
	else
	{
		allocation->x=0;
		allocation->y=0;
		allocation->width=0;
		allocation->height=0;
	}
	if (value.char_value)
	{
		XFree(value.char_value);
	}
}

static void avail_space(gint *width, gint *height, PersonalMenuNavigatorPlugin *plugin)
{
	GtkRequisition req;
	GdkScreen *screen=gtk_widget_get_screen(GTK_WIDGET(plugin->menu));
	int menu_height=0;
	int main_height=0;
	GtkAllocation workarea={0,0,800,480};
	GtkWidget *top_level;
	HildonDesktopPanelWindowOrientation orientation=HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_LEFT;
	
	/*get_workarea(&workarea);*/

	top_level=gtk_widget_get_ancestor(plugin->button,HILDON_DESKTOP_TYPE_WINDOW);

	if (HILDON_DESKTOP_IS_PANEL_WINDOW(top_level));
	{
		g_object_get(top_level,"orientation",&orientation,NULL);
	}
	gtk_widget_size_request(GTK_WIDGET(plugin->menu),&req);

	menu_height=req.height;
	main_height=gdk_screen_get_height(screen);

	switch (orientation)
	{
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_LEFT:
			*width=MIN(workarea.width-80,req.width);
			*height=MIN(req.height,menu_height);
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_RIGHT:
			*width=MIN(workarea.width-80,req.width);
			*height=MIN(req.height,menu_height);
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_TOP:
			*width=MIN(req.width,workarea.width);
			*height=MIN(req.height,workarea.height-80);
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_BOTTOM:
			*width=MIN(req.width,workarea.width);
			*height=MIN(req.height,workarea.height-80);
			break;
		default:
			g_assert_not_reached();
	}
}

static void menu_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, PersonalMenuNavigatorPlugin *plugin)
{
	GtkRequisition req;
	GdkScreen *screen=gtk_widget_get_screen(GTK_WIDGET(menu));
	int menu_height=0;
	int main_height=0;
	GtkAllocation workarea={0,0,0,0};
	GtkWidget *top_level;
	HildonDesktopPanelWindowOrientation orientation=HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_LEFT;
	
	get_workarea(&workarea);

	*push_in=FALSE;
	top_level=gtk_widget_get_toplevel(plugin->button);

	if (HILDON_DESKTOP_IS_PANEL_WINDOW(top_level));
	{
		g_object_get(top_level,"orientation",&orientation,NULL);
	}
	gtk_widget_size_request(GTK_WIDGET(menu),&req);

	menu_height=req.height;
	main_height=gdk_screen_get_height(screen);

	switch (orientation)
	{
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_LEFT:
			*x=workarea.x;
			if (main_height-plugin->button->allocation.y<menu_height)
				*y=MAX(0,((main_height-menu_height)/2));
			else
				*y=plugin->button->allocation.y;
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_RIGHT:
			*x=workarea.x+workarea.width-req.width;
			if (main_height-plugin->button->allocation.y<menu_height)
				*y=MAX(0,((main_height-menu_height)/2));
			else
				*y=plugin->button->allocation.y;
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_TOP:
			*x=plugin->button->allocation.x;
			*y=workarea.y;
			break;
		case HILDON_DESKTOP_PANEL_WINDOW_ORIENTATION_BOTTOM:
			*x=plugin->button->allocation.x;
			*y=workarea.y+workarea.height-req.height;
			break;
		default:
			g_assert_not_reached();
	}
}

static void menu_off(GtkMenu *menu, PersonalMenuNavigatorPlugin *plugin)
{
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plugin->button), FALSE);
	gtk_menu_popdown(GTK_MENU(plugin->menu));
	plugin->popped=FALSE;
}

static void
popup_menu (GtkWidget *button, PersonalMenuNavigatorPlugin *plugin)
{
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(plugin->button)))
	{
		gtk_menu_popup (GTK_MENU (plugin->menu),NULL,NULL,(GtkMenuPositionFunc)menu_position,plugin,0,gtk_get_current_event_time());
		plugin->popped=TRUE;
	}
}

static void header_select(GtkItem *item, gpointer user_data)
{
	gtk_item_deselect(item);
}

static gboolean header_activate(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	return TRUE;
}

static void create_menu (PersonalMenuNavigatorPlugin *plugin)
{
    GtkWidget  *menu_item;
    GdkPixbuf  *icon;
    GtkIconTheme *icon_theme;
    GtkWidget   *icon_image;
    gchar* filename;
    gboolean fileExists;
    GKeyFile* keyFile;
    GtkAllocation workarea={0,0,0,0};

    if (GTK_IS_MENU(plugin->menu))
	gtk_widget_destroy(plugin->menu);
    plugin->menu = gtk_menu_new ();

    keyFile=g_key_file_new();
    filename=g_strdup(HOME_DIR);
    filename=g_strconcat(filename,"/.personal_menu.rc",NULL);
    fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
    if (fileExists)
    {
      int i=0;
      int x=0;
      int y=0;
      int iconSize=40;
      int minRows=8;
      int numCols=4;
      int entryPerColumn;
      gchar *colourString;
      gchar *fontString;
      gboolean systemBool;
      gchar *headerColourString;
      gchar *headerFontString;
      gboolean headerSystemBool;
      gboolean layoutBool;
      gsize numEntry;
      GError *error=NULL;
      gchar **retVal=g_key_file_get_groups(keyFile,&numEntry);
      char entry[5];
      gboolean found;
      g_strfreev(retVal);

      get_workarea(&workarea);
      
	numEntry--;
	iconSize=g_key_file_get_integer(keyFile,"config","icon size",&error);
	if (error)
	{
		iconSize=40;
		g_error_free(error);
		error=NULL;
	}
	colourString=g_key_file_get_string(keyFile,"config","font colour",&error);
	if (error)
	{
		colourString="000000000000";
		g_error_free(error);
		error=NULL;
	}
	headerColourString=g_key_file_get_string(keyFile,"config","header font colour",&error);
	if (error)
	{
		headerColourString="000000000000";
		g_error_free(error);
		error=NULL;
	}
	fontString=g_key_file_get_string(keyFile,"config","font style",&error);
	if (error)
	{
		fontString="Sans 14";
		g_error_free(error);
		error=NULL;
	}
	headerFontString=g_key_file_get_string(keyFile,"config","header font style",&error);
	if (error)
	{
		headerFontString="Sans 14";
		g_error_free(error);
		error=NULL;
	}
	systemBool=g_key_file_get_boolean(keyFile,"config","system style",&error);
	if (error)
	{
		systemBool=TRUE;
		g_error_free(error);
		error=NULL;
	}
	headerSystemBool=g_key_file_get_boolean(keyFile,"config","header system style",&error);
	if (error)
	{
		headerSystemBool=TRUE;
		g_error_free(error);
		error=NULL;
	}
	layoutBool=g_key_file_get_boolean(keyFile,"config","auto layout",&error);
	if (error)
	{
		layoutBool=TRUE;
		g_error_free(error);
		error=NULL;
	}
	if (!layoutBool)
	{
		minRows=g_key_file_get_integer(keyFile,"config","num rows",&error);
		if (error)
		{
			minRows=8;
			g_error_free(error);
			error=NULL;
		}
		numCols=g_key_file_get_integer(keyFile,"config","num cols",&error);
		if (error)
		{
			numCols=4;
			g_error_free(error);
			error=NULL;
		}
	}
	else
	{
		if (iconSize!=0)
			minRows=(workarea.height-64)/(iconSize+2);
		else
		{
			GtkStyle *style=gtk_rc_get_style(GTK_WIDGET(plugin));
			PangoFontDescription *font=style->font_desc;
			gint fontSize=pango_font_description_get_size(font);
			if (!pango_font_description_get_size_is_absolute(font))
				fontSize=fontSize/(PANGO_SCALE);
			minRows=(workarea.height-64)/(fontSize+10);
		}
	}
	entryPerColumn=minRows;

	if (numEntry>(minRows)*numCols)
	{
		entryPerColumn=numEntry/numCols;
		if (numEntry%numCols>0)
			entryPerColumn++;
	}
      snprintf(entry,5,"%d",i);
      found=g_key_file_has_group(keyFile,entry);
      while (found)
      {
	gchar* appName;
	gchar* iconName;
	gchar* executable=NULL;

	sprintf(entry,"%d",i);
	appName=g_key_file_get_string(keyFile,entry,"app name",NULL);
	iconName=g_key_file_get_string(keyFile,entry,"icon name",NULL);
	executable=g_key_file_get_string(keyFile,entry,"executable",NULL);

	if (executable!=NULL && g_strcasecmp(executable,"")!=0)
	{
		menu_item=gtk_image_menu_item_new_with_label(appName);
		if (iconSize!=0)
		{
			icon_theme = gtk_icon_theme_get_default ();
			icon=gtk_icon_theme_load_icon(icon_theme,iconName,iconSize,0,NULL);
			if (icon==NULL)
			{
				icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",iconSize,0,NULL);
			}
			icon=gdk_pixbuf_scale_simple(icon,iconSize,iconSize,GDK_INTERP_NEAREST);
			icon_image=gtk_image_new_from_pixbuf(icon);
			g_object_unref(G_OBJECT(icon));
			gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item),icon_image);
		}
		if (!systemBool)
			gtk_label_set_markup(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))),g_strconcat("<span font_desc='",fontString,"' foreground='#",colourString,"'>",gtk_label_get_label(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item)))),"</span>",NULL));
		g_signal_connect(G_OBJECT(menu_item),"activate",G_CALLBACK(execute_command),i);
	}
	else
	{
		menu_item=gtk_menu_item_new_with_label(appName);
		gtk_label_set_use_markup(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))),TRUE);
		if (headerSystemBool)
			gtk_label_set_markup(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))),g_strconcat("<span weight='bold'>",g_markup_escape_text(appName,-1),"</span>",NULL));
		else
			gtk_label_set_markup(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))),g_strconcat("<span font_desc='",headerFontString,"' foreground='#",headerColourString,"'>",g_markup_escape_text(appName,-1),"</span>",NULL));
		g_signal_connect(G_OBJECT(menu_item),"select",G_CALLBACK(header_select),NULL);
		g_signal_connect(G_OBJECT(menu_item),"button-release-event",G_CALLBACK(header_activate),NULL);
	}


	gtk_menu_attach(GTK_MENU(plugin->menu),menu_item,x,x+1,y,y+1);
	y++;
	if (y>entryPerColumn-1)
	{
		y=0;
		x++;
	}
	snprintf(entry,5,"%d",++i);
	found=g_key_file_has_group(keyFile,entry);
	g_free(appName);
	g_free(iconName);
	g_free(executable);
      }
    }
	else
	{
		menu_item=gtk_image_menu_item_new_with_label("Create your personal menu!");
		icon_theme = gtk_icon_theme_get_default ();
		icon=gtk_icon_theme_load_icon(icon_theme,"personal-menu",40,0,NULL);
		if (icon==NULL)
		{
			icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",40,0,NULL);
		}
		icon_image=gtk_image_new_from_pixbuf(icon);
		g_object_unref(G_OBJECT(icon));
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item),icon_image);
		g_signal_connect(G_OBJECT(menu_item),"activate",G_CALLBACK(execute_command),-1);
		gtk_menu_append(GTK_MENU_SHELL(plugin->menu),menu_item);
	}
    g_key_file_free(keyFile);

    /* Name the menu to get the appropriate theming */
    gtk_widget_set_name (plugin->menu, "menu_from_navigator");

    gtk_widget_show_all (plugin->menu);
    g_signal_connect(G_OBJECT(plugin->menu), "hide", G_CALLBACK(menu_off), plugin);
    g_free(filename);

    gint width,height;
    gtk_widget_set_size_request(plugin->menu,-1,-1);
    avail_space(&width,&height,plugin);
    gtk_widget_set_size_request(plugin->menu,width,height);
}

gint dbus_req_handler(const gchar* interface, const gchar* method, GArray* arguments, PersonalMenuNavigatorPlugin *plugin, osso_rpc_t* retval)
{
  osso_rpc_t val;
  val=g_array_index(arguments,osso_rpc_t,0);
  if (DBUS_TYPE_STRING==val.type)
    gtk_button_clicked(GTK_BUTTON(plugin->button));

  return OSSO_OK;
}

static gboolean menu_changed(PersonalMenuNavigatorPlugin *plugin)
{
	plugin->updateTimeout=0;
	if (plugin->popped)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plugin->button),FALSE);
		create_menu(plugin);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plugin->button),TRUE);
	}
	else
	{
		create_menu(plugin);
	}
	return FALSE;
}

static void
file_changed (GnomeVFSMonitorHandle *handle,
                                  const gchar *monitor_uri,
                                  const gchar *info_uri,
                                  GnomeVFSMonitorEventType event_type,
                                  PersonalMenuNavigatorPlugin *plugin)
{
	if (!plugin->updateTimeout)
		plugin->updateTimeout=g_timeout_add(500,(GSourceFunc)menu_changed,plugin);
}

static gboolean key_release(GtkWidget *widget,GdkEventKey *event,PersonalMenuNavigatorPlugin *plugin)
{
    switch (event->keyval) 
    {
		case GDK_Escape:
			menu_off(GTK_MENU(plugin->menu),plugin);
			return TRUE;
    }
    return FALSE;
}

static void
personal_menu_navigator_plugin_init (PersonalMenuNavigatorPlugin *navigator_plugin)
{
  GtkWidget *button;
  osso_return_t result;
  gchar *conf_file;

  button = personal_menu_button_new (10);
  navigator_plugin->button=button;
  
  g_signal_connect(G_OBJECT(button),"toggled",G_CALLBACK(popup_menu),navigator_plugin);
  
  gtk_widget_set_size_request (button, 80, 80);
  gtk_widget_set_name (button, "hildon-navigator-button-one");

  gtk_widget_show_all (button);

  gtk_container_add (GTK_CONTAINER (navigator_plugin), button);
  gtk_widget_show_all (GTK_WIDGET(navigator_plugin));

  navigator_plugin->menu=gtk_menu_new();
  create_menu(navigator_plugin);

  navigator_plugin->context=osso_initialize("personalmenu","0.5",TRUE,NULL);
  result=osso_rpc_set_cb_f(navigator_plugin->context,"com.nokia.personalmenu","/com/nokia/personalmenu","com.nokia.personalmenu",dbus_req_handler,navigator_plugin);

  conf_file = g_build_filename (g_get_home_dir (), ".personal_menu.rc", NULL);
  gnome_vfs_init();
  gnome_vfs_monitor_add (&navigator_plugin->fileMon, 
                             conf_file,
                             GNOME_VFS_MONITOR_FILE,
                             (GnomeVFSMonitorCallback) file_changed,
                             navigator_plugin);
  navigator_plugin->updateTimeout=0;
  navigator_plugin->popped=FALSE;
  g_signal_connect(G_OBJECT(navigator_plugin->menu),"key-release-event",G_CALLBACK(key_release),navigator_plugin);
}

static void
personal_menu_navigator_plugin_finalize (GObject *gobject)
{
	PersonalMenuNavigatorPlugin *navigator_plugin=PERSONAL_MENU_NAVIGATOR_PLUGIN(gobject);
	if (navigator_plugin->fileMon)
		gnome_vfs_monitor_cancel(navigator_plugin->fileMon);
	if (navigator_plugin->updateTimeout)
		g_source_remove(navigator_plugin->updateTimeout);
	/*G_OBJECT_CLASS (personal_menu_navigator_plugin_class)->finalize (gobject);*/
}

static void
personal_menu_navigator_plugin_class_init (PersonalMenuNavigatorPluginClass *class)
{
}
