/*
 *  Copyright (c) 2008 Andrew Olmsted <andrew.olmsted@gmail.com>
 *
 *  Parts of this file are based on Command Navigator which is
 *  Copyright (c) 2007-2008 Jiri Benc <jbenc@upir.cz>
 *  Parts of this file are based on hello-world-app
 *  Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libosso.h>
#include <dirent.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>

#define HOME_DIR g_get_home_dir()

/* Common struct types declarations */
typedef struct _PersonalMenuSetup PersonalMenuSetup;

/* Instance struct */
struct _PersonalMenuSetup
{
	GtkTreeView *view;
	GtkWindow *setupwin;
	GtkListStore *entryList;
	GtkTreeIter entryIter;
	GtkWidget *iconSizeCombo;
	GtkWidget *fontButton;
	gchar *fontText;
	GtkWidget *systemCheck;
	GtkWidget *colourButton;
	GtkWidget *layoutCheck;
	GtkWidget *numRows;
	GtkWidget *numCols;
	gchar *context;
	GtkWidget *iconPreview;
};

void select_icon(GtkButton *button, PersonalMenuSetup *setup);

static GtkTreeView *setup_make_view(PersonalMenuSetup *setup)
{
	GtkTreeView *res;
	gchar* filename;
	gboolean fileExists;
	GKeyFile* keyFile;

	setup->entryList=gtk_list_store_new(7,GDK_TYPE_PIXBUF,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING);
	/*setup->entryList=gtk_list_store_new(4,GDK_TYPE_PIXBUF,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING);*/

	keyFile=g_key_file_new();
	filename=g_strdup(HOME_DIR);
	filename=g_strconcat(filename,"/.personal_menu.rc",NULL);
	fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
	if (fileExists)
	{
	  GtkIconTheme *icon_theme;
	  GdkPixbuf *icon;
	  int i=0;
	  char entry[5];
	  gboolean found;
	  snprintf(entry,5,"%d",i);
	  found=g_key_file_has_group(keyFile,entry);
	  while (found)
	  {
	    gchar* appName;
	    gchar* iconName;
	    gchar* executable;
	    gchar* service;
	    gchar* runAsRoot;
	    gchar* runInTerm;
	    sprintf(entry,"%d",i);
	    appName=g_key_file_get_string(keyFile,entry,"app name",NULL);
	    iconName=g_key_file_get_string(keyFile,entry,"icon name",NULL);
	    executable=g_key_file_get_string(keyFile,entry,"executable",NULL);
	    runAsRoot=g_key_file_get_string(keyFile,entry,"run as root",NULL);
	    if (runAsRoot==NULL)
		runAsRoot="false";
	    runInTerm=g_key_file_get_string(keyFile,entry,"run in term",NULL);
	    if (runInTerm==NULL)
		runInTerm="false";
	    service=g_key_file_get_string(keyFile,entry,"service",NULL);

	    icon_theme = gtk_icon_theme_get_default ();
	    icon=gtk_icon_theme_load_icon(icon_theme,iconName,40,0,NULL);
	    if (icon==NULL)
	    {
	      icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",40,0,NULL);
	    }
	    gtk_list_store_append(setup->entryList, &setup->entryIter);
	    gtk_list_store_set(setup->entryList, &setup->entryIter, 0, icon,1,appName,2,executable,3,iconName,4,runAsRoot,5,runInTerm,6,service,-1);
	    /*gtk_list_store_set(setup->entryList, &setup->entryIter, 0, icon,1,appName,2,executable,3,iconName,-1);*/
	    g_object_unref(G_OBJECT(icon));
	    snprintf(entry,5,"%d",++i);
	    found=g_key_file_has_group(keyFile,entry);
	  }
	}
	g_key_file_free(keyFile);
    

	res = GTK_TREE_VIEW(gtk_tree_view_new_with_model(GTK_TREE_MODEL(setup->entryList)));
	gtk_tree_view_append_column(res,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_pixbuf_new(),"pixbuf",0,NULL));
	gtk_tree_view_append_column(res,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_text_new(),"text",1,NULL));
	gtk_tree_view_append_column(res,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_text_new(),"text",2,NULL));
	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(res),GTK_SELECTION_BROWSE);

	return res;
}

void select_desktop_file(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkDialog *dialog;
	DIR *d;
	struct dirent *dir;
	GtkListStore *store;
	GtkTreeIter iter;
	GtkTreeView *res;
	GtkWidget *scroll;
	gboolean fileExists;
	GKeyFile* keyFile;
	int result;
	GtkTreeModel* sort;
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;

	dialog = GTK_DIALOG(gtk_dialog_new());
	if (!dialog)
		return;
	gtk_window_set_title(GTK_WINDOW(dialog), "Select Desktop File");
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	button = gtk_button_new_with_label("OK");
	gtk_dialog_add_action_widget(dialog, button, GTK_RESPONSE_ACCEPT);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_dialog_add_action_widget(dialog, gtk_button_new_with_label("Cancel"), GTK_RESPONSE_REJECT);

	store=gtk_list_store_new(5,GDK_TYPE_PIXBUF,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING);
	d=opendir("/usr/share/applications/hildon/");
	if (d)
	{
		while ((dir=readdir(d)) != NULL)
		{
			if ((g_strrstr(dir->d_name,".desktop"))||(g_strrstr(dir->d_name,".DESKTOP")))
			{
				gchar *group;
				gchar *appName,*iconName,*executable,*service;
				GtkIconTheme *icon_theme;
				GdkPixbuf *icon;
				gchar *filename;
				filename=g_strconcat("/usr/share/applications/hildon/",dir->d_name,NULL);
				keyFile=g_key_file_new();
				fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
				group=g_key_file_get_start_group(keyFile);
				appName=g_key_file_get_string(keyFile,group,"Name",NULL);
				iconName=g_key_file_get_string(keyFile,group,"Icon",NULL);
				executable=g_key_file_get_string(keyFile,group,"Exec",NULL);
				service=g_key_file_get_string(keyFile,group,"X-Osso-Service",NULL);
				icon_theme = gtk_icon_theme_get_default ();
				icon=gtk_icon_theme_load_icon(icon_theme,iconName,40,0,NULL);
				if (icon==NULL)
				{
				  icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",40,0,NULL);
				}
				gtk_list_store_append(store, &iter);
				gtk_list_store_set(store, &iter,0,icon,1,gettext(appName),2,executable,3,iconName,4,service,-1);
				g_object_unref(G_OBJECT(icon));
			}
		}
	}
	g_key_file_free(keyFile);
    
	res = gtk_tree_view_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(res),col);
	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(col,renderer,TRUE);
	gtk_tree_view_column_add_attribute(col,renderer, "pixbuf", 0);

	col = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(res), col);
	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col,renderer,TRUE);
	gtk_tree_view_column_add_attribute(col,renderer, "text", 1);

	col = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(res), col);
	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col,renderer,TRUE);
	gtk_tree_view_column_add_attribute(col,renderer, "text", 2);

	sort=gtk_tree_model_sort_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_sortable_set_sort_column_id(sort,1,GTK_SORT_ASCENDING);
	gtk_tree_view_set_model(res,sort);

	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(res),GTK_SELECTION_BROWSE);

	scroll = gtk_scrolled_window_new(NULL, GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 25, 0, 0, 0)));
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_widget_set_size_request(GTK_WIDGET(scroll), 400, 400);
	gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(res));

	gtk_container_add(GTK_CONTAINER(dialog->vbox), scroll);
	gtk_widget_show_all(GTK_WIDGET(dialog));
	result = gtk_dialog_run(dialog);
	if (result == GTK_RESPONSE_ACCEPT) {
		/*get the selected row and add the pertinant data to the entryList*/
		gchar *appName,*iconName,*executable,*service;
		GdkPixbuf *icon;
		GtkTreeSelection *select=gtk_tree_view_get_selection(res);
		GtkTreeModel *model=gtk_tree_view_get_model(res);
		gtk_tree_selection_get_selected(select,&model,&iter);
		gtk_tree_model_get(model,&iter,0,&icon,1,&appName,2,&executable,3,&iconName,4,&service,-1);
		gtk_list_store_append(setup->entryList, &setup->entryIter);
		gtk_list_store_set(setup->entryList,&setup->entryIter,0,icon,1,appName,2,executable,3,iconName,4,"false",5,"false",6,service,-1);
		g_object_unref(G_OBJECT(icon));
	}
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

void change_context(GtkComboBox *combo, PersonalMenuSetup *setup)
{
	GtkWidget *dialog;
	GtkButton *button;
	button=gtk_button_new();
	setup->context=gtk_combo_box_get_active_text(combo);
	dialog=gtk_widget_get_toplevel(GTK_WIDGET(combo));
	gtk_widget_destroy(GTK_WIDGET(dialog));
	select_icon(button,setup);
}

void select_icon(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkDialog *dialog;
	GtkIconTheme *iconTheme;
	GList *iconList;
	GtkListStore *store;
	GtkTreeIter iter;
	GtkTreeView *res;
	GtkWidget *scroll;
	GtkWidget *typeCombo;
	GtkWidget *caption;
	GtkWidget *vbox;
	int i,result;
	GtkTreeModel* sort;
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;

	dialog = GTK_DIALOG(gtk_dialog_new());
	if (!dialog)
		return;
	gtk_window_set_title(GTK_WINDOW(dialog), "Select Icon");
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	button = gtk_button_new_with_label("OK");
	gtk_dialog_add_action_widget(dialog, button, GTK_RESPONSE_ACCEPT);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_dialog_add_action_widget(dialog, gtk_button_new_with_label("Cancel"), GTK_RESPONSE_REJECT);

	vbox=gtk_vbox_new(FALSE,0);
	typeCombo=gtk_combo_box_new_text();
	gtk_combo_box_append_text(GTK_COMBO_BOX(typeCombo),"Applications");
	if (g_strcasecmp(setup->context,"Applications")==0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(typeCombo),0);
	gtk_combo_box_append_text(GTK_COMBO_BOX(typeCombo),"Devices");
	if (g_strcasecmp(setup->context,"Devices")==0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(typeCombo),1);
	gtk_combo_box_append_text(GTK_COMBO_BOX(typeCombo),"FileSystems");
	if (g_strcasecmp(setup->context,"FileSystems")==0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(typeCombo),2);
	gtk_combo_box_append_text(GTK_COMBO_BOX(typeCombo),"MimeTypes");
	if (g_strcasecmp(setup->context,"MimeTypes")==0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(typeCombo),3);
	gtk_combo_box_append_text(GTK_COMBO_BOX(typeCombo),"hildon");
	if (g_strcasecmp(setup->context,"hildon")==0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(typeCombo),4);
	g_signal_connect(G_OBJECT(typeCombo),"changed",G_CALLBACK(change_context),setup);

	caption=hildon_caption_new(NULL,"Type of icon",typeCombo,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);

	store=gtk_list_store_new(2,GDK_TYPE_PIXBUF,G_TYPE_STRING);

	iconTheme = gtk_icon_theme_get_default ();
	iconList=gtk_icon_theme_list_icons(iconTheme,setup->context);

	for (i=0;i<g_list_length(iconList);i++)
	{
		gchar* element;
		GdkPixbuf *icon;
		element=g_list_nth_data(iconList,i);
		icon=gtk_icon_theme_load_icon(iconTheme,element,40,0,NULL);
		icon=gdk_pixbuf_scale_simple(icon,40,40,GDK_INTERP_NEAREST);
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,0,icon,1,gettext(element),-1);
		g_free(element);
		g_object_unref(G_OBJECT(icon));
	}
	g_list_free(iconList);
    
	res = gtk_tree_view_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(res),col);
	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(col,renderer,TRUE);
	gtk_tree_view_column_add_attribute(col,renderer, "pixbuf", 0);

	col = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(res), col);
	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col,renderer,TRUE);
	gtk_tree_view_column_add_attribute(col,renderer, "text", 1);

	sort=gtk_tree_model_sort_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_sortable_set_sort_column_id(sort,1,GTK_SORT_ASCENDING);
	gtk_tree_view_set_model(res,sort);
	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(res),GTK_SELECTION_BROWSE);
	scroll = gtk_scrolled_window_new(NULL, GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 25, 0, 0, 0)));
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_widget_set_size_request(GTK_WIDGET(scroll), 400, 300);
	gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(res));

	gtk_box_pack_start(GTK_BOX(vbox),scroll,FALSE,FALSE,0);
	gtk_container_add(GTK_CONTAINER(dialog->vbox), vbox);
	gtk_widget_show_all(GTK_WIDGET(dialog));
	gtk_widget_grab_focus(res);
	result = gtk_dialog_run(dialog);
	if (result == GTK_RESPONSE_ACCEPT) {
		/*get the selected row and add the pertinant data to the entryList*/
		gchar *iconName;
		GtkTreeSelection *select=gtk_tree_view_get_selection(res);
		GtkTreeModel *model=gtk_tree_view_get_model(res);
		gtk_tree_selection_get_selected(select,&model,&iter);
		gtk_tree_model_get(model,&iter,1,&iconName,-1);
		gtk_image_set_from_icon_name(setup->iconPreview,iconName,GTK_ICON_SIZE_SMALL_TOOLBAR);
	}
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void dialog_add_edit(GtkTreeIter iter, gboolean edit, PersonalMenuSetup *setup)
{
	GtkDialog *dialog;
	GtkWidget *button, *vbox;
	GtkWidget *hbox;
	GtkWidget *caption;
	GtkWidget *addByDesktopFile;
	GtkWidget *appNameEntry, *executableEntry, *iconButton;
	GtkWidget *termCheckBox, *rootCheckBox;
	GtkWidget *scroll;
	int res;

	dialog = GTK_DIALOG(gtk_dialog_new());
	if (!dialog)
		return;
	gtk_window_set_transient_for(GTK_WINDOW(dialog), setup->setupwin);
	gtk_window_set_title(GTK_WINDOW(dialog), edit ? "Edit command" : "Add command");
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	button = gtk_button_new_with_label("OK");
	gtk_dialog_add_action_widget(dialog, button, GTK_RESPONSE_ACCEPT);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_dialog_add_action_widget(dialog, gtk_button_new_with_label("Cancel"), GTK_RESPONSE_REJECT);

	vbox=gtk_vbox_new(FALSE,0);
	scroll=gtk_scrolled_window_new(NULL,NULL);
	
	if (!edit)
	{
		addByDesktopFile=gtk_button_new_with_label("Select from installed applications");
		gtk_misc_set_padding(GTK_MISC(GTK_BIN(addByDesktopFile)->child),0,10);
		gtk_box_pack_start(GTK_BOX(vbox),addByDesktopFile,FALSE,FALSE,0);
		g_signal_connect(G_OBJECT(addByDesktopFile),"clicked",G_CALLBACK(select_desktop_file),setup);
	}

	appNameEntry=gtk_entry_new();
	caption=hildon_caption_new(NULL,"Name of the menu entry",appNameEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);

	executableEntry=gtk_entry_new();
	caption=hildon_caption_new(NULL,"Command to execute",executableEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);

	setup->context="hildon";
	hbox=gtk_hbox_new(FALSE,0);
	iconButton=gtk_button_new_with_label("Select Icon");
	setup->iconPreview=gtk_image_new_from_icon_name("qgn_list_gene_default_app",GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_box_pack_start(GTK_BOX(hbox),setup->iconPreview,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(hbox),iconButton,FALSE,FALSE,0);
	caption=hildon_caption_new(NULL,"Icon to display",hbox,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
	g_signal_connect(G_OBJECT(iconButton),"clicked",G_CALLBACK(select_icon),setup);

	termCheckBox=gtk_check_button_new_with_label("Run in terminal?");
	gtk_box_pack_start(GTK_BOX(vbox),termCheckBox,FALSE,FALSE,0);

	rootCheckBox=gtk_check_button_new_with_label("Run as root?");
	gtk_box_pack_start(GTK_BOX(vbox),rootCheckBox,FALSE,FALSE,0);

	if (edit)
	{
		gchar *appName,*iconName,*executable;
		gchar *runAsRoot,*runInTerm;
		gtk_tree_model_get(gtk_tree_view_get_model(setup->view),&iter,1,&appName,2,&executable,3,&iconName,4,&runAsRoot,5,&runInTerm,-1);
		/*gtk_tree_model_get(gtk_tree_view_get_model(setup->view),&iter,1,&appName,2,&executable,3,&iconName,-1);*/
		gtk_entry_set_text(GTK_ENTRY(appNameEntry),appName);
		gtk_image_set_from_icon_name(GTK_IMAGE(setup->iconPreview),iconName,GTK_ICON_SIZE_SMALL_TOOLBAR);
		gtk_entry_set_text(GTK_ENTRY(executableEntry),executable);
		if (g_strcasecmp(runAsRoot,"true")==0)
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rootCheckBox),TRUE);
		if (g_strcasecmp(runInTerm,"true")==0)
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(termCheckBox),TRUE);
	}

	gtk_scrolled_window_set_policy(scroll,GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(scroll,vbox);
	gtk_container_add(GTK_CONTAINER(dialog->vbox),scroll);
	gtk_widget_set_size_request(GTK_WIDGET(scroll), -1, 250);

	gtk_widget_show_all(GTK_WIDGET(dialog));
	res = gtk_dialog_run(dialog);
	if (res == GTK_RESPONSE_ACCEPT) 
	{
		gchar *appName,*iconName,*executable;
		gchar *runAsRoot="false", *runInTerm="false";
		GtkIconTheme *icon_theme;
		GdkPixbuf *icon;
		appName=gtk_entry_get_text(GTK_ENTRY(appNameEntry));
		gtk_image_get_icon_name(setup->iconPreview,&iconName,NULL);
		executable=gtk_entry_get_text(GTK_ENTRY(executableEntry));
		if (gtk_toggle_button_get_active(termCheckBox))
			runInTerm="true";
		if (gtk_toggle_button_get_active(rootCheckBox))
			runAsRoot="true";

		if ((g_strcasecmp(appName,"")!=0)||(g_strcasecmp(executable,"")!=0))
		{
			icon_theme=gtk_icon_theme_get_default();
			icon=gtk_icon_theme_load_icon(icon_theme,iconName,40,0,NULL);
			if (icon==NULL)
			{
				icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",40,0,NULL);
			}
			if (!edit)
			{
				gtk_list_store_append(setup->entryList,&setup->entryIter);
				gtk_list_store_set(setup->entryList,&setup->entryIter,0,icon,1,appName,2,executable,3,iconName,4,runAsRoot,5,runInTerm,-1);
				/*gtk_list_store_set(setup->entryList,&setup->entryIter,0,icon,1,appName,2,executable,3,iconName,-1);*/
			}
			else
			{
				gtk_list_store_set(setup->entryList,&iter,0,icon,1,appName,2,executable,3,iconName,4,runAsRoot,5,runInTerm,-1);
				/*gtk_list_store_set(setup->entryList,&iter,0,icon,1,appName,2,executable,3,iconName,-1);*/
			}
			g_object_unref(G_OBJECT(icon));
		}
	}

	gtk_widget_destroy(GTK_WIDGET(dialog));
	gtk_widget_grab_focus(GTK_WIDGET(setup->view));
}

void set_font_button_text(PersonalMenuSetup *setup, gchar *text)
{
	if (strlen(text)>15)
	{
		char result[16]="";
		char ell[3]="...";
		strncat(result,text,12);
		strncat(result,ell,3);
		gtk_button_set_label(setup->fontButton,result);
	}
	else
	{
		gtk_button_set_label(setup->fontButton,text);
	}
	setup->fontText=text;
}

void dialog_font_select(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkWidget *fsd;
	gint res;
	fsd=gtk_font_selection_dialog_new("Select Font");
	gtk_font_selection_dialog_set_font_name(fsd,setup->fontText);
	gtk_widget_show_all(fsd);
	res=gtk_dialog_run(GTK_DIALOG(fsd));
	if (res==GTK_RESPONSE_OK)
	{
		set_font_button_text(setup,gtk_font_selection_dialog_get_font_name(fsd));
	}
	gtk_widget_destroy(fsd);
}

void add_cmd(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkTreeIter iter;
	dialog_add_edit(iter,FALSE,setup);
}

void edit_cmd(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->view);
	GtkTreeIter iter;
	gtk_tree_selection_get_selected(select,&model,&iter);
	dialog_add_edit(iter,TRUE,setup);
}

void delete_cmd(GtkButton *button, PersonalMenuSetup *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->view);
	GtkTreeIter iter;
	gtk_tree_selection_get_selected(select,&model,&iter);
	gtk_list_store_remove(setup->entryList,&iter);
}

void move_up(GtkButton *button, PersonalMenuSetup *setup)
{
	gint pos;
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->view);
	GtkTreeIter iterFirst,iterSecond;
	GtkTreePath *path;
	gtk_tree_selection_get_selected(select,&model,&iterFirst);
	path=gtk_tree_model_get_path(model,&iterFirst);
	gtk_tree_path_prev(path);
	gtk_tree_model_get_iter(model,&iterSecond,path);
	gtk_list_store_swap(setup->entryList,&iterFirst,&iterSecond);
	gtk_tree_view_scroll_to_cell(setup->view,path,NULL,FALSE,0,0);
}

void move_down(GtkButton *button, PersonalMenuSetup *setup)
{
	gint pos;
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->view);
	GtkTreeIter iterFirst,iterSecond;
	GtkTreePath *path;
	gtk_tree_selection_get_selected(select,&model,&iterFirst);
	iterSecond=iterFirst;
	if (gtk_tree_model_iter_next(model,&iterSecond))
	{
		gtk_list_store_swap(setup->entryList,&iterFirst,&iterSecond);
		path=gtk_tree_model_get_path(model,&iterFirst);
		gtk_tree_view_scroll_to_cell(setup->view,path,NULL,FALSE,0,0);
	}
}

void write_key_file(PersonalMenuSetup *setup)
{
	gchar* filename;
	GKeyFile* keyFile;
	gchar* fileData;
	FILE* rcFile;
	gsize size;
	GtkTreeIter iter;
	GtkTreeModel *model=gtk_tree_view_get_model(setup->view);
	gboolean exists;
	gint i=0;
	GdkColor colour={0,0,0,0};
	gchar *colourText=NULL;

	keyFile=g_key_file_new();
	exists=gtk_tree_model_get_iter_first(model,&iter);
	while (exists)
	{
		gchar *appName,*iconName,*executable=NULL,*service;
		gchar *runAsRoot,*runInTerm;
		gchar entry[5];
		snprintf(entry,5,"%d",i++);
		gtk_tree_model_get(model,&iter,1,&appName,2,&executable,3,&iconName,4,&runAsRoot,5,&runInTerm,6,&service,-1);
		/*gtk_tree_model_get(model,&iter,1,&appName,2,&executable,3,&iconName,-1);*/
		g_key_file_set_string(keyFile,entry,"app name",appName);
		g_key_file_set_string(keyFile,entry,"icon name",iconName);
		if (executable!=NULL)
			g_key_file_set_string(keyFile,entry,"executable",executable);
		g_key_file_set_string(keyFile,entry,"run as root",runAsRoot);
		g_key_file_set_string(keyFile,entry,"run in term",runInTerm);
		if (service!=NULL)
			g_key_file_set_string(keyFile,entry,"service",service);
		exists=gtk_tree_model_iter_next(model,&iter);
	}

	g_key_file_set_string(keyFile,"config","icon size",gtk_combo_box_get_active_text(setup->iconSizeCombo));
	gtk_color_button_get_color(setup->colourButton,&colour);
	colourText=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
	g_key_file_set_string(keyFile,"config","font style",setup->fontText);
	g_key_file_set_string(keyFile,"config","font colour",colourText);
	g_key_file_set_boolean(keyFile,"config","system style",gtk_toggle_button_get_active(setup->systemCheck));
	g_key_file_set_boolean(keyFile,"config","auto layout",gtk_toggle_button_get_active(setup->layoutCheck));
	g_key_file_set_integer(keyFile,"config","num cols",hildon_number_editor_get_value(setup->numCols));
	g_key_file_set_integer(keyFile,"config","num rows",hildon_number_editor_get_value(setup->numRows));
	g_free(colourText);

	filename=g_strdup(HOME_DIR);
	filename=g_strconcat(filename,"/.personal_menu.rc",NULL);
	fileData=g_key_file_to_data(keyFile,&size,NULL);
	rcFile=fopen(filename,"w");
	fputs(fileData,rcFile);
	fclose(rcFile);
	g_key_file_free(keyFile);
	g_free(fileData);
}

static void system_check_toggle(GtkToggleButton *toggleButton, PersonalMenuSetup *setup)
{
	if (gtk_toggle_button_get_active(toggleButton))
	{
		gtk_widget_hide(gtk_widget_get_parent(setup->fontButton));
		gtk_widget_hide(gtk_widget_get_parent(setup->colourButton));
	}
	else
	{
		gtk_widget_show(gtk_widget_get_parent(setup->fontButton));
		gtk_widget_show(gtk_widget_get_parent(setup->colourButton));
	}
}

static void layout_check_toggle(GtkToggleButton *toggleButton, PersonalMenuSetup *setup)
{
	if (gtk_toggle_button_get_active(toggleButton))
	{
		gtk_widget_hide(gtk_widget_get_parent(setup->numCols));
		gtk_widget_hide(gtk_widget_get_parent(setup->numRows));
	}
	else
	{
		gtk_widget_show(gtk_widget_get_parent(setup->numCols));
		gtk_widget_show(gtk_widget_get_parent(setup->numRows));
	}
}

static void run_setup(GtkWindow *parent)
{
	GtkDialog *dialog;
	GtkWidget *button, *scroll, *hbox, *vbox;
	GtkWidget *vbox2, *hbox2, *caption;
	GtkWidget *scroll2;
	int res;
	gchar* filename;
	gboolean fileExists;
	GKeyFile* keyFile;
	PersonalMenuSetup *setup=g_new0(PersonalMenuSetup,1);

	dialog = GTK_DIALOG(gtk_dialog_new());
	if (!dialog)
		return;
	gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);
	gtk_window_set_title(GTK_WINDOW(dialog), "Change Menu Items");
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	
	setup->setupwin = GTK_WINDOW(dialog);
	setup->view = setup_make_view(setup);

	gtk_dialog_add_action_widget(dialog, gtk_button_new_with_label("OK"), GTK_RESPONSE_ACCEPT);
	gtk_dialog_add_action_widget(dialog, gtk_button_new_with_label("Cancel"), GTK_RESPONSE_REJECT);

	scroll = gtk_scrolled_window_new(NULL, GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 25, 0, 0, 0)));
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_widget_set_size_request(GTK_WIDGET(scroll), 300, -1);
	gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(setup->view));

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), scroll, TRUE, TRUE, 0);
	hbox2=gtk_hbox_new(TRUE,0);
	vbox2=gtk_vbox_new(FALSE,0);
	scroll2=gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll2),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll2),vbox2);
	gtk_box_pack_start(GTK_BOX(hbox), scroll2, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Edit");
	gtk_box_pack_start(GTK_BOX(hbox2),button,TRUE,TRUE,0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(edit_cmd), setup);
	button = gtk_button_new_with_label("Add");
	gtk_box_pack_start(GTK_BOX(hbox2),button,TRUE,TRUE,0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(add_cmd), setup);
	button = gtk_button_new_with_label("Delete");
	gtk_box_pack_start(GTK_BOX(hbox2),button,TRUE,TRUE,0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(delete_cmd), setup);
	gtk_box_pack_start(GTK_BOX(vbox2),hbox2,TRUE,TRUE,0);
	hbox2=gtk_hbox_new(FALSE,0);
	button = gtk_button_new();
	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_icon_name("qgn_indi_arrow_up", GTK_ICON_SIZE_BUTTON));
	gtk_box_pack_start(GTK_BOX(hbox2),button,TRUE,TRUE,0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(move_up), setup);
	button = gtk_button_new();
	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_icon_name("qgn_indi_arrow_down", GTK_ICON_SIZE_BUTTON));
	gtk_box_pack_start(GTK_BOX(hbox2),button,TRUE,TRUE,0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(move_down), setup);
	gtk_box_pack_start(GTK_BOX(vbox2),hbox2,TRUE,TRUE,0);

	setup->iconSizeCombo=gtk_combo_box_new_text();
	gtk_combo_box_append_text(setup->iconSizeCombo,"0");
	gtk_combo_box_append_text(setup->iconSizeCombo,"26");
	gtk_combo_box_append_text(setup->iconSizeCombo,"40");
	gtk_combo_box_append_text(setup->iconSizeCombo,"48");
	gtk_combo_box_append_text(setup->iconSizeCombo,"56");
	gtk_combo_box_append_text(setup->iconSizeCombo,"64");
	caption=hildon_caption_new(NULL,"Select icon size",setup->iconSizeCombo,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox2),caption,FALSE,FALSE,0);

	setup->systemCheck=gtk_check_button_new_with_label("Use system font style");
	gtk_box_pack_start(GTK_BOX(vbox2),setup->systemCheck,FALSE,FALSE,0);
	g_signal_connect(G_OBJECT(setup->systemCheck),"toggled",G_CALLBACK(system_check_toggle),setup);

	hbox2=gtk_hbox_new(FALSE,0);	
	setup->fontButton=gtk_button_new_with_label("Select");
		gtk_label_set_max_width_chars(GTK_LABEL(gtk_bin_get_child(setup->fontButton)),15);
	g_signal_connect(G_OBJECT(setup->fontButton),"clicked",G_CALLBACK(dialog_font_select),setup);
	caption=hildon_caption_new(NULL,"Font Style",setup->fontButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox2),caption,FALSE,FALSE,0);
	
	setup->colourButton=gtk_color_button_new();
	caption=hildon_caption_new(NULL,"Font Colour",setup->colourButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox2),caption,FALSE,FALSE,0);
	
	setup->layoutCheck=gtk_check_button_new_with_label("Use automatic layout");
	gtk_box_pack_start(GTK_BOX(vbox2),setup->layoutCheck,FALSE,FALSE,0);
	g_signal_connect(G_OBJECT(setup->layoutCheck),"toggled",G_CALLBACK(layout_check_toggle),setup);
	
	setup->numCols=hildon_number_editor_new(1,4);
	caption=hildon_caption_new(NULL,"Number of cols",setup->numCols,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox2),caption,FALSE,FALSE,0);
	
	setup->numRows=hildon_number_editor_new(1,30);
	caption=hildon_caption_new(NULL,"Number of rows",setup->numRows,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox2),caption,FALSE,FALSE,0);

	keyFile=g_key_file_new();
	filename=g_strdup(HOME_DIR);
	filename=g_strconcat(filename,"/.personal_menu.rc",NULL);
	fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
	if (fileExists)
	{
		GError *error=NULL;
		GdkColor colour={0,0,0,0};
		gint iconSize=g_key_file_get_integer(keyFile,"config","icon size",&error);
		if (error)
		{
			iconSize=40;
			g_error_free(error);
			error=NULL;
		}
		gchar *colourString=g_key_file_get_string(keyFile,"config","font colour",&error);
		if (error)
		{
			colourString="000000000000";
			g_error_free(error);
			error=NULL;
		}
		gchar *fontString=g_key_file_get_string(keyFile,"config","font style",&error);
		if (error)
		{
			fontString="Sans 14";
			g_error_free(error);
			error=NULL;
		}
		gboolean systemBool=g_key_file_get_boolean(keyFile,"config","system style",&error);
		if (error)
		{
			systemBool=TRUE;
			g_error_free(error);
			error=NULL;
		}
		gboolean layoutBool=g_key_file_get_boolean(keyFile,"config","auto layout",&error);
		if (error)
		{
			layoutBool=TRUE;
			g_error_free(error);
			error=NULL;
		}
		gint numberOfCols=g_key_file_get_integer(keyFile,"config","num cols",&error);
		if (error)
		{
			numberOfCols=4;
			g_error_free(error);
			error=NULL;
		}
		gint numberOfRows=g_key_file_get_integer(keyFile,"config","num rows",&error);
		if (error)
		{
			numberOfRows=6;
			g_error_free(error);
			error=NULL;
		}
		if (iconSize==0)
			gtk_combo_box_set_active(setup->iconSizeCombo,0);
		else if (iconSize==26)
			gtk_combo_box_set_active(setup->iconSizeCombo,1);
		else if (iconSize==48)
			gtk_combo_box_set_active(setup->iconSizeCombo,3);
		else if (iconSize==56)
			gtk_combo_box_set_active(setup->iconSizeCombo,4);
		else if (iconSize==64)
			gtk_combo_box_set_active(setup->iconSizeCombo,5);
		else
			gtk_combo_box_set_active(setup->iconSizeCombo,2);
			
		char s[5]="";
		guint16 x;
		strncat(s,colourString,strlen(colourString)/3);
		x=strtol(s,NULL,16);
		colour.red=x;
		s[0]=NULL;
		strncat(s,colourString+strlen(colourString)/3,strlen(colourString)/3);
		x=strtol(s,NULL,16);
		colour.green=x;
		s[0]=NULL;
		strncat(s,colourString+(strlen(colourString)/3)*2,strlen(colourString)/3);
		x=strtol(s,NULL,16);
		colour.blue=x;
		
		//gtk_button_set_label(setup->fontButton,fontString);
		setup->fontText=fontString;
		set_font_button_text(setup,fontString);
		gtk_color_button_set_color(setup->colourButton,&colour);
		gtk_toggle_button_set_active(setup->systemCheck,systemBool);
		gtk_toggle_button_set_active(setup->layoutCheck,layoutBool);
		hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(setup->numCols),numberOfCols);
		hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(setup->numRows),numberOfRows);
	}
	else
	{
		GdkColor colour={0,0,0,0};
		gtk_combo_box_set_active(setup->iconSizeCombo,2);
		//gtk_button_set_label(setup->fontButton,"Sans 14");
		setup->fontText="Sans 14";
		set_font_button_text(setup,"Sans 14");
		gtk_color_button_set_color(setup->colourButton,&colour);
		gtk_toggle_button_set_active(setup->systemCheck,TRUE);
		gtk_toggle_button_set_active(setup->layoutCheck,TRUE);
	}
	g_key_file_free(keyFile);
	gtk_container_add(GTK_CONTAINER(dialog->vbox), hbox);

	gtk_widget_set_size_request(dialog, -1, 400);
	gtk_widget_show_all(GTK_WIDGET(dialog));
	system_check_toggle(setup->systemCheck,setup);
	layout_check_toggle(setup->layoutCheck,setup);
	res = gtk_dialog_run(dialog);
	if (res == GTK_RESPONSE_ACCEPT) {
		write_key_file(setup);
	}

	gtk_widget_destroy(GTK_WIDGET(dialog));
}

osso_return_t execute(osso_context_t * osso, gpointer data)
{
	setlocale(LC_ALL,"");
	bindtextdomain("maemo-af-desktop","/usr/share/locale");
	textdomain("maemo-af-desktop");
	(void)osso;
	run_setup(GTK_WINDOW(data));
	return 0;
}
