/*
  PackageView - Show debian package details
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "PackageList.h"

#include <sstream>

// apt-pkg
#include <apt-pkg/error.h>
#include <apt-pkg/init.h>
#include <apt-pkg/cachefile.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/sourcelist.h>

#include <Lum/Base/String.h>

#include "Package.h"

class PkgProgress : public OpProgress
{
  PackageListTask* task;

public:
  PkgProgress(PackageListTask *task)
   : task(task)
  {
    // no code
  }

  void Update()
  {
    //task->SetCaption(Lum::Base::StringToWString(Op));
    task->SetAction(Lum::Base::StringToWString(Op));
    task->SetProgress(Percent);
  }
};

PackageListTask::PackageListTask(std::list<Package>& packages)
: packages(packages)
{
  // no code
}

void PackageListTask::Run()
{
  if (!pkgInitConfig(*_config) || !pkgInitSystem(*_config,_system)) {
    std::cout << "Cannot initialize!" << std::endl;
    return;
  }

  pkgSourceList *srcList=NULL;
  MMap          *map=NULL;

  if (_config->FindB("APT::Cache::Generate",true) == false) {
    map=new MMap(*new FileFd(_config->FindFile("Dir::Cache::pkgcache"),
                             FileFd::ReadOnly),MMap::Public|MMap::ReadOnly);
  }
  else {
    // Open the cache file
    srcList=new pkgSourceList;
    srcList->ReadMainList();

    // Generate it and map it
    PkgProgress prog(this);
    pkgMakeStatusCache(*srcList,prog,&map,true);
  }

  if (_error->PendingError()) {
    _error->DumpErrors();
    return;
  }

  pkgCache cache(map);

  for (pkgCache::PkgIterator iter=cache.PkgBegin(); !iter.end(); iter++) {
    if (iter->CurrentState!=pkgCache::State::Installed) {
      continue;
    }

    pkgCache::VerIterator version;

    if (iter.CurrentVer()!=NULL) {
      version=iter.CurrentVer();
    }
    else {
      version=iter.VersionList();
    }

    if (version.end()) {
      continue;
    }

    Package entry;

    pkgRecords Recs(cache);

    entry.SetName(Lum::Base::StringToWString(iter.Name()));
    entry.SetState((pkgCache::State::PkgCurrentState)iter->CurrentState);

    pkgRecords::Parser &P = Recs.Lookup(version.FileList());

    pkgCache::DepIterator dep=version.DependsList();
    std::stringstream     buffer;

    while (!dep.end()) {
      pkgCache::DepIterator s,a,b;

      dep.GlobOr(a,b);

      s=a;
      while (true) {
        if (a!=s) {
          buffer << " | ";
        }
        else {
          buffer << a.DepType() << " ";
        }

        buffer << a.SmartTargetPkg().Name();

        if (a.TargetVer()!=NULL) {
          buffer << " " << a.CompType() << " " << a.TargetVer();
        }

        if (a==b) {
          break;
        }
        ++a;
      }

      dep=a;
      ++dep;

      buffer << std::endl;

    }

    //Lum::Base::StringToWString(""/*P.Homepage()*/),
    entry.SetShortDescription(Lum::Base::StringToWString(P.ShortDesc()));
    entry.SetDescription(Lum::Base::StringToWString(P.LongDesc()));
    entry.SetMaintainer(Lum::Base::StringToWString(P.Maintainer()));

    entry.SetDependencies(Lum::Base::StringToWString(buffer.str()));
    entry.SetVersion(Lum::Base::StringToWString(version.VerStr()));

    if (iter.CurrentVer().Section()!=NULL) {
      entry.SetSection(Lum::Base::StringToWString(version.Section()));
    }
    entry.SetPackageSize(version->Size);
    entry.SetInstalledSize(version->InstalledSize);

    packages.push_back(entry);
  }
}

