/*
  PackageView - Show debian package details
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <fstream>
#include <iostream>

#include <Lum/OS/Main.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/Dialog.h>
#include <Lum/Space.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/WindowGroup.h>

#include "Package.h"
#include "PackageDetails.h"
#include "PackageList.h"

#include "config.h"

std::wstring        ossoVersion;

static Lum::AppInfo info;

class PackageFormatProvider : public Lum::TableView::FormatProvider
{
public:
  Lum::TableView::Alignment GetAlignment(size_t column, size_t row) const
  {
    switch (column) {
    case 1:
      return Lum::TableView::left;
      break;
    case 2:
      return Lum::TableView::right;
      break;
    default:
      assert(false);
      break;
    }
  }

  Lum::OS::Color GetTextColor(size_t column, size_t row) const
  {
    return Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor);
  }
};

class MainDialog : public Lum::Dialog
{
private:
  std::list<Package>                 packages;
  Lum::Model::ActionRef              showPackageAction;
  Lum::Model::ActionRef              aboutAction;
  std::wstring                       lastSearch;
  Lum::Model::StringRef              search;
  Lum::Model::ActionRef              searchTimerAction;
  Lum::Model::ListTableRef           currentPackages;
  Lum::Model::SingleLineSelectionRef selection;
  PackageListTask                    packageListTask;

public:
  MainDialog()
  : showPackageAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    search(new Lum::Model::String(L"")),
    searchTimerAction(new Lum::Model::Action()),
    currentPackages(new Lum::Model::ListTable(1)),
    selection(new Lum::Model::SingleLineSelection()),
    packageListTask(packages)
  {
    AttachModel(packageListTask.GetFinishedAction());

    AttachModel(GetOpenedAction());
    AttachModel(GetClosedAction());

    AttachModel(search);
    AttachModel(searchTimerAction);

    AttachModel(selection);

    AttachModel(showPackageAction);
    AttachModel(aboutAction);

    DetectOssoVersion();
  }

  ~MainDialog()
  {
    UnattachModel(aboutAction);
    UnattachModel(showPackageAction);

    UnattachModel(selection);

    UnattachModel(searchTimerAction);
    UnattachModel(search);

    UnattachModel(GetClosedAction());
    UnattachModel(GetOpenedAction());

    UnattachModel(packageListTask.GetFinishedAction());
  }

  virtual void PreInit()
  {
    Lum::Button           *button;
    Lum::Panel            *hpanel,*vpanel;
    Lum::String           *string;
    Lum::Table            *table;
    Lum::Text             *text;
    Lum::WindowGroup      *wGroup;
    Lum::Model::HeaderRef headerModel;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    hpanel=new Lum::HPanel();
    hpanel->SetFlex(true,true);

    vpanel=new Lum::VPanel();
    vpanel->SetFlex(true,true);

    string=new Lum::String();
    string->SetFlex(true,false);
    string->SetModel(search);

    vpanel->Add(string);
    vpanel->Add(new Lum::VSpace());

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Package",Lum::Base::Size::stdCharWidth,30,true);
    headerModel->AddColumn(L"Version",Lum::Base::Size::stdCharWidth,10,false);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetMinHeight(Lum::Base::Size::stdCharHeight,10);
    table->SetModel(currentPackages);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(showPackageAction);
    table->SetShowHeader(true);
    table->SetFormatProvider(new PackageFormatProvider());
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoHSize(true);

    vpanel->Add(table);
    hpanel->Add(vpanel);

    hpanel->Add(new Lum::HSpace());

    vpanel=new Lum::VPanel();
    vpanel->SetFlex(false,true);

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetText(ossoVersion,Lum::OS::Font::bold,Lum::Text::centered);
    vpanel->Add(text);

    vpanel->Add(new Lum::VSpace(true));

    button=new Lum::Button(L"_About");
    button->SetFlex(true,false);
    button->SetModel(aboutAction);
    vpanel->Add(button);

    hpanel->Add(vpanel);

    wGroup->SetMain(hpanel);

    SetTop(wGroup);

    packageListTask.SetParent(this);
    packageListTask.SetCaption(L"Reading package list...");

    Dialog::PreInit();
  }

  void DetectOssoVersion()
  {
    std::ifstream file;

    file.open("/etc/osso_software_version",std::ios::in);

    if (file) {
      std::string version;

      file >> version;
      file.close();

      if (version.find("2006")!=std::string::npos) {
        ossoVersion=L"OS 2006";
      }
      else if (version.find("2007")!=std::string::npos) {
        ossoVersion=L"OS 2007";
      }
      else if (version.find("2008")!=std::string::npos) {
        ossoVersion=L"OS 2008";
      }
    }

    if (ossoVersion.empty()) {
      ossoVersion=L"OS ????";
    }
  }

  void UpdateEntries()
  {
    currentPackages->Disable();
    currentPackages->Off();
    currentPackages->Clear();

    std::wstring search=this->search->Get();

    for (std::list<Package>::iterator iter=packages.begin();
         iter!=packages.end();
         ++iter) {
      if (search.empty() || iter->GetName().find(search)!=std::wstring::npos) {
        currentPackages->Append(new PackageEntry(currentPackages,&(*iter)));
      }
    }

    currentPackages->Sort(1);
    currentPackages->On();
    currentPackages->Enable();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      currentPackages->Disable();
      currentPackages->Off();
      packageListTask.Start();
    }
    else if (model==showPackageAction && showPackageAction->IsFinished()) {
      Package*       package=dynamic_cast<PackageEntry*>(currentPackages->GetEntry(selection->GetLine()))->GetPackage();
      PackageDetails *dialog;

      dialog=new PackageDetails(*package);
      dialog->SetParent(this);
      dialog->SetTitle(L"Details...");

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      /*if (configurationChanged && !SaveConfig()) {
        std::cerr << "Error while storing configuration!" << std::endl;
      }*/
    }
    else if (model==packageListTask.GetFinishedAction() && packageListTask.GetFinishedAction()->IsFinished()) {
      UpdateEntries();
    }
    else if (model==search) {
      Lum::OS::display->RemoveTimer(searchTimerAction);
      Lum::OS::display->AddTimer(1,500000,searchTimerAction);
    }
    else if (model==searchTimerAction && searchTimerAction->IsFinished()) {
      if (search->Get()!=lastSearch) {
        UpdateEntries();
        lastSearch=search->Get();
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Show installed applications..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2008, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }
};


LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))
