/*
 * This file is part of MTetrinet.
 *
 * MTetrinet is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * MTetrinet is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MTetrinet.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

#include <sys/time.h>
#include <time.h>
#include <errno.h>

#ifdef USE_HILDON
#include <hildon/hildon-program.h>
#include <libosso.h>
#endif

#include <gtk/gtk.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "io.h"
#include "tetrinet.h"
#include "tetris.h"
#include "misc.h"
#include "version.h"

enum { FIELDS=0, PARTYLINE, WINLIST };

#ifdef USE_HILDON
HildonProgram *program = NULL;
HildonWindow *window = NULL;
osso_context_t *osso_context = NULL;
#else
GtkWidget *window = NULL;
#endif

/* notebook index of fields/partyline/winlist */
int view_idx[3] = { -1, -1, -1};

GtkWidget *partyline_input_entry = NULL;
GtkWidget *gmsg_input_entry = NULL;

GtkWidget *notebook = NULL;
GdkPixbuf *blocks_pix = NULL;

/* screen setup stuff */
GtkWidget *playfield[6] = { NULL, NULL, NULL, NULL, NULL, NULL };
char field_state[6][FIELD_HEIGHT][FIELD_WIDTH];

GtkWidget *special_desc = NULL, *special_queue = NULL;
GtkWidget *preview = NULL;
GtkWidget *level_label = NULL, *lines_label = NULL;

GtkWidget *menu_item_view_fullscreen = NULL;

/* there are three text buffers: partyline, game messages, winlist */
GtkWidget *view[3] = { NULL, NULL, NULL };
GtkTextBuffer *buffer[3] = { NULL, NULL, NULL };

typedef struct key_event_s {
  int key;
  struct key_event_s *next;
} key_event_t;

key_event_t *key_event = NULL;

static void screen_refresh(void) { printf("screen_refresh()\n"); } 
static void screen_redraw(void) { printf("screen_redraw()\n"); } 

#if 0
static GtkWidget *gtk_label_small_new(char *name) {
  GtkWidget *label = gtk_label_new("");
  char *markup = g_markup_printf_escaped("<span size='x-small'>%s</span>", name);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}

static void gtk_label_small_set_text(GtkWidget *label, char *name) {
  char *markup = g_markup_printf_escaped("<span size='x-small'>%s</span>", name);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
}
#endif

struct {
  int code;
  GdkColor c;
  char *colorname;
  GtkTextTag *tag;  
} color_code[] = {
  { TATTR_CBLACK,                    {0, 0x0000, 0x0000, 0x0000}, "black",        NULL },
  { TATTR_CRED,                      {0, 0x7FFF, 0x0000, 0x0000}, "red",          NULL },
  { TATTR_CGREEN,                    {0, 0x0000, 0x7FFF, 0x0000}, "green",        NULL },
  { TATTR_CBROWN,                    {0, 0x7FFF, 0x7FFF, 0x0000}, "brown",        NULL },
  { TATTR_CBLUE,                     {0, 0x0000, 0x0000, 0x7FFF}, "blue",         NULL },
  { TATTR_CMAGENTA,                  {0, 0x7FFF, 0x0000, 0x7FFF}, "magenta",      NULL },
  { TATTR_CCYAN,                     {0, 0x0000, 0x7FFF, 0x7FFF}, "cyan",         NULL },
  { TATTR_CGREY,                     {0, 0x7FFF, 0x7FFF, 0x7FFF}, "grey",         NULL },
  { TATTR_CBLACK   | TATTR_CXBRIGHT, {0, 0x3FFF, 0x3FFF, 0x3FFF}, "darkgrey",     NULL },
  { TATTR_CRED     | TATTR_CXBRIGHT, {0, 0xFFFF, 0x0000, 0x0000}, "lightred",     NULL },
  { TATTR_CGREEN   | TATTR_CXBRIGHT, {0, 0x0000, 0xFFFF, 0x0000}, "lightgreen",   NULL },
  { TATTR_CBROWN   | TATTR_CXBRIGHT, {0, 0xFFFF, 0xFFFF, 0x0000}, "orange",       NULL },
  { TATTR_CBLUE    | TATTR_CXBRIGHT, {0, 0x0000, 0x0000, 0xFFFF}, "lightblue",    NULL },
  { TATTR_CMAGENTA | TATTR_CXBRIGHT, {0, 0xFFFF, 0x0000, 0xFFFF}, "lightmagenta", NULL },
  { TATTR_CCYAN    | TATTR_CXBRIGHT, {0, 0x0000, 0xFFFF, 0xFFFF}, "lightcyan",    NULL },
  { TATTR_CGREY    | TATTR_CXBRIGHT, {0, 0xAFFF, 0xAFFF, 0xAFFF}, "lightgrey",    NULL },
  { TATTR_RESET,                     {0, 0x0000, 0x0000, 0x0000}, "black",        NULL },
  { -1,                              {0, 0x0000, 0x0000, 0x0000}, NULL,           NULL }
};

#define FIELD2PLAYER(i)  ((!i)?my_playernum:((i<my_playernum)?i:(i+1)))

static void draw_text(int bufnum, const char *s) { 
  printf("draw_text(%d): ", bufnum); 

  g_assert(bufnum < 3);

  /* add all messages from buffer 2 to buffer 1 */
  if(bufnum == 2) bufnum = 1;

  /* start new line */
  GtkTextIter end;
  GtkTextBuffer *buf = buffer[bufnum];

  if(!buf) {
    printf("ERROR: buffer %d not yet setup, skipping message %s\n",
	   bufnum, s);

    /* TODO: queue messages until main screen has been drawn */
    /* or just initialize main screen now */

    return;
  }

  gtk_text_buffer_get_end_iter(buf, &end);  

  if(gtk_text_buffer_get_char_count(buf))
    gtk_text_buffer_insert(buf, &end, "\n", -1);

  /* translate control thingies */
  char *p = (char*)s;
  GtkTextTag *color_tag = NULL;
  while(*s) {

    /* walk over all valid characters */
    while(*p && *p > TATTR_MAX) {
      putchar(*p);
      p++;
    }

    /* write them to buffer */
    gtk_text_buffer_get_end_iter(buf, &end);  

    if(!color_tag)
      gtk_text_buffer_insert(buf, &end, s, p-s);
    else
      gtk_text_buffer_insert_with_tags(buf, &end, s, p-s, color_tag, NULL);

    s = p;

    /* process control codes */
    while(*p && *p <= TATTR_MAX) {
      int i = 0;
      while(color_code[i].colorname && color_code[i].code != *p-1)
	i++;
      
      /* found matching color */
      if(color_code[i].colorname) {
	
	/* tag needs to be created */
	if(!color_code[i].tag) 
	  color_code[i].tag = gtk_text_buffer_create_tag(buf, NULL,
				 "foreground-gdk", &color_code[i].c, NULL);  

	color_tag = color_code[i].tag;
      }
      p++;
    }
    s = p;
  }

  printf("\n");

#if 0
  if(gtk_text_buffer_get_char_count(buf))
    gtk_text_buffer_insert(buf, &end, "\r", -1);
#endif

  GtkTextMark *mark = gtk_text_buffer_get_mark(buf, "end");
  gtk_text_buffer_get_iter_at_mark(buf, &end, mark);
  gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(view[bufnum]), mark);
}

static void clear_text(int bufnum) { 
  printf("clear_text(%d)\n", bufnum);  

  g_assert(bufnum < 3);

  /* start new line */
  GtkTextIter start, end;
  GtkTextBuffer *buf = buffer[bufnum];
  if(!buf) return;

  gtk_text_buffer_get_start_iter(buf, &start);  
  gtk_text_buffer_get_end_iter(buf, &end);  
  gtk_text_buffer_delete(buf, &start, &end);
} 

static void setup_fields(void) { 
  printf("setup_fields()\n"); 

  /* make fields tab the active one if it isn't already */
  if(gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)) != view_idx[FIELDS])
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), view_idx[FIELDS]);
}


#define BLOCK_SIZE_PLAYER 16
#define BLOCK_SIZE_OPPONENT 8

#define PLAYFIELD_PLAYER_WIDTH    (FIELD_WIDTH *BLOCK_SIZE_PLAYER)
#define PLAYFIELD_PLAYER_HEIGHT   (FIELD_HEIGHT*BLOCK_SIZE_PLAYER)
#define PLAYFIELD_OPPONENT_WIDTH  (FIELD_WIDTH *BLOCK_SIZE_OPPONENT)
#define PLAYFIELD_OPPONENT_HEIGHT (FIELD_HEIGHT*BLOCK_SIZE_OPPONENT)

void draw_block(GtkWidget *widget, int index, int xo, int x, int yo, int y) {
  int i = (int)g_object_get_data(G_OBJECT(widget), "index");
  /* i is 0 for player, -1 for special queue and 1,2,... for opponents */

  int block_size = (i>0)?BLOCK_SIZE_OPPONENT:BLOCK_SIZE_PLAYER;

  if(!index) 
    gdk_draw_pixbuf(widget->window, 
		    widget->style->bg_gc[GTK_STATE_NORMAL],
		    blocks_pix, 
		    ((i>0)?PLAYFIELD_PLAYER_WIDTH:0) + ((i<0)?0:x), 
		    (BLOCK_SIZE_OPPONENT + BLOCK_SIZE_PLAYER) + ((i<0)?0:y), 
		    xo+x, yo+y, 
		    block_size, block_size,
		    GDK_RGB_DITHER_NONE, 0, 0);
  else 
    gdk_draw_pixbuf(widget->window, 
		    widget->style->bg_gc[GTK_STATE_NORMAL],
		    blocks_pix, 
		    (index-1)*block_size, 
		    (i>0)?BLOCK_SIZE_PLAYER:0, 
		    xo+x, yo+y, 
		    block_size, block_size,
		    GDK_RGB_DITHER_NONE, 0, 0);
}

static void draw_own_field(void) { 
  printf("draw_own_field()\n"); 

  /* calculate offsets to center field */
  gint xo = (playfield[0]->allocation.width - PLAYFIELD_PLAYER_WIDTH)/2;
  gint yo = (playfield[0]->allocation.height - PLAYFIELD_PLAYER_HEIGHT)/2;

  Field *f = &fields[my_playernum-1];

  int x, y;
  for (y = 0; y < FIELD_HEIGHT; y++) {
    for (x = 0; x < FIELD_WIDTH; x++) {
      int c = (int) (*f)[y][x];

      if(c != field_state[0][y][x]) { 
	draw_block(playfield[0], c, xo, BLOCK_SIZE_PLAYER*x, yo, BLOCK_SIZE_PLAYER*y);
	field_state[0][y][x] = c;
      }
    }
  }
}


static void draw_other_field(int player) {
  printf("draw_other_field(%d)\n", player); 
  Field *f = &fields[player-1];

  if(player > my_playernum)
    player--;

  printf("  real: %d\n", player); 

  /* calculate offsets to center field */
  gint xo = (playfield[player]->allocation.width - PLAYFIELD_OPPONENT_WIDTH)/2;
  gint yo = (playfield[player]->allocation.height - PLAYFIELD_OPPONENT_HEIGHT)/2;

  int x, y;
  for (y = 0; y < 22; y++) {
    for (x = 0; x < 12; x++) {
      int c = (int) (*f)[y][x];

      if(c != field_state[player][y][x]) { 
	draw_block(playfield[player], c, xo, BLOCK_SIZE_OPPONENT*x, yo, BLOCK_SIZE_OPPONENT*y);
	field_state[player][y][x] = c;
      }
    }
  }
}

int shape_offset[][2] = {
  { 0,                  0.5 * BLOCK_SIZE_PLAYER }, /* Straight bar */
  { 0,                       -BLOCK_SIZE_PLAYER },  /* Square block */
  { 0.5 * BLOCK_SIZE_PLAYER, -BLOCK_SIZE_PLAYER }, /* Reversed L block */    
  { 0.5 * BLOCK_SIZE_PLAYER, -BLOCK_SIZE_PLAYER }, /* L block */    
  { 0.5 * BLOCK_SIZE_PLAYER, -BLOCK_SIZE_PLAYER }, /* Z block */    
  { 0.5 * BLOCK_SIZE_PLAYER, -BLOCK_SIZE_PLAYER }, /* S block */        
  { 0.5 * BLOCK_SIZE_PLAYER, -BLOCK_SIZE_PLAYER }  /* T block */    
};

static void draw_status(void) { 
  printf("draw_status()\n");


  char *msg = g_strdup_printf("%d", lines>99999 ? 99999 : lines);
  gtk_label_set_text(GTK_LABEL(lines_label), msg);
  g_free(msg);

  msg = g_strdup_printf("%d", levels[my_playernum]);
  gtk_label_set_text(GTK_LABEL(level_label), msg);
  g_free(msg);

  gint xo = (preview->allocation.width - 5*BLOCK_SIZE_PLAYER)/2;
  gint yo = (preview->allocation.height - 3*BLOCK_SIZE_PLAYER)/2;

  /* erase background */
  gdk_draw_rectangle(preview->window,
            preview->style->black_gc, TRUE,
            xo, yo, 5*BLOCK_SIZE_PLAYER, 3*BLOCK_SIZE_PLAYER);
  
  int i,j;
  char shape[4][4];
  xo += shape_offset[next_piece][0] + BLOCK_SIZE_PLAYER/2;
  yo += shape_offset[next_piece][1] + BLOCK_SIZE_PLAYER/2;
  if(get_shape(next_piece, 0, shape) == 0) {
    for (j = 0; j < 4; j++) 
      for (i = 0; i < 4; i++) 
	if(shape[j][i] > 0)
	  draw_block(preview, shape[j][i], xo, BLOCK_SIZE_PLAYER*i,  yo, BLOCK_SIZE_PLAYER*j);
  }
}

static const char *descs[] = {
  "---",
  "Add Line",
  "Clear Line",
  "Nuke Field",
  "Clear Random Blocks",
  "Switch Fields",
  "Clear Special Blocks",
  "Block Gravity",
  "Blockquake",
  "Block Bomb"
};

#define VIEW_SPECIALS  18

static void draw_specials(void) { 
  printf("draw_specials()\n"); 

#ifdef USE_HILDON
  char *markup = g_markup_printf_escaped("<span size='x-small'>%s</span>", 
					 descs[specials[0]+1]);
  gtk_label_set_markup(GTK_LABEL(special_desc), markup);
  g_free(markup);
#else
  gtk_label_set_text(GTK_LABEL(special_desc), descs[specials[0]+1]);
#endif
  /* erase background */
  gdk_draw_rectangle(special_queue->window,
            special_queue->style->black_gc, TRUE,
            0, 0, BLOCK_SIZE_PLAYER * VIEW_SPECIALS, BLOCK_SIZE_PLAYER);

  int i = 0;
  while(i < special_capacity && specials[i] >= 0 && i < VIEW_SPECIALS) {
    draw_block(special_queue, specials[i]+6, 0, BLOCK_SIZE_PLAYER*i, 0, 0);
    i++;
  }
}

static const char *attdef_msgs[][2] = {
    { "cs1", "\0021 Line Added\025 to \005All\025" },
    { "cs2", "\0022 Lines Added\025 to \005All\025" },
    { "cs4", "\0024 Lines Added\025 to \005All\025" },
    { "a",   "\002Add Line\025" },
    { "c",   "\003Clear Line\025" },
    { "n",   "\003Nuke Field\025" },
    { "r",   "\002Clear Random Blocks\025" },
    { "s",   "\002Switch Fields\025" },
    { "b",   "\002Clear Special Blocks\025" },
    { "g",   "\003Block Gravity\025" },
    { "q",   "\002Blockquake\025" },
    { "o",   "\002Block Bomb\025" },
    { NULL }
};

static void draw_attdef(const char *type, int from, int to) { 
  printf("draw_attdef(\"%s\", %d, %d)\n", type, from, to);

  int i = 0;
  for(i=0; attdef_msgs[i][0]; i++) 
    if(strcmp(attdef_msgs[i][0], type) == 0)
      break;
  
  if(!attdef_msgs[i][0])
    return;

  char *message = NULL;
  if(to)
    message = g_strdup_printf("%s on \005%s\025", attdef_msgs[i][1], players[to-1]);
  if(!from)
    message = g_strdup_printf("%s by server", attdef_msgs[i][1]);
  else
    message = g_strdup_printf("%s by \005%s\025", attdef_msgs[i][1], players[from-1]);

  if(message) {
    draw_text(1, message);
    g_free(message);
  }
}

static void draw_gmsg_input(const char *s, int pos) { 
  printf("draw_gmsg_input(%s, %d)\n", s, pos);
}

static void clear_gmsg_input(void) { 
  printf("clear_gmsg_input()\n");

  /* remove entire input line if present */
  if(!gmsg_input_entry) return;

  gtk_entry_set_text(GTK_ENTRY(gmsg_input_entry), "");
  gtk_widget_hide(gmsg_input_entry);
}

static void draw_partyline_input(const char *s, int pos) { 
  printf("draw_partyline_input(%s, %d)\n", s, pos); 

  if(!strlen(s)) 
    gtk_entry_set_text(GTK_ENTRY(partyline_input_entry), "");
}

static void setup_winlist(void) { printf("setup_winlist()\n"); }

static void key_enqueue(int key) {
  key_event_t **keyP = &key_event;

  /* search end of key queue */
  while(*keyP)
    keyP = &(*keyP)->next;

  /* attach new entry */
  *keyP = g_new0(key_event_t,1);
  (*keyP)->key = key;  
}

static void key_enqueue_string(char *str) {
  while(*str)
    key_enqueue(*str++);
}

static gboolean playfield_clicked_event(GtkWidget *widget, GdkEventButton *event,
				      gpointer user_data) {
  int i = (int)g_object_get_data(G_OBJECT(widget), "index");

  /* issue "send special" command if not entering text */
  if(!GTK_WIDGET_VISIBLE(gmsg_input_entry)) {
    printf("my_playernum = %d\n", my_playernum);
    printf("clicked %d, player %d\n", i, FIELD2PLAYER(i));

    key_enqueue('0' + FIELD2PLAYER(i));
  }

  return FALSE;
}

static gint playfield_expose_event(GtkWidget *widget, GdkEventExpose *event, 
				   gpointer data) {

  int i = (int)g_object_get_data(G_OBJECT(widget), "index");
  printf("expose event for playfield %d\n", i);

  int size = i?BLOCK_SIZE_OPPONENT:BLOCK_SIZE_PLAYER;

  /* calculate offsets to center field */
  gint xo = (playfield[i]->allocation.width - size * FIELD_WIDTH)/2;
  gint yo = (playfield[i]->allocation.height - size * FIELD_HEIGHT)/2;

  int x, y;
  for (y = 0; y < 22; y++) 
    for (x = 0; x < 12; x++) 
      draw_block(playfield[i], field_state[i][y][x], xo, size*x, yo, size*y);

  return TRUE;
}
 
GtkWidget *playfield_new(int i) {
  playfield[i]  = gtk_drawing_area_new();
  
  gtk_drawing_area_size(GTK_DRAWING_AREA(playfield[i]), 
			i?PLAYFIELD_OPPONENT_WIDTH:PLAYFIELD_PLAYER_WIDTH,
			i?PLAYFIELD_OPPONENT_HEIGHT:PLAYFIELD_PLAYER_HEIGHT);

  g_object_set_data(G_OBJECT(playfield[i]), "index", (gpointer)i);

  gtk_signal_connect(GTK_OBJECT(playfield[i]), "expose_event",
		     G_CALLBACK(playfield_expose_event), NULL);

  gtk_widget_set_events(playfield[i], GDK_EXPOSURE_MASK);
  gtk_widget_add_events(playfield[i], GDK_BUTTON_PRESS_MASK);

  gtk_signal_connect(GTK_OBJECT(playfield[i]), "button_press_event",
		     (GtkSignalFunc)playfield_clicked_event, NULL);

  /* TODO: should be taken from game date? */
  int x, y;
  for (y = 0; y < 22; y++) 
    for (x = 0; x < 12; x++) 
      field_state[i][y][x] = 0;

  return playfield[i];
}

static gint special_queue_expose_event(GtkWidget *widget, GdkEventExpose *event, 
				 gpointer data) {

  printf("special queue expose event\n");

  draw_specials();
  return TRUE;
}

static gboolean special_queue_clicked_event(GtkWidget *widget, GdkEventButton *event,
					    gpointer user_data) {

  /* issue "delete special" command if not entering text */
  if(!GTK_WIDGET_VISIBLE(gmsg_input_entry)) {
    printf("clicked specials\n");
    key_enqueue('d');
  }

  return FALSE;
}

GtkWidget *special_queue_new(void) {
  special_queue = gtk_drawing_area_new();

  gtk_drawing_area_size(GTK_DRAWING_AREA(special_queue), 
	BLOCK_SIZE_PLAYER * VIEW_SPECIALS, BLOCK_SIZE_PLAYER);

  g_object_set_data(G_OBJECT(special_queue), "index", (gpointer)-1);

  gtk_signal_connect(GTK_OBJECT(special_queue), "expose_event",
 		     G_CALLBACK(special_queue_expose_event), NULL);

  gtk_widget_set_events(special_queue, GDK_EXPOSURE_MASK);
  gtk_widget_add_events(special_queue, GDK_BUTTON_PRESS_MASK);

  gtk_signal_connect(GTK_OBJECT(special_queue), "button_press_event",
		     (GtkSignalFunc)special_queue_clicked_event, NULL);

  return special_queue;
}

static gint preview_expose_event(GtkWidget *widget, GdkEventExpose *event, 
				 gpointer data) {
  printf("expose event for preview\n");

  draw_status();

  return TRUE;
}
 
GtkWidget *preview_new(void) {
  preview = gtk_drawing_area_new();
  
  gtk_drawing_area_size(GTK_DRAWING_AREA(preview), 
			BLOCK_SIZE_PLAYER*5, BLOCK_SIZE_PLAYER*3);

  g_object_set_data(G_OBJECT(preview), "index", (gpointer)-2);

  gtk_signal_connect(GTK_OBJECT(preview), "expose_event",
		     G_CALLBACK(preview_expose_event), NULL);
  
  return preview;
}

GdkPixbuf *blocks_load(void) {
  char *fullname = find_file("blocks.png");
  
  if(!fullname) {
    printf("Unable to locate \"blocks.png\"\n");
    return NULL;
  }

  GdkPixbuf *pix = gdk_pixbuf_new_from_file(fullname, NULL);
  if(!pix) 
    printf("Unable to load \"blocks.png\"\n");
  
  g_free(fullname);
  return pix;
}

void blocks_free(GdkPixbuf *pix) {
  gdk_pixbuf_unref(pix);
}

/******************** begin of menu *********************/

static void 
cb_menu_about(GtkWidget *window, gpointer data) {
  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());

  gtk_about_dialog_set_name(about, "MTetrinet");
  gtk_about_dialog_set_version(about, VERSION);
  gtk_about_dialog_set_copyright(about, 
	 _("MTetrinet (c) 2008 by\n" 
	 "Till Harbaum <till@harbaum.org>"));

  gtk_about_dialog_set_website(about,
       _("http://www.harbaum.org/till/maemo"));
  
  gtk_about_dialog_set_comments(about, 
	_("based on \"Tetrinet for Linux\""));

  gtk_widget_show_all(GTK_WIDGET(about));
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(GTK_WIDGET(about));
}

void on_window_destroy (GtkWidget *widget, gpointer data);

static void 
cb_menu_quit(GtkWidget *window, gpointer data) {
  on_window_destroy(window, data);
}

static void 
cb_menu_fullscreen(GtkWidget *widget, gpointer data) {

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
    gtk_window_fullscreen(GTK_WINDOW(window));
  else
    gtk_window_unfullscreen(GTK_WINDOW(window));
}

void menu_create(GtkWidget *vbox) { 
  GtkWidget *menu, *item;  
  menu = gtk_menu_new();

  menu_item_view_fullscreen = 
    item = gtk_check_menu_item_new_with_label( _("Fullscreen") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_fullscreen), 
		   NULL);

  item = gtk_menu_item_new_with_label( _("About...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_about), NULL);


  item = gtk_menu_item_new_with_label( _("Quit") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_quit), NULL);

#ifdef USE_HILDON
  hildon_window_set_menu(window, GTK_MENU(menu));
#else
  /* attach ordinary gtk menu */
  GtkWidget *menu_bar = gtk_menu_bar_new();

  GtkWidget *root_menu = gtk_menu_item_new_with_label (_("Menu"));
  gtk_widget_show(root_menu);

  gtk_menu_bar_append(menu_bar, root_menu); 
  gtk_menu_item_set_submenu(GTK_MENU_ITEM (root_menu), menu);

  gtk_widget_show(menu_bar);
  gtk_box_pack_start(GTK_BOX(vbox), menu_bar, 0, 0, 0);
#endif
}

/********************* end of menu **********************/


void cleanup(void) {
#ifdef USE_HILDON
  if(osso_context)
    osso_deinitialize(osso_context);

  program = NULL;
#endif

  blocks_free(blocks_pix);

  puts("everything is gone");
}

void on_window_destroy (GtkWidget *widget, gpointer data) {
  window = NULL;

  /* send F10 (quit) to tetrinet engine */
  key_enqueue(K_F10);
}

static int transtab[][2] = {
  { GDK_Up,    K_UP    },
  { GDK_Down,  K_DOWN  },
  { GDK_Left,  K_LEFT  },
  { GDK_Right, K_RIGHT },
  { ' ',       ' '     },    // block drop

#ifdef USE_HILDON
  { GDK_Return, ' ' },       // direction pad center is return key -> 
                             // map to space for block drop
#endif

  { 0, 0 }
};

gboolean on_window_key_press(GtkWidget *widget, 
			 GdkEventKey *event, gpointer data) {
  int handled = FALSE;

  /* map game keys only in field mode */
  if(gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)) == view_idx[FIELDS]) {

    int i = 0;
    while(event->keyval != transtab[i][0] && transtab[i][0])
      i++;
  
    if(transtab[i][0]) {
      printf("translated key\n");
      
      key_enqueue(transtab[i][1]);
      return TRUE;
    }
  }

  switch(event->keyval) {
#ifdef USE_HILDON

  case HILDON_HARDKEY_INCREASE:
    handled = TRUE;
    break;
    
  case HILDON_HARDKEY_DECREASE:
    handled = TRUE;
    break;
#endif

#ifdef USE_HILDON
  case HILDON_HARDKEY_FULLSCREEN:
#else
  case GDK_F11:
#endif
    {
      gboolean fullscreen = !gtk_check_menu_item_get_active(
	       GTK_CHECK_MENU_ITEM(menu_item_view_fullscreen));
      gtk_check_menu_item_set_active(
	       GTK_CHECK_MENU_ITEM(menu_item_view_fullscreen), 
	       fullscreen);

      if(fullscreen)
	gtk_window_fullscreen(GTK_WINDOW(window));
      else
	gtk_window_unfullscreen(GTK_WINDOW(window));

      handled = TRUE;
    }
    break;
  }
  
  return handled;
}


/* this is ugly and generates 100% cpu load. this really needs to */
/* be done more clever */
static int wait_for_input(int msec) { 
  int state = 0;

  printf("wait_for_input(%d)\n", msec); 

  if(key_event) {
    printf("key %d in queue\n", key_event->key);

    key_event_t *cur = key_event;
    state = key_event->key;
    key_event = key_event->next;
    g_free(cur);
    return state;
  }

  //  if(msec > 0) { while(1) gtk_main_iteration(); }
  //  g_assert(msec < 0);

  struct timeval start, end;
  if(msec > 0)
    gettimeofday(&start, NULL);

  while(!state) {
    /* handle server input */
    fd_set fds;
    struct timeval tv;

    FD_ZERO(&fds);
    FD_SET(server_sock, &fds);
    tv.tv_usec = tv.tv_sec = 0;
    while(select(server_sock+1, &fds, NULL, NULL, &tv) < 0) {
      if(errno != EINTR)
	perror("Warning: select() failed");
    }

    if(FD_ISSET(server_sock, &fds)) {
      printf("server msg pending\n");
      state = -1;
    }

    if(gtk_events_pending()) {
      while(gtk_events_pending()) {
	gtk_main_iteration();
	
	if(key_event) {
	  key_event_t *cur = key_event;
	  state = key_event->key;
	  key_event = key_event->next;
	  
	  g_free(cur);
	  //	printf("sending key %d (%c)\n", state, state); 
	  return state;
	}
      }
    } else
      usleep(1000);

    if(msec > 0) {
      gettimeofday(&end, NULL);

      int diff = 
	(end.tv_sec  - start.tv_sec) * 1000 +
	(end.tv_usec - start.tv_usec) / 1000;

      if(diff > msec)
	state = -2;
    } else if(!msec)
      /* timeout was 0, return immediately */
      state = -2;
  }    

  //  printf("returning %d\n", state);
  return state;
}

GtkWidget *text_window_new(int bufnum) {  
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
    				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), 
				      GTK_SHADOW_ETCHED_OUT);

  view[bufnum] = gtk_text_view_new();
  buffer[bufnum] = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view[bufnum]));
  gtk_text_view_set_editable(GTK_TEXT_VIEW(view[bufnum]), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(view[bufnum]), FALSE);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view[bufnum]), GTK_WRAP_CHAR);
  GtkTextIter end;
  gtk_text_buffer_get_end_iter(buffer[bufnum], &end);
  gtk_text_buffer_create_mark(buffer[bufnum], "end", &end, FALSE);

  /* Change default font throughout the widget */
  PangoFontDescription *font_desc = pango_font_description_from_string("Courier 10");
  gtk_widget_modify_font(view[bufnum], font_desc);
  pango_font_description_free(font_desc);

  gtk_container_add(GTK_CONTAINER(scrolled_window), view[bufnum]);
  return scrolled_window;
}

static void on_notebook_page_change(GtkNotebook     *notebook,
				    GtkNotebookPage *page,
				    guint            page_num,
				    gpointer         user_data) {

  printf("switched to page %d\n", page_num);

  int keys[] = { K_F1, K_F2, K_F3 };
  printf("enqueue key %d\n", keys[page_num]);
  key_enqueue(keys[page_num]);
}

static void screen_setup(void) { 
  printf("screen setup()\n");
} 

static void setup_partyline(void) { 
  printf("setup_partyline()\n"); 

  /* make partyline tab the active one if it isn't already */
  if(gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)) != 
     view_idx[PARTYLINE])
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook),
				  view_idx[PARTYLINE]);
}

/*************************************************************************/
/************************** Interface declaration ************************/
/*************************************************************************/

Interface tty_interface = {

  wait_for_input,
  
  screen_setup,
  screen_refresh,
  screen_redraw,
  
  draw_text,
  clear_text,
  
  setup_fields,
  draw_own_field,
  draw_other_field,
  draw_status,
  draw_specials,
  draw_attdef,
  draw_gmsg_input,
  clear_gmsg_input,
  
  setup_partyline,
  draw_partyline_input,
  
  setup_winlist
};

static gboolean gmsg_window_clicked_event(GtkWidget *widget, GdkEventButton *event,
					  gpointer user_data) {

  /* issue "start enter text" command if not already entering text */
  if(!GTK_WIDGET_VISIBLE(gmsg_input_entry)) {
    printf("clicked gmsg window\n");

    if(gmsg_input_entry) {
      gtk_widget_show(gmsg_input_entry);
      gtk_widget_grab_focus(gmsg_input_entry);
      return TRUE;
    }
  }

  return FALSE;
}

/* simple dialog to handle the original tetrinets parameters */ 
gboolean parameter_dialog(char **server, char **nickname) {
  GConfClient *gconf_client = gconf_client_get_default();

  /* fetch values from gconf if present */
  GConfValue *value = NULL;
  if((value = gconf_client_get(gconf_client, "/apps/" PACKAGE "/server", NULL))) {
    if(*server) g_free(*server);
    *server = gconf_client_get_string(gconf_client, "/apps/" PACKAGE "/server", NULL);
    gconf_value_free(value);
  }

  if((value = gconf_client_get(gconf_client, "/apps/" PACKAGE "/nickname", NULL))) {
    if(*nickname) g_free(*nickname);
    *nickname = gconf_client_get_string(gconf_client, "/apps/" PACKAGE "/nickname", NULL);
    gconf_value_free(value);
  }


  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Tetrinet setup"),
	  GTK_WINDOW(window), GTK_DIALOG_MODAL,
	  GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
	  NULL);

  GtkWidget *server_w, *nick_w, *table = gtk_table_new(2, 2, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new("Server:"), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), server_w = gtk_entry_new(), 1, 2, 0, 1);
  gtk_entry_set_activates_default(GTK_ENTRY(server_w), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new("Nickname:"), 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), nick_w = gtk_entry_new(), 1, 2, 1, 2);
  gtk_entry_set_activates_default(GTK_ENTRY(nick_w), TRUE);

  if(*server) gtk_entry_set_text(GTK_ENTRY(server_w), *server);
  if(*nickname) gtk_entry_set_text(GTK_ENTRY(nick_w), *nickname);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

  gtk_widget_show_all(dialog);
  if(GTK_RESPONSE_ACCEPT != gtk_dialog_run(GTK_DIALOG(dialog))) {
    gtk_widget_destroy(dialog);
    return FALSE;
  }

  /* set new values */
  if(*server) g_free(*server);
  if(*nickname) g_free(*nickname);

  *server = g_strdup(gtk_entry_get_text(GTK_ENTRY(server_w)));
  *nickname = g_strdup(gtk_entry_get_text(GTK_ENTRY(nick_w)));

  gtk_widget_destroy(dialog);

  /* save new values in gconf */
  gconf_client_set_string(gconf_client, "/apps/" PACKAGE "/server", *server, NULL);
  gconf_client_set_string(gconf_client, "/apps/" PACKAGE "/nickname", *nickname, NULL);

  return TRUE;
}

/* we intercept the fprintf calls from the original tetrinet to be able to */
/* display some error messages via the gui */
int tfprintf(FILE *stream, const char *format, ...) {
  va_list ap;

  printf("fprintf(%p-%p, %s)\n", stream, stderr, format);

  if(stream != stderr) {
    va_start(ap, format);
    int n = vfprintf(stream, format, ap);
    va_end(ap);
    return n;
  }

  va_start(ap, format);
  char *buf = g_strdup_vprintf(format, ap);
  va_end(ap);
  
  GtkWidget *dialog = gtk_message_dialog_new(
	     GTK_WINDOW(window),
	     GTK_DIALOG_DESTROY_WITH_PARENT,
	     GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
	     buf);

  gtk_window_set_title(GTK_WINDOW(dialog), "Message");

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  g_free(buf);

  return 0;
}

void on_partyline_activate(GtkEntry *entry, gpointer  user_data) {
  char *msg = (char*)gtk_entry_get_text(entry);

  printf("partyline activate with message \"%s\"\n", msg);

  key_enqueue_string(msg);
  key_enqueue('\r');
}

void on_gmsg_activate(GtkEntry *entry, gpointer  user_data) {
  char *msg = (char*)gtk_entry_get_text(entry);

  printf("gmsg activate with message \"%s\"\n", msg);

  key_enqueue('t');
  key_enqueue_string(msg);
  key_enqueue('\r');
}

/* this is a pretty ugly hack but it allows us to preempt the main() in tetrinet.c */
/* we interfere with normal startup to do three things: */
/* 1. forward the command line parameters to gtk */
/* 2. inject gui derived parameters into tetrinet.c */
/* 3. achieve 1. and 2. without modifying the "tetrinet for linux" source code */

#undef main
extern int tmain(int ac, char **av);

int main(int ac, char **av) {
  printf("gtk interface start\n");

  printf("Using locale for %s in %s\n", PACKAGE, LOCALEDIR);

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);

  gtk_init(&ac, &av);

#ifdef USE_HILDON
  printf("Installing osso context for \"org.harbaum." PACKAGE "\"\n");
  osso_context = osso_initialize("org.harbaum."PACKAGE, 
					 VERSION, TRUE, NULL);
  if(osso_context == NULL) {
    printf("error initiating osso context\n");
  }
#endif

  g_set_application_name("MTetrinet");

#ifdef USE_HILDON
  /* Create the hildon program and setup the title */
  program = HILDON_PROGRAM(hildon_program_get_instance());
  
  /* Create HildonWindow and set it to HildonProgram */
  window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(program, window);
#else
  /* Create a Window. */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), "MTetrinet");
  /* Set a decent default size for the window. */
  gtk_window_set_default_size(GTK_WINDOW(window), 640, 400);
#endif

  g_signal_connect(G_OBJECT(window), "destroy", 
		   G_CALLBACK(on_window_destroy), NULL);

  g_signal_connect(G_OBJECT(window), "key_press_event",
		   G_CALLBACK(on_window_key_press), NULL);

  GtkWidget *vbox = gtk_vbox_new(FALSE,0);
  menu_create(vbox);
  blocks_pix = blocks_load();

  notebook = gtk_notebook_new();

  /* +++++++++++++++++++++ create "field" view ++++++++++++++++++++++ */

  GtkWidget *table = gtk_table_new(3, 6, FALSE);  // x, y
  /* players view */
  gtk_table_attach(GTK_TABLE(table), playfield_new(0), 
		   0, 1, 0, 3,    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_set_row_spacing(GTK_TABLE(table), 0, 4);
  gtk_table_set_row_spacing(GTK_TABLE(table), 1, 4);
  
  /* ----------------- the game message window ---------------- */
  GtkWidget *ivbox = gtk_vbox_new(0, FALSE);
  gtk_box_pack_start(GTK_BOX(ivbox), text_window_new(1), TRUE, TRUE, 0);
  gmsg_input_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(gmsg_input_entry), "activate",
		     (GtkSignalFunc)on_gmsg_activate, NULL);
  
  gtk_box_pack_start(GTK_BOX(ivbox), gmsg_input_entry, FALSE, FALSE, 0);

  gtk_signal_connect(GTK_OBJECT(view[1]), "button_press_event",
		     (GtkSignalFunc)gmsg_window_clicked_event, NULL);

  gtk_table_attach(GTK_TABLE(table), ivbox, 
		   3, 6, 0, 1,    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  /* ---------------- status ----------------- */

  ivbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start(GTK_BOX(ivbox), preview_new(), TRUE, FALSE, 0);

  GtkWidget *itable = gtk_table_new(2, 2, FALSE);  // x, y
  gtk_table_attach_defaults(GTK_TABLE(itable), gtk_label_new("Lines:"), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(itable), lines_label = gtk_label_new(""), 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(itable), gtk_label_new("Level:"), 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(itable), level_label = gtk_label_new(""), 1, 2, 1, 2);

  gtk_box_pack_start_defaults(GTK_BOX(ivbox), itable);
  gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 1, 3, 0, 1);

  /* ---------------- "specials" queue ----------------- */

  special_desc = gtk_label_new("");
  gtk_table_attach(GTK_TABLE(table), special_desc,
		   1, 3, 1, 2,    GTK_FILL | GTK_EXPAND, 0, 0, 0);

  gtk_table_attach(GTK_TABLE(table), special_queue_new(), 
		   3, 6, 1, 2,    GTK_FILL | GTK_EXPAND, 0, 0, 0);

  /* ---------------- and five opponent views ----------------- */
  int i;
  for(i=0;i<5;i++) {
    gtk_table_attach(GTK_TABLE(table), playfield_new(i+1), 
		     i+1, i+2, 2, 3,    GTK_FILL, GTK_FILL, 0, 0);
    gtk_table_set_col_spacing(GTK_TABLE(table), i, 3);
  }

  view_idx[FIELDS] = gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
					      gtk_label_new("Field"));

  /* +++++++++++++++++++++ create "partyline" view ++++++++++++++++++++++ */

  ivbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(ivbox), text_window_new(0));

  partyline_input_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(partyline_input_entry), "activate",
		     (GtkSignalFunc)on_partyline_activate, NULL);
  gtk_box_pack_start(GTK_BOX(ivbox), partyline_input_entry, 0,0,0);

  view_idx[PARTYLINE] = gtk_notebook_append_page(GTK_NOTEBOOK(notebook), ivbox, 
						 gtk_label_new("Partyline"));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), notebook);
  
  gtk_container_add(GTK_CONTAINER(window), vbox);
  gtk_widget_show_all(GTK_WIDGET(window));
  gtk_widget_hide(gmsg_input_entry);

  g_signal_connect(G_OBJECT(notebook), "switch-page", 
		   G_CALLBACK(on_notebook_page_change), NULL);

  /* let gtk do its thing */
  while(gtk_events_pending())
    gtk_main_iteration();

  /* call original tetrinet main with fake arguments */
  char *argv[] = { av[0], NULL, NULL };
  argv[1] = g_strdup("Maemo");
  argv[2] = g_strdup("tetrinet.us");

  if(!parameter_dialog(&argv[2], &argv[1])) {
    cleanup();
    return 0;
  }

  tmain(3, argv);

  g_free(argv[1]);
  g_free(argv[2]);

  cleanup();

  printf("gtk interface end\n");

  return 0;
}
