from utils.Observable import Observable

import gobject


class MPlayerError(StandardError): pass
class MPlayerNotFoundError(MPlayerError): pass
class MPlayerDiedError(MPlayerError): pass
class FileNotFoundError(MPlayerError): pass
class InvalidFileError(MPlayerError): pass


# idle timeout in milliseconds
_IDLE_TIMEOUT = 1000 * 60 * 3


class AbstractPlayer(Observable):
    """
    Base class for media player implementations.
    """

    OBS_STARTED = 0
    OBS_KILLED = 1
    OBS_SUSPENDED = 2
    OBS_ERROR = 3
        
    OBS_PLAYING = 4
    OBS_STOPPED = 5
    OBS_NEW_STREAM_TRACK = 6
    OBS_EOF = 7
    
    OBS_POSITION = 8
    
    OBS_ASPECT = 9
    OBS_TAG_INFO = 10
    
    OBS_BUFFERING = 11
    
    # error codes
    ERR_INVALID = 0
    ERR_NOT_FOUND = 1
    ERR_CONNECTION_TIMEOUT = 2
    ERR_NOT_SUPPORTED = 3
    
    
    __context_id = 0
    
    
    def __init__(self):
        """
        Constructor to be invoked by subclasses. Do not instantiate this class
        directly.
        """

        # URI of the current file
        self.__uri = ""
        
        # point to resume from
        self.__suspension_point = None

        # current position and total length
        self.__position = (0, 0)

        # whether the player is currently playing
        self.__playing = False
        
        # whether media has video
        self.__has_video = False
        
        # whether media has audio
        self.__has_audio = False
        
        # current volume level (0..100)
        self.__volume = 50
        
        # Xid of the video window, or -1 if unset
        self.__window_id = -1
        
        
        self.__position_handler = None
        self.__idle_handler = None


    def __repr__(self):
        """
        Returns a readable string representation of this player object.
        """
    
        return self.__class__.__name__


    def _new_context_id(self):
        """
        Generates and returns a new context ID.
        
        @return: a new context ID
        """
    
        self.__context_id += 1
        ctx_id = self.__context_id
        return ctx_id

       
        
    def handle_expose(self, src, gc, x, y, w, h):
        """
        Override this method if the player has to handle GTK expose events
        on the player window.
        """
    
        pass


    def __on_eof(self):
        """
        Reacts on end-of-file.
        """

        self.__playing = False
        print "REACHED EOF"
        self.__suspension_point = (self.__uri, 0)
        self.update_observer(self.OBS_EOF, self.__context_id)


    def __on_idle_timeout(self):
        """
        Reacts on idle timeout and suspends the player.
        """

        print "IDLE TIMEOUT"
        gobject.source_remove(self.__idle_handler)
        self.__idle_handler = None
        
        pos, total = self._get_position()
        self.__suspension_point = (self.__uri, pos)
        self.stop()


    def __watch_progress(self):
        """
        Starts watching the time position progress.
        """
    
        if (not self.__position_handler):
            self.__position_handler = \
                  gobject.timeout_add(0, self.__update_position)
            

    def __update_position(self):
        """
        Regularly updates the position in the file.
        """

        if (self.__playing):
            pos, total = self._get_position()
            self.__position = (pos, total)
            
            self.update_observer(self.OBS_POSITION, self.__context_id,
                                 pos, total)

            self.__position_handler = \
                  gobject.timeout_add(500, self.__update_position)

            # detect EOF
            if (self._is_eof()):
                self.__on_eof()
                

        else:
            # stop updating when not playing
            self.__position_handler = None

        # reset idle timeout
        if (self.__idle_handler):
            gobject.source_remove(self.__idle_handler)
        self.__idle_handler = gobject.timeout_add(_IDLE_TIMEOUT,
                                                  self.__on_idle_timeout)
        pass

        
    def set_window(self, xid):
        """
        Sets the window for video output.
        
        @param xid: Xid of the window
        """
    
        self.__window_id = xid
        
        
    def set_options(self, opts):
    
        pass
               
        
    def load(self, uri, ctx_id = -1):
        """
        Loads and plays the given file.
        
        @param uri: URI of the file
        @param ctx_id: context ID to use; for internal use only
        @return: context ID
        """

        self.__has_video = True
        self.__has_audio = True
        self.__uri = uri
       
        self.stop()
        
        if (self.__window_id != -1):
            self._set_window(self.__window_id)

        self._load(uri)
        self._set_volume(self.__volume)
        
        self.play()

        if (ctx_id != -1):
            self.__context_id = ctx_id
        else:
            self.__context_id = self._new_context_id()
        print "CTX", self.__context_id
        return self.__context_id

       
       
    def resume(self, uri, pos):
    
        raise NotImplementedError


    def _report_aspect_ratio(self, ratio):
        """
        The subclass calls this to report the video aspect ratio.
        
        @ratio: the aspect ratio
        """
        
        self.update_observer(self.OBS_ASPECT, self.__context_id, ratio)


    def _report_tags(self, tags):
        """
        The subclass calls this to report ID tags.
        """
        
        pass
        
        
    def play(self):
        """
        Starts playback.
        """
    
        if (not self.__playing):
            self._play()
            self.__playing = True
            self.update_observer(self.OBS_PLAYING, self.__context_id)
            self.__watch_progress()
        
        
    def pause(self):
        """
        Pauses playback or starts it if it was paused.
        """

        if (self.__playing):
            self._stop()
            self.__playing = False
            self.update_observer(self.OBS_STOPPED, self.__context_id)
        else:
            self._play()
            self.__playing = True
            self.update_observer(self.OBS_PLAYING, self.__context_id)
            self.__watch_progress()
        
        
    def stop(self):
        """
        Stops playback.
        """

        if (self.__playing):
            self._stop()
            self.__playing = False
            self.update_observer(self.OBS_STOPPED, self.__context_id)
        
        
    def close(self):
        """
        Closes the player.
        """
    
        self._close()
        self.__context_id = 0
        
        
    def set_volume(self, volume):
        """
        Sets the audio volume.
        
        @param volume: volume as a value between 0 and 100
        """

        self._set_volume(volume)    
        
        
    def seek(self, pos):
        """
        Seeks to the given position.
        
        @param pos: position in seconds
        """
    
        self._seek(pos)
        
        
    def seek_percent(self, percent):
        """
        Seeks to the given position.
        
        @param percent: position in percents
        """
    
        nil, total = self.__position
        pos = total * percent / 100.0
        self.seek(pos)


    def has_video(self):
        """
        Returns whether there is a video track.
        """
    
        return self.__has_video
        

    def has_audio(self):
        """
        Returns whether there is an audio track.
        """

        return self.__has_audio

