from AbstractPlayer import *
    
import pygst; pygst.require("0.10")
import gst

import time


class _GstPlayer(AbstractPlayer):
    """
    Singleton class for a gstreamer media player backend.
    """

    def __init__(self):
        """
        Creates the GstPlayer singleton.
        """
        
        self.__is_eof = False
        
        # position, total length, and time stamp of the last position measuring
        self.__pos_time = (0, 0, 0)
        
        
        AbstractPlayer.__init__(self)

        self.__player = gst.element_factory_make("playbin", "player")
        bus = self.__player.get_bus()
        bus.add_signal_watch()
        bus.enable_sync_message_emission()
        bus.connect("message", self.__on_message)
        bus.connect("sync-message::element", self.__on_sync_message) 
        
        
    def __on_message(self, bus, message):
    
        t = message.type
        #print "Message Type", t
        if (t == gst.MESSAGE_EOS):        
            self.__player.set_state(gst.STATE_NULL)
            self.__is_eof = True

        elif (t == gst.MESSAGE_ERROR):
            self.__player.set_state(gst.STATE_NULL)
            err, debug = message.parse_error()
            self.update_observer(self.OBS_ERROR, self.__context_id,
                                 self.ERR_INVALID)


    def __on_sync_message(self, bus, message):
    
        if (message.structure == None): return
        
        name = message.structure.get_name()

        if (name == "prepare-xwindow-id" and self.__window_id != -1):
            imgsink = message.src
            imgsink.set_property("force-aspect-ratio", True)
            imgsink.set_xwindow_id(self.__window_id)


    def _set_window(self, xid):
    
        self.__window_id = xid
        
        
    def set_options(self, opts):
    
        pass

       
    def _load(self, uri):
       
        if (uri.startswith("/")): uri = "file://" + uri
        uri = uri.replace("\"", "\\\"")

        self.__player.set_state(gst.STATE_NULL)
        self.__player.set_property("uri", uri)
        


    def _set_volume(self, volume):

        self.__player.set_property("volume", volume / 100.0)

    
    def _is_eof(self):
    
        return self.__is_eof


    def _play(self):
           
        self.__is_eof = False
        self.__pos_time = (0, 0, 0)
        self.__player.set_state(gst.STATE_PLAYING)


    def _stop(self):

        self.__player.set_state(gst.STATE_PAUSED)


    def _close(self):
    
        self._stop()        


    def _seek(self, pos):
    
        self.__pos_time = (0, 0, 0)
        self.__player.seek_simple(gst.Format(gst.FORMAT_TIME),
                                  gst.SEEK_FLAG_FLUSH,
                                  pos * 1000000000)


    def _get_position(self):

        pos, total, mtime = self.__pos_time
        if (mtime > 0):
            pos += time.time() - mtime
            
        else:
            try:
                pos, format = self.__player.query_position(gst.FORMAT_TIME)
                total, format = self.__player.query_duration(gst.FORMAT_TIME)
                
                # gstreamer time is in nano seconds... weird! :o
                pos /= 1000000000
                total /= 1000000000
                self.__pos_time = (pos, total, time.time())
                
            except:
                pos, total = (0, 0)
        
        return (pos, total)
       
        


_singleton = _GstPlayer()
def GstPlayer(): return _singleton

