#ifndef PARSER_H
#define PARSER_H

/*
  MathJinni - A simple formular calculator
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <map>
#include <string>
#include <vector>

#include <math.h>

class Parser
{
public:
  // Erlaubte Operatoren
  enum TokenType {
    FINISHED,
    tokenError,
    OP4,
    OP3,
    OP2,
    OP1,
    LBRAC,
    RBRAC,
    PLUS,
    MINUS,
    NUMBER
  };

  // Error codes
  enum ReturnCode {
    OK,
    NODIGIT,
    NORBRAC,
    WRONGBRAC,
    UNKNOWNIDENT,
    ILLGVAL,
    UNKNOWN,
    NOOP,
    DIVBYZERO,
    NEGVAL
  };

public:
  class Stack
  {
  public:
    int        pos;
    ReturnCode error;
    double     stack[1000];

  public:
    Stack();
  };

  class Data
  {
  public:
    virtual ~Data() {};
  };

  class Variable : public Data
  {
  public:
    double value;

  public:
    Variable(double value);
  };

  typedef void (*MathFunct) (Stack& stack, Variable* variable);

  class Function : public Data
  {
  public:
    MathFunct function;
    TokenType type;

  public:
    Function(MathFunct function, TokenType type);
  };

  class Operation
  {
  public:
    MathFunct function;
    Variable  *data;
  };

public:
  class Expression
  {
  private:
    Stack                  stack;
    std::vector<Operation> code;

  public:
    void AddFunction(Function *function);
    void AddPushVar(Variable* variable);

    bool Calculate(double& value, ReturnCode& result);
  };

private:
  wchar_t                     decimalPoint;
  Function                    *plus;
  Function                    *minus;
  Function                    *negate;
  Function                    *times;
  Function                    *pushVar;

  std::map<std::string,Data*> tokens;

private:
  Variable* MakeNumber(double value);

  Data* GetToken(const std::string& name);

  TokenType GetToken(const std::string& string, size_t& pos,
                     Parser::Data*& data, Parser::ReturnCode& result);

void ParseFactor(const std::string& string,
                 size_t& pos,
                 Data*& data,
                 ReturnCode& result,
                 Expression *expression,
                 TokenType& token);

  void ParseBigTerm(const std::string& string,
                    size_t& pos,
                    Data*& data,
                    ReturnCode& result,
                    Expression *expression,
                    TokenType& token);

  void ParseTerm(const std::string& string,
                 size_t& pos,
                 Data*& data,
                 ReturnCode& result,
                 Expression *expression,
                 TokenType& token);

  void ParseExpression(const std::string& string,
                       size_t& pos,
                       Data*& data,
                       ReturnCode& result,
                       Expression *expression,
                       TokenType& token);

public:
  Parser();
  ~Parser();

  void AddConstant(const std::string& name, double value);

  Variable* AddVariable(const std::string& name, double value);
  Variable* GetVariable(const std::string& name);

  void AddFunction(const std::string& name, MathFunct function, TokenType type);

  Expression* Parse(const std::string& string, ReturnCode& result, size_t& pos);
};

#endif
