/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Plot.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/TextValue.h>
#include <Lum/WindowGroup.h>

#include "FunctionInput.h"
#include "NumberInput.h"
#include "Util.h"

static Plot::Prefs* prefs=new Plot::Prefs();

class FunctionDialog : public Lum::Dialog
{
private:
  Parser                &parser;
  Lum::Model::StringRef function;
  Lum::String           *functionControl;
  Lum::Model::ActionRef okAction;
  Parser::Expression*   expression;

public:
  FunctionDialog(Parser& parser, Lum::Model::String* function);
  ~FunctionDialog();

  void PreInit();

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg);

  std::wstring GetFunction() const;
  Parser::Expression* GetExpression() const;
};

FunctionDialog::FunctionDialog(Parser& parser, Lum::Model::String* function)
 : parser(parser),
   function(function),
   functionControl(NULL),
   okAction(new Lum::Model::Action()),
   expression(NULL)
{
  AttachModel(okAction);

  AttachModel(GetClosedAction());
}

FunctionDialog::~FunctionDialog()
{
  UnattachModel(GetClosedAction());
  UnattachModel(okAction);
}

void FunctionDialog::PreInit()
{
  FunctionInput    *input;
  Lum::Button      *button;
  Lum::ButtonRow   *buttonRow;
  Lum::Label       *label;
  Lum::Panel       *panel;
  Lum::WindowGroup *container;

  panel=new Lum::VPanel();
  panel->SetFlex(true,true);

  label=new Lum::Label();
  label->SetFlex(true,true);

  functionControl=new Lum::String();
  functionControl->SetFlex(true,false);
  functionControl->SetRequestsKeyboard(false);
  functionControl->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
  functionControl->RequestFocus();
  functionControl->SetModel(function);
  label->AddLabel(L"Function:",functionControl);

  panel->Add(label);

  input=new FunctionInput();
  input->SetFlex(true,true);
  input->SetString(functionControl);
  panel->Add(input);

  panel->Add(new Lum::VSpace());

  buttonRow=new Lum::ButtonRow();
  buttonRow->SetFlex(true,false);

  button=new Lum::Button(_ld(dlgButtonOk));
  button->RequestFocus();
  button->SetFlex(true,true);
  button->SetType(Lum::Button::typeCommit);
  button->SetModel(okAction);
  buttonRow->Add(button);

  button=new Lum::Button(_ld(dlgButtonCancel));
  button->RequestFocus();
  button->SetFlex(true,true);
  button->SetType(Lum::Button::typeCancel);
  button->SetModel(GetClosedAction());
  buttonRow->Add(button);
  panel->Add(buttonRow);

  container=new Lum::WindowGroup();
  container->SetMain(panel);

  SetTop(container);

  Lum::Dialog::PreInit();
}

void FunctionDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    delete expression;
    expression=NULL;
  }
  else if (model==okAction && okAction->IsFinished()) {
    Parser::ReturnCode result;
    size_t             pos;

    delete expression;

    expression=parser.Parse(Lum::Base::WStringToString(function->Get()),result,pos);

    if (result!=Parser::OK) {
      functionControl->SetCursorPos(pos);
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Parse error!",GetErrorText(result));

      delete expression;
      expression=NULL;
    }
    else {
      Exit();
    }
  }

  Lum::Dialog::Resync(model,msg);
}

std::wstring FunctionDialog::GetFunction() const
{
  return function->Get();
}

Parser::Expression* FunctionDialog::GetExpression() const
{
  return expression;
}

class FunctionsDialog : public Lum::Dialog
{
private:
  Parser                             &parser;
  FunctionsRef                       functions;
  Lum::Model::SingleLineSelectionRef selection;

  Lum::Model::ActionRef              addAction;
  Lum::Model::ActionRef              updateAction;
  Lum::Model::ActionRef              removeAction;

private:
  void AddFunction();

public:
  FunctionsDialog(Parser &parser, Functions* functions);
  ~FunctionsDialog();

  void PreInit();

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg);
};

FunctionsDialog::FunctionsDialog(Parser &parser, Functions* functions)
 : parser(parser),
   functions(functions),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   updateAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action())
{
  updateAction->Disable();
  removeAction->Disable();

  AttachModel(selection);

  AttachModel(addAction);
  AttachModel(updateAction);
  AttachModel(removeAction);

  AttachModel(GetClosedAction());
}

FunctionsDialog::~FunctionsDialog()
{
  UnattachModel(GetClosedAction());

  UnattachModel(removeAction);
  UnattachModel(updateAction);
  UnattachModel(addAction);

  UnattachModel(selection);
}

void FunctionsDialog::PreInit()
{
  Lum::Button           *button;
  Lum::ButtonRow        *buttonRow;
  Lum::Panel            *panel,*horiz,*vert;
  Lum::Table            *table;
  Lum::WindowGroup      *container;
  Lum::Model::HeaderRef headerModel;

  panel=new Lum::VPanel();
  panel->SetFlex(true,true);

  horiz=new Lum::HPanel();
  horiz->SetFlex(true,true);

  vert=new Lum::VPanel();
  vert->SetFlex(false,true);

  button=new Lum::Button(L"_Add");
  button->RequestFocus();
  button->SetFlex(true,false);
  button->SetModel(addAction);
  vert->Add(button);

  button=new Lum::Button(L"_Modify");
  button->RequestFocus();
  button->SetFlex(true,false);
  button->SetModel(updateAction);
  vert->Add(button);

  button=new Lum::Button(L"_Remove");
  button->RequestFocus();
  button->SetFlex(true,false);
  button->SetModel(removeAction);
  vert->Add(button);

  horiz->Add(vert);

  horiz->Add(new Lum::HSpace());

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Function",Lum::Base::Size::modePixel,32000);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->RequestFocus();
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,40);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->SetShowHeader(true);
  //table->GetTableView()->SetAutoHFit(true);
  table->SetModel(functions);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  horiz->Add(table);

  panel->Add(horiz);

  panel->Add(new Lum::VSpace());

  buttonRow=new Lum::ButtonRow();
  buttonRow->SetFlex(true,false);

  button=new Lum::Button(_ld(dlgButtonClose));
  button->RequestFocus();
  button->SetFlex(true,true);
  button->SetType(Lum::Button::typeCommit);
  button->SetModel(GetClosedAction());
  buttonRow->Add(button);

  panel->Add(buttonRow);

  container=new Lum::WindowGroup();
  container->SetMain(panel);

  SetTop(container);

  Dialog::PreInit();
}

void FunctionsDialog::AddFunction()
{
  FunctionDialog        *dialog;
  Lum::Model::StringRef function=new Lum::Model::String(L"");

  dialog=new FunctionDialog(parser,function);
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->GetExpression()!=NULL) {
    functions->AddFunction(dialog->GetFunction(),dialog->GetExpression());
  }

  delete dialog;
}
void FunctionsDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    //Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      updateAction->Enable();
      removeAction->Enable();
    }
    else {
      updateAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    AddFunction();
  }
  else if (model==updateAction && updateAction->IsFinished()) {
    if (selection->HasSelection()) {
      FunctionDialog        *dialog;
      Lum::Model::StringRef function=new Lum::Model::String(functions->GetString(1,selection->GetLine()));

      dialog=new FunctionDialog(parser,function);
      dialog->SetParent(GetWindow());

      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (dialog->GetExpression()!=NULL) {
        functions->SetFunction(selection->GetLine(),dialog->GetFunction(),dialog->GetExpression());
      }

      delete dialog;
    }
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    if (selection->HasSelection()) {
      functions->RemoveFunction(selection->GetLine());
    }
  }

  Dialog::Resync(model,msg);
}

class RegionDialog : public Lum::Dialog
{
private:
  Lum::Model::StringRef topValue;
  Lum::Model::StringRef bottomValue;
  Lum::Model::StringRef leftValue;
  Lum::Model::StringRef rightValue;

  double                top;
  double                bottom;
  double                left;
  double                right;

  Lum::Model::ActionRef okAction;

  bool                  success;

public:
  RegionDialog(double top,
               double bottom,
               double left,
               double right)
    : topValue(new Lum::Model::String(DoubleToWStringExp(top))),
      bottomValue(new Lum::Model::String(DoubleToWStringExp(bottom))),
      leftValue(new Lum::Model::String(DoubleToWStringExp(left))),
      rightValue(new Lum::Model::String(DoubleToWStringExp(right))),
      top(top),
      bottom(bottom),
      left(left),
      right(right),
      okAction(new Lum::Model::Action()),
      success(false)
  {
    AttachModel(GetClosedAction());
    AttachModel(okAction);
  }

  ~RegionDialog()
  {
    UnattachModel(okAction);
    UnattachModel(GetClosedAction());
  }

  bool GetInterval(double &top, double &bottom, double &left, double &right)
  {
    if (sscanf(Lum::Base::WStringToString(topValue->Get()).c_str(),"%lf",&top)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the top border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(bottomValue->Get()).c_str(),"%lf",&bottom)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the bottom border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(leftValue->Get()).c_str(),"%lf",&left)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the left border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(rightValue->Get()).c_str(),"%lf",&right)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the right border of the diagram is not valid!");
      return false;
    }

    if (left>=right) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"Left border value is bigger than right border value!");
      return false;
    }

    if (bottom>=top) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"Bottom border value is bigger than top border value!");
      return false;
    }

    return true;
  }

  void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *row;
    Lum::Panel       *vert,*horiz;
    Lum::String      *string;
    Lum::WindowGroup *wGroup;
    NumberInput      *numberInput;

    numberInput=new NumberInput();

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    vert=new Lum::VPanel();
    vert->SetFlex(true,true);

    // Top

    horiz=new Lum::HPanel();
    horiz->SetFlex(true,false);

    horiz->Add(new Lum::HSpace(true));

    string=new Lum::String();
    string->RequestFocus();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetModel(topValue);
    numberInput->AddString(string);
    horiz->Add(string);

    horiz->Add(new Lum::HSpace(true));

    vert->Add(horiz);

    vert->Add(new Lum::VSpace());

    // Left & right

    horiz=new Lum::HPanel();
    horiz->SetFlex(true,false);

    string=new Lum::String();
    string->RequestFocus();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetModel(leftValue);
    numberInput->AddString(string);
    horiz->Add(string);

    horiz->Add(new Lum::HSpace());

    string=new Lum::String();
    string->RequestFocus();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetModel(rightValue);
    numberInput->AddString(string);
    horiz->Add(string);

    vert->Add(horiz);

    vert->Add(new Lum::VSpace());

    // Bottom

    horiz=new Lum::HPanel();
    horiz->SetFlex(true,false);

    horiz->Add(new Lum::HSpace(true));

    string=new Lum::String();
    string->RequestFocus();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetModel(bottomValue);
    numberInput->AddString(string);
    horiz->Add(string);

    horiz->Add(new Lum::HSpace(true));

    vert->Add(horiz);

    vert->Add(new Lum::VSpace());

    vert->Add(numberInput);

    vert->Add(new Lum::VSpace());

    row=new Lum::ButtonRow();
    row->SetFlex(true,false);

    button=new Lum::Button(_ld(dlgButtonOk));
    button->RequestFocus();
    button->SetFlex(true,true);
    button->SetType(Lum::Button::typeCommit);
    button->SetModel(okAction);
    row->Add(button);

    button=new Lum::Button(_ld(dlgButtonCancel));
    button->RequestFocus();
    button->SetFlex(true,true);
    button->SetType(Lum::Button::typeCancel);
    button->SetModel(GetClosedAction());
    row->Add(button);

    vert->Add(row);

    wGroup->SetMain(vert);
    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      //Exit();
    }
    else if (model==okAction && okAction->IsFinished()) {
      if (GetInterval(top,bottom,left,right)) {
        success=true;
        Exit();
      }
    }

    Dialog::Resync(model,msg);
  }

  bool HasSuccess()
  {
    return success;
  }

  void GetIntervalValues(double& top, double& bottom, double& left, double& right)
  {
    top=this->top;
    bottom=this->bottom;
    left=this->left;
    right=this->right;
  }
};

Plot::Prefs::Prefs()
 : Lum::Component::Prefs(L"Plot")
{
  // no code
}

void Plot::Prefs::Initialize()
{
  Lum::Component::Prefs::Initialize();

  frame=new Lum::OS::EmptyFrame();
}

Plot::Plot()
: history(new Lum::Model::ListTable(1)),
  editFunctions(new Lum::Model::Action()),
  autosize(new Lum::Model::Action()),
  region(new Lum::Model::Action())
 {
  SetPrefs(::prefs);

  plotVar=parser.AddVariable("x",0.0);

  functions=new Functions(plotVar);

  AttachModel(editFunctions);
  AttachModel(autosize);
  AttachModel(region);
}

Plot::~Plot()
{
  UnattachModel(region);
  UnattachModel(autosize);
  UnattachModel(editFunctions);
}

void Plot::EditFunctions()
{
  FunctionsDialog *dialog;

  dialog=new FunctionsDialog(parser,functions);
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  delete dialog;
}

void Plot::Autosize()
{
  plotter->VAutosize();
}

void Plot::SetRegion()
{
  RegionDialog *dialog;
  double       top,bottom,left,right;

  dialog=new RegionDialog(functions->GetTop(),functions->GetBottom(),
                          functions->GetLeft(),functions->GetRight());
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  if (!dialog->HasSuccess()) {
    delete dialog;
    return;
  }

  dialog->GetIntervalValues(top,bottom,left,right);

  functions->SetRegion(top,bottom,left,right);

  delete dialog;
}

void Plot::CalcSize()
{
  Lum::Button    *button;
  Lum::ButtonRow *row;
  Lum::Panel     *panel;

  panel=new Lum::VPanel();
  panel->SetFlex(true,true);

  plotter=new Plotter();
  plotter->SetFlex(true,true);
  plotter->SetModel(functions);

  panel->Add(plotter);

  panel->Add(new Lum::VSpace());

  row=new Lum::ButtonRow();
  row->SetFlex(true,false);

  button=new Lum::Button();
  button->SetFlex(true,false);
  button->RequestFocus();
  button->SetModel(editFunctions);
  button->SetText(L"_f(x)");
  row->Add(button);

  button=new Lum::Button();
  button->SetFlex(true,false);
  button->RequestFocus();
  button->SetModel(autosize);
  button->SetText(L"_Autosize");
  row->Add(button);

  button=new Lum::Button();
  button->SetFlex(true,false);
  button->RequestFocus();
  button->SetModel(region);
  button->SetText(L"_Region");
  row->Add(button);

  panel->Add(row);

  container=panel;

  container->SetParent(this);
  container->CalcSize();

  minWidth=container->GetOMinWidth();
  minHeight=container->GetOMinHeight();
  width=container->GetOWidth();
  height=container->GetOHeight();

  Component::CalcSize();
}

void Plot::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==editFunctions && editFunctions->IsFinished()) {
    EditFunctions();
  }
  else if (model==autosize && autosize->IsFinished()) {
    Autosize();
  }
  else if (model==region && region->IsFinished()) {
    SetRegion();
  }

  Lum::Component::Resync(model,msg);
}

void Plot::LoadConfig(Lum::Config::Node *top)
{
  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    Lum::Config::Node *node=*iter;

    if (node->GetName()==L"function") {
      std::wstring function;

      if (node->GetAttribute(L"function",function)) {
        Parser::Expression* expression;
        Parser::ReturnCode  result;
        size_t              pos;

        expression=parser.Parse(Lum::Base::WStringToString(function),result,pos);

        if (result!=Parser::OK || expression==NULL) {
          delete expression;
        }
        else {
          functions->AddFunction(function,expression);
        }
      }
    }
  }
}

void Plot::StoreConfig(Lum::Config::Node *top)
{
  if (functions.Valid()) {
    for (size_t row=1; row<=functions->GetRows(); row++) {
      Lum::Config::Node *node;

      node=new Lum::Config::Node();
      node->SetName(L"function");
      node->SetAttribute(L"function",functions->GetString(1,row));

      top->Add(node);
    }
  }
}

