/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FormatConversion.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Array.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>

static FormatConversion::Prefs* prefs=new FormatConversion::Prefs();

static const wchar_t *digitString[16] = {
                                         L"0",
                                         L"1",
                                         L"2",
                                         L"3",
                                         L"4",
                                         L"5",
                                         L"6",
                                         L"7",
                                         L"8",
                                         L"9",
                                         L"A",
                                         L"B",
                                         L"C",
                                         L"D",
                                         L"E",
                                         L"F",
                                       };

static Lum::Button* createTextButton(const std::wstring& text, Lum::Base::Model *model)
{
  Lum::Button *button;

  button=new Lum::Button(text);
  button->SetFlex(true,true);
  button->SetModel(model);

  return button;
}

static Lum::String* CreateString()
{
  Lum::String *string;

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetRequestsKeyboard(false);
  string->RequestFocus();
  string->SetAlignment(Lum::String::right);

  return string;
}
FormatConversion::Prefs::Prefs()
 : Lum::Component::Prefs(L"FormatConversion")
{
  // no code
}

void FormatConversion::Prefs::Initialize()
{
  Lum::Component::Prefs::Initialize();

  frame=new Lum::OS::EmptyFrame();
}

FormatConversion::FormatConversion()
: binFormat(new Lum::Model::String(L"")),
  octFormat(new Lum::Model::String(L"")),
  decFormat(new Lum::Model::String(L"")),
  hexFormat(new Lum::Model::String(L"")),
  formatConversion(false),
  binFocusIn(new Lum::Model::Action()),
  octFocusIn(new Lum::Model::Action()),
  decFocusIn(new Lum::Model::Action()),
  hexFocusIn(new Lum::Model::Action()),
  focusOut(new Lum::Model::Action()),
  inputTarget(targetNone)
{
  SetPrefs(::prefs);

  AttachModel(binFormat);
  AttachModel(octFormat);
  AttachModel(decFormat);
  AttachModel(hexFormat);
  AttachModel(binFocusIn);
  AttachModel(octFocusIn);
  AttachModel(decFocusIn);
  AttachModel(hexFocusIn);
  AttachModel(focusOut);

  for (size_t i=0; i<17; i++) {
    digitsPressed[i]=new Lum::Model::Action();
    digitsPressed[i]->Disable();
    AttachModel(digitsPressed[i]);
  }
}

FormatConversion::~FormatConversion()
{
  for (size_t i=0; i<17; i++) {
    UnattachModel(digitsPressed[i]);
  }

  UnattachModel(focusOut);
  UnattachModel(binFocusIn);
  UnattachModel(octFocusIn);
  UnattachModel(decFocusIn);
  UnattachModel(hexFocusIn);
  UnattachModel(hexFormat);
  UnattachModel(decFormat);
  UnattachModel(octFormat);
  UnattachModel(binFormat);
}

void FormatConversion::ConvertFromBin()
{
  unsigned long value;

  if (binFormat->Empty() || !Lum::Base::WStringToNumber(binFormat->Get(),value,2)) {
    octFormat->Set(L"");
    decFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromOct()
{
  unsigned long value;

  if (octFormat->Empty() || !Lum::Base::WStringToNumber(octFormat->Get(),value,8)) {
    binFormat->Set(L"");
    decFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromDec()
{
  unsigned long value;

  if (decFormat->Empty() || !Lum::Base::WStringToNumber(decFormat->Get(),value,10)) {
    binFormat->Set(L"");
    octFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromHex()
{
  unsigned long value;

  if (hexFormat->Empty() || !Lum::Base::WStringToNumber(hexFormat->Get(),value,16)) {
    binFormat->Set(L"");
    octFormat->Set(L"");
    decFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  formatConversion=false;
}


void FormatConversion::CalcSize()
{
  Lum::Array  *array;
  Lum::Label  *label;
  Lum::Panel  *panel;

  panel=new Lum::VPanel();
  panel->SetFlex(true,true);

  label=new Lum::Label();
  label->SetFlex(true,false);

  bin=CreateString();
  bin->SetModel(binFormat);
  bin->SetFocusInAction(binFocusIn);
  bin->SetFocusOutAction(focusOut);
  label->AddLabel(L"Binary",bin);

  oct=CreateString();
  oct->SetModel(octFormat);
  oct->SetFocusInAction(octFocusIn);
  oct->SetFocusOutAction(focusOut);
  label->AddLabel(L"Octal",oct);

  dec=CreateString();
  dec->SetModel(decFormat);
  dec->SetFocusInAction(decFocusIn);
  dec->SetFocusOutAction(focusOut);
  label->AddLabel(L"Decimal",dec);

  hex=CreateString();
  hex->SetModel(hexFormat);
  hex->SetFocusInAction(hexFocusIn);
  hex->SetFocusOutAction(focusOut);
  label->AddLabel(L"Hex",hex);

  panel->Add(label);

  panel->Add(new Lum::VSpace());

  array=new Lum::Array();
  array->SetFlex(true,false);
  array->SetHorizontalCount(10);
  array->SetSpace(true,true);

  array->Add(createTextButton(L"0",digitsPressed[0]));
  array->Add(createTextButton(L"1",digitsPressed[1]));
  array->Add(createTextButton(L"2",digitsPressed[2]));
  array->Add(createTextButton(L"3",digitsPressed[3]));
  array->Add(createTextButton(L"4",digitsPressed[4]));
  array->Add(createTextButton(L"5",digitsPressed[5]));
  array->Add(createTextButton(L"6",digitsPressed[6]));
  array->Add(createTextButton(L"7",digitsPressed[7]));
  array->Add(createTextButton(L"8",digitsPressed[8]));
  array->Add(createTextButton(L"9",digitsPressed[9]));
  array->Add(createTextButton(L"A",digitsPressed[10]));
  array->Add(createTextButton(L"B",digitsPressed[11]));
  array->Add(createTextButton(L"C",digitsPressed[12]));
  array->Add(createTextButton(L"D",digitsPressed[14]));
  array->Add(createTextButton(L"E",digitsPressed[14]));
  array->Add(createTextButton(L"F",digitsPressed[15]));
  array->Add(new Lum::VSpace());
  array->Add(new Lum::VSpace());
  array->Add(new Lum::VSpace());
  array->Add(createTextButton(L"<-",digitsPressed[16]));

  panel->Add(array);

  panel->Add(new Lum::VSpace(Lum::Space::sizeEmpty,true));

  container=panel;

  container->SetParent(this);
  container->CalcSize();

  minWidth=container->GetOMinWidth();
  minHeight=container->GetOMinHeight();
  width=container->GetOWidth();
  height=container->GetOHeight();

  Component::CalcSize();
}

void FormatConversion::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==binFormat && !formatConversion) {
    ConvertFromBin();
  }
  else if (model==octFormat && !formatConversion) {
    ConvertFromOct();
  }
  else if (model==decFormat && !formatConversion) {
    ConvertFromDec();
  }
  else if (model==hexFormat && !formatConversion) {
    ConvertFromHex();
  }
  else if (model==binFocusIn && binFocusIn->IsFinished()) {
    inputTarget=targetBin;
    for (size_t i=0; i<=1; i++) {
      digitsPressed[i]->Enable();
    }
    digitsPressed[16]->Enable();
  }
  else if (model==octFocusIn && octFocusIn->IsFinished()) {
    inputTarget=targetOct;
    for (size_t i=0; i<=7; i++) {
      digitsPressed[i]->Enable();
    }
    digitsPressed[16]->Enable();
  }
  else if (model==decFocusIn && decFocusIn->IsFinished()) {
    inputTarget=targetDec;
    for (size_t i=0; i<=9; i++) {
      digitsPressed[i]->Enable();
    }
    digitsPressed[16]->Enable();
  }
  else if (model==hexFocusIn && hexFocusIn->IsFinished()) {
    inputTarget=targetHex;
    for (size_t i=0; i<=16; i++) {
      digitsPressed[i]->Enable();
    }
    digitsPressed[16]->Enable();
  }
  else if (model==focusOut && focusOut->IsFinished()) {
    inputTarget=targetNone;
    for (size_t i=0; i<17; i++) {
      digitsPressed[i]->Disable();
    }
  }
  else if (model==digitsPressed[16] && digitsPressed[16]->IsFinished()) {
    switch (inputTarget) {
    case targetNone:
      break;
    case targetBin:
      bin->ActionBackspace();
      break;
    case targetOct:
      oct->ActionBackspace();
      break;
    case targetDec:
      dec->ActionBackspace();
      break;
    case targetHex:
      hex->ActionBackspace();
      break;
    }
  }
  else {
    for (size_t i=0; i<16; i++) {
      if (model==digitsPressed[i] && digitsPressed[i]->IsFinished()) {
        switch (inputTarget) {
        case targetNone:
          break;
        case targetBin:
          bin->ActionInsertString(digitString[i]);
          break;
        case targetOct:
          oct->ActionInsertString(digitString[i]);
          break;
        case targetDec:
          dec->ActionInsertString(digitString[i]);
          break;
        case targetHex:
          hex->ActionInsertString(digitString[i]);
          break;
        }
      }
    }
  }

  Lum::Component::Resync(model,msg);
}
