/*
 * This file is part of maptile-loader
 * Copyright (C) 2007  Pekka Rönkkö (pronkko@gmail.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <config.h>

#include "maptile-repo.h"


G_DEFINE_TYPE (MaptileRepo, maptile_repo, G_TYPE_OBJECT);

static void      maptile_repo_get_property  (GObject      *object,
                                           guint         param_id,
                                           GValue       *value,
                                           GParamSpec   *pspec);
static void      maptile_repo_set_property  (GObject      *object,
                                           guint         param_id,
                                           const GValue *value,
                                           GParamSpec   *pspec);
enum {
  PROP_0,
  PROP_MAP_NAME,
  PROP_MAP_DATA_URI,
  PROP_MAP_CACHE_URI,
/*  PROP_TILE_AREA_WIDTH,
  PROP_TILE_AREA_HEIGHT,
  PROP_TILE_IMAGE_WIDTH,
  PROP_TILE_IMAGE_HEIGHT,
  PROP_ORIENTATION_SUPPORT,
  PROP_DEFAULT_ORIENTATION,
*/  PROP_CON_IC_NEEDED
};

static void
maptile_repo_class_init (MaptileRepoClass *class)
{
  GObjectClass     *object_class      = G_OBJECT_CLASS (class);

  object_class->get_property = maptile_repo_get_property;
  object_class->set_property = maptile_repo_set_property;


  g_object_class_install_property (object_class, PROP_MAP_NAME,
                                   g_param_spec_string ("map-name",
                                                        "Map name",
                                                        "The name of the map",
                                                        NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_MAP_DATA_URI,
                                   g_param_spec_string ("map-data-uri",
                                                        "Map data URI",
                                                        "The URI of the map server",
                                                        NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_MAP_CACHE_URI,
                                   g_param_spec_string ("map-cache-uri",
                                                        "Map cache URI",
                                                        "The URI of the map local cache directory",
                                                        NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));
/*
  g_object_class_install_property (object_class, PROP_TILE_AREA_WIDTH,
                                   g_param_spec_uint ("tile-area-width",
                                                        "Map tile area width",
                                                        "Map tile area width in pixels",
                                                        0,1024, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_TILE_AREA_HEIGHT,
                                   g_param_spec_uint ("tile-area-height",
                                                        "Map tile area height",
                                                        "Map tile area height in pixels",
                                                        0,1024, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_TILE_IMAGE_WIDTH,
                                   g_param_spec_uint ("tile-image-width",
                                                        "Map tile image width",
                                                        "Map tile image width in pixels",
                                                        0,1024, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_TILE_IMAGE_HEIGHT,
                                   g_param_spec_uint ("tile-image-height",
                                                        "Map tile image height",
                                                        "Map tile image height in pixels",
                                                        0,1024, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_ORIENTATION_SUPPORT,
                                   g_param_spec_boolean ("orientation-support",
                                                        "Map tile orientation support",
                                                        "Does map support orientation",
                                                        FALSE,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_DEFAULT_ORIENTATION,
                                   g_param_spec_float ("default-orientation",
                                                        "Map tile default orientation",
                                                        "Map tile default orientation",
                                                        0.0, 360.0, 0.0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));
*/
  g_object_class_install_property (object_class, PROP_CON_IC_NEEDED,
                                   g_param_spec_boolean ("con-ic-needed",
                                                        "Map tile internet connection needed",
                                                        "Is internet connection needed to download map data",
                                                        TRUE,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

}

static void
maptile_repo_init (MaptileRepo *repo)
{
  repo->map_name = NULL;
  repo->map_data_uri = NULL;
  repo->map_cache_uri = NULL;
/*
  repo->tile_area_width = 0;
  repo->tile_area_height = 0;
  repo->tile_image_width = 0;
  repo->tile_image_height = 0;

  repo->orientation_support = FALSE;
  repo->default_orientation = 0.0;
*/
  repo->con_ic_needed = FALSE;
}

MaptileRepo *
maptile_repo_new (GType type)
{
  g_return_val_if_fail (g_type_is_a (type, MAPTILE_TYPE_REPO), NULL);

  return g_object_new (type, NULL);
}

static void
maptile_repo_get_property  (GObject      *object,
                                          guint         param_id,
                                          GValue       *value,
                                          GParamSpec   *pspec)
{
  MaptileRepo *repo = MAPTILE_REPO (object);

  switch (param_id)
    {
    case PROP_MAP_NAME:
      g_value_set_string (value, repo->map_name);
      break;
    case PROP_MAP_DATA_URI:
      g_value_set_string (value, repo->map_data_uri);
      break;
    case PROP_MAP_CACHE_URI:
      g_value_set_string (value, repo->map_cache_uri);
      break;
/*    case PROP_TILE_AREA_WIDTH:
      g_value_set_uint (value, repo->tile_area_width);
      break;
    case PROP_TILE_AREA_HEIGHT:
      g_value_set_uint (value, repo->tile_area_height);
      break;
    case PROP_TILE_IMAGE_WIDTH:
      g_value_set_uint (value, repo->tile_image_width);
      break;
    case PROP_TILE_IMAGE_HEIGHT:
      g_value_set_uint (value, repo->tile_image_height);
      break;
    case PROP_ORIENTATION_SUPPORT:
      g_value_set_boolean (value, repo->orientation_support);
      break;
    case PROP_DEFAULT_ORIENTATION:
      g_value_set_float (value, repo->default_orientation);
      break;
*/    case PROP_CON_IC_NEEDED:
      g_value_set_boolean (value, repo->con_ic_needed);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

static void
maptile_repo_set_property  (GObject      *object,
                                          guint         param_id,
                                          const GValue *value,
                                          GParamSpec   *pspec)
{
  MaptileRepo *repo = MAPTILE_REPO (object);

  switch (param_id)
    {
    case PROP_MAP_NAME:
      if ( repo->map_name )
        g_free (repo->map_name);
      repo->map_name = g_value_dup_string (value);
      break;
    case PROP_MAP_DATA_URI:
      if ( repo->map_data_uri )
        g_free (repo->map_data_uri);
      repo->map_data_uri = g_value_dup_string (value);
      break;
    case PROP_MAP_CACHE_URI:
      if (repo->map_cache_uri)
        g_free (repo->map_cache_uri);
      repo->map_cache_uri = g_value_dup_string (value);
      break;
/*    case PROP_TILE_AREA_WIDTH:
      repo->tile_area_width = g_value_get_uint(value);
      break;
    case PROP_TILE_AREA_HEIGHT:
      repo->tile_area_height = g_value_get_uint(value);
      break;
    case PROP_TILE_IMAGE_WIDTH:
      repo->tile_image_width = g_value_get_uint(value);
      break;
    case PROP_TILE_IMAGE_HEIGHT:
      repo->tile_image_height = g_value_get_uint(value);
      break;
    case PROP_ORIENTATION_SUPPORT:
      repo->orientation_support = g_value_get_boolean(value);
      break;
    case PROP_DEFAULT_ORIENTATION:
      repo->default_orientation = g_value_get_float(value);
      break;
*/    case PROP_CON_IC_NEEDED:
      repo->con_ic_needed = g_value_get_boolean(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
      break;
    }
}

void
maptile_repo_get_tile (MaptileRepo     *repo,
                        guint tilex,
                        guint tiley,
                        guint zoom,
                        gchar **tile)
{
  g_return_if_fail (MAPTILE_IS_REPO (repo));

  if (MAPTILE_REPO_GET_CLASS (repo)->get_tile)
    MAPTILE_REPO_GET_CLASS (repo)->get_tile (repo, tilex, tiley, zoom, tile);
  else
    g_warning ("%s: repo class %s doesn't implement MaptileRepo::get_tile()\n",
               G_STRFUNC, g_type_name (G_TYPE_FROM_INSTANCE (repo)));
}

