/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdlib.h>
#include <ctype.h>

#include "obd-chan.h"
#include "obd-chan-sim.h"
#include "obd-chan-file.h"
#include "obd-chan-elm.h"
#include "obd-chan-ozen.h"
#include "obd-log.h"

int obdchan_init(struct obdchan* chan)
{
	if(chan == NULL)
		return OBDCHANERR_INVALID_ARGS;

	chan->con = NULL;
	chan->type = OBDCHAN_UNDEFINED;
	chan->name = NULL;
	chan->sendpid = NULL;
	chan->senddtc = NULL;
	chan->close = NULL;
	return 0;
}

int obdchan_name(struct obdchan* chan, char* buffer, unsigned int len)
{
	if(chan == NULL || buffer == NULL || len == 0)
		return OBDCHANERR_INVALID_ARGS;

	if(chan->name == NULL)
		return OBDCHANERR_NOT_SETUP;

	return chan->name(buffer, len);
}

int obdchan_setup_simulator(struct obdchan* chan)
{
	if(chan == NULL) {
		LOG_ERROR("Channel is null?");
		return OBDCHANERR_INVALID_ARGS;
	}

	LOG_INFO("Using OBD Simulator");

	return obdchansim_setup(chan);
}

int obdchan_setup_file(struct obdchan* chan, char* filename)
{
	if(chan == NULL) {
		LOG_ERROR("Channel is null?");
		return OBDCHANERR_INVALID_ARGS;
	}

	LOG_INFO("Using OBD File");

	return obdchanfile_setup(chan, filename);
}

int obdchan_setup_channel(struct obdchan* chan, struct obdcon* con)
{
	int ret;

	if(chan == NULL) {
		LOG_ERROR("Channel is null?");
		return OBDCHANERR_INVALID_ARGS;
	}

	LOG_INFO("Trying ELM ...");
	if((ret = obdchanelm_setup(chan, con)) == 0) {
		LOG_INFO("ELM setup operation succeed!");
		return 0;
	}
	LOG_INFO("ELM communication failed!");

	if(ret == OBDCHANERR_SETUP_ERROR)
		return ret;

	LOG_INFO("Trying Ozen ...");

	if((ret = obdchanozen_setup(chan, con)) == 0) {
		LOG_INFO("Ozen setup operation succeed!");
		return 0;
	}
	LOG_INFO("Ozen communication failed!");
	return ret;
}

int obdchan_sendpid(struct obdchan* chan, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	if(chan == NULL || sensor_data == NULL || 
			!(sensor_data->pid >= 0x00 && sensor_data->pid < 0xFF)) {
		LOG_ERROR("Invalid parameter");
		return OBDCHANERR_INVALID_ARGS;
	}

	if(chan->sendpid == NULL) {
		LOG_ERROR("Sendpid function pointer is null");
		return OBDCHANERR_NOT_SETUP;
	}

	return chan->sendpid(chan, sensor_data, timeout);
}

int obdchan_senddtc(struct obdchan* chan, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	if(chan == NULL || dtc_data == NULL)
		return OBDCHANERR_INVALID_ARGS;

	if(chan->senddtc == NULL)
		return OBDCHANERR_NOT_SETUP;

	return chan->senddtc(chan, dtc_data, timeout);
}

int obdchan_close(struct obdchan* chan)
{
	if(chan == NULL)
		return OBDCHANERR_INVALID_ARGS;
	
	if(chan->close == NULL)
		return OBDCHANERR_NOT_SETUP;

	return chan->close(chan);
}
