/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE /* for strndup */

#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "obd-conn.h"
#include "obd-log.h"

#define MAX_TIMEOUT 120
#define MAX_DEVNAME_LEN 128
#define POLL_PIPE 0
#define POLL_SERIAL 1
#define PIPE_READ 0
#define PIPE_WRITE 1

static int obdcon_check_devname(const char* devname)
{
	int slen;
	char* pstr;

	slen = strnlen(devname, MAX_DEVNAME_LEN + 1);
	if (slen == MAX_DEVNAME_LEN + 1 || slen == 0)
		return -1;

	pstr = strstr(devname, "/dev/tty");
	if (pstr == devname)
		return 0;

	pstr = strstr(devname, "/dev/rfcomm");
	if (pstr == devname)
		return 0;

	return -1;
}


static int obdcon_open_serial(struct obdcon* con, const char* devname)
{
	con->handle = open(devname, O_RDWR | O_NOCTTY);
	if (con->handle < 0){
		LOG_ERROR("Unable to open serial device (%s) - %s", devname, strerror(errno));
		con->handle = 0;
		return OBDCONERR_ERROR;
	}

	con->devname = strndup(devname, MAX_DEVNAME_LEN);
	if (con->devname == NULL) {
		LOG_ERROR("Unable to alloc memory to hold the device name");
		goto error;
	}

	/* get current port settings and store it */
	tcgetattr(con->handle, &con->oldtio);

	con->newtio.c_iflag = IGNPAR;
	con->newtio.c_cflag = (CS8 | CLOCAL | CREAD);
	con->newtio.c_oflag = 0;
	con->newtio.c_lflag = 0;
	con->newtio.c_line = 0;

	memset(&con->newtio.c_cc, '\x00', NCCS);
	con->newtio.c_cc[VMIN] = 0;
        con->newtio.c_cc[VTIME] = 0;

	cfsetspeed (&con->newtio, con->baudrate);
	tcflush (con->handle, TCIFLUSH);
	tcsetattr (con->handle, TCSANOW, &con->newtio);
	return 0;

error:
	if (con->handle > 0) {
		close(con->handle);
		con->handle = 0;
	}
	if (con->devname != NULL) {
		free(con->devname);
		con->devname = NULL;
	}
	return -1;

}

static int obdcon_close_serial(struct obdcon* con)
{
	tcsetattr(con->handle, TCSANOW, &con->oldtio);
	if (close(con->handle))
		return -1;
	free(con->devname);
	con->devname = NULL;
	con->status = OBDCON_NOT_CONNECTED;
	return 0;
}

int obdcon_init(struct obdcon* con)
{
	if (con == NULL)
		return -1;

	con->devname = NULL;
	con->baudrate = B9600;
	con->handle = 0;
	con->status = OBDCON_NOT_CONNECTED;
	con->closepipe[PIPE_READ] = 0;
	con->closepipe[PIPE_WRITE] = 0;
	con->poll[POLL_PIPE].fd = 0;
	con->poll[POLL_PIPE].events = 0;
	con->poll[POLL_PIPE].revents = 0;
	con->poll[POLL_SERIAL].fd = 0;
	con->poll[POLL_SERIAL].events = 0;
	con->poll[POLL_SERIAL].revents = 0;

	return 0;
}

int obdcon_opendev(struct obdcon* con, const char* devname)
{
	int ret = 0;

	if (con == NULL) {
		ret = OBDCONERR_INVALID_ARGS;
		LOG_ERROR("Connection param is not valid!");
		goto error;
	}

	if (con->handle != 0 || con->devname != NULL) {
		ret = OBDCONERR_ALREADY_CONNECTED;
		LOG_ERROR("Connection already established!");
		goto error;
	}

	if (obdcon_check_devname(devname)) {
		ret = OBDCONERR_INVALID_DEVNAME;
		LOG_ERROR("%s is not valid devname. Expecting /dev/ttyX or "
			"/dev/rfcommX", devname);
		goto error;
	}

	con->status = OBDCON_CONNECTING;

	if (obdcon_open_serial(con, devname)) {
		ret = OBDCONERR_ERROR;
		goto error;
	}

	LOG_INFO("Connected to serial \"%s\"", devname);

	if(pipe(con->closepipe)) {
		LOG_ERROR("Unable to create close pipe - %s", strerror(errno));
		ret = OBDCONERR_ERROR;
		goto error;
	}

	con->poll[POLL_PIPE].fd = con->closepipe[PIPE_READ];
	con->poll[POLL_PIPE].events = POLLIN | POLLPRI;

	con->poll[POLL_SERIAL].fd = con->handle;
	con->poll[POLL_SERIAL].events = POLLIN | POLLPRI | POLLERR | POLLHUP | POLLNVAL;

	con->status = OBDCON_CONNECTED;

	return 0;

error:
	if (con->devname) {
		free(con->devname);
		con->devname = NULL;
	}

	if (con->handle) {
		tcsetattr(con->handle, TCSANOW, &con->oldtio);
		close(con->handle);
		con->handle = 0;
	}

	if (con->closepipe[PIPE_WRITE]) {
		close(con->closepipe[PIPE_WRITE]);
		con->closepipe[PIPE_WRITE] = 0;
	}

	if (con->closepipe[PIPE_READ]) {
		close(con->closepipe[PIPE_READ]);
		con->closepipe[PIPE_READ] = 0;
	}
	con->status = OBDCON_NOT_CONNECTED;

	return ret;
}

int obdcon_closedev(struct obdcon* con)
{
	LOG_DEBUG("Trying to close serial connection");
	if (con == NULL)
		return OBDCONERR_INVALID_ARGS;

	if (con->status == OBDCON_NOT_CONNECTED) {
		LOG_ERROR("Close on a non connected connection");
		return OBDCONERR_NOT_CONNECTED;
	}

	if (obdcon_close_serial(con))
		return OBDCONERR_ERROR;

	write(con->closepipe[PIPE_WRITE], "c", 1);
	close(con->closepipe[PIPE_WRITE]);
	con->closepipe[PIPE_WRITE] = 0;

	/* FIXME: should we close it here? does it give an error in the poll? 
	 * If so, is it enough to close the handle of the serial itself?*/
	close(con->closepipe[PIPE_READ]);
	con->closepipe[PIPE_READ] = 0;

	con->poll[POLL_PIPE].fd = 0;
	con->poll[POLL_PIPE].events = 0;

	con->poll[POLL_SERIAL].fd = 0;
	con->poll[POLL_SERIAL].events = 0;

	LOG_INFO("Serial connection closed");
	return 0;
}

int obdcon_read(struct obdcon* con, unsigned char* buffer, unsigned int size, unsigned int timeout)
{
	int poll_ret;
	int bytes_readed;
	unsigned int total_readed;
	int read_len;

	if (con == NULL) {
		LOG_ERROR("BUG - Null connection pointer");
		return OBDCONERR_INVALID_ARGS;
	}

	if (con->status == OBDCON_NOT_CONNECTED || con->status == OBDCON_ERROR) {
		LOG_ERROR("Trying to read but not connected");
		return OBDCONERR_NOT_CONNECTED;
	}

	if (timeout > MAX_TIMEOUT)
		timeout = MAX_TIMEOUT;
	
	do{
		poll_ret = poll(con->poll, sizeof(con->poll)/sizeof(struct pollfd),
				timeout * 1000);
		if (poll_ret < 0 && errno != EINTR) {
			LOG_ERROR("Poll error - %s", strerror(errno));
			return OBDCONERR_ERROR; /*error*/
		}
	} while (poll_ret < 0);

	if (poll_ret == 0) { /* timeout */
		LOG_ERROR("Read timeout");
		return OBDCONERR_TIMEOUT;
	}

	if (con->poll[POLL_PIPE].revents != 0)
		return 0; /* connection close*/

	if (con->poll[POLL_SERIAL].revents != 0){
		if (con->poll[POLL_SERIAL].revents &
				(POLLERR | POLLNVAL | POLLHUP)) {
			LOG_DEBUG("Serial connection error");
			return OBDCONERR_ERROR; /*connection error */
		}

		if (con->poll[POLL_SERIAL].revents &
				(POLLIN | POLLPRI)){
			if(ioctl(con->handle, FIONREAD, &read_len))
				return OBDCONERR_ERROR;

			if(read_len > size)
				return OBDCONERR_ERROR;

			total_readed = bytes_readed = 0;
			while(total_readed < read_len) {
				bytes_readed = read(con->handle,
						buffer + total_readed,
						read_len - total_readed);
				if (bytes_readed >=0)
					total_readed += bytes_readed;
				else
					if (errno != EINTR)
						return OBDCONERR_ERROR;
			}
			/* LOG_DEBUG("Readed (%d) - %s", total_readed, buffer) */;
			return total_readed;
		}
		return 0;

	}
	return OBDCONERR_ERROR;
}

int obdcon_write(struct obdcon* con, const unsigned char* buffer, unsigned int size)
{
	unsigned int twritten;
	int written;

	if (size == 0)
		return 0;

	if (con == NULL)
		return OBDCONERR_INVALID_ARGS;

	if (con->status == OBDCON_NOT_CONNECTED || con->status == OBDCON_ERROR)
		return -1;

	twritten = written = 0;
	while (twritten < size) {
		written = write(con->handle, buffer + twritten, size - written);
		if (written >= 0)
			twritten += written;
		else {
			if (errno != EINTR) {
				LOG_ERROR("Error writing to serial - %s", strerror(errno));
				return OBDCONERR_ERROR;
			}
		}
	}
	/* LOG_DEBUG("Wrote (%d) - %s",twritten, buffer); */
	return twritten;
}
