/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>

#include "obd-chan-elm.h"
#include "obd-log.h"
#include "obd-sensor.h"

#define DEVICE_TIMEOUT 100
#define BUFFER_LEN 512
#define DATA_LEN (2 * MAX_DTCS + 2)
#define READ_DEADLINE_FACTOR 4

static const char* name = "ELM";

static const char* elm_errors[] = {
	"BUFFER FULL",
	"BUS BUSY",
	"BUS ERROR",
	"CAN ERROR",
	"FB ERROR",
	"DATA ERROR",
	"<RX ERROR",
	"UNABLE TO CONNECT",
	".ERROR"
};

struct obdchanelm {
	unsigned int blen;
	char buffer[BUFFER_LEN];
};

static int obdchanelm_error(struct obdchan* chan)
{
	int i;
	struct obdchanelm *pelm = chan->data;

	for(i=0; i < sizeof(elm_errors)/sizeof(char*); i++)
		if(strstr(pelm->buffer, elm_errors[i])) {
			LOG_ERROR("ELM error: \"%s\"", elm_errors[i]);
			return -1;
		}
	return 0;
}

static int obdchanelm_checkdata(char* data)
{
	int i;
	for(i = 0; i < strlen(data); i++)
		if(data[i] != ' ' && !isxdigit(data[i]))
			return -1;

	return 0;

}

static void obdchanelm_devicetimeout(struct obdchan* chan, unsigned int timeout)
{
	struct obdchanelm *pelm = chan->data;
	if(timeout > 255)
		timeout = 255;

	sprintf(pelm->buffer, "ATST %02X", timeout/4);
	pelm->blen = strlen(pelm->buffer);
}

static int obdchanelm_sendcmd(struct obdchan* chan, unsigned int timeout)
{
	int readed;
	int total_readed;
	struct obdchanelm *pelm = chan->data;
	time_t tread;

	if(pelm->blen >= BUFFER_LEN - 1) {
		LOG_ERROR("Trying to send more than the buffer size (%d)", BUFFER_LEN);
		return OBDCHANERR_ERROR;
	}

	pelm->buffer[pelm->blen] = '\r';
	pelm->buffer[pelm->blen + 1] = '\0';

	pelm->blen++;

	LOG_NOESCAPE(OBDLOG_DEBUG, "Sending to ELM -> ",pelm->buffer, pelm->blen);

	if (obdcon_write(chan->con, (unsigned char *)pelm->buffer, pelm->blen) < 0) {
		LOG_ERROR("Unable to write to serial connection");
		return OBDCHANERR_ERROR;
	}

	memset(pelm->buffer, 0, BUFFER_LEN); /* avoid a '>' char from previus read */

	total_readed = readed = 0;

	tread = time(0);

	while (!strchr(pelm->buffer, '>') &&
			time(0) - tread < READ_DEADLINE_FACTOR * timeout &&
			total_readed < BUFFER_LEN - 1) {

		readed = obdcon_read(chan->con, (unsigned char *)pelm->buffer + total_readed, 
				BUFFER_LEN - total_readed - 1,
				timeout);
		if(readed < 0) {
			LOG_ERROR("Error reading data from the serial");
			return OBDCHANERR_ERROR;
		}

		total_readed += readed;
	}

	if(!strchr(pelm->buffer, '>')) {
		LOG_ERROR("ELM timeout - %d secs.", timeout);
		return OBDCHANERR_TIMEOUT;
	}

	pelm->blen = total_readed;

	LOG_NOESCAPE(OBDLOG_DEBUG, "Recieved from ELM -> ",pelm->buffer, pelm->blen);

	return 0;
}

static inline int obdchanelm_ssendcmd(struct obdchan* chan, char* str, unsigned int timeout)
{
	struct obdchanelm* pelm = chan->data;

	memcpy(pelm->buffer, str, strlen(str));
	pelm->blen = strlen(str);

	return obdchanelm_sendcmd(chan, timeout);
}

static int obdchanelm_str2int(char* data, unsigned char* buffer, unsigned int size)
{
	unsigned int buf_len = 0;
	unsigned int data_len = strlen(data);
	int i;

	if(data_len < 3)
		return -1;

	for(i = 0; i + 2 < data_len && buf_len < size; i++){
		if(isxdigit(data[i]) && isxdigit(data[i + 1])) {
			data[i + 2] = '\0';
			buffer[buf_len] = (unsigned char) strtol(data + i, NULL, 16);
			buf_len++;
		}
	}

	return buf_len;
}

static int obdchanelm_name(char* buffer, unsigned int len)
{
	strncpy(buffer, name, len - 1);
	buffer[len - 1] = '\0';

	return 0;
}

static int obdchanelm_sendpid(struct obdchan* chan, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	struct obdchanelm* pelm = chan->data;
	unsigned char buffer[DATA_LEN];
	int buf_len;
	char* lastline;
	char* lastret;
	char* end;
	int ret;

	if(pelm == NULL)
		return OBDCHANERR_NOT_SETUP;

	sprintf(pelm->buffer, "%02X%02X", 1, sensor_data->pid); /* (Mode 01)(OBD PID) */
	pelm->blen = 4;

	if((ret = obdchanelm_sendcmd(chan, timeout)) < 0) {
		LOG_ERROR("Error sending command");
		return ret;
	}

	if(obdchanelm_error(chan) < 0)
		return OBDCHANERR_DEVICE_ERROR;

	if(strstr(pelm->buffer, "NO DATA")) { /* FIXME: Add a limit for NO DATA ;
						 check if a NO DATA can happen in the middle and after a normal data comes*/
		LOG_DEBUG("Got a NO DATA for PID %02X", sensor_data->pid);
		sensor_data->valid |= OBD_NO_DATA;
		return 0;
	}

	if(pelm->blen < 4) {
		LOG_ERROR("Got less then 3 bytes reply");
		return OBDCHANERR_PROTOCOL_ERROR;
	}

	end = lastret = strchr(pelm->buffer, '>');

	if(lastret == NULL || lastret - pelm->buffer < 4)
		return OBDCHANERR_ERROR;

	while(*end != ' ' && end != pelm->buffer)
		end--;

	end[1] = '\0';

	lastline = pelm->buffer;
	lastret = strrchr(pelm->buffer, '\r');

	if(lastret != NULL) {
		lastline = lastret;
		while(!isxdigit(*lastline) && lastline != end)
			lastline++;
	}

	if(obdchanelm_checkdata(lastline)) {
		LOG_WARNING("Invalid ELM data returned");
		sensor_data->valid |= OBD_INVALID_DATA;
		return 0;
	}

	if((buf_len = obdchanelm_str2int(lastline, buffer, DATA_LEN)) <= 0) {
		LOG_WARNING("Invalid ELM data returned");
		sensor_data->valid |= OBD_INVALID_DATA;
		return 0;
	}

	return obdsensor_process_obd(buffer, buf_len, sensor_data);
}

static int obdchanelm_senddtc(struct obdchan* chan, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	struct obdchanelm* pelm = chan->data;
	unsigned char buffer[DATA_LEN];
	int buf_len;
	char* lastline;
	char* lastret;
	char* end;

	if(pelm == NULL)
		return OBDCHANERR_NOT_SETUP;

	sprintf(pelm->buffer, "%02X", 3); /* OBD MODE 03 */
	pelm->blen = 2;

	if(obdchanelm_sendcmd(chan, timeout) < 0) {
		LOG_ERROR("Error sending command");
		return OBDCHANERR_ERROR;
	}

	if(obdchanelm_error(chan))
		return OBDCHANERR_ERROR;

	if(strstr(pelm->buffer, "NO DATA")) { /* FIXME: Add a limit for NO DATA ;
						 check if a NO DATA can happen in the middle and after a normal data comes*/
		dtc_data->valid |= OBD_NO_DATA;
		return 0;
	}

	if(pelm->blen < 4) {
		LOG_ERROR("Got less then 3 bytes reply");
		return OBDCHANERR_PROTOCOL_ERROR;
	}

	dtc_data->valid |= OBD_INVALID_DATA;

	lastret = strchr(pelm->buffer, '>');

	if(lastret == NULL || lastret - pelm->buffer < 4)
		return OBDCHANERR_ERROR;

	if(strchr(pelm->buffer, ':')) { /* ELM returnis a 0: line if more than 3 DTC errors (this sucks!) */
		char cat[2 * DATA_LEN];
		unsigned int pos = 0;
		char* dot;
		char* lastspace;

		memset(cat, '\0', 2 * DATA_LEN);
		LOG_DEBUG("Multiple DTCs replies");

		dot = strchr(pelm->buffer, ':');
		while (dot && pos < (2 * MAX_DTCS) + 1) {
			end = strchr(dot, '\r');
			if(end == NULL)
				end = lastret - 1;

			lastspace = end;

			while(lastspace[0] != ' ' && lastspace != pelm->buffer)
				lastspace--;

			lastspace[1] = '\0';

			strncpy(cat + pos, dot + 1, (2 * DATA_LEN) - pos - 1);
			dot = strchr(end + 1, ':');
			pos = strlen(cat);
		}

		if(obdchanelm_checkdata(cat)) {
			LOG_WARNING("Invalid ELM DTC data returned");
			dtc_data->valid |= OBD_INVALID_DATA;
			return 0;
		}

		if((buf_len = obdchanelm_str2int(cat, buffer, DATA_LEN)) <= 0) {
			LOG_WARNING("Invalid ELM DTC data returned from str2int");
			dtc_data->valid |= OBD_INVALID_DATA;
			return 0;
		}

		return obdsensor_process_dtc(buffer, buf_len, dtc_data);
	}

	LOG_DEBUG("Single DTCs reply");

	end = lastret;

	while(*end != ' ' && end != pelm->buffer)
		end--;

	end[1] = '\0';
	lastline = strrchr(pelm->buffer, '\r');

	if(lastline == NULL)
		lastline = pelm->buffer;


	while(!isxdigit(lastline[0]) && lastline != end)
		lastline++;

	if(obdchanelm_checkdata(lastline)) {
		LOG_WARNING("Invalid ELM DTC data returned");
		dtc_data->valid |= OBD_INVALID_DATA;
		return 0;
	}

	if((buf_len = obdchanelm_str2int(lastline, buffer, DATA_LEN)) <= 0) {
		LOG_WARNING("Invalid ELM DTC data returned from str2int");
		dtc_data->valid |= OBD_INVALID_DATA;
		return 0;
	}
	return obdsensor_process_dtc(buffer, buf_len, dtc_data);

}

static int obdchanelm_close(struct obdchan* chan)
{
	if(chan->type == OBDCHAN_UNDEFINED)
		return OBDCHANERR_NOT_SETUP;

	if(chan->data)
		free(chan->data);
	chan->data = NULL;

	chan->name = NULL;
	chan->sendpid = NULL;
	chan->senddtc = NULL;
	chan->close = NULL;

	return 0;
}

int obdchanelm_setup(struct obdchan* chan, struct obdcon* con)
{
	struct obdchanelm* pelm = NULL;
	int ret = 0;

	if(chan->type != OBDCHAN_UNDEFINED) {
		LOG_ERROR("Channel was already setup!");
		return OBDCHANERR_ALREADY_SETUP;
	}
	
	if(con == NULL) {
		LOG_ERROR("(BUG!!) Connection pointer is NULL!");
		return OBDCHANERR_INVALID_ARGS;
	}

	chan->con = con;

	chan->data = malloc(sizeof(struct obdchanelm));

	if(chan->data == NULL) {
		LOG_ERROR("Unable to alloc memory for ELM data!");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	pelm = chan->data;

	if(obdchanelm_ssendcmd(chan, "AT", 2) < 0) {/* Flush the bluetooth input buffer */
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	if(obdchanelm_ssendcmd(chan, "ATZ", 5) < 0) { /* Resetting the elm device */
		LOG_ERROR("Unable to reset ELM device");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	if(strstr(pelm->buffer, "AGV4000")) {
		LOG_INFO("Found a AGV4000 device");

		if(obdchanelm_ssendcmd(chan, "0100", 5) < 0) {; /* Selecting protocol */
			LOG_ERROR("AGV4000 - Unable to select OBD protocol");
			ret = OBDCHANERR_ERROR;
			goto error;
		}

		if(!strstr(pelm->buffer, "41")) {
			LOG_ERROR("AGV4000 - Unable to select OBD protocol");
			ret = OBDCHANERR_ERROR; /* Could not select the AVG4000 protocol */
			goto error;
		}

	} else
		if(!strstr(pelm->buffer, "ELM") && 
				!strstr(pelm->buffer, "OBDKey")) {
				LOG_ERROR("Unknown ELM based device found! Please ask the support of your device in the community");
				ret = OBDCHANERR_ERROR;
				goto error;
		}

	if(obdchanelm_ssendcmd(chan, "ATE0", 5) < 0) { /* Echo off */
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	if(!strstr(pelm->buffer, "OK")) {
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	LOG_DEBUG("ELM echo disabled");

	if(obdchanelm_ssendcmd(chan, "ATL0", 5) < 0) { /* Linefeed off */
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	if(!strstr(pelm->buffer, "OK"))
		LOG_WARNING("Unable to set ELM linefeed off");

	LOG_DEBUG("ELM linefeed off");

	obdchanelm_devicetimeout(chan, DEVICE_TIMEOUT);
	if(obdchanelm_sendcmd(chan, 5)) {
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	if(!strstr(pelm->buffer, "OK"))
		LOG_WARNING("Unable to set ELM timeout (is it a ELM 327 1.1 device?)");

	if(obdchanelm_ssendcmd(chan, "ATAT2", 6)) { /* Adaptive_timing - ELM 1.2 */
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	if(!strstr(pelm->buffer, "OK"))
		LOG_WARNING("Unable to set ELM to use adaptive timing - ELM 1.1-?");

	if(obdchanelm_ssendcmd(chan, "0101", 10)) { /* Test an obd command */
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	if(obdchanelm_error(chan) < 0) {
		ret = OBDCHANERR_SETUP_ERROR;
		goto error;
	}

	chan->type = OBDCHAN_ELM;
	chan->name = obdchanelm_name;
	chan->sendpid = obdchanelm_sendpid;
	chan->senddtc = obdchanelm_senddtc;
	chan->close = obdchanelm_close;
	return 0;

error:
	if(chan->data) {
		free(chan->data);
		chan->data = NULL;
	}

	chan->con = NULL;

	return ret;
}
