/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef __LIBOBD__H__
#define __LIBOBD__H__

#include <features.h>
__BEGIN_DECLS

#include <stdio.h>

#define MAX_DTCS 32

#define OBD_ERROR 1
#define OBD_WARNING 2
#define OBD_INFO 3
#define OBD_DEBUG 4

#define OBD_INVALID_DATA	1
#define OBD_NO_DATA		2
#define OBD_HAS_FIRST_VALUE	4
#define OBD_HAS_SECOND_VALUE	8
#define OBD_HAS_FIRST_VALUE_IMPERIAL	16
#define OBD_HAS_SECOND_VALUE_IMPERIAL	32


struct obd_sensor_data {
	int valid;
	int pid;
	double result[2];
	double rimperial[2];
};

struct obd_dtc_data {
	int valid;
	char code_list[MAX_DTCS][6];
};


/*
 * Allocs the necessary struct to use the library. The lib can deal with
 * different connections using different structs. So if you want to deal with
 * different obd connections, you must alloc different structs and call the
 * open method bellow for each serial device. This function retuns a void
 * pointer to a structure that should be used by the functions bellow 
 *
 * @return void* pointer to the obd structure or NULL if error.
 */
extern void* obd_alloc();

/*
 * Opens a serial device. Once the device is opened, you must call the
 * function obd_chan_setup for the obd device setup. Only after you may
 * call obd_sendcmd.
 *
 * @param obd a void pointer to a structure used through out the this lib
 * @param devname the name of the device to be open (e.g. /dev/rfcomm0)
 * @return 0 for success; -1 for error
 */
extern int obd_open_dev(void* obd, const char *devname);

/*
 * Uses a software simulator as a device. As we are using software simulation,
 * there is no need to call obd_open_dev to establish a serial connection. If
 * a serial connection has been established, it will be closed.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @return 0 for success; -1 for error
 */
extern int obd_open_simulator(void* obd);

/*
 * Use a file to act as an obd device. This file should contain the mapping of
 * PID -> value and DTC -> value.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @param filename a pointer to a string containing the file name
 */
extern int obd_open_file(void* obd, char* filename);

/*
 * Establish the protocol conversation with an obd scantool.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @return 0 for success; -1 for error
 */
extern int obd_channel_setup(void* obd);

/*
 * Closes the serial connection. This must be called even for the software
 * simulator.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @return 0 for success; -1 for error
 */
extern int obd_close(void* obd);

/*
 * Dealloc the obd structure. The connection must be already closed for
 * this function to succeed. Once called, discard the void pointer as it
 * is not valid anymore.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @return 0 for success; -1 for error
 */
extern int obd_dealloc(void* obd);

/*
 * Sends a real time request to the obd device.
 * The PID of the command should be a 2 character string (e.g. '0C').
 * Use the pid field of obd_sensor_data to inform the OBD PID to send.
 * This function is blocking.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @param sensor_data an already allocated struct that contains the pid and where
 * the return will put the data.
 * @param timeout the timeout is seconds
 * @return 0 for success; -1 for error
 */
extern int obd_sendpid (void* obd, struct obd_sensor_data* sensor_data, unsigned int timeout);

/*
 * Sends a request to get the  Diagnostic Trouble Codes (DTCs) from the obd device.
 * This function is blocking.
 *
 * @param obd a void pointer to a structure used through out this lib
 * @param dtc_data an already allocated struct that contains the pid and where
 * the return will put the data.
 * @param timeout the timeout is seconds
 */
extern int obd_senddtc (void* obd, struct obd_dtc_data* dtc_data, unsigned int timeout);

/*
 * Returns the current device type name or null if an error 
 *
 * @param obd a void pointer to a structure used through out this lib
 * @param buffer a pointer to the buffer where this function should put the name
 * @param size the buffer size in bytes
 * @return 0 for success; -1 for error
 */
extern int obd_channel_name(void* obd, char* buffer, unsigned int size);

/*
 * Setup the log system.
 *
 * @param handle the handle of where the log system will write messages
 * @param level how verbose should we be - OBD_ERROR, OBD_WARNING,
 * OBD_INFO, OBD_DEBUG. If the number is not one of those,
 * ERROR is assumed.
 */
extern int obd_log_setup(FILE* file, int level);

__END_DECLS

#endif /* __LIBOBD__H__ */
