/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2008 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef _KMPLAYER_SURFACE_H_
#define _KMPLAYER_SURFACE_H_

#include "kmplayertypes.h"
#include "kmplayerplaylist.h"

typedef struct _cairo_surface cairo_surface_t;

namespace KMPlayer {

class ViewArea;
class Surface;

/**                                   a  b  0
 * Matrix for coordinate transforms   c  d  0
 *                                    tx ty 1     */
class KMPLAYER_NO_EXPORT Matrix {
    float a, b, c, d;
    Single tx, ty;
public:
    Matrix ();
    Matrix (const Matrix & matrix);
    Matrix (Single xoff, Single yoff, float xscale, float yscale);
    void getXY (Single & x, Single & y) const;
    void getXYWH (Single & x, Single & y, Single & w, Single & h) const;
    void invXYWH (Single & x, Single & y, Single & w, Single & h) const;
    void transform (const Matrix & matrix);
    void scale (float sx, float sy);
    void translate (Single x, Single y);
    // void rotate (float phi); // add this when needed
};


class KMPLAYER_NO_EXPORT Surface : public TreeNode <Surface> {
public:
    Surface (ViewArea *widget);
    ~Surface();

    void clear ();
    Surface *createSurface (NodePtr owner, const SRect & rect);
    IRect toScreen (Single x, Single y, Single w, Single h);
    IRect clipToScreen (Single x, Single y, Single w, Single h);
    void resize (const SRect & rect, bool parent_resized=false);
    void repaint ();
    void repaint (const SRect &rect);
    void remove ();                // remove from parent, mark ancestors dirty
    void markDirty ();             // mark this and ancestors dirty
    void updateChildren (bool parent_resized=false);
    SSize dimension () const { return size.isEmpty () ? bounds.size : size; }
    ViewArea *viewArea () { return view_widget; }

    NodePtrW node;
    SRect bounds;                  // bounds in in parent coord.
    SSize size;                    // if set, dimension in Surface coord.
    float xscale, yscale;          // internal scaling
    unsigned int background_color; // rgba background color
    cairo_surface_t *surface;
    bool dirty;                    // a decendant is removed

private:
    ViewArea *view_widget;
};

typedef Item<Surface>::SharedType SurfacePtr;
typedef Item<Surface>::WeakType SurfacePtrW;
ITEM_AS_POINTER(KMPlayer::Surface)

} // namespace

#endif
