/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <string.h>
#include <map>

#include "actor.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "viewarea.h"

using namespace KMPlayer;

namespace {
    typedef std::map <String, ImageDataPtrW> ImageDataMap;
    static ImageDataMap *image_data_map;
}

//------------------------%<----------------------------------------------------

ActorAgent::ActorAgent (Control *ctrl) : m_control (ctrl) {
    if (!image_data_map)
        image_data_map = new ImageDataMap;
}

ActorAgent::~ActorAgent () {
    if (image_data_map->size () == 0) {
        delete image_data_map;
        image_data_map = 0;
    }
}

Actor *ActorAgent::hire (ActorType type, Node *node) {
    switch (type) {
        case Audio:
        case AudioVideo:
            //if (!m_control->source()->authoriseUrl (
            //            node->mrl()->absolutePath ()))
            //    return NULL;

            return new AudioVideoActor (this, node);
        case Image:
            return new ImageActor (this, node);
        case Text:
            return new TextActor (this, node);
    }
    return NULL;
}

//------------------------%<----------------------------------------------------

Actor::Actor (ActorAgent *manager, Node *node)
 : m_agent (manager), m_node (node), job (NULL) {
   //manager->medias ().push_back (this);
}

Actor::~Actor () {
    clearData ();
    //m_agent->medias ().remove (this);
}

KDE_NO_EXPORT void Actor::killWGet () {
    if (job) {
        job->kill (); // quiet, no result signal
        job = 0L;
        clearData (); // assume data is invalid
    }
}

/**
 * Gets contents from url and puts it in m_data
 */
KDE_NO_EXPORT bool Actor::wget (const String &str) {
    clearData ();
    url = str;
    URL uri (str);
    debugLog () << "Actor::wget " << str << endl;
    if (uri.isLocalFile ()) {
        File file (uri.path ());
        if (file.exists () && file.open (IO_ReadOnly)) {
            data = file.readAll ();
            file.close ();
        }
        ready (str);
        return true;
    } else
        job = asyncGet (this, uri.url ());
    return false;
}

Mrl *Actor::mrl () {
    return m_node ? m_node->mrl () : NULL;
}

KDE_NO_EXPORT String Actor::mimetype () {
    if (data.size () > 0 && mime.isEmpty ())
        mime = MimeType::findByContent (data.data (), data.size ());
    return mime;
}

KDE_NO_EXPORT void Actor::clearData () {
    killWGet ();
    url.truncate (0);
    mime.truncate (0);
    data.resize (0);
}

KDE_NO_EXPORT void Actor::dismiss () {
    delete this;
}

KDE_NO_EXPORT bool Actor::downloading () const {
    return !!job;
}

KDE_NO_EXPORT void Actor::ready (const String &) {
    if (m_node)
        m_node->document()->post (m_node.ptr(), new Posting (m_node, MsgMediaReady));
}

KDE_NO_EXPORT void Actor::jobResult (Job *jb) {
    if (jb->error ())
        data.resize (0);
    job = 0L; // signal KIO::Job::result deletes itself
    ready (url);
}

KDE_NO_EXPORT void Actor::jobData (Job *jb, ByteArray &buf) {
    //debugLog () << "ActorTypeRuntime::jobData " << data.size () << endl;
    if (data.size ())
        mime = jb->contentType ();
    if (buf.size ()) {
        int old_size = data.size ();
        data.resize (old_size + buf.size ());
        memcpy (data.data () + old_size, buf.data (), buf.size ());
    }
}

//------------------------%<----------------------------------------------------

AudioVideoActor::AudioVideoActor (ActorAgent *manager, Node *node)
 : Actor (manager, node),
   ignore_pause (false) {
    debugLog() << "AudioVideoActor::AudioVideoActor" << endl;
}

AudioVideoActor::~AudioVideoActor () {
    stop ();
    debugLog() << "AudioVideoActor::~AudioVideoActor" << endl;
}

bool AudioVideoActor::play () {
    Process *process = m_agent->control ()->process ();
    if (process && process->playing () && process->mrl () == mrl ()) {
        errorLog () << "play: already playing " << mrl ()->src << endl;
    } else if (m_agent->control ()->requestPlayURL (m_node)) {
        if (!mrl ()->audio_only)
            m_agent->control ()->viewArea ()->setAudioVideoNode (m_node);
        ignore_pause = true;
        m_node->defer ();
        ignore_pause = false;
        return true;
    }
    return false;
}

void AudioVideoActor::stop () {
    Process *process = m_agent->control ()->process ();
    if (process)
        process->stop ();
    m_agent->control ()->viewArea ()->setAudioVideoNode (NULL);
}

void AudioVideoActor::pause () {
    Process *process = m_agent->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

void AudioVideoActor::unpause () {
    Process *process = m_agent->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

//------------------------%<----------------------------------------------------

#include <cairo.h>

ImageData::ImageData( const String & img)
 : width (0),
   height (0),
   flags (0),
   has_alpha (false),
   image (0L),
   surface (NULL),
   url (img) {
    //if (img.isEmpty ())
    //    //debugLog() << "New ImageData for " << this << endl;
    //else
    //    //debugLog() << "New ImageData for " << img << endl;
}

ImageData::~ImageData() {
    if (!url.isEmpty ())
        image_data_map->erase (url);
    if (surface)
        cairo_surface_destroy (surface);
    delete image;
}

void ImageData::setImage (Image *img) {
    if (image != img) {
        delete image;
        if (surface)
            cairo_surface_destroy (surface);
        image = img;
        width = img->width ();
        height = img->height ();
        has_alpha = false;
    }
}

ImageActor::ImageActor (ActorAgent *manager, Node *node)
 : Actor (manager, node),
   img_movie (NULL),
   img_movie_iter (NULL),
   pix_loader (NULL),
   img_movie_timer (0),
   timeout (0) {}

ImageActor::~ImageActor () {
    stop ();
    if (img_movie_iter)
        g_object_unref (img_movie_iter);
    if (pix_loader)
        //gdk_pixbuf_loader_close (pix_loader, 0L);
        g_object_unref (pix_loader);
    else if (img_movie)
        g_object_unref (img_movie);
}

KDE_NO_EXPORT bool ImageActor::play () {
    if (img_movie)
        unpause ();
    return img_movie;
}

KDE_NO_EXPORT void ImageActor::stop () {
    pause ();
}

void ImageActor::pause () {
    if (img_movie_timer)
        g_source_remove (img_movie_timer);
    img_movie_timer = 0;
}

gboolean movieUpdateTimeout (void * p) {
   return ((ImageActor *)p)->movieTimer ();
}

bool ImageActor::movieTimer () {
    bool ret = false;
    if (gdk_pixbuf_animation_iter_advance (img_movie_iter, 0L)) {
        timeout = 0;
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
        int to = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (to == timeout) {
            ret = true; // re-use timer
        } else {
            timeout = to;
            if (to > 0)
                img_movie_timer = g_timeout_add (to, movieUpdateTimeout, this);
        }
        if (m_node)
            m_node->document()->post (m_node, new Posting (m_node, MsgMediaUpdated));
    } else if (m_node) {
        m_node->document()->post (m_node, new Posting(m_node, MsgMediaFinished));
    }
    return ret;
}

void ImageActor::unpause () {
    if (img_movie_iter && !img_movie_timer) {
        timeout = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (timeout > 0)
            img_movie_timer = g_timeout_add (timeout, movieUpdateTimeout, this);
    }
}

KDE_NO_EXPORT void ImageActor::setupImage () {
    GError *gerror = 0L;

    pix_loader = gdk_pixbuf_loader_new ();
    if (gdk_pixbuf_loader_write (pix_loader,
                (const guchar *) data.data (),
                data.size (), &gerror) &&
            gdk_pixbuf_loader_close (pix_loader, 0L)) {
        cached_img = ImageDataPtr (new ImageData (url));
        img_movie = gdk_pixbuf_loader_get_animation (pix_loader);
    }

    if (img_movie && !gdk_pixbuf_animation_is_static_image (img_movie)) {
        img_movie_iter = gdk_pixbuf_animation_get_iter (img_movie, NULL);
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
    } else if (img_movie) {
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_get_static_image (img_movie))));
        cached_img->flags = (int) ImageData::ImagePixmap;
        (*image_data_map)[url] = ImageDataPtrW (cached_img);
    }
    g_free (gerror);
}

KDE_NO_EXPORT void ImageActor::ready (const String &url) {
    if (data.size ()) {
        String mime = mimetype ();
        if (!mime.startsWith ("text/")) { // FIXME svg
            ImageDataMap::iterator i = image_data_map->find (url);
            if (i == image_data_map->end ())
                setupImage ();
            else
                cached_img = i->second;
            data = ByteArray ();
        }
    }
    Actor::ready (url);
}

bool ImageActor::isEmpty () const {
    return !cached_img || cached_img->isEmpty ();
}

KDE_NO_EXPORT bool ImageActor::wget (const String &str) {
    ImageDataMap::iterator i = image_data_map->find (str);
    if (i != image_data_map->end ()) {
        cached_img = i->second;
        Actor::ready (str);
        return true;
    } else {
        return Actor::wget (str);
    }
}

//------------------------%<----------------------------------------------------

TextActor::TextActor (ActorAgent *manager, Node *node)
 : Actor (manager, node), default_font_size (14)
{}

TextActor::~TextActor () {
}

KDE_NO_EXPORT void TextActor::ready (const String &url) {
    if (data.size ())
        text = String (data.data (), data.size ());
    debugLog() << "TextActor::ready " << data.size () << " " << text << endl;
    Actor::ready (url);
}

bool TextActor::play () {
    return !text.isEmpty ();
}
