//#include <config.h>

#include <string.h>
#include <errno.h>
#include <glob.h>

#define DISABLE_PRINTING
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
extern "C" {
#include <gtkhtml/gtkhtml.h>
}
#ifdef BUILD_BORA
# include <hildon-widgets/gtk-infoprint.h>
# include <hildon-widgets/hildon-seekbar.h>
#else
# include <hildon/hildon-banner.h>
# include <hildon/hildon-seekbar.h>
#endif
//#include <gdk/gdk.h>
//#include <pango/pango-font.h>

#include "kmplayerplaylist.h"
#include "mediaobject.h"
#include "kmplayer_lists.h"
#include "kmplayer_smil.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "kmplayer.h"
#include "viewarea.h"

using namespace KMPlayer;

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

static bool isPlayListMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    //debugLog () << "isPlayListMime  mime:" << mimestr << endl;
    return mimestr && (!strcmp (mimestr, "audio/mpegurl") ||
            !strcmp (mimestr, "audio/x-mpegurl") ||
            !strncmp (mimestr, "video/x-ms", 10) ||
            !strncmp (mimestr, "audio/x-ms", 10) ||
            //!strcmp (mimestr, "video/x-ms-wmp") ||
            //!strcmp (mimestr, "video/x-ms-asf") ||
            //!strcmp (mimestr, "video/x-ms-wmv") ||
            //!strcmp (mimestr, "video/x-ms-wvx") ||
            //!strcmp (mimestr, "video/x-msvideo") ||
            !strcmp (mimestr, "video/x-real") ||
            !strcmp (mimestr, "audio/x-scpls") ||
            !strcmp (mimestr, "audio/x-pn-realaudio") ||
            !strcmp (mimestr, "audio/vnd.rn-realaudio") ||
            !strcmp (mimestr, "audio/m3u") ||
            !strcmp (mimestr, "audio/x-m3u") ||
            !strncmp (mimestr, "text/", 5) ||
            (!strncmp (mimestr, "application/", 12) &&
             strstr (mimestr + 12,"+xml")) ||
            !strncasecmp (mimestr, "application/smil", 16) ||
            !strncasecmp (mimestr, "application/xml", 15) ||
            !strcmp (mimestr, "application/news_reader") ||
            !strcmp (mimestr, "application/rss+xml") ||
            !strcmp (mimestr, "application/atom+xml") ||
            !strcmp (mimestr, "application/x-mplayer2"));
}

#ifdef KMPLAYER_DEBUG
static Application *debug_app;
static Control *debug_control;

void sig_usr1 (int) {
    debugLog () << "Start dump" << endl;
    debugLog () << debug_control->process () << " " <<
        debug_control->blanking_timer << endl;
    if (debug_control->process () && debug_control->process ()->mrl ())
        debugLog () << debug_control->process ()->playing () <<
            debug_control->process ()->mrl ()->src <<
            debug_control->process ()->mrl ()->audio_only <<
            debug_control->process ()->hasVideo () << endl;
    debugLog () << "End dump" << endl;
}
#endif

static bool isAudioOnlyMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    return mimestr && (!strncmp ((const char *) mime, "audio/", 6));
}

static gboolean playSchedule (void * ctr) {
    static_cast <Control *> (ctr)->play ();
    return false;
}

static gboolean playCurrentSchedule (void * ctr) {
    static_cast <Control *> (ctr)->playCurrent ();
    return false;
}

Control::Control (Application * a)
    : m_process (0L), m_doc_timer (0), progress_banner (NULL),
      playlist_data (0L), m_config (new Config), m_app (a),
      paint_timer (0), update_tree_timer (0), blanking_timer (0),
      in_progress_update (false),
      media_manager (new MediaManager (this)) {
#ifdef KMPLAYER_DEBUG
    debug_app = a;
    debug_control = this;
#endif
}

Control::~Control () {
    delete m_process;
    if (playlist_data->document)
        playlist_data->document->document()->dispose ();
    delete playlist_data;
    delete m_config;
    m_process = 0L;
}

void Control::init () {
    unsigned int ver, rel, req, evb, err;
    glob_t gt;
    bool has_xv = !glob ("/usr/lib/libXv.so*", GLOB_ERR|GLOB_NOSORT, 0, &gt);
    globfree (&gt);
    media_server = new OssoMediaServer (this, m_app->osso_context, has_xv);
    mplayer = new MPlayer (this, has_xv);
#ifdef __ARMEL__
    npp = new NpPlayer (this);
#endif
    String p = m_config->readEntry (GCONF_KEY_PLAYER, OssoMediaServer::myname);
    setProcess ((const char *) p);

    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();
    addTree (m_app, 0L, //gdk_pixbuf_new_from_xpm_data((const char**)www_xpm));
            gdk_pixbuf_scale_simple (
                    gtk_icon_theme_load_icon (icon_theme,
                        "qgn_grid_tasknavigator_web", 26,
                        (GtkIconLookupFlags)0, 0L),
                    26, 26, GDK_INTERP_BILINEAR));
    playlists = new KMPlayer::Playlist (m_app, 0L);
    m_app->playlists_tree_id = addTree (m_app, playlists,
            gtk_icon_theme_load_icon(icon_theme,
                "qgn_list_gene_playlist", 26, (GtkIconLookupFlags)0, 0L));
    recents = new KMPlayer::Recents (m_app);
    m_app->recents_tree_id = addTree (m_app, recents,
            gtk_icon_theme_load_icon(icon_theme,
                "qgn_list_gene_url_history", 26, (GtkIconLookupFlags)0, 0L));
}

void Control::deinit () {
    stop ();
    setProcess ((Process *) NULL);
    Recents *rec = static_cast <Recents *> (recents.ptr());
    if (rec->resolved &&
            rec->load_tree_version != rec->m_tree_version)
        rec->writeToFile (m_config->readEntry
                (GCONF_KEY_RECENT_FILE, "/home/user/recent.xml"));
    rec->dispose ();
    Playlist *playlist = static_cast<Playlist *>(playlists.ptr());
    if (playlist->resolved &&
            playlist->load_tree_version != playlist->m_tree_version)
        playlist->writeToFile (m_config->readEntry
                (GCONF_KEY_PLAYLIST_FILE, "/home/user/playlist.xml"));
    playlist->dispose ();
}

gboolean cb_blankingTimer (void * p) {
    Application * app = static_cast <Application *> (p);
    if (CONTROL(app)->playlist_data->current && 
            (CONTROL(app)->playlist_data->current->id == SMIL::id_node_smil ||
             (CONTROL(app)->process ()->playing () &&
              CONTROL(app)->process ()->hasVideo ()))) {
        osso_display_blanking_pause (app->osso_context);
        return 1;
    }
    CONTROL(app)->blanking_timer = 0;
    return 0;
}

gboolean cb_docTimer (void * p) {
    return static_cast <Control *> (p)->docTimerEvent ();
}

gboolean cb_scheduledUpdateTree (void * vp) {
    Control *ctr = static_cast <Control *> (vp);
    if (ctr->process ()) {
        if (!ctr->playlist_data->document)
            ctr->playlist_data->document = new Document(String(), ctr);
        updateTree (ctr->m_app, 0);
    }
    ctr->update_tree_timer = 0;
    return false;
}

void Control::play () {
    if (!playlist_data->document)                // nothing to play
        return;
    if (playlist_data->document->active ()) {    // is playing
        if (playlist_data->document->state == Node::state_deferred)
            playlist_data->document->undefer (); // unpause
        else
            playlist_data->document->defer ();   // pause
    } else {
        debugLog () << "play" << endl;
        //gtk_paned_set_position (m_app->vsplit_view, VSPLIT);
        //gtk_container_resize_children (GTK_CONTAINER (m_app->vsplit_view));
        if (!applyBackRequest () && playlist_data->document) {
            playCurrent ();
        }
    }
}

void Control::stop () {
    debugLog () << "stop" << endl;
    for (SharedPtr<ResolveInfo> rinfo =m_resolve_info; rinfo; rinfo=rinfo->next)
        rinfo->job->kill ();
    m_resolve_info = 0L;
    setLoading (100); // no pogress bar
    if (blanking_timer) {
        g_source_remove (blanking_timer);
        blanking_timer = 0;
    }
    back_request = 0L;
    if (playlist_data->document &&
            playlist_data->document->state > Node::state_init) {
        playlist_data->document->reset ();
        //gtk_paned_set_position (m_app->vsplit_view, 40);
        setCurrent (0L);
    }
    VIEWAREA(m_app)->setAspect (0.0);
    m_process->quit ();
    if (VIEWAREA(m_app)->viewWidget ()->window) {
        gdk_window_clear (VIEWAREA(m_app)->videoWindow ());
        gdk_window_clear (VIEWAREA(m_app)->viewWidget ()->window);
        gtk_widget_queue_draw (m_app->infopanel_view);
    }
    app_normal_screen (m_app);
    setLength (0);
    setPosition (0);
    updatePlaylistView ();
}

String Control::currentMrl () {
    Mrl * mrl = playlist_data->current ? playlist_data->current->mrl () : 0L;
    if (!mrl)
        return String ();
    return mrl->absolutePath ();
}

void Control::openDocument (NodePtr doc, NodePtr cur) {
    stop ();
    if (playlist_data->document)
        playlist_data->document->document()->dispose ();
    doc->document()->notify_listener = this;
    playlist_data->document = doc;
    cur_url = doc->mrl()->pretty_name;
    app_set_title ((const char *) cur_url, m_app);
    back_request = cur;
    g_timeout_add (0, playSchedule, static_cast <void *> (this));
    updatePlaylistView (true);
}

ViewArea *Control::viewArea () const {
    return (ViewArea *)m_app->viewarea;
}

unsigned long Control::videoOutput () const {
    return GDK_DRAWABLE_XID (GDK_DRAWABLE (VIEWAREA (m_app)->videoWindow ()));
}

void Control::playCurrent () {
    String url = currentMrl ();
    // set title url;
    aspect = 0.0;
    width = height = 0;
    //debugLog () << "playCurrent" << !!playlist_data->document << !playlist_data->document->active () << endl;
    // view ())->videoStop (); //show buttonbar
    if (playlist_data->document &&
            !playlist_data->document->active ()) {
        if (!playlist_data->current)
            setCurrent (playlist_data->document);
        else { // ugly code duplicate w/ back_request
            for (NodePtr p = playlist_data->current->parentNode(); p; p = p->parentNode())
                p->setState (Element::state_activated);
        }
        playlist_data->current->activate ();
        updateViewAreaSizes ();
    } else if (!playlist_data->current) {
        //debugLog () << "playCurrent no current" << endl;
        stop ();
        ; //emit endOfPlayItems ();
    } else if (playlist_data->current->state ==Element::state_deferred) {
        //debugLog () << "playCurrent set undefer" << endl;
        ;//playlist_data->current->undefer ();
    } else if (m_process->state () == Process::NotRunning) {
        //debugLog () << "playCurrent set ready" << endl;
        m_process->ready (GDK_DRAWABLE_XID (GDK_DRAWABLE (
                        VIEWAREA(m_app)->videoWindow ())));
    } else if (!playlist_data->current->active ()) {
        playlist_data->current->activate ();
        updateViewAreaSizes ();
    } else {
        Mrl *mrl = back_request
            ? back_request->mrl ()->linkNode ()
            : playlist_data->current->mrl ()->linkNode ();
        //debugLog () << "playCurrent " << m_process->playing () << " " << mrl->mrl ()->resolved << endl;
        if (m_process->playing ()) {
            errorLog() << "playCurrent and backend still playing" << endl;
            if (m_process->mrl () != mrl)
                m_process->stop ();
            else
                return; // FIXME
        }
        back_request = 0L;
        if (!mrl->src.isEmpty () && mrl->resolved) {
            updateViewAreaSizes ();
            if (m_process->hasXv ())
                XSync (GDK_DISPLAY_XDISPLAY (gtk_widget_get_display (VIEWAREA(m_app)->viewWidget ())), false);
            VIEWAREA(m_app)->setEventFiltering (true);
            m_process->play (mrl);
        } else
            errorLog() << "playCurrent and mrl not resolved" << endl;
    }
}

bool Control::applyBackRequest() {
    if (!back_request || back_request->mrl ()->view_mode == Mrl::WindowMode)
        return false;
    if (playlist_data->current)
        playlist_data->document->reset ();
    playlist_data->current = 0L;
    if (back_request->document() != playlist_data->document) {
        //debugLog () << "playlist/recent" << endl;
        NodePtr n = back_request;
        back_request = 0L;
        n->activate();
    } else {
        setCurrent (back_request);
        for (NodePtr p = back_request->parentNode(); p; p = p->parentNode())
            p->setState (Element::state_activated);
        back_request = 0L;
        g_timeout_add(0, playCurrentSchedule, static_cast<void*>(this));
    }
    return true;
}

static String processForMimetype (Config *config, const String & mime) {
    String pname = config->readEntry (String (GCONF_KEY_MIME) + mime, "");
    if (pname.isEmpty ()) {
        // hardcoded defaults FIXME
        if (mime == "application/x-shockwave-flash" ||
                mime == "application/futuresplash")
            pname = "npp";
    }
    return pname;
}

bool Control::resolveURL (NodePtr m) {
  //debugLog() << "resolveURL " << !!back_request << " " << m->mrl()->src<<endl;
    if (back_request && back_request->mrl ()->view_mode == Mrl::SingleMode)
        return false; // block auto playlist traversal on jump
    Mrl * mrl = m->mrl ();
    if (!mrl || mrl->src.isEmpty ())
        return true;
    int depth = 0;
    for (NodePtr e = m->parentNode (); e; e = e->parentNode ())
        ++depth;
    if (depth > 40)
        return true;
    URL url (mrl->absolutePath ());
    String mimestr = mrl->mimetype;
    if (!mimestr.isEmpty () &&
            processForMimetype (m_config, mimestr) == String ("npp"))
        return true; // FIXME
    if (mimestr.isEmpty ()) {
        if (url.isLocalFile ())
            mimestr = MimeType::findByURL (url);
        if (!mimestr.isEmpty ())
            mrl->mimetype = mimestr;
    }
    debugLog() << "resolveURL " << url << " mime:" << mimestr << endl;
    if (mimestr.isEmpty ()) {
        for (NodePtr p = mrl->parentNode (); p; p = p->parentNode ()) {
            Mrl *m = p->mrl ();
            if (m && m->linkNode ()->audio_only) {
                mrl->linkNode ()->audio_only = true;
                break;
            }
        }
    } else
        mrl->linkNode ()->audio_only |= isAudioOnlyMime (mimestr);
    bool maybe_playlist = mimestr.isEmpty() || isPlayListMime (mimestr);
    if (url.isLocalFile ()) {
        File file (url.path ());
        struct stat stats;
        if (!file.exists ()) {
            errorLog () << url.path () << " not found" << endl;
            return true;
        }
        //debugLog () << "opening " << url << " mime:" << mimestr << endl;
        if (maybe_playlist && file.size () < 2000000) {
            char buf[256];
            off_t nr = file.readBlock (buf, sizeof (buf) - 1);
            if (nr > 0) {
                buf[nr] = 0;
                String mimestr = MimeType::findByContent (buf, nr);
                if (mimestr.hasPrefix (String ("text/")) &&
                        strncmp (buf, "RIFF", 4)) {
                    ByteArray ba = file.readAll ();
                    String data (ba.data (), ba.size ());
                    if (!data.isEmpty ()) {
                        TextStream in (data);
                        readPlaylist (m, in);
                    }
                }
            }
        }
    } else if ((maybe_playlist && // exclude mms/rtsp/rtp 
                url.protocol ().compare (String ("mms")) &&
                url.protocol ().compare (String ("rtsp")) &&
                url.protocol ().compare (String ("rtp"))) ||
            (mimestr.isEmpty () && // try http(s) ones too
             url.protocol ().hasPrefix (String ("http")))) {
        debugLog () << "maybe playlist, network " << endl;
        Job * job = asyncGet (this, url.url ());
        m_resolve_info = new ResolveInfo (m, job, m_resolve_info);
        return false; // wait for network operation ..
    }
    return true;
}

void Control::jobData (Job * job, ByteArray & data) {
    SharedPtr <ResolveInfo> rinfo = m_resolve_info;
    while (rinfo && rinfo->job != job)
        rinfo = rinfo->next;
    if (!rinfo) {
        warningLog () << "Spurious kioData" << endl;
        return;
    }
    //debugLog () << "jobData of " << data.size () << endl;
    if (rinfo->data.isEmpty ()) {
        String mimestr = MimeType::findByContent (data.data (), data.size ());
        if (rinfo->resolving_mrl->mrl ()->mimetype.isEmpty ())
            rinfo->resolving_mrl->mrl ()->mimetype =
                job->contentType ().isEmpty () ? mimestr : job->contentType ();
        if (mimestr.isEmpty() || !mimestr.hasPrefix (String ("text/")) ||
                (data.size () > 4 && !strncmp (data.data (), "RIFF", 4)) ||
                !rinfo->resolving_mrl) {
            job->kill (false); // not quiet, wants jobResult
            return;
        }
    }
    rinfo->data.readRawBytes (data.data (), data.size ());
}

void Control::jobResult (Job * job) {
    debugLog () << "jobResult" << endl;
    SharedPtr <ResolveInfo> previnfo, rinfo = m_resolve_info;
    while (rinfo && rinfo->job != job) {
        previnfo = rinfo;
        rinfo = rinfo->next;
    }
    if (!rinfo) {
        warningLog () << "Spurious kioData" << endl;
        return;
    }
    if (previnfo)
        previnfo->next = rinfo->next;
    else
        m_resolve_info = rinfo->next;
    if (rinfo->resolving_mrl) {
        Mrl *mrl = rinfo->resolving_mrl->mrl ();
        if (!rinfo->data.isEmpty ())
            readPlaylist (mrl, rinfo->data);
    //debugLog () << "set resolved " << rinfo->resolving_mrl->mrl ()->src << endl;
        mrl->resolved = true;
        if (mrl->mimetype.isEmpty ())
            mrl->mimetype = job->contentType ();
        mrl->audio_only = isAudioOnlyMime (mrl->mimetype);
        mrl->undefer ();
    }
}

void Control::readPlaylist (NodePtr root, TextStream & in) {
    String line;
    Mrl * mrl = root->mrl ();
    do {
        line = in.readLine ().stripWhiteSpace ();
    } while (!in.atEnd () && line.isEmpty ());
    if (!mrl || line.isEmpty ())
        return;
    NodePtr cur_elm = root;
    if (mrl->isPlayable ())
        cur_elm = mrl->linkNode ();
    if (mrl->mimetype.hasPrefix ("text/") && line.find ("[playlist]") > -1)
        mrl->mimetype = "audio/x-scpls";
    if (mrl->mimetype == String ("audio/x-scpls")) {
        bool groupfound = false;
        int nr = -1;
        struct Entry {
            String url, title;
        } * entries = 0L;
        do {
            if (line.hasPrefix (String ("[")) && line.hasSuffix (String ("]"))) {
                if (!groupfound && line.mid (1, line.length () - 2).stripWhiteSpace () == String ("playlist"))
                    groupfound = true;
                else
                    break;
                //debugLog () << "Group found: " << line << endl;
            } else if (groupfound) {
                int eq_pos = line.find (Char ('='));
                if (eq_pos > 0) {
                    if (line.lower ().hasPrefix (String ("numberofentries"))) {
                        nr = line.mid (eq_pos + 1).stripWhiteSpace ().toInt ();
                        debugLog () << "numberofentries : "<< nr<< endl;
                        if (nr > 0 && nr < 1024)
                            entries = new Entry[nr];
                        else
                            nr = 0;
                    } else if (nr > 0) {
                        String ll = line.lower ();
                        if (ll.hasPrefix (String ("file"))) {
                            int i = line.mid (4, eq_pos-4).toInt ();
                            if (i > 0 && i <= nr)
                                entries[i-1].url = line.mid (eq_pos + 1).stripWhiteSpace ();
                        } else if (ll.hasPrefix (String ("title"))) {
                            int i = line.mid (5, eq_pos-5).toInt ();
                            if (i > 0 && i <= nr)
                                entries[i-1].title = line.mid (eq_pos + 1).stripWhiteSpace ();
                        }
                    }
                }
            }
            line.truncate (0);
            while (!in.atEnd () && line.isEmpty ())
                line = in.readLine ().stripWhiteSpace ();
        } while (!line.isEmpty ());
        for (int i = 0; i < nr; i++)
            if (!entries[i].url.isEmpty ()) {
                Mrl * m = new GenericURL (playlist_data->document, URL::decode_string (entries[i].url), entries[i].title);
                cur_elm->appendChild (m);
                m->linkNode ()->audio_only = true;
            }
        delete [] entries;
    } else if (line.stripWhiteSpace ().hasPrefix (Char ('<'))) {
        readXML (cur_elm, in, line);
        //cur_elm->normalize ();
        if (playlist_data->document && playlist_data->document->firstChild ()) {
            // SMIL documents have set its size of root-layout
            Mrl * mrl = playlist_data->document->firstChild ()->mrl ();
            if (mrl) {
                width = mrl->width;
                height = mrl->height;
                aspect = height > 0 ? width/height : 0.0;
            }
        }
    } else if (line.lower () != String ("[reference]")) do {
        if (line.lower ().hasPrefix (String ("asf ")))
            line = line.mid (4).stripWhiteSpace ();
        if (!line.isEmpty () && !line.hasPrefix (Char ('#')))
            cur_elm->appendChild(new GenericURL(playlist_data->document, line));
        line.truncate (0);
        while (!in.atEnd () && line.isEmpty ())
            line = in.readLine ().stripWhiteSpace ();
    } while (!line.isEmpty () && line != String ("--stop--")); /* TODO && m_document.size () < 1024 / * support 1k entries * /);*/
}

MediaManager *Control::mediaManager () const {
    return media_manager;
}

/* dbus-send --print-reply /com/nokia/osso_browser \
 * --dest=com.nokia.osso_browser com.nokia.osso_browser.load_url \
 *  string:http://maemo.org/
 */
bool Control::requestPlayURL (NodePtr mrl) {
    debugLog () << "requestPlayURL " << mrl->mrl()->src << (mrl->mrl ()->view_mode == Mrl::SingleMode) << endl;
    if (process ()->state () > Process::Ready) {
        if (process ()->mrl () == mrl->mrl ()->linkNode ())
            return true;
        back_request = mrl; // still playing, schedule it
        if (!setProcess (mrl->mrl ()))
            process ()->stop ();
    } else {
        if (mrl->mrl ()->view_mode == Mrl::SingleMode)
            setCurrent (mrl);
        else
            back_request = mrl;
        setProcess (mrl->mrl ());
        g_timeout_add (0, playCurrentSchedule, static_cast <void *> (this));
    }
    return true;
}

bool Control::setProcess (const char *pname) {
    GtkWidget *menu;
    Process *old_process = m_process;
    if (!strcmp (pname, mplayer->name ())) {
        m_process = mplayer;
        menu = m_app->menu_item_mplayer;
#ifdef __ARMEL__
    } else if (!strcmp (pname, npp->name ())) {
        m_process = npp;
        menu = m_app->menu_item_npp;
#endif
    } else {
        m_process = media_server;
        menu = m_app->menu_item_media_server;
    }
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu), true);
    if (old_process)
        old_process->quit ();
}

bool Control::setProcess (Mrl *mrl) {
    String pname;
    bool changed = false;
    if (!mrl->mimetype.isEmpty ())
        pname = processForMimetype (m_config, mrl->mimetype);
    if (pname.isEmpty ())
        pname = m_config->readEntry (GCONF_KEY_PLAYER, OssoMediaServer::myname);
    if (!m_process || pname != m_process->name ()) {
        setProcess ((const char *) pname);
        changed = true;
    }
    return changed;
}

void Control::setTimeout (int ms) {
    //debugLog () << "Control::setTimeout " << ms << endl;
    if (m_doc_timer)
        if (!g_source_remove (m_doc_timer))
            errorLog () << "failed to kill doc timer " << m_doc_timer << endl;
    m_doc_timer = ms > -1 ? g_timeout_add (ms, cb_docTimer, static_cast <void *> (this)) : 0;
}

void Control::openUrl (const URL &, const String &, const String & /*srv*/) {
    errorMsg ("Open for target and mimetype is not supported");
}

bool Control::needsUpdate () const {
    return viewArea ()->needsUpdateScreen ();
}

void Control::update () {
    viewArea ()->updateScreen ();
}

bool Control::docTimerEvent () {
    int id = m_doc_timer;
    if (!playlist_data->document || !playlist_data->document->active ())
        m_doc_timer = 0;
    else
        playlist_data->document->document ()->timer (); //will call setTimeout()
    return id == m_doc_timer;
}

bool Control::setCurrent (NodePtr n) {
    debugLog() << "setCurrent " << (n && n->mrl() ? n->mrl()->src : "-")<< endl;
    if (playlist_data->current == n)
        return true;
    playlist_data->current = n;
    updatePlaylistView ();
    Mrl * mrl = n ? n->mrl () : 0L;
    if (mrl && (playlist_data->current->id == SMIL::id_node_smil ||
                !mrl->linkNode ()->audio_only)) {
        if (!blanking_timer)
            blanking_timer = g_timeout_add (25000, cb_blankingTimer, m_app);
        osso_display_blanking_pause (m_app->osso_context); // 60s delay
    } else if (blanking_timer) {
        g_source_remove (blanking_timer);
        blanking_timer = 0;
    }
    if (mrl) {
        String url;
        while (mrl) {
            if (mrl->src.hasPrefix ("Playlist://"))
                break;
            if (mrl->view_mode == Mrl::SingleMode)
                url = mrl->absolutePath ();
            NodePtr p = mrl->parentNode ();
            for (mrl = 0L; p; p = p->parentNode ())
                if (p->mrl ()) {
                    mrl = p->mrl ();
                    break;
                }
        }
        app_insert_location (m_app, (const char *)url);
    }
    return true;
}

static char * stateDesc [] = {
    "init", "deferred", "activated", "began", "finished", "deactivated"
};

void Control::stateElementChanged (Node* n, Node::State os, Node::State ns) {
    //debugLog () << "state " << n->nodeName () << " changed to " << stateDesc [int (ns)] << endl;
    if (!m_process) // destructing
        return;
    if (ns == Node::state_deactivated &&
            n == playlist_data->document && !back_request) {
        stop ();
        app_set_playing (m_app, false);
    } else if ((ns == Node::state_deactivated || ns == Node::state_finished) &&
            n->mrl () && process ()->mrl () == n->mrl()->linkNode ()) {
        process ()->stop ();
        updateViewAreaSizes (); // move away video widget
    } else if ((ns == Node::state_deferred ||
                (os == Node::state_deferred && ns > Node::state_deferred)) &&
            n == playlist_data->document) {
    debugLog () << "doc state " << stateDesc [int (os)] << " changed to " << stateDesc [int (ns)] << endl;
        if ((ns == Node::state_deferred &&
                    m_process->state() != Process::Paused) ||
                (ns > Node::state_deferred &&
                 m_process->state() == Process::Paused))
            m_process->pause ();
        app_set_playing (m_app, ns > Node::state_deferred);
    } else if (ns == Node::state_activated) {
        if (n == playlist_data->document)
            app_set_playing (m_app, true);
        if (n->id == SMIL::id_node_body)
            updateViewAreaSizes (); // move away video widget
        if (n->isPlayable () &&
                n->mrl ()->view_mode == Mrl::SingleMode) {
            Node *p = n->parentNode();
            if (!p || !p->mrl () || p->mrl ()->view_mode == Mrl::SingleMode)
                // make sure we don't set current to nested document
                setCurrent (n);
        }
    }
    if (n == playlist_data->current &&
            (ns == Node::state_activated || ns == Node::state_deactivated))
        updatePlaylistView ();
}

void Control::updateViewAreaSizes () {
    if (!VIEWAREA(m_app)->videoWindow () || !m_process)
        return;
    VIEWAREA(m_app)->resizeEvent();
    Rect video_rect = VIEWAREA(m_app)->videoGeometry ();
    int x = video_rect.x ();
    int y = video_rect.y ();
#ifdef __ARMEL__
    if (process () == npp) {
        m_process->setVideoWindow (x, y,
                video_rect.width (), video_rect.height ());
    } else
#endif
    {
        for (GdkWindow * win = gdk_window_get_parent (
                    VIEWAREA(m_app)->videoWindow ());
                win;
                win = gdk_window_get_parent (win)) {
            int x1, y1;
            gdk_window_get_position (win, &x1, &y1);
            x += x1;
            y += y1;
        }
    }
    debugLog () << "updateViewAreaSizes " << x << ", " << y << " " << video_rect.width () << "x" << video_rect.height () << endl;
    m_process->setVideoWindow (x, y,
            video_rect.width (), video_rect.height ());
}

void Control::setInfoMessage (const String & msg) {
    app_set_info_content (m_app, (const char *) msg, false);
}

Surface *Control::getSurface (Mrl *mrl) {
    return VIEWAREA(m_app)->getSurface (mrl);
}

void Control::updatePlaylistView (bool force) {
    if (force) {
        if (update_tree_timer) {
            g_source_remove (update_tree_timer);
            update_tree_timer = 0;
        }
        updateTree (m_app, 0);
    } else if (!update_tree_timer)
        update_tree_timer = g_timeout_add (200, cb_scheduledUpdateTree, this);
}

static const char * statemap [] = {
        "Not Running", "Ready", "Buffering", "Playing", "Paused"
};

void Control::stateChanged (Process * p, Process::State os, Process::State ns) {
    debugLog () << "Process stateChanged " << statemap [(int)os] << " -> " << statemap [(int)ns] << endl;
    if (!m_process) // destructing
        return;
    if (!p->mrl () && ns > Process::Ready) {
        p->stop (); // reschedule to the NotRunning state
    } else if (ns == Process::NotRunning) {
        VIEWAREA(m_app)->setEventFiltering (false);
        if (p == m_process)
            if (playlist_data->document &&
                    playlist_data->document->state > Node::state_init) {
                stop (); // this should not happen
            } else if (blanking_timer) {
                g_source_remove (blanking_timer);
                blanking_timer = 0;
            }
        // else process changed
        if (!playlist_data->edit_mode)
            app_set_info_content (m_app, "", false);
    } else if (ns == Process::Ready) {
        setLoading (100); // move possible buffer info dialog
        if (os > Process::Ready) {
            setLength (0);
            setPosition (0);
            VIEWAREA(m_app)->setAspect (0.0);
            NodePtr node = p->mrl (); // p->mrl is weak, needs check
            Mrl * mrl = node ? node->mrl () : 0L;
            if (mrl && !applyBackRequest () &&  // if cause is no jump
                    mrl->active ()) {        // if cause is eof
                mrl->endOfFile ();           // set node to finished
                if (back_request)            // overlapping audio/video in SMIL
                    g_timeout_add(0, playCurrentSchedule, (void *) this);
            }
        } else
            g_timeout_add (0, playCurrentSchedule, static_cast <void *> (this));
    } else if (ns == Process::Buffering) {
        if (p->mrl ()->mrl ()->view_mode != Mrl::SingleMode) {
            AudioVideoMedia *media =
                static_cast <AudioVideoMedia *> (p->mrl()->mrl()->media_object);
            if (media)
                media->ignore_pause = true;
            p->mrl ()->defer ();
            if (media)
                media->ignore_pause = false;
        }
    } else if (ns == Process::Paused &&
            playlist_data->document->state != Node::state_deferred &&
            !playlist_data->document->document ()->postponed ()) {
        if ((m_app->window_fullscreen &&
                gtk_window_has_toplevel_focus (GTK_WINDOW (m_app->window_fullscreen))) ||
                gtk_window_has_toplevel_focus (m_app->window))
            m_process->pause ();
        else
            playlist_data->document->defer ();
    } else if (ns == Process::Playing) {
        p->mrl ()->audio_only |= !p->hasVideo ();
        if (p->mrl ()->state == Element::state_deferred) {
            AudioVideoMedia *media =
                static_cast <AudioVideoMedia *> (p->mrl()->media_object);
            if (media)
                media->ignore_pause = true;
            p->mrl ()->undefer ();
            if (media)
                media->ignore_pause = false;
        }
        setLoading (100); // move possible buffer info dialog
        recents->defer ();
        NodePtr c = recents->firstChild ();
        String url = playlist_data->document->mrl()->src;
        if (url != String ("Playlist://") && (!c || c->mrl()->src != url)) {
            recents->insertBefore (new Recent (recents, m_app, url), c);
            c = recents->firstChild ()->nextSibling ();
            int count = 1;
            KMPlayer::NodePtr more;
            while (c) {
                if (c->id == id_node_recent_node &&
                        c->mrl ()->src == url) {
                    KMPlayer::NodePtr tmp = c->nextSibling ();
                    recents->removeChild (c);
                    c = tmp;
                } else {
                    if (c->id == id_node_group_node)
                        more = c;
                    c = c->nextSibling ();
                    count++;
                }
            }
            if (!more && count > 10) {
                more = new Group (recents, m_app, "More ...");
                recents->appendChild (more);
            }
            if (more) {
                KMPlayer::NodePtr item;
                if (count > 10) {
                    KMPlayer::NodePtr item = more->previousSibling ();
                    recents->removeChild (item);
                    more->insertBefore (item, more->firstChild ());
                }
                if (more->firstChild ())
                    c = more->firstChild ()->nextSibling ();
                count = 0;
                while (c) {
                    if (c->id == id_node_recent_node &&
                            c->mrl ()->src == url) {
                        KMPlayer::NodePtr tmp = c->nextSibling ();
                        more->removeChild (c);
                        c = tmp;
                    } else {
                        c = c->nextSibling ();
                        count++;
                    }
                }
                if (count > 50)
                    more->removeChild (more->lastChild ());
            }
            updateTree (m_app, m_app->recents_tree_id);
        }
        updatePlayTree (m_app);
    }
}

void Control::errorMsg (const String & msg) {
    hildon_banner_show_information (GTK_WIDGET (m_app->window),
            NULL, (const char *) msg);
}

void Control::infoMsg (const String & msg) {
    app_set_info_content (m_app, (const char *) msg, false);
}

void Control::setLoading (int perc) {
    if (perc >= 100) {
        if (progress_banner)
            gtk_widget_destroy (progress_banner);
        progress_banner = NULL;
    } else {
        if (!progress_banner)
            progress_banner = hildon_banner_show_progress (
                    GTK_WIDGET (m_app->window), NULL, "Buffering");
        hildon_banner_set_fraction (
                HILDON_BANNER (progress_banner), 1.0 * perc / 100);
    }
}

void Control::setPosition (int p) {
    if (p < hildon_seekbar_get_total_time(HILDON_SEEKBAR(m_app->progress_bar))){
        in_progress_update = true;
        if (progress_banner)
            setLoading (100);
        hildon_seekbar_set_position (HILDON_SEEKBAR (m_app->progress_bar), p);
        in_progress_update = false;
    }
}

void Control::setLength (int len) {
    hildon_seekbar_set_total_time (HILDON_SEEKBAR (m_app->progress_bar), len);
    hildon_seekbar_set_fraction (HILDON_SEEKBAR (m_app->progress_bar), len);
    if (len > 50) {
        gtk_widget_hide (GTK_WIDGET (m_app->tb_location));
        gtk_widget_show (GTK_WIDGET (m_app->tb_progress));
    } else {
        gtk_widget_show (GTK_WIDGET (m_app->tb_location));
        gtk_widget_hide (GTK_WIDGET (m_app->tb_progress));
    }
}

void Control::setAspect (float aspect) {
    debugLog () << "Control::setAspect " << aspect << endl;
    VIEWAREA(m_app)->setAspect (aspect);
    updateViewAreaSizes ();
}

void cb_view_area_configure (GtkWidget *widget, GdkEventConfigure *event, Application * app) {
    //debugLog () << "cb_view_area_configure " << event->width<< "x" << event->height << endl;
    CONTROL(app)->updateViewAreaSizes ();
}

void Control::bitRates (int & prefered, int & maximal) {
}

PlayListData::PlayListData(short i, PlayListData * n)
    : id (i), show_all (false), edit_mode (false), next (n) {}

Application *app_new () {
    Application *app = new Application;
    memset (app, 0, sizeof (Application));
    app->control = new Control (app);
    StringPool::init();
    return app;
}

void app_destroy (Application *app) {
    delete CONTROL(app);
    delete VIEWAREA(app);
    StringPool::reset ();
}

void app_init_control (Application *app) {
    app->viewarea = new ViewArea;
    CONTROL(app)->init ();
}

void KMPlayer::jump (Node *n, Control *ctr) {
    Mrl * mrl = n ? n->mrl () : 0L;
    if (mrl && mrl->view_mode == Mrl::SingleMode) {
        if (!mrl->isPlayable ())
            return;
        if (ctr->process ()->playing () &&
                ctr->process ()->mrl() == mrl->linkNode ()) {
            if (ctr->playlist_data->document->state == Node::state_deferred)
                ctr->playlist_data->document->undefer (); // unpause
            else
                ctr->playlist_data->document->defer ();   // pause
        } else {
            //debugLog() << "jump " << mrl->src << endl;
            ctr->back_request = n;
            if (ctr->process ()->playing ())
                ctr->process ()->stop ();
            else
                ctr->applyBackRequest();
        }
    } else
        ctr->updatePlaylistView ();
}

void app_set_info_content (Application *app, const char *utf8, bool plaintext) {
    if (!plaintext && utf8[0]) {
        gtk_widget_hide (app->info_viewport);
        gtk_widget_show (app->html_viewport);
        gtk_html_load_from_string (GTK_HTML (app->html_infopanel_view),utf8,-1);
    } else {
        gtk_widget_hide (app->html_viewport);
        gtk_widget_show (app->info_viewport);
        gtk_text_buffer_set_text (gtk_text_view_get_buffer
                (GTK_TEXT_VIEW (app->infopanel_view)), utf8, -1);
    }
}

void cb_playlist_prev (GtkButton * button, void *p) {
    Control *ctr = (Control *)p;
    if (ctr->playlist_data->current &&
            ctr->playlist_data->current->previousSibling ())
        jump (ctr->playlist_data->current->previousSibling (), ctr); //FIXME
}

void cb_playlist_play (GtkButton * button, void *p) {
    Control *ctr = (Control *)p;
    g_timeout_add (0, playSchedule, static_cast <void *> (ctr));
}

void cb_playlist_next (GtkButton * button, void *p) {
    Control *ctr = (Control *)p;
    if (ctr->process ()->playing ())
        ctr->process ()->stop ();
}

void cb_playlist_stop (GtkButton * button, void *p) {
    Control *ctr = (Control *)p;
    ctr->stop ();
}

void openFile (Application * app, const gchar * path) {
    CONTROL(app)->stop ();
    PlayListData *pd = CONTROL(app)->playlist_data;
    if (pd->document)
        pd->document->document()->dispose ();
    pd->document = new Document (String (path), CONTROL(app));
    CONTROL(app)->cur_url = path;
    g_timeout_add (0, playSchedule, static_cast <void *> (CONTROL(app)));
    CONTROL(app)->updatePlaylistView (true);
}

void cb_openUrls (void *app, int argc, gchar **argv) {
}

const char * saveFile (void *p, const gchar *path) {
    Control *ctr = (Control *)p;
    if (!path)
        path = (const char *) ctr->cur_url;
    int fd = open (path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
    if (fd < 0)
        return NULL;
    PlayListData *pd = ctr->playlist_data;
    String str = pd->document
        ? pd->document->childNodes ()->length () == 1
            ? pd->document->firstChild ()->outerXML ()
            : pd->document->outerXML ()
        : "";
    ::write (fd, (const char *) str, str.length ());
    g_print ("saved %s %d bytes\n", path, str.length ());
    ::close (fd);
    ctr->cur_url = path;
    return (const char *) ctr->cur_url;
}

void newFile (Application * app) {
    PlayListData *pd = CONTROL(app)->playlist_data;
    if (pd->document) {
        pd->document->reset ();
        pd->document->document ()->dispose ();
    }
    pd->document = new Document (String (), 0L);
    CONTROL(app)->updatePlaylistView (true);
    CONTROL(app)->cur_url = String ();
}

void cb_mainview_deleted (GtkObject *, Application * app) {
    CONTROL(app)->deinit ();
    gtk_main_quit ();
    debugLog () << "destoyed" << endl;
}

gboolean cb_keyPressed (GtkWidget *, GdkEventKey * e, Application *app) {
    switch (e->keyval) {
        case GDK_F6: //HILDON_HARDKEY_FULLSCREEN:
            //if (app->control->process ()->playing ()) {
             //   int w, h;
            if (!app->fullscreen) {
                if (CONTROL(app)->playlist_data->document &&
                        CONTROL(app)->playlist_data->document->active ())
                    app_full_screen (app);
            } else {
                app_normal_screen (app);
            }
            break;
        case GDK_F7: //HILDON_HARDKEY_INCREASE
            if (CONTROL(app)->process ()->playing ())
                CONTROL(app)->process ()->volume (2, false);
            break;
        case GDK_F8: //HILDON_HARDKEY_DECREASE
            if (CONTROL(app)->process ()->playing ())
                CONTROL(app)->process ()->volume (-2, false);
            break;
        case GDK_Left:
            if (CONTROL(app)->process ()->playing ())
                CONTROL(app)->process ()->seek (-200, false); // 20s
            break;
        case GDK_Right:
            if (CONTROL(app)->process ()->playing ())
                CONTROL(app)->process ()->seek (200, false); // 20s
            break;
        case GDK_Return: // HILDON_HARDKEY_SELECT
            rowSelected (app);
            break;
        default:
            KMPlayer::debugLog () << "Keypress " << e->keyval << KMPlayer::endl;
    }
    return false;
}

void progressValueChanged(GtkWidget *w, Application * app) {
    if (!CONTROL(app)->in_progress_update &&
            CONTROL(app)->process ()->playing ()) {
        int p = hildon_seekbar_get_position(HILDON_SEEKBAR (app->progress_bar));
        CONTROL(app)->process ()->seek (p, true);
        //debugLog () << "progressBarClicked " << p << endl;
    }
}

void toggleMediaServer (GtkCheckMenuItem *radio, Application *app) {
    if (!gtk_check_menu_item_get_active (radio))
        return;
    Process *p;
    if (radio == GTK_CHECK_MENU_ITEM (app->menu_item_media_server))
        p = CONTROL(app)->media_server;
#ifdef __ARMEL__
    else if (radio == GTK_CHECK_MENU_ITEM (app->menu_item_npp))
        p = CONTROL(app)->npp;
#endif
    else
        p = CONTROL(app)->mplayer;
    CONTROL(app)->setProcess (p);
#ifdef __ARMEL__
    if (p != CONTROL(app)->npp)
#endif
        CONTROL(app)->m_config->writeEntry(GCONF_KEY_PLAYER, String(p->name()));
}

void app_set_suspend (Application *a, bool suspend) {
    PlayListData *pd = CONTROL(a)->playlist_data;
    if (pd->document &&
            pd->document->active () &&
            (!CONTROL(a)->process ()->playing () ||
             CONTROL(a)->process()->mrl()->view_mode ==Mrl::WindowMode ||
             CONTROL(a)->process ()->hasVideo ())) {
        if (pd->document->state == Node::state_deferred) {
            if (!suspend)
                ; // do nothing  a->playlist_data->document->undefer ();
        } else if (suspend)
            pd->document->defer ();
    }
}

static bool locationbar_initialized;

gboolean
locationbarGainFocus (GtkWidget *, GdkEventFocus *, Application * app) {
    int n = gtk_toolbar_get_n_items (GTK_TOOLBAR (app->main_toolbar));
    for (int i = 5; i < n; i++)
        gtk_widget_hide (GTK_WIDGET (gtk_toolbar_get_nth_item (GTK_TOOLBAR (app->main_toolbar), i)));
    app_set_playing (app, app->playing);
    if (!locationbar_initialized) {
        KMPlayer::debugLog() << "locationbar_initializing" << KMPlayer::endl;
        const char * defs[6] = {
            "http://www.cwi.nl/SMIL/EUROshow/EUROshowB.smi",
            "http://homepages.cwi.nl/~media/SMIL/fiets/fiets.smil",
            "http://www.theworld.org/rss/tech.xml",
            "http://www.etherbeat.com/shoutcast/64.pls",
            "http://www.swissgroove.ch/listen.m3u",
            "file:///home/user/MyDocs/.videos/Discovery.avi"
        };
        int defpos = 0;
        locationbar_initialized = true;
        KMPlayer::Config * config = CONTROL(app)->m_config;
        KMPlayer::String cfg_prefix (GCONF_KEY_LOCATION_URL);
        GtkTreeIter iterator;
        GtkListStore * model = GTK_LIST_STORE (gtk_combo_box_get_model (
                    GTK_COMBO_BOX (app->location_bar)));
        for (int i = 1; i < 7; i++) {
            gtk_list_store_append (model, &iterator);
            KMPlayer::String cfg = config->readEntry (cfg_prefix +
                    KMPlayer::String::number(i), "");
            if (cfg.isEmpty ())
                cfg = defs[defpos++];
            gtk_list_store_set (model, &iterator,
                0, (const char *) cfg, -1);
        }
    }
    gtk_editable_select_region (GTK_EDITABLE (GTK_BIN (app->location_bar)->child), 0, -1);
    //g_print ("callback_locationbar_gain_focus %d\n", n);
    return false;
}

void save_locationbar (Application *app) {
    if (!locationbar_initialized)
        return;
    KMPlayer::Config * config = CONTROL(app)->m_config;
    int idx = 1;
    KMPlayer::String cfg_pfx (GCONF_KEY_LOCATION_URL);
    GtkTreeIter iterator;
    GtkTreeModel * model = GTK_TREE_MODEL (gtk_combo_box_get_model (
                GTK_COMBO_BOX (app->location_bar)));
    for (bool b = gtk_tree_model_get_iter_first (model, &iterator);
            b;
            b = gtk_tree_model_iter_next (model, &iterator)) {
        GValue value = { 0 , };
        gtk_tree_model_get_value (model, &iterator, 0, &value);
        KMPlayer::String val (g_value_get_string (&value));
        g_value_unset (&value);
        KMPlayer::debugLog() << val << " " << idx << KMPlayer::endl;
        if (!val.isEmpty ())
            config->writeEntry (cfg_pfx + KMPlayer::String::number(idx++), val);
    }
}

/*
 * TODO
static void cb_iap (struct iap_event_t * event, void * arg) {
    if (event->type == OSSO_IAP_CONNECTED) {
    }
}

osso_iap_cb (cb_iap);
osso_iap_connect (OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, 0L);

'dbus-send --print-reply --dest=com.nokia.osso_browser /usr/bin/browser com.nokia.osso_browser.load_url string:%s' % url
*/
