/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <string.h>
#include <map>

#include "mediaobject.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "viewarea.h"

namespace KMPlayer {

const unsigned int event_media_ready = 100;
const unsigned int event_img_updated = 101;
const unsigned int event_img_anim_finished = 102;

}

using namespace KMPlayer;

namespace {
    typedef std::map <String, ImageDataPtrW> ImageDataMap;
    static ImageDataMap *image_data_map;
}

//------------------------%<----------------------------------------------------

MediaManager::MediaManager (Control *ctrl) : m_control (ctrl) {
    if (!image_data_map)
        image_data_map = new ImageDataMap;
}

MediaManager::~MediaManager () {
    if (image_data_map->size () == 0) {
        delete image_data_map;
        image_data_map = 0;
    }
}

MediaObject *MediaManager::createMedia (MediaType type, Node *node) {
    switch (type) {
        case Audio:
        case AudioVideo:
            //if (!m_control->source()->authoriseUrl (
            //            node->mrl()->absolutePath ()))
            //    return NULL;

            return new AudioVideoMedia (this, node);
        case Image:
            return new ImageMedia (this, node);
        case Text:
            return new TextMedia (this, node);
    }
    return NULL;
}

//------------------------%<----------------------------------------------------

MediaObject::MediaObject (MediaManager *manager, Node *node)
 : m_manager (manager), m_node (node), job (NULL) {
   //manager->medias ().push_back (this);
}

MediaObject::~MediaObject () {
    clearData ();
    //m_manager->medias ().remove (this);
}

KDE_NO_EXPORT void MediaObject::killWGet () {
    if (job) {
        job->kill (); // quiet, no result signal
        job = 0L;
        clearData (); // assume data is invalid
    }
}

/**
 * Gets contents from url and puts it in m_data
 */
KDE_NO_EXPORT bool MediaObject::wget (const String &str) {
    url = str;
    URL uri (str);
    clearData ();
    debugLog () << "MediaObject::wget " << str << endl;
    if (uri.isLocalFile ()) {
        File file (uri.path ());
        if (file.exists () && file.open (IO_ReadOnly)) {
            data = file.readAll ();
            file.close ();
        }
        ready (str);
        return true;
    } else
        job = asyncGet (this, uri.url ());
    return false;
}

Mrl *MediaObject::mrl () {
    return m_node ? m_node->mrl () : NULL;
}

KDE_NO_EXPORT String MediaObject::mimetype () {
    if (data.size () > 0 && mime.isEmpty ())
        mime = MimeType::findByContent (data.data (), data.size ());
    return mime;
}

KDE_NO_EXPORT void MediaObject::clearData () {
    killWGet ();
    url.truncate (0);
    mime.truncate (0);
    data.resize (0);
}

KDE_NO_EXPORT void MediaObject::destroy () {
    delete this;
}

KDE_NO_EXPORT bool MediaObject::downloading () const {
    return !!job;
}

KDE_NO_EXPORT void MediaObject::ready (const String &) {
    if (m_node)
        m_node->handleEvent (new Event (event_media_ready));
}

KDE_NO_EXPORT void MediaObject::jobResult (Job *jb) {
    if (jb->error ())
        data.resize (0);
    job = 0L; // signal KIO::Job::result deletes itself
    ready (url);
}

KDE_NO_EXPORT void MediaObject::jobData (Job *jb, ByteArray &buf) {
    //debugLog () << "MediaTypeRuntime::jobData " << data.size () << endl;
    if (data.size ())
        mime = jb->contentType ();
    if (buf.size ()) {
        int old_size = data.size ();
        data.resize (old_size + buf.size ());
        memcpy (data.data () + old_size, buf.data (), buf.size ());
    }
}

//------------------------%<----------------------------------------------------

AudioVideoMedia::AudioVideoMedia (MediaManager *manager, Node *node)
 : MediaObject (manager, node),
   ignore_pause (false) {
    debugLog() << "AudioVideoMedia::AudioVideoMedia" << endl;
}

AudioVideoMedia::~AudioVideoMedia () {
    stop ();
    debugLog() << "AudioVideoMedia::~AudioVideoMedia" << endl;
}

bool AudioVideoMedia::play () {
    Process *process = m_manager->control ()->process ();
    if (process && process->playing () && process->mrl () == mrl ()) {
        errorLog () << "play: already playing " << mrl ()->src << endl;
    } else if (m_manager->control ()->requestPlayURL (m_node)) {
        if (!mrl ()->audio_only)
            m_manager->control ()->viewArea ()->setAudioVideoNode (m_node);
        ignore_pause = true;
        m_node->defer ();
        ignore_pause = false;
        return true;
    }
    return false;
}

void AudioVideoMedia::stop () {
    Process *process = m_manager->control ()->process ();
    if (process)
        process->stop ();
    m_manager->control ()->viewArea ()->setAudioVideoNode (NULL);
}

void AudioVideoMedia::pause () {
    Process *process = m_manager->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

void AudioVideoMedia::unpause () {
    Process *process = m_manager->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

//------------------------%<----------------------------------------------------

ImageData::ImageData( const String & img) : image (0L), url (img) {
    //if (img.isEmpty ())
    //    //debugLog() << "New ImageData for " << this << endl;
    //else
    //    //debugLog() << "New ImageData for " << img << endl;
}

ImageData::~ImageData() {
    if (!url.isEmpty ())
        image_data_map->erase (url);
    delete image;
}

ImageMedia::ImageMedia (MediaManager *manager, Node *node)
 : MediaObject (manager, node),
   img_movie (NULL),
   img_movie_iter (NULL),
   pix_loader (NULL),
   img_movie_timer (0),
   timeout (0) {}

ImageMedia::~ImageMedia () {
    stop ();
    if (img_movie_iter)
        g_object_unref (img_movie_iter);
    if (pix_loader)
        //gdk_pixbuf_loader_close (pix_loader, 0L);
        g_object_unref (pix_loader);
    else if (img_movie)
        g_object_unref (img_movie);
}

KDE_NO_EXPORT bool ImageMedia::play () {
    if (img_movie)
        unpause ();
    return img_movie;
}

KDE_NO_EXPORT void ImageMedia::stop () {
    pause ();
}

void ImageMedia::pause () {
    if (img_movie_timer)
        g_source_remove (img_movie_timer);
    img_movie_timer = 0;
}

gboolean movieUpdateTimeout (void * p) {
   return ((ImageMedia *)p)->movieTimer ();
}

bool ImageMedia::movieTimer () {
    bool ret = false;
    if (gdk_pixbuf_animation_iter_advance (img_movie_iter, 0L)) {
        setUrl (String ());
        timeout = 0;
        cached_img->image = new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter)));
        int to = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (to == timeout) {
            ret = true; // re-use timer
        } else {
            timeout = to;
            if (to > 0)
                img_movie_timer = g_timeout_add (to, movieUpdateTimeout, this);
        }
        if (m_node)
            m_node->handleEvent (new Event (event_img_updated));
    } else if (m_node) {
        m_node->handleEvent (new Event (event_img_anim_finished));
    }
    return ret;
}

void ImageMedia::unpause () {
    if (img_movie_iter && !img_movie_timer) {
        setUrl (String ());
        timeout = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (timeout > 0)
            img_movie_timer = g_timeout_add (timeout, movieUpdateTimeout, this);
    }
}

KDE_NO_EXPORT void ImageMedia::ready (const String &url) {
    if (data.size ()) {
        String mime = mimetype ();
        if (!mime.hasPrefix ("text/")) { // FIXME svg
            GError *gerror = 0L;
            setUrl (url);
            pix_loader = gdk_pixbuf_loader_new ();
            if (gdk_pixbuf_loader_write (pix_loader,
                        (const guchar *) data.data (),
                        data.size (), &gerror) &&
                    gdk_pixbuf_loader_close (pix_loader, 0L))
                img_movie = gdk_pixbuf_loader_get_animation (pix_loader);
            if (img_movie && !gdk_pixbuf_animation_is_static_image(img_movie)) {
                img_movie_iter = gdk_pixbuf_animation_get_iter(img_movie, NULL);
                setUrl (String ());
                cached_img->image = new Image (GPixbuf (
                     gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter)));
            } else if (img_movie) {
                cached_img->image = new Image (GPixbuf (
                            gdk_pixbuf_animation_get_static_image (img_movie)));
            }
            g_free (gerror);
        }
    }
    MediaObject::ready (url);
}

bool ImageMedia::isEmpty () {
    return !cached_img || !cached_img->image;
}

void ImageMedia::setUrl (const String & url) {
    if (url.isEmpty ()) {
        cached_img = ImageDataPtr (new ImageData (url));
    } else {
        ImageDataMap::iterator i = image_data_map->find (url);
        if (i == image_data_map->end ()) {
            cached_img = ImageDataPtr (new ImageData (url));
            (*image_data_map)[url] = ImageDataPtrW (cached_img);
        } else {
            cached_img = i->second;
        }
    }
}

//------------------------%<----------------------------------------------------

TextMedia::TextMedia (MediaManager *manager, Node *node)
 : MediaObject (manager, node), default_font_size (14)
{}

TextMedia::~TextMedia () {
}

KDE_NO_EXPORT void TextMedia::ready (const String &url) {
    if (data.size ())
        text = String (data.data (), data.size ());
    debugLog() << "TextMedia::ready " << data.size () << " " << text << endl;
    MediaObject::ready (url);
}

bool TextMedia::play () {
    return !text.isEmpty ();
}
