/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@xs4all.nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include <config.h>
#include <math.h>
#include <sys/types.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#include "kmplayerprocess.h"
#ifdef __ARMEL__
# include "npplayer.h"
# include "kmplayer_lists.h"
#endif

using namespace KMPlayer;


Process::Process (ProcessNotify * ctrl, const char * n, bool xv)
 : m_notify (ctrl), m_viewer (0L), m_name (n),
      m_state (NotRunning), m_old_state (NotRunning),// m_process (0L),
      start_timer (0), no_video (false), has_xv (xv)
      {}

Process::~Process () {
    stop ();
    if (start_timer)
        g_source_remove (start_timer);
}

//WId Process::widget () {
//    return 0;
//}

bool Process::playing () const {
    return m_state > Ready;
}

//void Process::setAudioLang (int, const QString &) {}

//void Process::setSubtitle (int, const QString &) {}

void Process::setPosition (int p) {
    m_position = p;
    m_notify->setPosition (m_position);
}

bool Process::pause () {
    return false;
}

bool Process::seek (int /*pos*/, bool /*absolute*/) {
    return false;
}

bool Process::volume (int /*pos*/, bool /*absolute*/) {
    return false;
}

bool Process::saturation (int /*pos*/, bool /*absolute*/) {
    return false;
}

bool Process::hue (int /*pos*/, bool /*absolute*/) {
    return false;
}

bool Process::contrast (int /*pos*/, bool /*absolute*/) {
    return false;
}

bool Process::brightness (int /*pos*/, bool /*absolute*/) {
    return false;
}

void Process::setVideoWindow (int, int, int, int) {}

//bool Process::grabPicture (const KURL & /*url*/, int /*pos*/) {
//    return false;
//}

//bool Process::supports (const char * source) const {
//    for (const char ** s = m_supported_sources; s[0]; ++s) {
//        if (!strcmp (s[0], source))
//            return true;
//    }
//    return false;
//}

bool Process::stop () {
    if (!playing ()) return true;
    return false;
}

bool Process::quit () {
    debugLog() << "quit" << endl;
    stop ();
    setState (NotRunning);
    return !playing ();
}

static gboolean cb_scheduledUpdateAspects (void * p) {
    static_cast <Process *> (p)->updateAspects ();
    return false; // single shot
}

void Process::updateAspects () {
    if (m_notify)
        m_notify->setAspect (m_height > 0 ? 1.0 * m_width / m_height : 0);
}

static gboolean cb_scheduledStateChanged (void * p) {
    static_cast <Process *> (p)->scheduledStateChanged ();
    return false; // single shot
}

void Process::setState (State newstate) {
    if (m_state != newstate) {
        m_old_state = m_state;
        m_state = newstate;
        start_timer = g_timeout_add (0, cb_scheduledStateChanged, static_cast <void *> (this));
    }
}

void Process::scheduledStateChanged () {
    m_notify->stateChanged (this, m_old_state, m_state);
    start_timer = 0;
}

KMPLAYER_NO_EXPORT bool Process::play (NodePtr _mrl) {
    m_mrl = _mrl;
    Mrl * m = _mrl ? _mrl->mrl () : 0L;
    m_url = m ? m->absolutePath () : String ();
    no_video = m ? m->audio_only : false;
    m_position = m_length = 0;
    m_width = m_height = 0;
    return false;
}

bool Process::ready (unsigned int viewer) {
    m_viewer = viewer;
    setState (Ready);
    return true;
}

unsigned int Process::viewer () const {
    return m_viewer;
}

//-----------------------------------------------------------------------------

static void setupMPlayerFunc (gpointer data) {
    static_cast <MPlayer *> (data)->setState (Process::Playing); //FIXME
}

static void watchMPlayerFunc (GPid pid, gint status, gpointer data) {
    //debugLog () << "watchMPlayerFunc " << endl;
    g_spawn_close_pid (pid);
    static_cast <MPlayer *> (data)->processExited (pid);
}

static gboolean watchMPlayerStdin (GIOChannel *, GIOCondition, gpointer data) {
    static_cast <MPlayer *> (data)->writeEvent ();
    return true;
}

static gboolean watchMPlayerStdout (GIOChannel *src, GIOCondition, gpointer data) {
    static_cast <MPlayer *> (data)->readEvent (src);
    return true;
}

#define NO_SEEK 0x7fffffff

MPlayer::MPlayer (ProcessNotify * ctrl, bool xv)
 : Process (ctrl, "mplayer", xv),
   pin (0L), pout (0L), win (0), wout (0),
   wait_for_pause (false) {
    m_rect = Rect (434, 60, 342, 260);
}

MPlayer::~MPlayer () {
}

bool MPlayer::play (NodePtr node) {
    debugLog () << "MPlayer::play" << endl;
    if (playing ())
        return pause ();
    wait_for_pause = false;
    setState (Buffering);
    Mrl *mrl = node ? node->mrl () : 0L;
    if (!mrl) {
        errorLog() << "Not a Mrl " << (node ? node->nodeName() :"null") << endl;
        setState (Ready);
        return false;
    }
    String strurl = mrl->absolutePath ();
    if (strurl.isEmpty ()) {
        setState (Ready);
        return false;
    }
    Process::play (node);
    request_seek = NO_SEEK;
    String wd ("/home/user");
    URL url (strurl);
    if (url.isLocalFile ()) {
        strurl = url.path ();
        wd = url.path ();
        int pos = wd.findRev (String ("/"));
        if (pos > -1)
            wd.truncate (pos);
    }
    request_quit = false;
    outbuf[0] = 0;
    String cmdline ="mplayer -nofs -slave -framedrop -identify -vo ";
    char buf[128];
    //if (has_xv)
    //    snprintf (buf, sizeof(buf), "xv -wid %u ", m_viewer);
    //else
        snprintf (buf, sizeof(buf),
                "xv,nokia770:fb_overlay_only:x=%d:y=%d:w=%d:h=%d",
                m_rect.x() - (m_rect.x() % 2),
                m_rect.y() - (m_rect.y() % 2),
                m_rect.width() - (m_rect.width() % 4),
                m_rect.height() - (m_rect.height() % 4));
    cmdline += String (buf);
    snprintf (buf, sizeof(buf), " -wid %u ", m_viewer);
    cmdline += String (buf);
    if (m_mrl->mrl ()->repeat > 0)
        cmdline += String (" -loop ") + String::number (m_mrl->mrl()->repeat+1);
    cmdline += Char (' ');
    char *quoted = g_shell_quote ((const char *) strurl);
    cmdline += String (quoted);
    g_free (quoted);
    debugLog () << cmdline << endl;
    int argc;
    gchar **argv;
    GError *err = 0L;
    int fdin, fdout, fderr;
    if (g_shell_parse_argv ((const char *) cmdline, &argc, &argv, &err) &&
            g_spawn_async_with_pipes ((const char *) wd, 
                argv,
                0L,
                (GSpawnFlags)(G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD),
                setupMPlayerFunc,
                static_cast <void*> (this),
                &process_id,
                &fdin,
                &fdout,
                0L /*&fderr*/,
                &err)) {
        //darn glib: assertion err!=null g_error_free (err);
        g_strfreev (argv);
        m_mrl = node;
        g_child_watch_add(process_id,watchMPlayerFunc,static_cast<void*>(this));
        pin = g_io_channel_unix_new (fdin);
        pout = g_io_channel_unix_new (fdout);
        g_io_channel_set_encoding (pout, 0L, 0L);
        GIOFlags flags = g_io_channel_get_flags (pout);
        g_io_channel_set_flags (pout, (GIOFlags)(flags|G_IO_FLAG_NONBLOCK), 0L);
        g_io_channel_set_buffer_size (pout, 80);
        wout = g_io_add_watch (pout, G_IO_IN, watchMPlayerStdout, this);
        return true;
    }
    errorLog () << err->message << endl;
    g_error_free (err);
    setState (Ready);
    return false;
}

bool MPlayer::pause () {
    wait_for_pause = !wait_for_pause;
    debugLog() << "MPlayer::pause schedule " << wait_for_pause << endl;
    for (StringList::iterator i = commands.begin (); i != commands.end (); ++i)
        if (i->hasPrefix ("pause")) {
            commands.erase (i);
            return true;
        }
    return sendCommand (String ("pause"));
}

bool MPlayer::ready (unsigned int viewer) {
    if (m_state < Ready)
        return Process::ready (viewer);
    return false;
}

bool MPlayer::stop () {
    debugLog() << "stop mplayer" << endl;
    return sendCommand (String ("quit"));
}

bool MPlayer::quit () {
    if (playing ()) {
        request_quit = true;
        stop ();
        while (g_main_context_iteration (0L, false))
            if (!pin)
                break;
        for (int n = 0; pin && n < 50; n++) {
            usleep (10000); // 10ms
            while (g_main_context_iteration (0L, false))
                if (!pin)
                    break;
        }
        if (pin) {
            errorLog () << "stop MPlayer" << endl;
            ::kill (process_id, SIGINT);
            for (int n = 0; pin && n < 100; n++) {
                usleep (10000); // 10ms
                while (g_main_context_iteration (0L, false))
                    if (!pin)
                        break;
            }
            if (pin) {
                errorLog () << "interupt MPlayer" << endl;
                ::kill (process_id, SIGTERM);
                for (int n = 0; pin && n < 50; n++) {
                    usleep (10000); // 10ms
                    while (g_main_context_iteration (0L, false))
                        if (!pin)
                            break;
                }
            }
            if (pin)
                processExited (process_id); // give up
        }
        commands.clear ();
        request_quit = false;
    }
    setState (NotRunning);
    return true;
}

bool MPlayer::seek (int pos, bool absolute) {
    if (/*request_seek != NO_SEEK &&*/ commands.size () > 1) {
        StringList::iterator i = commands.begin ();
        StringList::iterator end ( commands.end () );
        for (++i; i != end; ++i)
            if ((*i).hasPrefix (String ("seek"))) {
                i = commands.erase (i);
                request_seek = NO_SEEK;
                break;
            }
    }
    if (request_seek != NO_SEEK)
        return false;
    if (absolute && m_length > 0) {
        pos -= m_position;
        absolute = false;
    }
    request_seek = pos;
    char buf[32];
    snprintf (buf, sizeof (buf), "seek %d %d", pos/10, absolute ? 2 : 0);
    return sendCommand (String (buf));
}

bool MPlayer::volume (int pos, bool /*absolute*/) {
    return sendCommand (String ("volume ") + String::number (pos));
}

bool MPlayer::saturation (int pos, bool absolute) {
    return true;
}

bool MPlayer::hue (int pos, bool absolute) {
    return true;
}

bool MPlayer::contrast (int pos, bool absolute) {
    return true;
}

bool MPlayer::brightness (int pos, bool absolute) {
    return true;
}

void MPlayer::setVideoWindow (int x, int y, int w, int h) {
    debugLog() << "setVideoWindow " << w << " " << h << endl;
    if (x < 0 || y < 0 || w <= 0 || h <= 1 /* hack !!*/ ||
                x > 800 || y > 480 || w > 800 || h > 480)
        return;
    m_rect = Rect (x, y, w, h);
    if (!hasVideo ())
        return;
    char buf[64];
    snprintf (buf, sizeof (buf), "vo_change_rectangle %d %d %d %d",
            x - (x % 2), y - (y % 2), w - (w % 4), h - (h % 4));
    sendCommand (String (buf));
}

void MPlayer::processExited (GPid pid) {
    if (pid == process_id) {
        process_id = 0;
        commands.clear ();
        if (win) {
            g_source_remove (win);
            win = 0;
        }
        g_io_channel_shutdown (pin, true, 0L);
        g_io_channel_unref (pin);
        pin = 0L;
        g_source_remove (wout);
        wout = 0;
        g_io_channel_shutdown (pout, true, 0L);
        g_io_channel_unref (pout);
        pout = 0L;
        if (m_state > Ready && !request_quit)
            setState (Ready);
    }
}

void MPlayer::writeEvent () {
    if (!pin)
        return;
    String data;
    if (command_chunk.isEmpty ()) {
        if (commands.size ()) {
            data = *commands.begin ();
            commands.pop_front ();
        }
    } else
        data = command_chunk;
    debugLog () << "eval " << data /*<< endl*/;
    gsize nr = 0;
    if (!data.isEmpty ()) {
        g_io_channel_write_chars(pin, (const char*)data, data.length(), &nr,0L);
        g_io_channel_flush (pin, 0L);
    }
    if (nr < data.length ()) {
        //debugLog () << "partial " << nr << " " << data.length() << endl;
        command_chunk = String (((const char*) data) + nr);
    } else if (!commands.size ()) {
        // FIXME: always remove and readd when new data from stdin arrives
        g_source_remove (win);
        win = 0;
    }
}

void MPlayer::readEvent (GIOChannel * src) {
    gsize nr;
    int inbuf = strlen (outbuf);
    g_io_channel_read_chars (src, outbuf+inbuf, sizeof (outbuf)-inbuf, &nr, 0L);
    //debugLog() << "read " << nr << " chars had " << inbuf << endl;
    outbuf[inbuf + nr] = 0;
    if (src == pout) {
        char *tok = outbuf;
        for (int i = 0; i < inbuf + nr; i++) {
            if (outbuf[i] == '\r' && outbuf[i+1] != '\n') {
                outbuf[i] = 0;
                char * c = strstr (tok, "A:");
                if (!c)
                    c = strstr (tok, "V:");
                if (c) {
                    setPosition ((int)(10 * strtof (c + 2, 0L)));
                    request_seek = NO_SEEK;
                    setState (Playing);
                } else {
                    c = strstr (tok, "Cache fill:");
                    if (c) {
                        char *p = strchr (tok + 11, '%');
                        if (p)
                            *p = 0;
                        //debugLog() << "c " << (c + 11) << endl;
                        m_notify->setLoading ((int)strtof (c + 11, 0L));
                    }
                    if (m_state == Paused)
                        setState (Playing);
                }
                tok = outbuf + i + 1;
            } else if (outbuf[i] == '\n') {
                bool handled = false;
                outbuf[i] = 0;
                debugLog() << "> " << tok << endl;
                if (m_state == Buffering) {
                    char * p = strstr (tok, "Start");
                    if (p && strstr (p, "play")) { // find a 'Start.*play' pattern
                        m_notify->setLoading (99);
                        handled = true;
                    } else {
                        char * c = strstr (tok, "Cache fill:");
                        if (c) {
                            char *p = strchr (tok + 11, '%');
                            if (p)
                                *p = 0;
                            //debugLog() << "c2 " << (c + 11) << endl;
                            m_notify->setLoading ((int)strtof (c + 11, 0L));
                            handled = true;
                        } else {
                            char * v = strstr (tok, "Video: no video");
                            if (v) {
                                no_video = true;
                                handled = true;
                            }
                        }
                    }
                }
                if (!handled) {
                    if (!strncmp (tok, "ID_", 3)) {
                        handled = true;
                        if (!strncmp (tok + 3, "LENGTH", 6)) {
                            m_length = (int)(10 * strtof (tok + 10, 0L));
                            m_notify->setLength (m_length);
                        } else if (!strncmp (tok + 3, "PAUSED", 6)) {
                            if (wait_for_pause) {
                                wait_for_pause = false;
                                setState (Paused);
                            } // else ignore, a second 'pause' cmd should follow
                        } else if (!strncmp (tok + 3, "VIDEO_WIDTH", 11)) {
                            m_width = (int) strtof (tok + 15, 0L);
                            if (m_height > 0)
                                updateAspects ();
                        } else if (!strncmp (tok + 3, "VIDEO_HEIGHT", 12)) {
                            m_height = (int) strtof (tok + 16, 0L);
                            if (m_width > 0)
                                updateAspects ();
                        }
                    } else {
                        char * icy = strstr (tok, "ICY Info");
                        if (icy) {
                            //debugLog () << "icy found" << endl;
                            char *t = strstr (icy + 8, "StreamTitle=");
                            if (t) {
                                char *e = strchr (t + 12, ';');
                                if (e)
                                    *e = 0;
                                processNotify ()->infoMsg (String (t + 12));
                            }
                            handled = true;
                        }
                    }
                }
                tok = outbuf + i + 1;
            }
        }
        inbuf = tok - outbuf;
        if (inbuf > 0)
            memmove (outbuf, tok, inbuf);
        outbuf[inbuf] = 0;
    }
}

bool MPlayer::sendCommand (const String & cmd) {
    if (playing ()) {
        commands.push_back (cmd + Char ('\n'));
        if (!win)
            win = g_io_add_watch (pin, G_IO_OUT, watchMPlayerStdin, this);
        return true;
    }
    return false;
}

//-----------------------------------------------------------------------------

static const char *media_server_filter = "type='signal',sender='com.nokia.osso_media_server'";

gboolean cb_stoppedTimer (void * p) {
    OssoMediaServer * process = static_cast <OssoMediaServer *> (p);
    process->processStopped ();
    return 0;
}

static gboolean cb_retryAudioOnly (void * p) {
    Process *proc = static_cast <Process *> (p);
    if (proc->mrl ())
        proc->play (proc->mrl ());
    return false; // single shot
}

/*
  AddMatch
  sender=com.nokia.osso_media_server if=com.nokia.soso_media_server.video
  sender=com.nokia.osso_media_server if=com.nokia.soso_media_server.video.error
  set_volume if=com.nokia.soso_media_server.video double:1.0
  signal if=com.nokia.soso_media_server.video mtd=state_changed playing/pauses/stopped/end_of_stream/restart
  signal if=com.nokia.soso_media_server.video mtd=details_received
  signal if=com.nokia.soso_media_server.video mtd=info_buffering double:perc
  signal if=com.nokia.soso_media_server.video mtd=end_buffering double:0.0
  com.nokia.osso_media_server.sound
  com.nokia.osso_media_server.music
  run-standalone.sh dbus-send --dest=com.nokia.osso_media_server --print-reply /com/nokia/osso_media_server com.nokia.osso_media_server.music.play_media string:file:///home/user/MyDocs/.sounds/Everyday.mp3
*/

static DBusHandlerResult filter_func (DBusConnection * connection,
        DBusMessage *message, void * user_data) {
    const char *sender = dbus_message_get_sender (message);
    int message_type = dbus_message_get_type (message);
    bool handled = false;
    OssoMediaServer *process = static_cast <OssoMediaServer *> (user_data);

    if (!process->dbusFilter ())
        return DBUS_HANDLER_RESULT_HANDLED;

    if (message_type == DBUS_MESSAGE_TYPE_SIGNAL &&
            !strcmp (dbus_message_get_interface (message),
                       process->currentMediaServer ())) {
        DBusMessageIter iter;
        char *str;
        dbus_message_iter_init (message, &iter);
        do {
            int type = dbus_message_iter_get_arg_type (&iter);
            const char * method = dbus_message_get_member (message);
            handled = true;
            if (!strcmp (method, "info_buffering")) {
                if (type == DBUS_TYPE_DOUBLE) {
                    double perc;
                    dbus_message_iter_get_basic (&iter, &perc);
                    // printf ("buffering %0.2f\n", perc);
                    process->processBuffering ((int) perc);
                } else
                    printf ("buffering wrong type\n");
            } else if (!strcmp (method, "begin_buffering")) {
                printf ("begin_buffering\n");
                process->processBuffering (0);
            } else if (!strcmp (method, "end_buffering")) {
                printf ("end_buffering\n");
                process->processBuffering (100);
            } else if (!strcmp (method, "state_changed")) {
                if (type == DBUS_TYPE_STRING) {
                    const char * state;
                    dbus_message_iter_get_basic (&iter, &state);
                    printf ("state_changed %s\n", state);
                    if (!strcmp (state, "stopped")) {
                        if (!process->play_pending && !process->retry ())
                            process->processStopped ();
                    } else if (!strcmp (state, "playing"))
                        process->processPlaying ();
                    else if (!strcmp (state, "paused"))
                        process->processPaused ();
                } else
                    printf ("state_changed wrong type\n");
            } else if (!strcmp (method, "end_of_stream")) {
                printf ("end_of_stream\n");
                process->processStopped ();
            } else if (!strcmp (method, "restart")) {
                printf ("restart\n");
                process->processStopped ();
            } else if (!strcmp (method, "iradio_update")) {
                if (type == DBUS_TYPE_STRING) {
                    const char * info;
                    dbus_message_iter_get_basic (&iter, &info);
                    printf ("iradio_update %s\n", info);
                    process->processNotify ()->infoMsg (info);
                } else
                    printf ("iradio_update %d\n", type);
            } else if (!strcmp (method, "details_received")) {
                debugLog() << "details_received " << type << endl;
                bool aspect_changed = false;
                if (type == DBUS_TYPE_ARRAY) {
                    bool eof = false;
                    DBusMessageIter sub;
                    dbus_message_iter_recurse (&iter, &sub);
                    do {
                        type = dbus_message_iter_get_arg_type (&sub);
                        if (type == DBUS_TYPE_DICT_ENTRY) {
                            DBusMessageIter dictsub;
                            dbus_message_iter_recurse (&sub, &dictsub);
                            type = dbus_message_iter_get_arg_type (&dictsub);
                            String key, value;
                            const char * detail;
                            if (type == DBUS_TYPE_STRING) {
                                dbus_message_iter_get_basic (&dictsub, &detail);
                                key = detail;
                            }
                            if (dbus_message_iter_next (&dictsub)) {
                                type = dbus_message_iter_get_arg_type (&dictsub);
                                if (type == DBUS_TYPE_STRING) {
                                    dbus_message_iter_get_basic (&dictsub, &detail);
                                    value = detail;
                                }
                            }
                            if (!key.isEmpty ()) {
                                debugLog()<<"detail " <<key << "=" << value << endl;
                                if (key == "width") {
                                    process->setWidth (value.toInt ());
                                    aspect_changed = true;
                                } else if (key == "height") {
                                    process->setHeight (value.toInt ());
                                    aspect_changed = true;
                                } else if (key == "length") {
                                    int len = value.toInt () / 100;
                                    process->setLength (len);
                                }
                            }
                        } else
                            debugLog() << "details_received sub " << type << endl;
                        eof = !dbus_message_iter_next (&sub);
                    } while (!eof);
                }
                if (aspect_changed)
                    g_timeout_add (0, cb_scheduledUpdateAspects, process);
                //process->updateVideoSize();
            } else {
                debugLog() << "Unhandled " << method << endl;
                handled = false;
            }
        } while (dbus_message_iter_next (&iter));
    } else if (message_type == DBUS_MESSAGE_TYPE_SIGNAL &&
            !strcmp (dbus_message_get_interface (message),
                "com.nokia.osso_media_server.video.error")) {
        const char * method = dbus_message_get_member (message);
        warningLog() << "Unsupported format " << method << endl;
        if (!strcmp (method, "unsupported_type")) {
            if (!process->retry ())
                process->processStopped ();
            handled = true;
        }
    }
    if (!handled /*&&
            strstr (dbus_message_get_interface (message), "osso_media_server")*/)
        switch (message_type) {
            case DBUS_MESSAGE_TYPE_INVALID:
                printf ("invalid message");
                break;
            case DBUS_MESSAGE_TYPE_METHOD_CALL:
                printf ("message interface=%s; member=%s; sender=%s\n",
                        dbus_message_get_interface (message),
                        dbus_message_get_member (message),
                        sender ? sender : "(no sender)");
                break;
            case DBUS_MESSAGE_TYPE_SIGNAL:
                printf ("signal interface=%s; member=%s; sender=%s\n",
                        dbus_message_get_interface (message),
                        dbus_message_get_member (message),
                        sender ? sender : "(no sender)");
                break;
            case DBUS_MESSAGE_TYPE_METHOD_RETURN:
                //printf ("message return");
                break;
            case DBUS_MESSAGE_TYPE_ERROR: {
                DBusError err;
                dbus_error_init(&err);
                if (err.message) {
                    printf ("message error: %s\n", err.message);
                    process->processStopped ();
                }
                break;
            }
            default:
                printf ("filter_func unknown\n");
        }
    //print_message (message);

    if (dbus_message_is_signal (message,
                DBUS_INTERFACE_LOCAL,
                "Disconnected"))
        ;//exit (0);

    /* Conceptually we want this to be
     * DBUS_HANDLER_RESULT_NOT_YET_HANDLED, but this raises
     * some problems.  See bug 1719.
     */
    return DBUS_HANDLER_RESULT_HANDLED;
}

static void cb_playMedia (DBusPendingCall * pending, void * user_data) {
    OssoMediaServer * process = static_cast <OssoMediaServer *> (user_data);
    DBusMessage *reply = dbus_pending_call_steal_reply (pending);
    if (reply) {
        int type = dbus_message_get_type (reply);
        if (type == DBUS_MESSAGE_TYPE_ERROR) {
            DBusError error;
            dbus_error_init (&error);
            dbus_set_error_from_message (&error, reply);
            errorLog () << error.name << ": " << error.message << endl;
            process->processNotify ()->errorMsg (error.message);
            if (process->state () > Process::Ready)
                process->setState (Process::Ready);
            dbus_error_free (&error);
        }
        dbus_message_unref (reply);
    }
    process->play_pending = 0L;
}

static void cb_posMedia (DBusPendingCall * pending, void * user_data) {
    OssoMediaServer * process = static_cast <OssoMediaServer *> (user_data);
    DBusMessage *reply = dbus_pending_call_steal_reply (pending);
    if (reply) {
        int type = dbus_message_get_type (reply);
        if (type == DBUS_MESSAGE_TYPE_METHOD_RETURN) {
            DBusMessageIter iter;
            dbus_message_iter_init (reply, &iter);
            type = dbus_message_iter_get_arg_type (&iter);
            if (type == DBUS_TYPE_INT32) {
                int pos;
                dbus_message_iter_get_basic (&iter, &pos);
                if (dbus_message_iter_next (&iter)) {
                    type = dbus_message_iter_get_arg_type (&iter);
                    if (type == DBUS_TYPE_INT32) {
                        int len;
                        dbus_message_iter_get_basic (&iter, &len);
                        process->setLength (len/100);
                    }
                }
                process->setPosition (pos/100);
            }
        }
        dbus_message_unref (reply);
    }
    process->pos_pending = 0L;
}

static void cb_seekMedia (DBusPendingCall * pending, void * user_data) {
    OssoMediaServer * process = static_cast <OssoMediaServer *> (user_data);
    DBusMessage *reply = dbus_pending_call_steal_reply (pending);
    if (reply)
        dbus_message_unref (reply);
    process->seek_pending = 0L;
}

static gboolean cb_getPosition (void * p) {
    OssoMediaServer * proc = static_cast <OssoMediaServer *> (p);
    if (!proc->seek_pending && !proc->pos_pending) {
        DBusConnection * con = (DBusConnection *)osso_get_dbus_connection (
                proc->ossoContext ());
        DBusMessage *msg = dbus_message_new_method_call (0L,
                "/com/nokia/osso_media_server",
                proc->currentMediaServer (),
                "get_position");
        dbus_message_set_destination (msg, "com.nokia.osso_media_server");
        if (dbus_connection_send_with_reply(con, msg, &proc->pos_pending, 5000))
            dbus_pending_call_set_notify (proc->pos_pending, cb_posMedia, p, 0);
        dbus_message_unref (msg);
    }
    return true; // repeat
}

const char * OssoMediaServer::myname = "osso-media-server";

KMPLAYER_NO_CDTOR_EXPORT
OssoMediaServer::OssoMediaServer (ProcessNotify * ctrl, osso_context_t * ctx, bool xv)
    : Process (ctrl, myname, xv),
      play_pending (0L), pos_pending (0L), seek_pending (0L),
      m_osso_context (ctx),
      current_service (""),
      dbus_filter (NULL),
      have_played (false),
      try_audio_only (false),
      first_time (true),
      retry_timer (0), pos_timer (0) {
}

KDE_NO_CDTOR_EXPORT OssoMediaServer::~OssoMediaServer () {
    if (have_played)
        setVideoWindow (200, 60, 560, 380); // something sane
    if (!first_time) {
        DBusConnection *conn = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
        if (conn)
            dbus_connection_remove_filter (conn, filter_func, this);
    }
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::stop () {
    if (play_pending) {
        dbus_pending_call_cancel (play_pending);
        play_pending = 0L;
    }
    if (pos_pending) {
        dbus_pending_call_cancel (pos_pending);
        pos_pending = 0L;
    }
    if (seek_pending) {
        dbus_pending_call_cancel (seek_pending);
        seek_pending = 0L;
    }
    try_audio_only = false;
    if (retry_timer) {
        g_source_remove (retry_timer);
        retry_timer = 0;
    }
    if (pos_timer) {
        g_source_remove (pos_timer);
        pos_timer = 0;
    }
    if (!playing ()) return true;
    DBusConnection * connection = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
    DBusMessage *message = dbus_message_new_method_call ("com.nokia.osso_media_server", "/com/nokia/osso_media_server", current_service, "stop");
    DBusMessage *reply;
    DBusError error;
    dbus_error_init (&error);
    reply = dbus_connection_send_with_reply_and_block (connection,
            message, 500, &error);
    if (!reply) {
        errorLog() << "stop error: " << error.message << endl;
        dbus_error_free (&error);
    } else
        dbus_message_unref (reply);
    //dbus_connection_send (connection, message, NULL);
    dbus_connection_flush (connection);
    dbus_message_unref (message);
    // m_state will be set back to Ready via dbus callback ..
    return true;
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::quit () {
    if (m_state > NotRunning) {
        if (playing ()) {
            stop ();
        }
    }
    setState (NotRunning);
    return !playing ();
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::ready (unsigned int viewer) {
    if (m_state < Ready)
        return Process::ready (viewer);
    return false;
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::retry () {
    if (!current_service || !strcmp (current_service, "com.nokia.osso_media_server.music")) {
        return false;
    } else if (retry_timer) {
        return true;
    } else if (m_mrl && !m_mrl->mrl ()->audio_only && m_mrl->mrl ()->media_object) {
        try_audio_only = true;
        retry_timer = g_timeout_add (100, cb_retryAudioOnly, this);
        return true;
    }
    return false;
}

KMPLAYER_NO_EXPORT void OssoMediaServer::updateVideoSize () {
    //debugLog () <<  "OssoMediaServer::updateVideoWindow " << (!m_mrl) << " " <<  (has_xv && !m_viewer) << " " << (!has_xv && m_rect == m_cur_rect) << endl;
    if (!m_mrl || (has_xv && !m_viewer)  /*hasVideo () */ ||
            (!has_xv && m_rect == m_cur_rect))
        return;
    DBusConnection * connection = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
    DBusMessage *message = dbus_message_new_method_call ("com.nokia.osso_media_server", "/com/nokia/osso_media_server", "com.nokia.osso_media_server.video", "set_video_window");
    dbus_message_set_auto_start (message, true);
    if (has_xv) {
        unsigned x = m_viewer;
        unsigned y = 0;
        unsigned w = 0;
        unsigned h = 0;
        debugLog () << "OssoMediaServer::updateVideoWindow " << m_viewer <<endl;
        dbus_message_append_args (message, DBUS_TYPE_UINT32, &x, DBUS_TYPE_UINT32, &y, DBUS_TYPE_UINT32, &w, DBUS_TYPE_UINT32, &h, DBUS_TYPE_INVALID);
    } else {
        int x = m_rect.x();
        int y = m_rect.y();
        int w = m_rect.width();
        int h = m_rect.height();
        debugLog () <<  "OssoMediaServer::updateVideoWindow " << x << ", " << y
            << " " << w << "x" << h << endl;
        dbus_message_append_args (message, DBUS_TYPE_INT32, &x, DBUS_TYPE_INT32, &y,
                DBUS_TYPE_INT32, &w, DBUS_TYPE_INT32, &h, DBUS_TYPE_INVALID);
    }
    dbus_connection_send (connection, message, NULL);
    dbus_connection_flush (connection);
    dbus_message_unref (message);
    m_cur_rect = m_rect;
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::play (NodePtr node) {
    Mrl * mrl = node->mrl ();
    //debugLog() << "OssoMediaServer::play " << m_viewer << endl;
    if (retry_timer) // are we called by the retry timer
        retry_timer = 0;
    else
        try_audio_only = false;
    if (!mrl)
        return false;
    String abspath = mrl->absolutePath ();
    if (abspath.isEmpty ()) {
        errorLog () << "empty path " << mrl->nodeName () << " " << mrl->src << endl;
        if (!mrl->src.isEmpty ()) // bug in absolutePath
            abspath = mrl->src;
        else
            return false;
    }
    Process::play (node);
    DBusConnection * connection = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
    if (m_state < Buffering) {
        have_played = true;
        updateVideoSize ();
        if (!dbus_filter) {
            DBusError err;
            dbus_error_init (&err);
            dbus_filter = media_server_filter;
            dbus_bus_add_match (connection, dbus_filter, &err);
            if (dbus_error_is_set (&err)) {
                errorLog () << "add match error: " << err.message << endl;
                dbus_error_free (&err);
            }
        }
        if (first_time) {
            dbus_connection_add_filter (connection, filter_func, this, NULL);
            first_time = false;
        }
    }
    no_video = mrl->audio_only || try_audio_only;
    current_service = no_video
        ? "com.nokia.osso_media_server.music"
        : "com.nokia.osso_media_server.video";
    DBusMessage *message = dbus_message_new_method_call (0L, "/com/nokia/osso_media_server", current_service, "play_media");
    dbus_message_set_destination (message, "com.nokia.osso_media_server");
    const char * path = (const char *) abspath;
    debugLog () << "OssoMediaServer::play " << current_service << " " << path << endl;
    dbus_message_append_args(message, DBUS_TYPE_STRING,&path,DBUS_TYPE_INVALID);
    //dbus_connection_send (connection, message, NULL);
    //dbus_connection_flush (connection);

    // first set to Buffering, as we might get an error signal before returning
    if (m_state < Buffering)
        setState (Buffering); // TODO: add timeout 

    if (dbus_connection_send_with_reply (connection,
                message, &play_pending, 5000))
        dbus_pending_call_set_notify (play_pending,cb_playMedia,(void*)this,0L);
    else
        setState (Ready);
    dbus_message_unref (message);
    return m_state == Buffering;
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::pause () {
    //debugLog() << "OssoMediaServer::pause" << endl;
    if (m_state > Ready) {
        if (!has_xv)
            updateVideoSize ();
        DBusConnection * connection = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
        DBusMessage *message = dbus_message_new_method_call (NULL, "/com/nokia/osso_media_server", "com.nokia.osso_media_server.video", "pause");
        dbus_message_set_destination (message, "com.nokia.osso_media_server");
        dbus_connection_send (connection, message, NULL);
        dbus_connection_flush (connection);
        dbus_message_unref (message);
    }
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::seek (int pos, bool absolute) {
    if (seek_pending || pos_pending || !playing ())
        return false;
    pos *= 100;
    DBusConnection * con = (DBusConnection *)osso_get_dbus_connection (
            ossoContext ());
    DBusMessage *msg = dbus_message_new_method_call (0L,
            "/com/nokia/osso_media_server", currentMediaServer (), "seek");
    dbus_message_set_destination (msg, "com.nokia.osso_media_server");
    int abs = absolute ? 1 : 0;
    dbus_message_append_args (msg, DBUS_TYPE_INT32, &abs,
            DBUS_TYPE_INT32, &pos, DBUS_TYPE_INVALID);
    if (dbus_connection_send_with_reply (con, msg, &seek_pending, 5000))
        dbus_pending_call_set_notify (seek_pending,cb_seekMedia,(void*)this,0L);
    dbus_message_unref (msg);
    return true;
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::volume (int incdec, bool absolute) {
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::saturation (int val, bool absolute) {
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::hue (int val, bool absolute) {
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::contrast (int val, bool /*absolute*/) {
}

KMPLAYER_NO_EXPORT bool OssoMediaServer::brightness (int val, bool /*absolute*/) {
}

KMPLAYER_NO_EXPORT
void OssoMediaServer::setVideoWindow (int x, int y, int w, int h) {
    //debugLog() << "OssoMediaServer::setVideoWindow " << w << "x" << h << endl;
    if (!(x < 0 || y < 0 || w <= 0 || h <= 1 /* hack !!*/ ||
                x > 800 || y > 480 || w > 800 || h > 480))
        m_rect = Rect (x, y, w, h);
}

KMPLAYER_NO_EXPORT void OssoMediaServer::processStopped () {
    if (dbus_filter) {
        DBusError err;
        dbus_error_init (&err);
        DBusConnection * conn = (DBusConnection *)osso_get_dbus_connection (m_osso_context);
        dbus_bus_remove_match (conn, dbus_filter, &err);
        if (dbus_error_is_set (&err)) {
            errorLog () << "remove match error: " << err.message << endl;
            dbus_error_free (&err);
        }
        dbus_filter = NULL;
    }
    if (retry_timer) {
        g_source_remove (retry_timer);
        retry_timer = 0;
        try_audio_only = true;
    }
    if (m_state > Ready)
        setState (Ready);
}

KMPLAYER_NO_EXPORT void OssoMediaServer::processBuffering (int perc) {
    if (perc < 100)
        setState (Buffering);
    else
        setState (Playing);
    m_notify->setLoading (perc);
}

KMPLAYER_NO_EXPORT void OssoMediaServer::processPlaying () {
    setState (Playing);
    if (!pos_timer)
        pos_timer = g_timeout_add (5000, cb_getPosition, this);
}

KMPLAYER_NO_EXPORT void OssoMediaServer::processPaused () {
    if (m_state == Paused) // ugh
        pause ();
    else
        setState (Paused);
}

KMPLAYER_NO_EXPORT void OssoMediaServer::setLength (int l) {
    if (l != m_length) {
        m_length = l;
        m_notify->setLength (l);
    }
}

/*void OssoMediaServer::setAudioLang (int id, const QString &) {
    SharedPtr <LangInfo> li = alanglist;
    for (; id > 0 && li; li = li->next)
        id--;
    if (li)
        aid = li->id;
    m_needs_restarted = true;
    sendCommand (QString ("quit"));
}

void OssoMediaServer::setSubtitle (int id, const QString &) {
    SharedPtr <LangInfo> li = slanglist;
    for (; id > 0 && li; li = li->next)
        id--;
    if (li)
        sid = li->id;
    m_needs_restarted = true;
    sendCommand (QString ("quit"));
}*/

#ifdef __ARMEL__

static NSNotify ns_notify;

extern "C" {
static void npEmbedded (void *ndata) {
    ((NpPlayer *) ndata)->pluginEmbedded ();
}

static void npSetDimension (void *ndata, int w, int h) {
    NpPlayer *player = (NpPlayer *) ndata;
    player->setWidth (w);
    player->setHeight (h);
    player->updateAspects ();
}

static void npGetUrl (void *inst, uint32_t stream,
        const char *url, const char *target, void *ndata) {
    ((NpPlayer *) ndata)->requestStream (inst, stream, url, target);
}

static char *nsEvaluate (const char *script, void *ndata) {
    String result = ((NpPlayer *) ndata)->evaluateScript (script);
    if (result.isEmpty ())
        return g_strdup ("");
    return g_strdup ((const char *)result);
}

static void npFinish (void *inst, uint32_t stream, void *ndata) {
    ((NpPlayer *) ndata)->destroyStream (inst, stream);
}

static gint streamCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

static gboolean deleteStreams (gpointer key, gpointer value, gpointer data) {
    (void) key;
    delete (NpStream *)value;
    return false; // continue
}

static gboolean copyParams (gpointer key, gpointer value, gpointer data) {
    char ***tuple = (char ***)data;
    tuple[0][0] = (char *)key;
    tuple[1][0] = (char *)value;
    tuple[0]++;
    tuple[1]++;
    return false; // continue
}

static gboolean openStreams (gpointer key, gpointer value, gpointer data) {
    int active = (int)(long)data;
    NpStream *ns = (NpStream *)value;
    (void) key;
    if (ns->finish_reason == NpStream::NoReason && !ns->job) {
        ++active;
        ns->open ();
    }
    return active > 4;
}

static gboolean findClosedStreams (gpointer key, gpointer value, gpointer data) {
    NpStream *ns = (NpStream *)value;
    (void) key;
    if (ns->finish_reason > NpStream::NoReason) {
        *(NpStream **)data = ns;
        return true;
    }
    return false;
}

} // extern "C"

KDE_NO_CDTOR_EXPORT NpPlayer::NpPlayer (ProcessNotify * ctrl)
 : Process (ctrl, "npp", false),
   streams (NULL),
   mime (NULL),
   npdata (NULL),
   argc (0),
   argn (NULL),
   argv (NULL),
   processing(false) {
    plugins = g_tree_new ((GCompareFunc)::strcmp);
    if (!ns_notify.embedded) {
        ns_notify.embedded = npEmbedded;
        ns_notify.setDimension = npSetDimension;
        ns_notify.getUrl = npGetUrl;
        ns_notify.evaluate = nsEvaluate;
        ns_notify.finishStream = npFinish;
    }
}

KDE_NO_CDTOR_EXPORT NpPlayer::~NpPlayer () {
    if (mime)
        g_free (mime);
}

KMPLAYER_NO_EXPORT bool NpPlayer::ready (unsigned int viewer) {
    debugLog () << "NpPlayer::ready" << endl;
    m_viewer = viewer;
    if (m_state < Ready) {
        // only flash for now
        mime = g_strdup ("application/x-shockwave-flash");
        const char *lib = "/usr/lib/browser/plugins/libflashplayer.so";
        NPPluginLib *nplib = (NPPluginLib *) g_tree_search (plugins,
                (GCompareFunc)::strcmp, mime);
        if (!nplib) {
            nplib = nppPluginInit (lib);
            if (!nplib) {
                warningLog() << "failed to initialize plugin lib" << endl;
                return false;
            }
            g_tree_insert (plugins, (char *) mime, nplib);
        }
        nppInstanceWindowParent (npdata, (void *)(long)viewer);
        setState (Ready);
        return true;
    }
    return false;
}

KMPLAYER_NO_EXPORT bool NpPlayer::play (NodePtr node) {
    debugLog () << "NpPlayer::play" << endl;
    if (playing ())
        return true;
    setState (Buffering);
    Mrl *mrl = node ? node->mrl () : 0L;
    if (!mrl) {
        errorLog() << "Not a Mrl " << (node ? node->nodeName() :"null") << endl;
        setState (Ready);
        return false;
    }
    String strurl = mrl->absolutePath ();
    if (strurl.isEmpty ()) {
        setState (Ready);
        return false;
    }
    streams = g_tree_new (streamCompare);
    Process::play (node);

    GTree *params = g_tree_new ((GCompareFunc)::strcmp);
    if (mrl->id == id_node_html_object) {
        for (Node *n = mrl->firstChild().ptr(); n; n = n->nextSibling().ptr()) {
            /*if (n->id == KMPlayer::id_node_param) {
                KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
                String name = e->getAttribute (KMPlayer::StringPool::attr_name);
                if (!g_tree_lookup (params, (const char *) name))
                    g_tree_insert (params,
                            strdup ((const char *) name),
                            strdup ((const char *) e->getAttribute (KMPlayer::StringPool::attr_value)));
            } else*/ if (n->id == KMPlayer::id_node_html_embed) {
                KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
                AttributePtr a = e->attributes ()->first ();
                for (; a; a = a->nextSibling ()) {
                    String nm = a->name().toString();
                    if (!g_tree_lookup (params, (const char *) nm))
                        g_tree_insert (params,
                                strdup ((const char *) nm),
                                strdup ((const char *) a->value ()));
                }
            }
        }
    } else {
        g_tree_insert (params, strdup ("SRC"), strdup ((const char *) strurl));
    }
    argc = g_tree_nnodes (params);
    argn = (char **) malloc (argc * sizeof (char *));
    argv = (char **) malloc (argc * sizeof (char *));
    char **tuple[2] = { argn, argv };
    g_tree_traverse (params, copyParams, G_IN_ORDER, tuple);
    g_tree_destroy (params);

    npdata = nppInstanceOpen (mime, argc, argn, argv, (void *)this, &ns_notify);
    if (!npdata) {
        g_tree_destroy (streams);
        streams = NULL;
        setState (Ready);
        warningLog() << "failed to create new instance" << endl;
        return false;
    }
    nppInstanceStart ((const char *)strurl, mime, npdata);
    return true;
}

KMPLAYER_NO_EXPORT void NpPlayer::pluginEmbedded () {
    setState (Playing);
}

KMPLAYER_NO_EXPORT void NpPlayer::requestStream (void *inst, uint32_t sid,
        const String & url, const String & target) {
    URL uri (m_url, url);
    debugLog () << "NpPlayer::request " << sid << " '" << uri << "'" << endl;
    if (!target.isEmpty ()) {
        debugLog () << "new page request " << target << endl;
        if (url.hasPrefix ("javascript:")) {
            String result = evaluateScript (url.mid (11));
            debugLog() << "result is " << result << endl;
            if (result == "undefined")
                uri = URL ();
            else
                uri = URL (m_url, result); // probably wrong ..
        }
        /*if (!uri.url ().isEmpty ())
            emit openUrl (uri, target);*/
        nppStreamFinished (sid, NpStream::BecauseDone, this);
    } else if (url.hasPrefix ("javascript:")) {
        String result = evaluateScript (url.mid (11));
        const char *data = (const char *)result;
        nppStreamData (sid, data, strlen (data), this);
        nppStreamFinished (sid, NpStream::BecauseDone, this);
    } else {
        NpStream * ns = new NpStream (this, sid, uri);
        g_tree_insert (streams, (gpointer) sid, ns);
        if (url != uri.url ())
            nppStreamRedirected (sid, (const char *)uri.url ());
        processStreams ();
    }
}

KMPLAYER_NO_EXPORT void NpPlayer::destroyStream (void *inst, uint32_t sid) {
    NpStream * ns = (NpStream *) g_tree_lookup (streams, (gpointer) sid);
    if (ns) {
        nppStreamFinished (sid, NpStream::BecauseStopped, this);
        g_tree_remove (streams, (gpointer) sid);
        delete (ns);
    }
}

KMPLAYER_NO_EXPORT bool NpPlayer::stop () {
    if (npdata) {
        g_tree_foreach (streams, deleteStreams, NULL);
        g_tree_destroy (streams);
        streams = NULL;
        nppInstanceClose (npdata);
        npdata = NULL;
        setState (Ready);
    }
    for (int i = 0; i < argc; i++) {
        free (argn[i]);
        free (argv[i]);
    }
    if (argn) {
        free (argn);
        free (argv);
        argn = argv = NULL;
    }
    argc = 0;
    return true;
}

KMPLAYER_NO_EXPORT bool NpPlayer::quit () {
    stop ();
    if (mime)
        g_free (mime);
    mime = NULL;
    nppInstanceWindowClose (npdata); //FIXME
    setState (NotRunning);
    return true;
}

KMPLAYER_NO_EXPORT
void NpPlayer::setVideoWindow (int x, int y, int w, int h) {
    //debugLog() << "NpPlayer::setVideoWindow " << w << "x" << h << endl;
    if (!(x < 0 || y < 0 || w <= 0 || h <= 1 /* hack !!*/ ||
                x > 800 || y > 480 || w > 800 || h > 480))
        nppInstanceWindowDimension (npdata, x, y, w, h);
}

KMPLAYER_NO_EXPORT void NpPlayer::processStreams () {
    if (!processing) {
        processing = true;
        streams_deleted = 0;
        int active = 0;
        g_tree_traverse (streams, openStreams, G_IN_ORDER, &active);
        while (streams_deleted > 0) {
            NpStream *ns = NULL;
            g_tree_traverse (streams, findClosedStreams, G_IN_ORDER, &ns);
            if (ns) {
                g_tree_remove (streams, (gpointer) ns->stream);
                delete ns;
            } else {
                errorLog() << "findClosedStreams didn't find a stream" << endl;
                break;
            }
            streams_deleted--;
        }
        processing = false;
    }
}

KMPLAYER_NO_EXPORT void NpPlayer::streamFinished (NpStream *ns) {
    nppStreamFinished (ns->stream, ns->finish_reason, this);
    if (!processing) {
        g_tree_remove (streams, (gpointer) ns->stream);
        delete ns;
        processStreams ();
    } else {
        streams_deleted++;
    }
}

KMPLAYER_NO_EXPORT String NpPlayer::evaluateScript (const String &script) {
    debugLog() << "evaluateScript " << script << endl;
    if (script == "window.location+\"__flashplugin_unique__\"" ||
            script == "top.location+\"__flashplugin_unique__\"")
        return "http://www.example.com/__flashplugin_unique__";
    return "undefined";
}

KDE_NO_CDTOR_EXPORT NpStream::NpStream (NpPlayer *p, uint32_t id, const URL & u)
 : player (p), stream (id), url (u), job (NULL), finish_reason (NoReason) {
}

KDE_NO_CDTOR_EXPORT NpStream::~NpStream () {
    close ();
}

KMPLAYER_NO_EXPORT void NpStream::open () {
    debugLog () << "NpStream::open " << url.url () << endl;
    if (url.url().hasPrefix ("javascript:")) {
        String result = player->evaluateScript (url.url().mid (11));
        const char *data = (const char *)result;
        nppStreamData (stream, data, strlen (data), NULL /*FIXME*/);
        finish_reason = BecauseDone;
        player->streamFinished (this);
    } else {
        bytes = 0;
        job = asyncGet (this, url.url ());
    }
}

KMPLAYER_NO_EXPORT void NpStream::close () {
    if (job)
        job->kill ();
    job = NULL;
}

KMPLAYER_NO_EXPORT void NpStream::jobData (Job *jb, ByteArray & data) {
    //debugLog () << "NpStream::jobData " << url.url () << endl;
    if (!bytes)
        nppStreamInfo (stream, jb->contentType (), jb->contentLength ());
    bytes += data.size ();
    nppStreamData (stream, data.data (), data.size (), NULL /*FIXME*/);
}

KMPLAYER_NO_EXPORT void NpStream::jobResult (Job *jb) {
    debugLog () << "NpStream::jobResult " << bytes << endl;
    finish_reason = jb->error () || !bytes ? BecauseError : BecauseDone;
    job = NULL;
    player->streamFinished (this);
}

KMPLAYER_NO_EXPORT void NpStream::redirected (Job *, const String & uri) {
    url = URL (url, uri);
    nppStreamRedirected (stream, (const char *)url.url ());
}

#endif //__ARMEL__
